% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/network.generalizability.R
\name{network.generalizability}
\alias{network.generalizability}
\title{Estimate the Generalizability of Network}
\usage{
network.generalizability(
  data,
  method = c("split", "cv", "loocv"),
  number,
  corr = c("auto", "cor_auto", "pearson", "spearman"),
  na.data = c("pairwise", "listwise"),
  model = c("BGGM", "glasso", "TMFG"),
  algorithm = c("leiden", "louvain", "walktrap"),
  uni.method = c("expand", "LE", "louvain"),
  seed = NULL,
  ...
)
}
\arguments{
\item{data}{Matrix or data frame.
Should consist only of variables to be used in the analysis.
Can be raw data or a correlation matrix}

\item{method}{Character (length = 1).
Generalizability method.
Available options:

\itemize{

\item \code{"split"} --- Performs train/test data split on the data using
\code{number} to adjust the size of the \strong{training} split

\item \code{"cv"} --- (default) Performs \emph{k}-folds cross-validation using
\code{number} to adjust the number of folds (e.g., 5 = 80/20 splits; 10 = 90/10 splits)

\item \code{"loocv"} --- Performs leave-one-out cross-validation. Leave-one-out
has a tendency to \strong{overestimate} the generalizability of the model and
is not recommended (\emph{k}-folds cross-validation should be preferred)

}}

\item{number}{Numeric (length = 1).
Parameter to adjust the \code{method} argument. Ranges 0-1
for \code{method = "split"} and 1-N for \code{method = "cv"}.
Defaults to \code{0.80} and \code{5}, respectively}

\item{corr}{Character (length = 1).
Method to compute correlations.
Defaults to \code{"auto"}.
Available options:

\itemize{

\item \code{"auto"} --- Automatically computes appropriate correlations for
the data using Pearson's for continuous, polychoric for ordinal,
tetrachoric for binary, and polyserial/biserial for ordinal/binary with
continuous. To change the number of categories that are considered
ordinal, use \code{ordinal.categories}
(see \code{\link[EGAnet]{polychoric.matrix}} for more details)

\item \code{"cor_auto"} --- Uses \code{\link[qgraph]{cor_auto}} to compute correlations.
Arguments can be passed along to the function

\item \code{"pearson"} --- Pearson's correlation is computed for all
variables regardless of categories

\item \code{"spearman"} --- Spearman's rank-order correlation is computed
for all variables regardless of categories

}

For other similarity measures, compute them first and input them
into \code{data} with the sample size (\code{n})}

\item{na.data}{Character (length = 1).
How should missing data be handled?
Defaults to \code{"pairwise"}.
Available options:

\itemize{

\item \code{"pairwise"} --- Computes correlation for all available cases between
two variables

\item \code{"listwise"} --- Computes correlation for all complete cases in the dataset

}}

\item{model}{Character (length = 1).
Defaults to \code{"glasso"}.
Available options:

\itemize{

\item \code{"BGGM"} --- Computes the Bayesian Gaussian Graphical Model.
Set argument \code{ordinal.categories} to determine
levels allowed for a variable to be considered ordinal.
See \code{?BGGM::estimate} for more details

\item \code{"glasso"} --- Computes the GLASSO with EBIC model selection.
See \code{\link[EGAnet]{EBICglasso.qgraph}} for more details

\item \code{"TMFG"} --- Computes the TMFG method.
See \code{\link[EGAnet]{TMFG}} for more details

}}

\item{algorithm}{Character or
\code{\link{igraph}} \code{cluster_*} function (length = 1).
Defaults to \code{"walktrap"}.
Three options are listed below but all are available
(see \code{\link[EGAnet]{community.detection}} for other options):

\itemize{

\item \code{"leiden"} --- See \code{\link[igraph]{cluster_leiden}} for more details

\item \code{"louvain"} --- By default, \code{"louvain"} will implement the Louvain algorithm using
the consensus clustering method (see \code{\link[EGAnet]{community.consensus}}
for more information). This function will implement
\code{consensus.method = "most_common"} and \code{consensus.iter = 1000}
unless specified otherwise

\item \code{"walktrap"} --- See \code{\link[igraph]{cluster_walktrap}} for more details

}}

\item{uni.method}{Character (length = 1).
What unidimensionality method should be used?
Defaults to \code{"louvain"}.
Available options:

\itemize{

\item \code{"expand"} --- Expands the correlation matrix with four variables correlated 0.50.
If number of dimension returns 2 or less in check, then the data
are unidimensional; otherwise, regular EGA with no matrix
expansion is used. This method was used in the Golino et al.'s (2020)
\emph{Psychological Methods} simulation

\item \code{"LE"} --- Applies the Leading Eigenvector algorithm
(\code{\link[igraph]{cluster_leading_eigen}})
on the empirical correlation matrix. If the number of dimensions is 1,
then the Leading Eigenvector solution is used; otherwise, regular EGA
is used. This method was used in the Christensen et al.'s (2023)
\emph{Behavior Research Methods} simulation

\item \code{"louvain"} --- Applies the Louvain algorithm (\code{\link[igraph]{cluster_louvain}})
on the empirical correlation matrix. If the number of dimensions is 1,
then the Louvain solution is used; otherwise, regular EGA is used.
This method was validated Christensen's (2022) \emph{PsyArXiv} simulation.
Consensus clustering can be used by specifying either
\code{"consensus.method"} or \code{"consensus.iter"}

}}

\item{seed}{Numeric (length = 1).
Defaults to \code{NULL} or random results.
Set for reproducible results.
See \href{https://r-ega.net/articles/reproducibility-prng.html}{Reproducibility and PRNG}
for more details on random number generation in \code{\link{EGAnet}}}

\item{...}{Additional arguments to be passed on to
\code{\link[EGAnet]{auto.correlate}},
\code{\link[EGAnet]{network.estimation}},
\code{\link[EGAnet]{community.detection}},
\code{\link[EGAnet]{community.consensus}}, and
\code{\link[EGAnet]{community.unidimensional}}}
}
\value{
Returns a list containing:

\item{node}{Node-wise metrics output from \code{\link[EGAnet]{network.predictability}}}

\item{community}{Community-wise metrics output from \code{\link[EGAnet]{tefi}}}
}
\description{
General function to compute a network's predictive power on new data,
following Haslbeck and Waldorp (2018) and Williams and Rodriguez (2022) and using
generalizability methods of data splitting, \emph{k}-folds cross-validation,
and leave-one-out cross-validation

Uses \code{\link[EGAnet]{network.predictability}} as the basis to then perform
generalizability methods over
}
\details{
This implementation of network predictability proceeds in several steps
with important assumptions:

1. Network was estimated using (partial) correlations (not regression like the
\code{mgm} package!)

2. Original data that was used to estimate the network in 1. is necessary to
apply the original scaling to the new data

3. (Linear) regression-like coefficients are obtained by reserve engineering the
inverse covariance matrix using the network's partial correlations (i.e.,
by setting the diagonal of the network to -1 and computing the inverse
of the opposite signed partial correlation matrix; see \code{EGAnet:::pcor2inv})

4. Predicted values are obtained by matrix multiplying the new data with these
coefficients

5. \strong{Dichotomous and polytomous} data are given categorical values based
on the \strong{original data's} thresholds and these thresholds are used to
convert the continuous predicted values into their corresponding categorical values

6. Evaluation metrics:

\itemize{

\item dichotomous --- Accuracy or the percent correctly predicted for the 0s and 1s

\item polytomous --- Accuracy based on the correctly predicting the ordinal category exactly
(i.e., 1 = 1, 2, = 2, etc.) and a weighted accuracy such that absolute distance of the
predicted value from the actual value (e.g., |prediction - actual| = 1) is used
as the power of 0.5. This weighted approach provides an overall distance in terms of
accuracy where each predicted value away from the actual value is given a harsher
penalty (absolute difference = accuracy value): 0 = 1.000, 1 = 0.500, 2 = 0.2500,
3 = 0.1250, 4 = 0.0625, etc.

\item continuous --- R-sqaured and root mean square error

}
}
\examples{
# Data splitting
network.generalizability(
  data = wmt2[,7:24], method = "split",
  number = 0.80 # 80/20 training/testing
)

# k-folds cross-validation
network.generalizability(
  data = wmt2[,7:24], method = "cv",
  number = 5 # 5-fold cross-validation
)

\dontrun{
# Leave-one-out cross-validation
network.generalizability(
  data = wmt2[,7:24], method = "loocv"
)}

}
\references{
\strong{Original Implementation of Node Predictability} \cr
Haslbeck, J. M., & Waldorp, L. J. (2018).
How well do network models predict observations? On the importance of predictability in network models.
\emph{Behavior Research Methods}, \emph{50}(2), 853–861.

\strong{Derivation of Regression Coefficients Used (Formula 3)} \cr
Williams, D. R., & Rodriguez, J. E. (2022).
Why overfitting is not (usually) a problem in partial correlation networks.
\emph{Psychological Methods}, \emph{27}(5), 822–840.
}
\author{
Hudson Golino <hfg9s at virginia.edu> and Alexander P. Christensen <alexpaulchristensen@gmail.com>
}
