% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/forecast_infections.R
\name{forecast_infections}
\alias{forecast_infections}
\title{Forecast Infections and the Time-Varying Reproduction Number}
\usage{
forecast_infections(
  infections,
  rts,
  gt_mean,
  gt_sd,
  gt_max = 30,
  ensemble_type = "mean",
  forecast_model,
  horizon = 14,
  samples = 1000
)
}
\arguments{
\item{infections}{A data frame of cases by date of infection containing the following variables: date, mean, sd}

\item{rts}{A data frame of Rt estimates by date of infection containing the following variables: date, mean, sd}

\item{gt_mean}{Numeric, the mean of the gamma distributed generation time.}

\item{gt_sd}{Numeric, the standard deviation of the gamma distributed generation time.}

\item{gt_max}{Numeric, the maximum allowed value of the gamma distributed generation time.}

\item{ensemble_type}{Character string indicating the type of ensemble to use. By default this is
an unweighted ensemble ("mean") with no other types currently supported.}

\item{forecast_model}{An uninitialised forecast model function to be passed to \code{EpiSoon::forecast_rt}. Used
for forecasting future Rt and case co An example of the required structure is: \code{function(ss, y){bsts::AddSemilocalLinearTrend(ss, y = y)}}.}

\item{horizon}{Numeric, defaults to 14. The horizon over which to forecast Rts and cases.}

\item{samples}{Numeric, the number of forecast samples to take.}
}
\value{
A list of \code{data.tables}. The first entry ("samples") contains raw forecast samples and the second entry ("summarised") contains
summarised forecasts.
}
\description{
This function provides optional tools for forecasting cases and Rt estimates using the timeseries methods
(via the \code{EpiSoon} package). It requires the \code{Episoon} package. Installation instructions for the EpiSoon package are
available \href{https://epiforecasts.io/EpiSoon/}{here}.
}
\examples{
\donttest{

if(requireNamespace("EpiSoon")){
   if(requireNamespace("forecastHybrid")){

## Example case data
reported_cases <- EpiNow2::example_confirmed[1:40]
 
generation_time <- list(mean = EpiNow2::covid_generation_times[1, ]$mean,
                        mean_sd = EpiNow2::covid_generation_times[1, ]$mean_sd,
                        sd = EpiNow2::covid_generation_times[1, ]$sd,
                        sd_sd = EpiNow2::covid_generation_times[1, ]$sd_sd,
                        max = 30)
                          
incubation_period <- list(mean = EpiNow2::covid_incubation_period[1, ]$mean,
                          mean_sd = EpiNow2::covid_incubation_period[1, ]$mean_sd,
                          sd = EpiNow2::covid_incubation_period[1, ]$sd,
                          sd_sd = EpiNow2::covid_incubation_period[1, ]$sd_sd,
                          max = 30)
                   
reporting_delay <- list(mean = log(5),
                        mean_sd = log(2),
                        sd = log(2),
                        sd_sd = log(1.5),
                        max = 30)
                        
## Estimate Rt and infections from data
out <- estimate_infections(reported_cases, family = "negbin",
                           generation_time = generation_time,
                           delays = list(incubation_period, reporting_delay),
                           rt_prior = list(mean = 1, sd = 1),
                           samples = 1000, warmup = 500,
                           cores = ifelse(interactive(), 4, 1), chains = 4,
                           estimate_rt = TRUE,
                           verbose = FALSE, return_fit = TRUE)

## Forecast Rt and infections from estimates
forecast <- forecast_infections(
    infections = out$summarised[variable == "infections"],
    rts = out$summarised[variable == "R"],
     gt_mean = out$summarised[variable == "gt_mean"]$mean,
     gt_sd = out$summarised[variable == "gt_sd"]$mean,
     gt_max = 30,
     forecast_model = function(y, ...){
       EpiSoon::forecastHybrid_model(y = y[max(1, length(y) - 21):length(y)],
       model_params = list(models = "aefz", weights = "equal"),
       forecast_params = list(PI.combination = "mean"), ...)},
     horizon = 14, 
     samples = 1000)
                                
forecast$summarised
  }
 }
}                              
}
