% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mrunLWFB90.R
\name{run_multi_LWFB90}
\alias{run_multi_LWFB90}
\title{Make a multirun simulation using a set of variable input parameters.}
\usage{
run_multi_LWFB90(
  paramvar,
  param_b90,
  paramvar_nms = names(paramvar),
  cores = 2,
  show_progress = TRUE,
  ...
)
}
\arguments{
\item{paramvar}{Data.frame of variable input parameters. For each row, a
simulation is performed, with the elements in \code{param_b90} being
replaced by the respective column of \code{paramvar}. All parameter names
(column names) in \code{paramvar} must be found in \code{param_b90}. See
section \code{Parameter updating}.}

\item{param_b90}{Named list of parameters, in which the parameters defined in
\code{paramvar} will be replaced.}

\item{paramvar_nms}{Names of the parameters in \code{paramvar} to be replaced
in \code{param_b90}.}

\item{cores}{Number of CPUs to use for parallel processing. Default is 2.}

\item{show_progress}{Logical: Show progress bar? Default is TRUE. See also
section \code{Progress bar} below.}

\item{...}{Additional arguments passed to \code{\link{run_LWFB90}}: provide
at least the arguments that have no defaults such as \code{options_b90} and
\code{climate}).}
}
\value{
A named list with the results of the single runs as returned by
  \code{\link{run_LWFB90}}. Simulation or processing errors are passed on.
}
\description{
Wrapper function for \code{\link{run_LWFB90}} to make multiple simulations
parallel, with varying input parameters.
}
\section{Parameter updating}{

The transfer of values from a row in \code{paramvar} to \code{param_b90}
before each single run simulation is done by matching names from
\code{paramvar} and \code{param_b90}. In order to address data.frame or
vector elements in \code{param_b90} by a column name in \code{paramvar}, the
respective column name has to be set up from its name and index in
\code{param_b90}. To replace, e.g., the 2nd value of \code{ths} in the
\code{soil_materials} data.frame, the respective column name in
\code{paramvar} has to be called 'soil_materials.ths2'. In order to replace
the 3rd value of \code{maxlai} vector in \code{param_b90}, the column has to
be named 'maxlai3'.
}

\section{Data management}{

The returned list of single run results can become very large, if many
simulations are performed and the selected output contains daily resolution
data sets, especially daily layer-wise soil moisture data. To not overload
memory, it is advised to reduce the returned simulation results to a minimum,
by carefully selecting the output, and make use of the option to pass a list
of functions to \code{\link{run_LWFB90}} via argument \code{output_fun}.
These functions perform directly on the output of a single run simulation,
and can be used for aggregating model output on-the-fly, or for writing
results to a file or database. The regular output of
\code{\link{run_LWFB90}} can be suppressed by setting \code{rtrn.output =
FALSE}, for exclusively returning the output of such functions.
}

\section{Progress bar}{

This function provides a progress bar via the package \CRANpkg{progressr} if
\code{show_progress=TRUE}. The parallel computation is then wrapped with
\code{progressr::with_progress()} to enable progress reporting from
distributed calculations. The appearance of the progress bar (including
audible notification) can be customized by the user for the entire session
using \code{progressr::handlers()} (see \code{vignette('progressr-intro')}).
}

\examples{
\donttest{
data("slb1_meteo")
data("slb1_soil")

# Set up lists containing model control options and model parameters:
parms <- set_paramLWFB90()
# choose the 'Coupmodel' shape option for the annual lai dynamic,
# with fixed budburst and leaf fall dates:
opts <- set_optionsLWFB90(startdate = as.Date("2003-06-01"),
                                 enddate = as.Date("2003-06-30"),
                                 lai_method = 'Coupmodel',
                                 budburst_method = 'fixed',
                                 leaffall_method = 'fixed')

# Derive soil hydraulic properties from soil physical properties using pedotransfer functions
soil <- cbind(slb1_soil, hydpar_wessolek_tab(slb1_soil$texture))

#set up data.frame with variable parameters
n <- 10
set.seed(2021)
vary_parms <- data.frame(shp_optdoy = runif(n,180,240),
                         shp_budburst = runif(n, 0.1,1),
                         winlaifrac = runif(n, 0,0.5),
                         budburstdoy = runif(n,100,150),
                         soil_materials.ths3 = runif(n, 0.3,0.5), # ths of material 3
                         maxlai = runif(n,2,7))

# add the soil as soil_nodes and soil materials to param_b90, so ths3 can be looked up
parms[c("soil_nodes", "soil_materials")] <- soil_to_param(soil)

# Make a Multirun-Simulation
b90.multi <- run_multi_LWFB90(paramvar = vary_parms,
                        param_b90 = parms,
                        options_b90 = opts,
                        climate = slb1_meteo)
names(b90.multi)

# extract results
evapday <- data.table::rbindlist(
  lapply(b90.multi, FUN = function(x) { x$output[,c("yr", "doy", "evap")] }),
  idcol = "srun")

evapday$dates <- as.Date(paste(evapday$yr, evapday$doy),"\%Y \%j")

srun_nms <- unique(evapday$srun)

with(evapday[evapday$srun == srun_nms[1], ],
     plot(dates, cumsum(evap), type = "n",
          ylim = c(0,100))
)
for (i in 1:length(b90.multi)){
  with(evapday[evapday$srun == srun_nms[i], ],
       lines(dates, cumsum(evap)))
}

}
}
