% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_GrowthCurve.R
\name{plot_GrowthCurve}
\alias{plot_GrowthCurve}
\title{Fit and plot a dose-response curve for luminescence data (Lx/Tx against dose)}
\usage{
plot_GrowthCurve(
  sample,
  mode = "interpolation",
  fit.method = "EXP",
  fit.force_through_origin = FALSE,
  fit.weights = TRUE,
  fit.includingRepeatedRegPoints = TRUE,
  fit.NumberRegPoints = NULL,
  fit.NumberRegPointsReal = NULL,
  fit.bounds = TRUE,
  NumberIterations.MC = 100,
  output.plot = TRUE,
  output.plotExtended = TRUE,
  output.plotExtended.single = FALSE,
  cex.global = 1,
  txtProgressBar = TRUE,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{sample}{\link{data.frame} (\strong{required}):
data frame with three columns for \code{x = Dose},\code{y = LxTx},\code{z = LxTx.Error}, \code{y1 = TnTx}.
The column for the test dose response is optional, but requires \code{'TnTx'} as
column name if used. For exponential fits at least three dose points
(including the natural) should be provided.}

\item{mode}{\link{character} (\emph{with default}):
selects calculation mode of the function.
\itemize{
\item \code{"interpolation"} (default) calculates the De by interpolation,
\item \code{"extrapolation"} calculates the equivalent dose by extrapolation (useful for MAAD measurements) and
\item \code{"alternate"} calculates no equivalent dose and just fits the data points.
}

Please note that for option \code{"regenerative"} the first point is considered
as natural dose}

\item{fit.method}{\link{character} (\emph{with default}):
function used for fitting. Possible options are:
\itemize{
\item \code{LIN},
\item \code{QDR},
\item \code{EXP},
\item \verb{EXP OR LIN},
\item \code{EXP+LIN},
\item \code{EXP+EXP},
\item \code{GOK},
\item \code{LambertW}
}

See details.}

\item{fit.force_through_origin}{\link{logical} (\emph{with default})
allow to force the fitted function through the origin.
For \code{method = "EXP+EXP"} the function will be fixed through
the origin in either case, so this option will have no effect.}

\item{fit.weights}{\link{logical} (\emph{with default}):
option whether the fitting is done with or without weights. See details.}

\item{fit.includingRepeatedRegPoints}{\link{logical} (\emph{with default}):
includes repeated points for fitting (\code{TRUE}/\code{FALSE}).}

\item{fit.NumberRegPoints}{\link{integer} (\emph{optional}):
set number of regeneration points manually. By default the number of all (!)
regeneration points is used automatically.}

\item{fit.NumberRegPointsReal}{\link{integer} (\emph{optional}):
if the number of regeneration points is provided manually, the value of the
real, regeneration points = all points (repeated points) including reg 0,
has to be inserted.}

\item{fit.bounds}{\link{logical} (\emph{with default}):
set lower fit bounds for all fitting parameters to 0. Limited for the use
with the fit methods \code{EXP}, \code{EXP+LIN}, \verb{EXP OR LIN}, \code{GOK}, \code{LambertW}
Argument to be inserted for experimental application only!}

\item{NumberIterations.MC}{\link{integer} (\emph{with default}):
number of Monte Carlo simulations for error estimation. See details.}

\item{output.plot}{\link{logical} (\emph{with default}):
plot output (\code{TRUE/FALSE}).}

\item{output.plotExtended}{\link{logical} (\emph{with default}):
If' \code{TRUE}, 3 plots on one plot area are provided:
\enumerate{
\item growth curve,
\item histogram from Monte Carlo error simulation and
\item a test dose response plot.
}

If \code{FALSE}, just the growth curve will be plotted.
\strong{Requires:} \code{output.plot = TRUE}.}

\item{output.plotExtended.single}{\link{logical} (\emph{with default}):
single plot output (\code{TRUE/FALSE}) to allow for plotting the results in
single plot windows. Requires \code{output.plot = TRUE} and
\code{output.plotExtended = TRUE}.}

\item{cex.global}{\link{numeric} (\emph{with default}):
global scaling factor.}

\item{txtProgressBar}{\link{logical} (\emph{with default}):
enables or disables \code{txtProgressBar}. If \code{verbose = FALSE} also no
\code{txtProgressBar} is shown.}

\item{verbose}{\link{logical} (\emph{with default}):
enables or disables terminal feedback.}

\item{...}{Further arguments and graphical parameters to be passed. Note:
Standard arguments will only be passed to the growth curve plot. Supported:
\code{xlim}, \code{ylim}, \code{main}, \code{xlab}, \code{ylab}}
}
\value{
Along with a plot (so far wanted) an \code{RLum.Results} object is returned containing,
the slot \code{data} contains the following elements:

\tabular{lll}{
\strong{DATA.OBJECT} \tab \strong{TYPE} \tab \strong{DESCRIPTION} \cr
\code{..$De} : \tab  \code{data.frame} \tab Table with De values \cr
\code{..$De.MC} : \tab \code{numeric} \tab Table with De values from MC runs \cr
\code{..$Fit} : \tab \link{nls} or \link{lm} \tab object from the fitting for \code{EXP}, \code{EXP+LIN} and \code{EXP+EXP}.
In case of a resulting  linear fit when using \code{LIN}, \code{QDR} or \verb{EXP OR LIN} \cr
\code{..$Formula} : \tab \link{expression} \tab Fitting formula as R expression \cr
\code{..$call} : \tab \code{call} \tab The original function call\cr
}
}
\description{
A dose-response curve is produced for luminescence measurements using a
regenerative or additive protocol. The function supports interpolation and
extrapolation to calculate the equivalent dose.
}
\details{
\strong{Fitting methods}

For all options (except for the \code{LIN}, \code{QDR} and the \verb{EXP OR LIN}),
the \link[minpack.lm:nlsLM]{minpack.lm::nlsLM} function with the \code{LM} (Levenberg-Marquardt algorithm)
algorithm is used. Note: For historical reasons for the Monte Carlo
simulations partly the function \link{nls} using the \code{port} algorithm.

The solution is found by transforming the function or using \link{uniroot}.

\code{LIN}: fits a linear function to the data using
\link{lm}: \deqn{y = mx + n}

\code{QDR}: fits a linear function to the data using
\link{lm}: \deqn{y = a + bx + cx^2}

\code{EXP}: tries to fit a function of the form
\deqn{y = a(1 - exp(-\frac{(x+c)}{b}))}
Parameters b and c are approximated by a linear fit using \link{lm}. Note: b = D0

\verb{EXP OR LIN}: works for some cases where an \code{EXP} fit fails.
If the \code{EXP} fit fails, a \code{LIN} fit is done instead.

\code{EXP+LIN}: tries to fit an exponential plus linear function of the
form:
\deqn{y = a(1-exp(-\frac{x+c}{b}) + (gx))}
The \eqn{D_e} is calculated by iteration.

\strong{Note:} In the context of luminescence dating, this
function has no physical meaning. Therefore, no D0 value is returned.

\code{EXP+EXP}: tries to fit a double exponential function of the form
\deqn{y = (a_1 (1-exp(-\frac{x}{b_1}))) + (a_2 (1 - exp(-\frac{x}{b_2})))}
This fitting procedure is not robust against wrong start parameters and
should be further improved.

\code{GOK}: tries to fit the general-order kinetics function after
Guralnik et al. (2015) of the form of

\deqn{y = a (d - (1 + (\frac{1}{b}) x c)^{(-1/c)})}

where \strong{c > 0} is a kinetic order modifier
(not to be confused with \strong{c} in \code{EXP} or \code{EXP+LIN}!).

\code{LambertW}: tries to fit a dose-response curve based on the Lambert W function
according to Pagonis et al. (2020). The function has the form

\deqn{y ~ (1 + (W((R - 1) * exp(R - 1 - ((x + D_{int}) / D_{c}))) / (1 - R))) * N}

with \eqn{W} the Lambert W function, calculated using the package \link[lamW:lamW]{lamW::lambertW0},
\eqn{R} the dimensionless retrapping ratio, \eqn{N} the total concentration
of trappings states in cm^-3 and \eqn{D_{c} = N/R} a constant. \eqn{D_{int}} is
the offset on the x-axis. Please not that finding the root in \code{mode = "extrapolation"}
is a non-easy task due to the shape of the function and the results might be
unexpected.

\strong{Fit weighting}

If the option \code{fit.weights =  TRUE} is chosen, weights are calculated using
provided signal errors (Lx/Tx error):
\deqn{fit.weights = \frac{\frac{1}{error}}{\Sigma{\frac{1}{error}}}}

\strong{Error estimation using Monte Carlo simulation}

Error estimation is done using a parametric bootstrapping approach. A set of
\code{Lx/Tx} values is constructed by randomly drawing curve data sampled from normal
distributions. The normal distribution is defined by the input values (\code{mean = value}, \code{sd = value.error}). Then, a dose-response curve fit is attempted for each
dataset resulting in a new distribution of single \code{De} values. The standard
deviation of this distribution is becomes then the error of the \code{De}. With increasing
iterations, the error value becomes more stable. However, naturally the error
will not decrease with more MC runs.

Alternatively, the function returns highest probability density interval
estimates as output, users may find more useful under certain circumstances.

\strong{Note:} It may take some calculation time with increasing MC runs,
especially for the composed functions (\code{EXP+LIN} and \code{EXP+EXP}).\cr
Each error estimation is done with the function of the chosen fitting method.

\strong{Subtitle information}

To avoid plotting the subtitle information, provide an empty user \code{mtext}
\code{mtext = ""}. To plot any other subtitle text, use \code{mtext}.
}
\section{Function version}{
 1.11.13
}

\examples{

##(1) plot growth curve for a dummy data.set and show De value
data(ExampleData.LxTxData, envir = environment())
temp <- plot_GrowthCurve(LxTxData)
get_RLum(temp)

##(1b) horizontal plot arrangement
layout(mat = matrix(c(1,1,2,3), ncol = 2))
plot_GrowthCurve(LxTxData, output.plotExtended.single = TRUE)

##(1c) to access the fitting value try
get_RLum(temp, data.object = "Fit")

##(2) plot the growth curve only - uncomment to use
##pdf(file = "~/Desktop/Growth_Curve_Dummy.pdf", paper = "special")
plot_GrowthCurve(LxTxData)
##dev.off()

##(3) plot growth curve with pdf output - uncomment to use, single output
##pdf(file = "~/Desktop/Growth_Curve_Dummy.pdf", paper = "special")
plot_GrowthCurve(LxTxData, output.plotExtended.single = TRUE)
##dev.off()

##(4) plot resulting function for given interval x
x <- seq(1,10000, by = 100)
plot(
 x = x,
 y = eval(temp$Formula),
 type = "l"
)

##(5) plot using the 'extrapolation' mode
LxTxData[1,2:3] <- c(0.5, 0.001)
print(plot_GrowthCurve(LxTxData,mode = "extrapolation"))

##(6) plot using the 'alternate' mode
LxTxData[1,2:3] <- c(0.5, 0.001)
print(plot_GrowthCurve(LxTxData,mode = "alternate"))

##(7) import and fit test data set by Berger & Huntley 1989
QNL84_2_unbleached <-
read.table(system.file("extdata/QNL84_2_unbleached.txt", package = "Luminescence"))

results <- plot_GrowthCurve(
 QNL84_2_unbleached,
 mode = "extrapolation",
 plot = FALSE,
 verbose = FALSE)

#calculate confidence interval for the parameters
#as alternative error estimation
confint(results$Fit, level = 0.68)


\dontrun{
QNL84_2_bleached <-
read.table(system.file("extdata/QNL84_2_bleached.txt", package = "Luminescence"))
STRB87_1_unbleached <-
read.table(system.file("extdata/STRB87_1_unbleached.txt", package = "Luminescence"))
STRB87_1_bleached <-
read.table(system.file("extdata/STRB87_1_bleached.txt", package = "Luminescence"))

print(
 plot_GrowthCurve(
 QNL84_2_bleached,
 mode = "alternate",
 plot = FALSE,
 verbose = FALSE)$Fit)

print(
 plot_GrowthCurve(
 STRB87_1_unbleached,
 mode = "alternate",
 plot = FALSE,
 verbose = FALSE)$Fit)

print(
 plot_GrowthCurve(
 STRB87_1_bleached,
 mode = "alternate",
 plot = FALSE,
 verbose = FALSE)$Fit)
 }

} 

\section{How to cite}{
Kreutzer, S., Dietze, M., 2024. plot_GrowthCurve(): Fit and plot a dose-response curve for luminescence data (Lx/Tx against dose). Function version 1.11.13. In: Kreutzer, S., Burow, C., Dietze, M., Fuchs, M.C., Schmidt, C., Fischer, M., Friedrich, J., Mercier, N., Philippe, A., Riedesel, S., Autzen, M., Mittelstrass, D., Gray, H.J., Galharret, J., Colombo, M., 2024. Luminescence: Comprehensive Luminescence Dating Data Analysis. R package version 0.9.25. https://r-lum.github.io/Luminescence/
}

\references{
Berger, G.W., Huntley, D.J., 1989. Test data for exponential fits. Ancient TL 7, 43-46.

Guralnik, B., Li, B., Jain, M., Chen, R., Paris, R.B., Murray, A.S., Li, S.-H., Pagonis, P.,
Herman, F., 2015. Radiation-induced growth and isothermal decay of infrared-stimulated luminescence
from feldspar. Radiation Measurements 81, 224-231.

Pagonis, V., Kitis, G., Chen, R., 2020. A new analytical equation for the dose response of dosimetric materials,
based on the Lambert W function. Journal of Luminescence 225, 117333. \doi{10.1016/j.jlumin.2020.117333}
}
\seealso{
\link{nls}, \linkS4class{RLum.Results}, \link{get_RLum}, \link[minpack.lm:nlsLM]{minpack.lm::nlsLM},
\link{lm}, \link{uniroot}, \link[lamW:lamW]{lamW::lambertW0}
}
\author{
Sebastian Kreutzer, Institute of Geography, Heidelberg University (Germany)\cr
Michael Dietze, GFZ Potsdam (Germany)
, RLum Developer Team}
