% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/createLocationID.R
\name{createLocationID}
\alias{createLocationID}
\title{Create one or more unique locationIDs}
\usage{
createLocationID(
  longitude = NULL,
  latitude = NULL,
  algorithm = c("geohash", "digest"),
  invalidID = as.character(NA)
)
}
\arguments{
\item{longitude}{Vector of longitudes in decimal degrees E.}

\item{latitude}{Vector of latitudes in decimal degrees N.}

\item{algorithm}{Algorithm to use -- either \code{"geohash"} or \code{"digest"}.}

\item{invalidID}{Identifier to use for invalid locations. This can be a
character string or \code{NA}.}
}
\value{
Vector of character locationIDs.
}
\description{
A unique locationID is created for each incoming
\code{longitude} and \code{latitude}.

When the \code{"geohash"} algorithm is specified,
the following code is used:

\preformatted{
  locationID <-
    geohashTools::gh_encode(latitude, longitude, precision = 10)
}

When the \code{"digest"} algorithm is specified,
the following code is used to generate each locationID.

\preformatted{
# Retain accuracy up to ~.1m
locationString <- paste0(
  sprintf("\%.7f", longitude),
  "_",
  sprintf("\%.7f", latitude)
)
# Avoid collisions until billions of records
locationID <- digest::digest(locationString, algo = "xxhash64")
}

See the references for details on either algorithm.

The locationID for any invalid location can be specified by the user with
the \code{invalidID} argument.
}
\note{
The \code{"geohash"} algorithm is preferred but the \code{"digest"}
algorithm is retained because several existing databases
use the \code{"digest"} algorithm as a unique identifier.
}
\examples{
library(MazamaCoreUtils)

longitude <- c(-122.5, 0, NA, -122.5, -122.5)
latitude <- c( 47.5, 0, 47.5,   NA, 47.5)

createLocationID(longitude, latitude)
createLocationID(longitude, latitude, invalidID = "bad")
createLocationID(longitude, latitude, algorithm = "digest")

}
\references{
\url{https://en.wikipedia.org/wiki/Decimal_degrees}

\url{https://www.johndcook.com/blog/2017/01/10/probability-of-secure-hash-collisions/}

\url{https://michaelchirico.github.io/geohashTools/index.html}
}
