% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/option_valuation.R
\name{American_option_value}
\alias{American_option_value}
\title{N-factor model American put option pricing}
\usage{
American_option_value(
  x_0,
  parameters,
  N_simulations,
  option_maturity,
  dt,
  K,
  r,
  orthogonal = "Power",
  degree = 2,
  verbose = FALSE,
  debugging = FALSE
)
}
\arguments{
\item{x_0}{Initial values of the state vector.}

\item{parameters}{Named vector of parameter values of a specified N-factor model. Function \code{NFCP_parameters} is recommended.}

\item{N_simulations}{total number of simulated price paths}

\item{option_maturity}{Time to expiration of the option (in years)}

\item{dt}{discrete time step of simulation}

\item{K}{Strike price of the American put option}

\item{r}{Risk-free interest rate.}

\item{orthogonal}{The orthogonal polynomial used to approximate the continuation value of the option in the LSM simulation method.
Orthogonal polynomial arguments available are: "Power", "Laguerre", "Jacobi", "Legendre", "Chebyshev", "Hermite". See \code{help(LSM.AmericanOption)}}

\item{degree}{The degree of polynomials used in the least squares fit. See \code{help(LSM.AmericanOption)}}

\item{verbose}{\code{logical} Should additional information be output?}

\item{debugging}{\code{logical} Should additional simulation information be output?}
}
\value{
The 'American_option_value' function by default returns a \code{numeric} object corresponding to the calculated value of the American put option.

When \code{verbose = T}, 6 objects are returned within a \code{list} class object. The objects returned are:

\tabular{ll}{

\code{Value} \tab The calculated option value. \cr
\code{Standard Error} \tab The standard error of the calculated option value. \cr
\code{Expected Timing} \tab The expected time of early exercise.. \cr
\code{Expected Timing SE} \tab The standard error of the expected time of early exercise. \cr
\code{Exercise Probability} \tab The probability of early exercise of the option being exercised. \cr
\code{Cumulative Exercise Probability} \tab \code{vector}. The cumulative probability of option exercise at each discrete observation point \cr
}

When \code{debugging = T}, an additional 2 objects are returned within the \code{list} class object. These are the results output by both the 'Spot.Price.Simulate' and
'LSM.AmericanOption' of the 'LSMRealOptions' package respectively. The objects returned are:

\tabular{ll}{
\code{State_Variables} \tab A matrix of simulated state variables for each factor is returned when \code{verbose = T}. The number of factors returned corresponds to the number of factors in the specified N-factor model. \cr
\code{Prices} \tab A matrix of simulated price paths. Each column represents one simulated price path and each row represents one simulated observation. \cr
}
}
\description{
Value American put options under the parameters of an N-factor model through the Least-Squares Monte Carlo (LSM) Simulation Method.
This function is a wrapper to the 'LSM_American_option' function of the 'LSMRealOptions' package.
}
\details{
The 'American_option_value' function is a wrapper to the 'spot_price_simulate' and 'LSM_American_option' of the 'LSMRealOptions' package that
returns the value of American put options under a given N-factor model.

The least-squares Monte Carlo (LSM) simulation method is an option valuation method first presented by Longstaff and Schwartz (2001) that
approximates the value of American options.

Methods to solve for the value of options with early exercise opportunities include partial differential equations, lattice-based methods
and Monte-Carlo simulation. LSM simulation is the optimal solution method to value American options under an N-factor model due
to the multiple factors that can make up the spot price process and influence the option value. Whilst lattice and partial differential equation
approaches suffer from the 'curse of dimensionality', LSM simulation may be readily applied under multi-factor settings.

Longstaff and Schwartz (2001) state that as the conditional expectation of the continuation value belongs to a Hilbert space,
it can be represented by a combination of orthogonal basis functions. Increasing the number of stochastic state variables
therefore increases the number of required basis functions exponentially.
}
\examples{

# Example 1 - An American put option on a stock following 'GBM'
# growing at the risk-free rate:
American_option_value(x_0 = log(36),
                     parameters = c(mu_rn = (0.06 - (1/2) * 0.2^2), sigma_1 = 0.2),
                     N_simulations = 1e2,
                     option_maturity = 1,
                     dt = 1/50,
                     K = 40,
                     r = 0.05,
                     verbose = FALSE,
                     orthogonal = "Laguerre",
                     degree = 3)

# Example 2 - An American put option under a two-factor crude oil model:

## Step 1 - Obtain current (i.e. most recent) state vector by filtering the
## two-factor oil model:
Schwartz_Smith_oil <- NFCP_Kalman_filter(parameter_values = SS_oil$two_factor,
                                        parameter_names = names(SS_oil$two_factor),
                                        log_futures = log(SS_oil$stitched_futures),
                                        dt = SS.Oil$dt,
                                        futures_TTM = SS_oil$stitched_TTM,
                                        verbose = TRUE)

##Step 2 - Calculate 'put' option price:
American_option_value(x_0 = Schwartz_Smith_oil$x_t,
                     parameters = SS_oil$two_factor,
                     N_simulations = 1e2,
                     option_maturity = 1,
                     dt = 1/12,
                     K = 20,
                     r = 0.05,
                     verbose = FALSE,
                     orthogonal = "Power",
                     degree = 2)
}
\references{
Longstaff, F.A., and E.S. Schwartz. 2001. Valuing American Options by Simulation: A Simple Least-Squares Approach. The Review of Financial Studies. 14:113-147.

Schwartz, E. S., and J. E. Smith, (2000). Short-Term Variations and Long-Term Dynamics in Commodity Prices. \emph{Manage. Sci.}, 46, 893-911.

Cortazar, G., and L. Naranjo, (2006). An N-factor Gaussian model of oil futures prices. \emph{Journal of Futures Markets: Futures, Options, and Other Derivative Products}, 26(3), 243-268.

Aspinall, T., A. Gepp, G. Harris, S. Kelly, C. Southam, and B. Vanstone, (2021). LSMRealOptions: Value
American and Real Options Through LSM Simulation. R package version 0.1.0.
}
