#' NNS Normalization
#'
#' Normalizes a matrix of variables based on nonlinear scaling normalization method.
#' @param A Matrix of variables.
#' @param chart.type  Defaults to NULL.  'l' for line, 'b' for boxplot
#' @param Linear Performs a linear scaling normalization, resulting in eqaul means for all variables.  Defaults to FALSE.
#' @return Returns a matrix of normalized values.
#' @author Fred Viole, OVVO Financial Systems
#' @references Viole, F. and Nawrocki, D. (2013) "Nonlinear Nonparametric Statistics: Using Partial Moments"
#' \url{http://amzn.com/1490523995}
#' @examples
#' set.seed(123)
#' x<-rnorm(100); y<-rnorm(100)
#' A<-cbind(x,y)
#' NNS.norm(A)
#' @export

NNS.norm <- function(A,chart.type=NULL,Linear=F) {
  m  <- colMeans(A)
  RG <- m %o% (1/m)
  if(Linear==FALSE){
  scales <- colMeans(RG * as.matrix(abs(NNS.cor(A))))}else{scales <- colMeans(RG)}
  A_Normalized <- t(t(A) * scales)

  n <- ncol(A)
  i <- seq_len(n)
  labels <- c(colnames(A),
              paste0(colnames(A)," Normalized"))


if(!is.null(chart.type)){
  if(chart.type== 'b' ){
  par(mar = c(10,4,3,2) + 0.1)
  boxplot(cbind(A, A_Normalized),
          las = 2, names = labels,
          col = c(rep("grey", n), rainbow(n)))

  }

  if(chart.type== 'l' ){

    par(mar = c(3,2,2,2))

  par(mfrow=c(2,1))

  matplot(A,type = 'l',col=c('steelblue',rainbow(n)),ylab='',xaxt='n',lwd=2)
   legend('top', inset=c(0,0),c(colnames(A)),lty=1,col=c('steelblue',rainbow(n)),bty='n',horiz=TRUE,lwd=2)
   axis(1, at=seq(length(A[,1]),1,-round(sqrt(length(A[,1])))), labels=rownames(A[seq(length(A[,1]),1,-round(sqrt(length(A[,1])))),]),las=1,cex.axis=1)
  matplot(A_Normalized,type = 'l',col=c('steelblue',rainbow(n)),ylab='',xaxt='n',lwd=2)
  axis(1, at=seq(length(A[,1]),1,-round(sqrt(length(A[,1])))), labels=rownames(A[seq(length(A[,1]),1,-round(sqrt(length(A[,1])))),]),las=1,cex.axis=1)
  legend('top',c(paste0(colnames(A)," Normalized")),lty=1,col=c('steelblue',rainbow(n)),bty='n',horiz=TRUE,lwd=2)
  }}

  par(mfrow=c(1,1))
  return(A_Normalized)

}
