#' @keywords internal
#' @export
#' @title Create Empty Site Location Metadata Dataframe
#' @param rows number of rows
#' @return A empty \code{meta} dataframe for use in a \emph{ws_monitor} object.
#' @description The list of columns in the returned \code{meta} dataframe is:
#' 
#' \preformatted{
#' > names(meta)
#'  [1] "monitorID"             "longitude"             "latitude"             
#'  [4] "elevation"             "timezone"              "countryCode"          
#'  [7] "stateCode"             "siteName"              "agencyName"           
#' [10] "countyName"            "msaName"               "monitorType"          
#' [13] "monitorInstrument"     "aqsID"                 "pwfslID"              
#' [16] "pwfslDataIngestSource" "telemetryAggregator"   "telemetryUnitID"      
#' }
#' 
#' For each row, all values will be set to \code{NA} with the appropriate type,
#' either \code{numeric} or \code{character}.
#' 
#' An empty dataframe can be generated by setting \code{rows=0}.
#' @examples
#' meta_3 <- createEmptyMetaDataframe(rows=3)
#' all(is.na(meta_3))

createEmptyMetaDataframe <- function(rows=0) {
  
  # The PWFSLSmoke v1.0 data model contains the following parameters
  # 
  # > names(meta)
  #  [1] "monitorID"             "longitude"             "latitude"              "elevation"            
  #  [5] "timezone"              "countryCode"           "stateCode"             "siteName"             
  #  [9] "agencyName"            "countyName"            "msaName"               "monitorType"          
  # [13] "siteID"                "instrumentID"          "aqsID"                 "pwfslID"              
  # [17] "pwfslDataIngestSource" "telemetryAggregator"   "telemetryUnitID"      
  
  # NOTE:  'meta' must be a dataframe because it has rownames which are deprecated in tibbles
  # Create empty dataframe
  meta <- as.data.frame(matrix(nrow=rows,ncol=19), stringsAsFactors=FALSE)
  
  colNames <- c("monitorID", "longitude", "latitude",
                "elevation", "timezone", "countryCode",
                "stateCode", "siteName", "agencyName",
                "countyName", "msaName", "monitorType",
                "siteID", "instrumentID", "aqsID", "pwfslID",
                "pwfslDataIngestSource", "telemetryAggregator", "telemetryUnitID")
  
  names(meta) <- colNames
  
  # Assign missing values 
  if ( rows > 0 ) {
    
    meta$monitorID <- as.character(NA)
    meta$longitude <- as.numeric(NA)
    meta$latitude <- as.numeric(NA)
    meta$elevation <- as.numeric(NA)
    meta$timezone <- as.character(NA)
    meta$countryCode <- as.character(NA)
    meta$stateCode <- as.character(NA)
    meta$siteName <- as.character(NA)
    meta$countyName <- as.character(NA)
    meta$msaName <- as.character(NA)
    meta$agencyName <- as.character(NA)
    meta$monitorType <- as.character(NA)
    meta$siteID <- as.character(NA)
    meta$instrumentID <- as.character(NA)
    meta$aqsID <- as.character(NA)
    meta$pwfslID <- as.character(NA)
    meta$pwfslDataIngestSource <- as.character(NA)
    meta$telemetryAggregator <- as.character(NA)
    meta$telemetryUnitID <- as.character(NA)
    
  }
  
  return(meta)
  
}
