#' @title Opens/Creates a new file, writes common file header.
#'
#' @param filename Name of the file, with extension
#' @param description File Description (e.g., Basin or project information, R script name)
#' @param filetype File extension, Encoding, Raven version (e.g. "rvp ASCII Raven 2.9.1")
#' @param author Name of file author (optional)
#' @param creationDate Bool of whether creation date should be added to header. (default \code{TRUE})
#' @param linelen length (width) of header, in text characters (default: 74)
#' @param textlen Length of textlines (default: 40, used to right-align text)
#'
#' @return \item{TRUE}{returns \code{TRUE} if executed successfully}
#'
#' @author Leland Scantlebury, \email{leland@@scantle.com}
#'
#' @examples
#' tf <- file.path(tempdir(), "HogwartsBasin.rvp")
#' rvn_write_Raven_newfile(filename = tf,
#'                 description = "Hogwarts River Basin RVP File Generated by HP_FileGen.R",
#'                 filetype = "rvp ASCII Raven 2.9.1",
#'                 author = 'Harry Potter')
#'
#' # view file
#' readLines(tf)
#'
#' @export rvn_write_Raven_newfile
rvn_write_Raven_newfile <- function(filename, description, filetype, author=NULL,
                            creationDate=TRUE, linelen=74, textlen=40)
{
  # Some top filler
  write(strrep('#', linelen), append=FALSE, file=filename)

  # Write the important stuff
  rvn_write_Raven_header(filename, filetype=filetype, author=author,
                      creationDate=creationDate, textlen=textlen)

  # Write a little custom description
  write('#', append=TRUE, file=filename)
  write(paste("#",description), append=TRUE, file=filename)
  write(paste("#",strrep('-',linelen-2)), append=TRUE, file=filename)
  write('#', append=TRUE, file=filename)

  return(TRUE)
}
