% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitdependentCensorigModels.R
\name{fitDepCens}
\alias{fitDepCens}
\title{Fit Dependent Censoring Models}
\usage{
fitDepCens(
  resData,
  X,
  W,
  cop = c("Frank", "Gumbel", "Normal"),
  dist = c("Weibull", "lognormal"),
  start = NULL,
  n.iter = 20,
  bootstrap = TRUE,
  n.boot = 50,
  eps = 0.001
)
}
\arguments{
\item{resData}{Data matrix with three columns;  Z = the observed survival time, d1 = the censoring indicator of T
and  d2 =  the censoring indicator of C.}

\item{X}{Data matrix with covariates related to T.}

\item{W}{Data matrix with covariates related to C. First column of W should be a vector of ones.}

\item{cop}{Which copula should be computed to account for dependency between T and C. This argument can take
one of the values from \code{c("Gumbel", "Frank", "Normal")}.}

\item{dist}{The distribution to be used for the censoring time C. Only two distributions are allowed, i.e, Weibull
and lognormal distributions. With the value \code{"Weibull"} as the
default.}

\item{start}{Initial values for the finite dimensional parameters. If \code{start} is NULL, the initial values will be obtained
by fitting a Cox model for survival time T and a Weibull model for dependent censoring C.}

\item{n.iter}{Number of iterations; the default is \code{n.iter = 20}. The larger the number of iterations, the longer the computational time.}

\item{bootstrap}{A boolean indicating whether to compute bootstrap standard errors for making inferences.}

\item{n.boot}{Number of bootstrap samples to use in the estimation of bootstrap standard errors if \code{bootstrap = TRUE}. The default is n.boot = 50. But, higher
values  of \code{n.boot} are recommended for obtaining good estimates of bootstrap standard errors.
@importFrom copula pCopula frankCopula gumbelCopula tau}

\item{eps}{Convergence error. This is set by the user in such away that the desired convergence is met; the default is \code{eps = 1e-3}.}
}
\value{
This function returns a fit of dependent censoring model; parameter estimates, estimate of the cumulative hazard function, bootstrap standard
errors for finite-dimensional parameters, the nonparametric cumulative hazard function, etc.
}
\description{
This function allows to estimate the dependency parameter along all other model parameters. First, estimates the cumulative hazard function, and
then at the second stage it estimates other model parameters assuming that the cumulative hazard function is known. The details for
implementing the dependent censoring methodology can be found in Deresa and Van Keilegom (2023).
}
\examples{
\donttest{

# Toy data example to illustrate implementation
n = 300
beta = c(0.5)
lambd = 0.35
eta = c(0.9,0.4)
X = cbind(rbinom(n,1,0.5))
W = cbind(rep(1,n),rbinom(n,1,0.5))
# generate dependency structure from Frank
frank.cop <- copula::frankCopula(param = 5,dim = 2)
U = copula::rCopula(n,frank.cop)
T1 = (-log(1-U[,1]))/(lambd*exp(X*beta))           # Survival time
T2 = (-log(1-U[,2]))^(1.1)*exp(W\%*\%eta)            # Censoring time
A = runif(n,0,15)                                  # administrative censoring time
Z = pmin(T1,T2,A)
d1 = as.numeric(Z==T1)
d2 = as.numeric(Z==T2)
resData = data.frame("Z" = Z,"d1" = d1, "d2" = d2)   # should be data frame

# Fit dependent censoring model

fit <- fitDepCens(resData = resData, X = X, W = W, bootstrap = FALSE)

# parameter estimates

fit$parameterEstimates

# summary fit results
summary(fit)

# plot cumulative hazard vs time

plot(fit$observedTime, fit$cumhazardFunction, type = "l",xlab = "Time",
ylab = "Estimated cumulative hazard function")
}
}
\references{
Deresa and Van Keilegom (2023). Copula based Cox proportional hazards models for dependent censoring, Journal of the American Statistical Association (in press).
}
