% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/neighbourhood.R
\docType{methods}
\name{spokes}
\alias{spokes}
\alias{spokes,RasterLayer,SpatialPoints,missing-method}
\title{Identify outward radiating spokes from initial points}
\usage{
spokes(landscape, coords, loci, maxRadius = ncol(landscape)/4,
  minRadius = maxRadius, allowOverlap = TRUE, stopRule = NULL,
  includeBehavior = "includePixels", returnDistances = FALSE,
  angles = NA_real_, nAngles = NA_real_, returnAngles = FALSE,
  returnIndices = TRUE, ...)

\S4method{spokes}{RasterLayer,SpatialPoints,missing}(landscape, coords, loci,
  maxRadius = ncol(landscape)/4, minRadius = maxRadius,
  allowOverlap = TRUE, stopRule = NULL, includeBehavior = "includePixels",
  returnDistances = FALSE, angles = NA_real_, nAngles = NA_real_,
  returnAngles = FALSE, returnIndices = TRUE, ...)
}
\arguments{
\item{landscape}{Raster on which the circles are built.}

\item{coords}{Either a matrix with 2 (or 3) columns, x and y (and id), representing the
coordinates (and an associated id, like cell index),
or a \code{SpatialPoints*} object around which to make circles. Must be same
coordinate system as the \code{landscape} argument. Default is missing,
meaning it uses the default to \code{loci}}

\item{loci}{Numeric. An alternative to \code{coords}. These are the indices on
\code{landscape} to initiate this function. See \code{coords}. Default is one
point in centre of \code{landscape}..}

\item{maxRadius}{Numeric vector of length 1 or same length as coords}

\item{minRadius}{Numeric vector of length 1 or same length as \code{coords}. Default is
\code{maxRadius}, meaning return all cells that are touched
by the narrow ring at that exact radius. If smaller than \code{maxRadius},
then this will create a buffer or donut or ring.}

\item{allowOverlap}{Logical. Should duplicates across id be removed or kept. Default TRUE.}

\item{stopRule}{A function. If the spokes are to stop. This can be a function
of \code{landscape}, \code{fromCell}, \code{toCell}, \code{x}
(distance from coords cell), or any other named argument passed
into the \code{...} of this function. See examples.}

\item{includeBehavior}{Character string. Currently accepts only "includePixels", the default,
and "excludePixels". See details.}

\item{returnDistances}{Logical. If TRUE, then a column will be added to the returned
data.table that reports the distance from \code{coords} to every
point that was in the circle/donut surrounding \code{coords}. Default
FALSE, which is faster.}

\item{angles}{Numeric. Optional vector of angles, in radians, to use. This will create
"spokes" outward from coords. Default is NA, meaning, use internally
derived angles that will "fill" the circle.}

\item{nAngles}{Numeric, length one. Alternative to angles. If provided, the function
will create a sequence of angles from \code{0} to \code{2*pi},
with a length \code{nAngles}, and not including \code{2*pi}.
Will not be used if \code{angles} is provided, and will show
warning of both are given.}

\item{returnAngles}{Logical. If TRUE, then a column will be added to the returned
data.table that reports the angle from \code{coords} to every
point that was in the circle/donut surrounding \code{coords}. Default
FALSE.}

\item{returnIndices}{Logical. Should the function return a \code{data.table}
with indices and values of successful spread events, or
return a raster with values. See Details.}

\item{...}{Objects to be used by \code{stopRule()}. See examples.}
}
\value{
A matrix containing columns id (representing the row numbers of \code{coords}),
angles (from \code{coords} to each point along the spokes), x and y coordinates
of each point along the spokes, the corresponding indices on the \code{landscape}
Raster, dists (the distances between each \code{coords} and each point along the
spokes), and stop, indicating if it was a point that caused a spoke to stop
going outwards due to \code{stopRule}.
}
\description{
This is a generalized version of a notion of a viewshed.
The main difference is that there can be many "viewpoints".
}
\examples{
library(sp)
library(raster)
library(quickPlot)

set.seed(1234)

ras <- raster(extent(0, 10, 0, 10), res = 1, val = 0)
rp <- randomPolygons(ras, numTypes = 10)

clearPlot()
Plot(rp)

angles <- seq(0, pi * 2, length.out = 17)
angles <- angles[-length(angles)]
n <- 2
loci <- sample(ncell(rp), n)
coords <- SpatialPoints(xyFromCell(rp, loci))
stopRule <- function(landscape) landscape < 3
d2 <- spokes(rp, coords = coords, stopRule = stopRule,
             minRadius = 0, maxRadius = 50,
             returnAngles = TRUE, returnDistances = TRUE,
             allowOverlap = TRUE, angles = angles, returnIndices = TRUE)

# Assign values to the "patches" that were in the viewshed of a ray
rasB <- raster(ras)
rasB[] <- 0
rasB[d2[d2[, "stop"] == 1, "indices"]] <- 1

Plot(rasB, addTo = "rp", zero.color = "transparent", cols = "red")

if (NROW(d2) > 0) {
  sp1 <- SpatialPoints(d2[, c("x", "y")])
  Plot(sp1, addTo = "rp", pch = 19, size = 5, speedup = 0.1)
}
Plot(coords, addTo = "rp", pch = 19, size = 6, cols = "blue", speedup = 0.1)

clearPlot()
}
\author{
Eliot McIntire
}
