% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dist_sinkhorn.R
\name{sinkhorn}
\alias{sinkhorn}
\alias{sinkhornD}
\title{Wasserstein Distance via Entropic Regularization and Sinkhorn Algorithm}
\usage{
sinkhorn(X, Y, p = 2, wx = NULL, wy = NULL, lambda = 0.1, ...)

sinkhornD(D, p = 2, wx = NULL, wy = NULL, lambda = 0.1, ...)
}
\arguments{
\item{X}{an \eqn{(M\times P)} matrix of row observations.}

\item{Y}{an \eqn{(N\times P)} matrix of row observations.}

\item{p}{an exponent for the order of the distance (default: 2).}

\item{wx}{a length-\eqn{M} marginal density that sums to \eqn{1}. If \code{NULL} (default), uniform weight is set.}

\item{wy}{a length-\eqn{N} marginal density that sums to \eqn{1}. If \code{NULL} (default), uniform weight is set.}

\item{lambda}{a regularization parameter (default: 0.1).}

\item{...}{extra parameters including \describe{
\item{maxiter}{maximum number of iterations (default: 496).}
\item{abstol}{stopping criterion for iterations (default: 1e-10).}
}}

\item{D}{an \eqn{(M\times N)} distance matrix \eqn{d(x_m, y_n)} between two sets of observations.}
}
\value{
a named list containing\describe{
\item{distance}{\eqn{\mathcal{W}_p} distance value.}
\item{plan}{an \eqn{(M\times N)} nonnegative matrix for the optimal transport plan.}
}
}
\description{
To alleviate the computational burden of solving the exact optimal transport problem via linear programming,
Cuturi (2013) introduced an entropic regularization scheme that yields a smooth approximation to the
Wasserstein distance. Let \eqn{C:=\|X_m - Y_n\|^p} be the cost matrix, where \eqn{X_m} and \eqn{Y_n} are the observations from two distributions \eqn{\mu} and \eqn{nu}. 
Then, the regularized problem adds a penalty term to the objective function:
\deqn{
  W_{p,\lambda}^p(\mu, \nu) = \min_{\Gamma \in \Pi(\mu, \nu)} \langle \Gamma, C \rangle + \lambda \sum_{m,n} \Gamma_{m,n} \log (\Gamma_{m,n}),
}
where \eqn{\lambda > 0} is the regularization parameter and \eqn{\Gamma} denotes a transport plan.
As \eqn{\lambda \rightarrow 0}, the regularized solution converges to the exact Wasserstein solution,
but small values of \eqn{\lambda} may cause numerical instability due to underflow.
In such cases, the implementation halts with an error; users are advised to increase \eqn{\lambda}
to maintain numerical stability.
}
\examples{
\donttest{
#-------------------------------------------------------------------
#  Wasserstein Distance between Samples from Two Bivariate Normal
#
# * class 1 : samples from Gaussian with mean=(-1, -1)
# * class 2 : samples from Gaussian with mean=(+1, +1)
#-------------------------------------------------------------------
## SMALL EXAMPLE
set.seed(100)
m = 20
n = 10
X = matrix(rnorm(m*2, mean=-1),ncol=2) # m obs. for X
Y = matrix(rnorm(n*2, mean=+1),ncol=2) # n obs. for Y

## COMPARE WITH WASSERSTEIN 
outw = wasserstein(X, Y)
skh1 = sinkhorn(X, Y, lambda=0.05)
skh2 = sinkhorn(X, Y, lambda=0.25)

## VISUALIZE : SHOW THE PLAN AND DISTANCE
pm1 = paste0("Exact Wasserstein:\n distance=",round(outw$distance,2))
pm2 = paste0("Sinkhorn (lbd=0.05):\n distance=",round(skh1$distance,2))
pm5 = paste0("Sinkhorn (lbd=0.25):\n distance=",round(skh2$distance,2))

opar <- par(no.readonly=TRUE)
par(mfrow=c(1,3), pty="s")
image(outw$plan, axes=FALSE, main=pm1)
image(skh1$plan, axes=FALSE, main=pm2)
image(skh2$plan, axes=FALSE, main=pm5)
par(opar)
}

}
\references{
\insertRef{cuturi_2013_SinkhornDistancesLightspeed}{T4transport}
}
\concept{dist}
