% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aem.weight.edges.R, R/aem.weight.time.R
\name{aem.weight.edges}
\alias{aem.weight.edges}
\alias{aem.weight.time}
\title{Weight edges when constructing AEM variables}
\usage{
aem.weight.edges(
  nb.object,
  coords,
  distmat = NULL,
  alpha = 2,
  beta = NULL,
  max.d = NULL,
  unit.angle = "degrees",
  rot.angle = 0,
  rm.same.y = TRUE,
  plot.connexions = TRUE
)

aem.weight.time(
  dates,
  distmat = NULL,
  alpha = 2,
  beta = NULL,
  max.d = NULL,
  unit.angle = "degrees",
  rot.angle = 0,
  rm.same.y = TRUE,
  plot.connexions = TRUE
)
}
\arguments{
\item{nb.object}{Object with class 'nb', computed by the spdep package,
containing a list of neighbours for each sampling unit (site or time).}

\item{coords}{A three-column matrix or data frame. Column 1: identifiers of
the points (must be numeric). Columns 2 and 3: the X and Y coordinates of
the points.}

\item{distmat}{Class 'matrix' or 'dist' object containing a dissimilarity or
distance matrix. (See details).}

\item{alpha}{Numeric. Exponent of the first weighting function. (See details).}

\item{beta}{Numeric. Exponent of the second weighting function. (See details).}

\item{max.d}{Numeric. Maximum distance for weighting. Default value if
max.d=NULL: the maximum distance among a set of sites divided by 2 or the
full span of a time series divided by 2 (not recommended in most problems,
see details). A warning is given if \code{max.d = NULL} and the default
value is used.}

\item{unit.angle}{Character. The measurement units in which the angle is
defined: either "degrees" (default) or "radians".}

\item{rot.angle}{Numeric. Angle of the vector describing the process
influencing the sites. This argument generates a rotation of the site
coordinates. The set of coordinates is rotated counterclockwise. Negative
values will produce a clockwise rotation.}

\item{rm.same.y}{Logical (\code{TRUE}, \code{FALSE}). Determines if the links
perpendicular to the gradient should be removed. Default value: \code{TRUE}.
If these links have already been removed, this argument put to \code{TRUE}
will make the function crash. See details for more information.}

\item{plot.connexions}{Logical (\code{TRUE}, \code{FALSE}). Determines if the
sites and the associated connexion diagram should be plotted after rotation
of the coordinates by \code{gradient.angle}.}

\item{dates}{A vector of dates, class 'numeric' or 'Date'.}
}
\value{
A vector of weights associating a value to each edge of the graph.
}
\description{
These functions construct a vector of weights that can be
 associated to the edges of the connexion diagram used as a basis to build
 AEM eigenfunctions. \code{aem.weight.edges} is general and can be used for 1 or
 2 dimensional problems. \code{aem.weight.time} is meant to be used only for time
 series. It is a wrapper for \code{aem.weight.edges}.
}
\details{
These functions should be used in close relationship with
\code{\link{aem.build.binary}}, consequently many of the arguments in this
function and in \code{\link{aem.build.binary}} are the same.

The argument \code{distmat} may contain general forms of dissimilarity, for
example the difficulty of transfer of individuals, matter or energy among the
sampling units through space or time.

In \code{aem.weight.edges}, two weighting functions, described in Legendre and
Legendre (2012, eqs. 114.3 and 14.4) have been implemented, where \eqn{d_{ij}}
is the distance between sites \eqn{i} and \eqn{j}:

\tabular{ll}{ \code{Weighting function 1:} \tab \eqn{1 -
(d_{ij}/max(d))^\alpha}{1 - (d_{ij}/max(d))^alpha} \cr \code{Weighting
function 2:} \tab \eqn{1/d_{ij}^\beta}{1/d_{ij}^beta} }

Also note that if a value is provided for \code{beta} (that is, if it is not
\code{NULL}), weighting function 2 is used regardless of whether \code{alpha}
is defined or not.

In most applications, the default value of \code{max.d} is not optimal. A more
meaningful solution in many applications is to compute a Moran's I correlogram
(for univariate data) or a Mantel correlogram (for multivariate data), and
provide the distance where the correlation becomes 0 as the value for max.d.
}
\section{Functions}{
\itemize{
\item \code{aem.weight.time}: 
}}

\examples{

### Time serie example
### Example - 12 dates (days from January 1st of year 1) 
### in a 6-year study starting September 5, 2000
if(require("spdep", quietly = TRUE)){
dates <- as.Date(c(129,269,500,631,864,976,1228,1352,1606,1730,1957,2087),origin="2000/1/1")
autocor.limit <- 522  # Limit of autcorrelation in the correlogram

### Using aem.weight.time()
(wtime <- aem.weight.time(dates, alpha=2, max.d=autocor.limit))
### Using aem.weight.edges()
n <- length(dates)
nb <- cell2nb(1, n)
xy.dates <- cbind(1:n, rep(1, n), dates)
(wtime <- aem.weight.edges(nb, xy.dates, alpha=2, max.d=autocor.limit))

n <- length(dates)
nb <- cell2nb(1, n)
xy.dates <- cbind(1:n, dates, rep(1, n)) ## Note the inversion of 'dates' and 'rep(1,n)'
wtime <- aem.weight.edges(nb, xy.dates, alpha=2, 
max.d=autocor.limit,rot.angle=90) # Note that 'rot.angle=90' was used

### Spatial example using default d.max (notice the warning)
###########################################################################
nb<-cell2nb(5,5,"queen")
xy <- cbind(1:25,expand.grid(1:5,1:5))
(wspace <- aem.weight.edges(nb,xy))
}

}
\references{
Legendre, P. and L. Legendre (2012) \emph{Numerical Ecology}, 3rd English
edition. Elsevier Science BV, Amsterdam.

Legendre, P. and O. Gauthier (2014) Statistical methods for temporal and
space-time analysis of community composition data. \emph{Proceedings of the
Royal Society B - Biological Sciences}, 281, 20132728.
}
\seealso{
\code{\link{aem.build.binary}}, \code{\link[spdep]{sp.correlogram}},
 \code{\link[vegan]{mantel.correlog}}
}
\author{
Olivier Gauthier, Pierre Legendre and F. Guillaume Blanchet
}
\keyword{spatial}
\keyword{ts}
