% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_param_computed.R
\name{derive_param_computed}
\alias{derive_param_computed}
\title{Adds a Parameter Computed from the Analysis Value of Other Parameters}
\usage{
derive_param_computed(
  dataset = NULL,
  dataset_add = NULL,
  by_vars,
  parameters,
  analysis_var = AVAL,
  analysis_value,
  set_values_to,
  filter = NULL,
  constant_by_vars = NULL,
  constant_parameters = NULL
)
}
\arguments{
\item{dataset}{Input dataset

The variables specified by the \code{by_vars} parameter are expected.

The variable specified by \code{by_vars} and \code{PARAMCD} must be a unique key of
the input dataset after restricting it by the filter condition (\code{filter}
parameter) and to the parameters specified by \code{parameters}.}

\item{dataset_add}{Additional dataset

The variables specified by the \code{by_vars} parameter are expected.

The variable specified by \code{by_vars} and \code{PARAMCD} must be a unique key of
the additional dataset after restricting it to the parameters specified by
\code{parameters}.

If the argument is specified, the observations of the additional dataset
are considered in addition to the observations from the input dataset
(\code{dataset} restricted by \code{filter}).}

\item{by_vars}{Grouping variables

For each group defined by \code{by_vars} an observation is added to the output
dataset. Only variables specified in \code{by_vars} will be populated
in the newly created records.

\emph{Permitted Values:} list of variables}

\item{parameters}{Required parameter codes

It is expected that all parameter codes (\code{PARAMCD}) which are required to
derive the new parameter are specified for this parameter or the
\code{constant_parameters} parameter.

If observations should be considered which do not have a parameter code,
e.g., if an SDTM dataset is used, temporary parameter codes can be derived
by specifying a list of expressions. The name of the element defines the
temporary parameter code and the expression the condition for selecting the
records. For example \code{parameters = exprs(HGHT = VSTESTCD == "HEIGHT")}
selects the observations with \code{VSTESTCD == "HEIGHT"} from the input data
(\code{dataset} and \code{dataset_add}), sets \code{PARAMCD = "HGHT"} for these
observations, and adds them to the observations to consider.

Unnamed elements in the list of expressions are considered as parameter
codes. For example, \code{parameters = exprs(WEIGHT, HGHT = VSTESTCD == "HEIGHT")} uses the parameter code \code{"WEIGHT"} and creates a temporary
parameter code \code{"HGHT"}.

\emph{Permitted Values:} A character vector of \code{PARAMCD} values or a list of expressions}

\item{analysis_var}{Analysis variable

The specified variable is set to the value of \code{analysis_value} for the new
observations.

\emph{Permitted Values}: An unquoted symbol}

\item{analysis_value}{Definition of the analysis value

An expression defining the analysis value (\code{AVAL}) of the new parameter is
expected. The values of variables of the parameters specified by
\code{parameters} can be accessed using \verb{<variable name>.<parameter code>},
e.g., \code{AVAL.SYSBP}.

Variable names in the expression must not contain more than one dot.

\emph{Permitted Values:} An unquoted expression}

\item{set_values_to}{Variables to be set

The specified variables are set to the specified values for the new
observations. For example \code{exprs(PARAMCD = "MAP")} defines the parameter
code for the new parameter.

\emph{Permitted Values:} List of variable-value pairs}

\item{filter}{Filter condition

The specified condition is applied to the input dataset before deriving the
new parameter, i.e., only observations fulfilling the condition are taken
into account.

\emph{Permitted Values:} a condition}

\item{constant_by_vars}{By variables for constant parameters

The constant parameters (parameters that are measured only once) are merged
to the other parameters using the specified variables. (Refer to Example 2)

\emph{Permitted Values:} list of variables}

\item{constant_parameters}{Required constant parameter codes

It is expected that all the parameter codes (\code{PARAMCD}) which are required
to derive the new parameter and are measured only once are specified here.
For example if BMI should be derived and height is measured only once while
weight is measured at each visit. Height could be specified in the
\code{constant_parameters} parameter. (Refer to Example 2)

If observations should be considered which do not have a parameter code,
e.g., if an SDTM dataset is used, temporary parameter codes can be derived
by specifying a list of expressions. The name of the element defines the
temporary parameter code and the expression the condition for selecting the
records. For example \code{constant_parameters = exprs(HGHT = VSTESTCD == "HEIGHT")} selects the observations with \code{VSTESTCD == "HEIGHT"} from the
input data (\code{dataset} and \code{dataset_add}), sets \code{PARAMCD = "HGHT"} for these
observations, and adds them to the observations to consider.

Unnamed elements in the list of expressions are considered as parameter
codes. For example, \code{constant_parameters = exprs(WEIGHT, HGHT = VSTESTCD == "HEIGHT")} uses the parameter code \code{"WEIGHT"} and creates a temporary
parameter code \code{"HGHT"}.

\emph{Permitted Values:} A character vector of \code{PARAMCD} values or a list of expressions}
}
\value{
The input dataset with the new parameter added. Note, a variable will only
be populated in the new parameter rows if it is specified in \code{by_vars}.
}
\description{
Adds a parameter computed from the analysis value of other parameters. It is
expected that the analysis value of the new parameter is defined by an
expression using the analysis values of other parameters. For example mean
arterial pressure (MAP) can be derived from systolic (SYSBP) and diastolic
blood pressure (DIABP) with the formula
\deqn{MAP = \frac{SYSBP + 2 DIABP}{3}}{MAP = (SYSBP + 2 DIABP) / 3}
}
\details{
For each group (with respect to the variables specified for the
\code{by_vars} parameter) an observation is added to the output dataset if the
filtered input dataset (\code{dataset}) or the additional dataset
(\code{dataset_add}) contains exactly one observation for each parameter code
specified for \code{parameters}.

For the new observations \code{AVAL} is set to the value specified by
\code{analysis_value} and the variables specified for \code{set_values_to} are set to
the provided values. The values of the other variables of the input dataset
are set to \code{NA}.
}
\examples{
library(tibble)

# Example 1: Derive MAP
advs <- tribble(
  ~USUBJID,      ~PARAMCD, ~PARAM,                            ~AVAL, ~AVALU, ~VISIT,
  "01-701-1015", "DIABP",  "Diastolic Blood Pressure (mmHg)",    51, "mmHg", "BASELINE",
  "01-701-1015", "DIABP",  "Diastolic Blood Pressure (mmHg)",    50, "mmHg", "WEEK 2",
  "01-701-1015", "SYSBP",  "Systolic Blood Pressure (mmHg)",    121, "mmHg", "BASELINE",
  "01-701-1015", "SYSBP",  "Systolic Blood Pressure (mmHg)",    121, "mmHg", "WEEK 2",
  "01-701-1028", "DIABP",  "Diastolic Blood Pressure (mmHg)",    79, "mmHg", "BASELINE",
  "01-701-1028", "DIABP",  "Diastolic Blood Pressure (mmHg)",    80, "mmHg", "WEEK 2",
  "01-701-1028", "SYSBP",  "Systolic Blood Pressure (mmHg)",    130, "mmHg", "BASELINE",
  "01-701-1028", "SYSBP",  "Systolic Blood Pressure (mmHg)",    132, "mmHg", "WEEK 2"
)

derive_param_computed(
  advs,
  by_vars = exprs(USUBJID, VISIT),
  parameters = c("SYSBP", "DIABP"),
  analysis_value = (AVAL.SYSBP + 2 * AVAL.DIABP) / 3,
  set_values_to = exprs(
    PARAMCD = "MAP",
    PARAM = "Mean Arterial Pressure (mmHg)",
    AVALU = "mmHg"
  )
)

# Example 2: Derive BMI where height is measured only once
advs <- tribble(
  ~USUBJID,      ~PARAMCD, ~PARAM,        ~AVAL, ~AVALU, ~VISIT,
  "01-701-1015", "HEIGHT", "Height (cm)", 147.0, "cm",   "SCREENING",
  "01-701-1015", "WEIGHT", "Weight (kg)",  54.0, "kg",   "SCREENING",
  "01-701-1015", "WEIGHT", "Weight (kg)",  54.4, "kg",   "BASELINE",
  "01-701-1015", "WEIGHT", "Weight (kg)",  53.1, "kg",   "WEEK 2",
  "01-701-1028", "HEIGHT", "Height (cm)", 163.0, "cm",   "SCREENING",
  "01-701-1028", "WEIGHT", "Weight (kg)",  78.5, "kg",   "SCREENING",
  "01-701-1028", "WEIGHT", "Weight (kg)",  80.3, "kg",   "BASELINE",
  "01-701-1028", "WEIGHT", "Weight (kg)",  80.7, "kg",   "WEEK 2"
)

derive_param_computed(
  advs,
  by_vars = exprs(USUBJID, VISIT),
  parameters = "WEIGHT",
  analysis_value = AVAL.WEIGHT / (AVAL.HEIGHT / 100)^2,
  set_values_to = exprs(
    PARAMCD = "BMI",
    PARAM = "Body Mass Index (kg/m^2)",
    AVALU = "kg/m^2"
  ),
  constant_parameters = c("HEIGHT"),
  constant_by_vars = exprs(USUBJID)
)

# Example 3: Using data from an additional dataset and other variables than AVAL
qs <- tibble::tribble(
  ~USUBJID, ~AVISIT,   ~QSTESTCD, ~QSORRES, ~QSSTRESN,
  "1",      "WEEK 2",  "CHSF112", NA,               1,
  "1",      "WEEK 2",  "CHSF113", "Yes",           NA,
  "1",      "WEEK 2",  "CHSF114", NA,               1,
  "1",      "WEEK 4",  "CHSF112", NA,               2,
  "1",      "WEEK 4",  "CHSF113", "No",            NA,
  "1",      "WEEK 4",  "CHSF114", NA,               1
)

adchsf <- tibble::tribble(
  ~USUBJID, ~AVISIT,  ~PARAMCD, ~QSORRES, ~QSSTRESN, ~AVAL,
  "1",      "WEEK 2", "CHSF12", NA,       1,             6,
  "1",      "WEEK 2", "CHSF14", NA,       1,             6,
  "1",      "WEEK 4", "CHSF12", NA,       2,            12,
  "1",      "WEEK 4", "CHSF14", NA,       1,             6
)

derive_param_computed(
  adchsf,
  dataset_add = qs,
  by_vars = exprs(USUBJID, AVISIT),
  parameters = exprs(CHSF12, CHSF13 = QSTESTCD \%in\% c("CHSF113", "CHSF213"), CHSF14),
  analysis_value = case_when(
    QSORRES.CHSF13 == "Not applicable" ~ 0,
    QSORRES.CHSF13 == "Yes" ~ 38,
    QSORRES.CHSF13 == "No" ~ if_else(
      QSSTRESN.CHSF12 > QSSTRESN.CHSF14,
      25,
      0
    )
  ),
  set_values_to = exprs(PARAMCD = "CHSF13")
)
}
\seealso{
BDS-Findings Functions for adding Parameters/Records: 
\code{\link{default_qtc_paramcd}()},
\code{\link{derive_expected_records}()},
\code{\link{derive_extreme_event}()},
\code{\link{derive_extreme_records}()},
\code{\link{derive_locf_records}()},
\code{\link{derive_param_bmi}()},
\code{\link{derive_param_bsa}()},
\code{\link{derive_param_doseint}()},
\code{\link{derive_param_exist_flag}()},
\code{\link{derive_param_exposure}()},
\code{\link{derive_param_extreme_record}()},
\code{\link{derive_param_framingham}()},
\code{\link{derive_param_map}()},
\code{\link{derive_param_qtc}()},
\code{\link{derive_param_rr}()},
\code{\link{derive_param_wbc_abs}()},
\code{\link{derive_summary_records}()}
}
\concept{der_prm_bds_findings}
\keyword{der_prm_bds_findings}
