% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AnnData.R
\name{AnnData}
\alias{AnnData}
\alias{AnnDataR6}
\title{Create an Annotated Data Matrix}
\usage{
AnnData(
  X = NULL,
  obs = NULL,
  var = NULL,
  uns = NULL,
  obsm = NULL,
  varm = NULL,
  layers = NULL,
  raw = NULL,
  dtype = "float32",
  shape = NULL,
  filename = NULL,
  filemode = NULL,
  obsp = NULL,
  varp = NULL
)
}
\arguments{
\item{X}{A #observations × #variables data matrix. A view of the data is used if the data type matches, otherwise, a copy is made.}

\item{obs}{Key-indexed one-dimensional observations annotation of length #observations.}

\item{var}{Key-indexed one-dimensional variables annotation of length #variables.}

\item{uns}{Key-indexed unstructured annotation.}

\item{obsm}{Key-indexed multi-dimensional observations annotation of length #observations. If passing a \code{~numpy.ndarray}, it needs to have a structured datatype.}

\item{varm}{Key-indexed multi-dimensional variables annotation of length #variables. If passing a \code{~numpy.ndarray}, it needs to have a structured datatype.}

\item{layers}{Key-indexed multi-dimensional arrays aligned to dimensions of \code{X}.}

\item{raw}{Store raw version of \code{X} and \code{var} as \verb{$raw$X} and \verb{$raw$var}.}

\item{dtype}{Data type used for storage.}

\item{shape}{Shape list (#observations, #variables). Can only be provided if \code{X} is \code{NULL}.}

\item{filename}{Name of backing file. See \href{https://docs.h5py.org/en/latest/high/file.html#h5py.File}{h5py.File}.}

\item{filemode}{Open mode of backing file. See \href{https://docs.h5py.org/en/latest/high/file.html#h5py.File}{h5py.File}.}

\item{obsp}{Pairwise annotation of observations, a mutable mapping with array-like values.}

\item{varp}{Pairwise annotation of observations, a mutable mapping with array-like values.}
}
\description{
\code{AnnData} stores a data matrix \code{X} together with annotations
of observations \code{obs} (\code{obsm}, \code{obsp}), variables \code{var} (\code{varm}, \code{varp}),
and unstructured annotations \code{uns}.

An \code{AnnData} object \code{adata} can be sliced like a data frame,
for instance \code{adata_subset <- adata[, list_of_variable_names]}. \code{AnnData}’s
basic structure is similar to R's ExpressionSet.

If setting an \code{h5ad}-formatted HDF5 backing file \code{filename},
data remains on the disk but is automatically loaded into memory if needed.
See this \href{http://falexwolf.de/blog/171223_AnnData_indexing_views_HDF5-backing/}{blog post} for more details.
}
\details{
\code{AnnData} stores observations (samples) of variables/features in the rows of a matrix.
This is the convention of the modern classics of statistic and machine learning,
the convention of dataframes both in R and Python and the established statistics
and machine learning packages in Python (statsmodels, scikit-learn).

Single dimensional annotations of the observation and variables are stored
in the \code{obs} and \code{var} attributes as data frames.
This is intended for metrics calculated over their axes.
Multi-dimensional annotations are stored in \code{obsm} and \code{varm},
which are aligned to the objects observation and variable dimensions respectively.
Square matrices representing graphs are stored in \code{obsp} and \code{varp},
with both of their own dimensions aligned to their associated axis.
Additional measurements across both observations and variables are stored in
\code{layers}.

Indexing into an AnnData object can be performed by relative position
with numeric indices,  or by labels.
To avoid ambiguity with numeric indexing into observations or variables,
indexes of the AnnData object are converted to strings by the constructor.

Subsetting an AnnData object by indexing into it will also subset its elements
according to the dimensions they were aligned to.
This means an operation like \code{adata[list_of_obs, ]} will also subset \code{obs},
\code{obsm}, and \code{layers}.

Subsetting an AnnData object returns a view into the original object,
meaning very little additional memory is used upon subsetting.
This is achieved lazily, meaning that the constituent arrays are subset on access.
Copying a view causes an equivalent “real” AnnData object to be generated.
Attempting to modify a view (at any attribute except X) is handled
in a copy-on-modify manner, meaning the object is initialized in place.
Here’s an example\preformatted{batch1 <- adata[adata$obs["batch"] == "batch1", ]
batch1$obs["value"] = 0 # This makes batch1 a “real” AnnData object
}

At the end of this snippet: \code{adata} was not modified,
and \code{batch1} is its own AnnData object with its own data.

Similar to Bioconductor’s \code{ExpressionSet} and \code{scipy.sparse} matrices,
subsetting an AnnData object retains the dimensionality of its constituent arrays.
Therefore, unlike with the classes exposed by \code{pandas}, \code{numpy},
and \code{xarray}, there is no concept of a one dimensional AnnData object.
AnnDatas always have two inherent dimensions, \code{obs} and \code{var}.
Additionally, maintaining the dimensionality of the AnnData object allows for
consistent handling of \code{scipy.sparse} matrices and \code{numpy} arrays.
}
\examples{
\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
  layers = list(
    spliced = matrix(c(4, 5, 6, 7), nrow = 2),
    unspliced = matrix(c(8, 9, 10, 11), nrow = 2)
  ),
  obsm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  ),
  varm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  ),
  uns = list(a = 1, b = 2, c = list(c.a = 3, c.b = 4))
)

value <- matrix(c(1,2,3,4), nrow = 2)
ad$X <- value
ad$X

ad$layers
ad$layers["spliced"]
ad$layers["test"] <- value
ad$layers

ad$to_df()
ad$uns

as.matrix(ad)
as.matrix(ad, layer = "unspliced")
dim(ad)
rownames(ad)
colnames(ad)
}

## ------------------------------------------------
## Method `AnnDataR6$new`
## ------------------------------------------------

\dontrun{
# use AnnData() instead of AnnDataR6$new()
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2"))
)
}

## ------------------------------------------------
## Method `AnnDataR6$obs_keys`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2"))
)
ad$obs_keys()
}

## ------------------------------------------------
## Method `AnnDataR6$obs_names_make_unique`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(rep(1, 6), nrow = 3),
  obs = data.frame(field = c(1, 2, 3))
)
ad$obs_names <- c("a", "a", "b")
ad$obs_names_make_unique()
ad$obs_names
}

## ------------------------------------------------
## Method `AnnDataR6$obsm_keys`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  obsm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  )
)
ad$obs_keys()
}

## ------------------------------------------------
## Method `AnnDataR6$var_keys`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2"))
)
ad$var_keys()
}

## ------------------------------------------------
## Method `AnnDataR6$var_names_make_unique`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(rep(1, 6), nrow = 2),
  var = data.frame(field = c(1, 2, 3))
)
ad$var_names <- c("a", "a", "b")
ad$var_names_make_unique()
ad$var_names
}

## ------------------------------------------------
## Method `AnnDataR6$varm_keys`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
  varm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  )
)
ad$varm_keys()
}

## ------------------------------------------------
## Method `AnnDataR6$uns_keys`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
  uns = list(a = 1, b = 2, c = list(c.a = 3, c.b = 4))
)
}

## ------------------------------------------------
## Method `AnnDataR6$chunk_X`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(runif(10000), nrow = 50)
)

ad$chunk_X(select = 10L) # 10 random samples
ad$chunk_X(select = 1:3) # first 3 samples
}

## ------------------------------------------------
## Method `AnnDataR6$chunked_X`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(runif(10000), nrow = 50)
)
ad$chunked_X(10)
}

## ------------------------------------------------
## Method `AnnDataR6$copy`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2)
)
ad$copy()
ad$copy("file.h5ad")
}

## ------------------------------------------------
## Method `AnnDataR6$rename_categories`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2"))
)
ad$rename_categories("group", c(a = "A", b = "B")) # ??
}

## ------------------------------------------------
## Method `AnnDataR6$strings_to_categoricals`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
)
ad$strings_to_categoricals() # ??
}

## ------------------------------------------------
## Method `AnnDataR6$to_df`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
  layers = list(
    spliced = matrix(c(4, 5, 6, 7), nrow = 2),
    unspliced = matrix(c(8, 9, 10, 11), nrow = 2)
  )
)

ad$to_df()
ad$to_df("unspliced")
}

## ------------------------------------------------
## Method `AnnDataR6$transpose`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2"))
)

ad$transpose()
}

## ------------------------------------------------
## Method `AnnDataR6$write_csvs`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
  varm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  ),
  uns = list(a = 1, b = 2, c = list(c.a = 3, c.b = 4))
)

ad$to_write_csvs("output")

unlink("output", recursive = TRUE)
}

## ------------------------------------------------
## Method `AnnDataR6$write_h5ad`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
  varm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  ),
  uns = list(a = 1, b = 2, c = list(c.a = 3, c.b = 4))
)

ad$write_h5ad("output.h5ad")

file.remove("output.h5ad")
}

## ------------------------------------------------
## Method `AnnDataR6$write_loom`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
  varm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  ),
  uns = list(a = 1, b = 2, c = list(c.a = 3, c.b = 4))
)

ad$write_loom("output.loom")

file.remove("output.loom")
}

## ------------------------------------------------
## Method `AnnDataR6$print`
## ------------------------------------------------

\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
  layers = list(
    spliced = matrix(c(4, 5, 6, 7), nrow = 2),
    unspliced = matrix(c(8, 9, 10, 11), nrow = 2)
  ),
  obsm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  ),
  varm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  ),
  uns = list(a = 1, b = 2, c = list(c.a = 3, c.b = 4))
)

ad$print()
print(ad)
}
}
\seealso{
\code{\link[=read_h5ad]{read_h5ad()}} \code{\link[=read_csv]{read_csv()}} \code{\link[=read_excel]{read_excel()}} \code{\link[=read_hdf]{read_hdf()}} \code{\link[=read_loom]{read_loom()}} \code{\link[=read_mtx]{read_mtx()}} \code{\link[=read_text]{read_text()}} \code{\link[=read_umi_tools]{read_umi_tools()}} \code{\link[=write_h5ad]{write_h5ad()}} \code{\link[=write_csvs]{write_csvs()}} \code{\link[=write_loom]{write_loom()}}
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{X}}{Data matrix of shape \code{n_obs} × \code{n_vars}.}

\item{\code{filename}}{Name of the backing file.

Change to backing mode by setting the filename of a \code{.h5ad} file.
\itemize{
\item Setting the filename writes the stored data to disk.
\item Setting the filename when the filename was previously another name
moves the backing file from the previous file to the new file.
If you want to copy the previous file, use \code{copy(filename='new_filename')}.
}}

\item{\code{layers}}{A list-like object with values of the same dimensions as \code{X}.
Layers in AnnData are inspired by \href{https://linnarssonlab.org/loompy/apiwalkthrough/index.html#loomlayers}{loompy's layers}.

Overwrite the layers:\preformatted{adata$layers <- list(spliced = spliced, unspliced = unspliced)
}

Return the layer named \code{"unspliced"}:\preformatted{adata$layers["unspliced"]
}

Create or replace the \code{"spliced"} layer:\preformatted{adata$layers["spliced"] = example_matrix
}

Assign the 10th column of layer \code{"spliced"} to the variable a:\preformatted{a <- adata$layers["spliced"][, 10]
}

Delete the \code{"spliced"}:\preformatted{adata$layers["spliced"] <- NULL
}

Return layers' names:\preformatted{names(adata$layers)
}}

\item{\code{T}}{Transpose whole object.

Data matrix is transposed, observations and variables are interchanged.

Ignores \code{.raw}.}

\item{\code{is_view}}{\code{TRUE} if object is view of another AnnData object, \code{FALSE} otherwise.}

\item{\code{isbacked}}{\code{TRUE} if object is backed on disk, \code{FALSE} otherwise.}

\item{\code{n_obs}}{Number of observations.}

\item{\code{obs}}{One-dimensional annotation of observations (data.frame).}

\item{\code{obs_names}}{Names of observations.}

\item{\code{obsm}}{Multi-dimensional annotation of observations (matrix).

Stores for each key a two or higher-dimensional matrix with \code{n_obs} rows.}

\item{\code{obsp}}{Pairwise annotation of observations, a mutable mapping with array-like values.

Stores for each key a two or higher-dimensional matrix whose first two dimensions are of length \code{n_obs}.}

\item{\code{n_var}}{Number of variables.}

\item{\code{var}}{One-dimensional annotation of variables (data.frame).}

\item{\code{var_names}}{Names of variables.}

\item{\code{varm}}{Multi-dimensional annotation of variables (matrix).

Stores for each key a two or higher-dimensional matrix with \code{n_var} rows.}

\item{\code{varp}}{Pairwise annotation of variables, a mutable mapping with array-like values.

Stores for each key a two or higher-dimensional matrix whose first two dimensions are of length \code{n_var}.}

\item{\code{shape}}{Shape of data matrix (\code{n_obs}, \code{n_vars}).}

\item{\code{uns}}{Unstructured annotation (ordered dictionary).}

\item{\code{raw}}{Store raw version of \code{X} and \code{var} as \verb{$raw$X} and \verb{$raw$var}.

The \code{raw} attribute is initialized with the current content of an object
by setting:\preformatted{adata$raw = adata
}

Its content can be deleted:\preformatted{adata$raw <- NULL
}

Upon slicing an AnnData object along the obs (row) axis, \code{raw} is also
sliced. Slicing an AnnData object along the vars (columns) axis
leaves \code{raw} unaffected. Note that you can call:\preformatted{adata$raw[, 'orig_variable_name']$X
}

\verb{to retrieve the data associated with a variable that might have been filtered out or "compressed away" in}X`.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{AnnDataR6$new()}}
\item \href{#method-obs_keys}{\code{AnnDataR6$obs_keys()}}
\item \href{#method-obs_names_make_unique}{\code{AnnDataR6$obs_names_make_unique()}}
\item \href{#method-obsm_keys}{\code{AnnDataR6$obsm_keys()}}
\item \href{#method-var_keys}{\code{AnnDataR6$var_keys()}}
\item \href{#method-var_names_make_unique}{\code{AnnDataR6$var_names_make_unique()}}
\item \href{#method-varm_keys}{\code{AnnDataR6$varm_keys()}}
\item \href{#method-uns_keys}{\code{AnnDataR6$uns_keys()}}
\item \href{#method-chunk_X}{\code{AnnDataR6$chunk_X()}}
\item \href{#method-chunked_X}{\code{AnnDataR6$chunked_X()}}
\item \href{#method-concatenate}{\code{AnnDataR6$concatenate()}}
\item \href{#method-copy}{\code{AnnDataR6$copy()}}
\item \href{#method-rename_categories}{\code{AnnDataR6$rename_categories()}}
\item \href{#method-strings_to_categoricals}{\code{AnnDataR6$strings_to_categoricals()}}
\item \href{#method-to_df}{\code{AnnDataR6$to_df()}}
\item \href{#method-transpose}{\code{AnnDataR6$transpose()}}
\item \href{#method-write_csvs}{\code{AnnDataR6$write_csvs()}}
\item \href{#method-write_h5ad}{\code{AnnDataR6$write_h5ad()}}
\item \href{#method-write_loom}{\code{AnnDataR6$write_loom()}}
\item \href{#method-print}{\code{AnnDataR6$print()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Create a new AnnData object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$new(
  X = NULL,
  obs = NULL,
  var = NULL,
  uns = NULL,
  obsm = NULL,
  varm = NULL,
  layers = NULL,
  raw = NULL,
  dtype = "float32",
  shape = NULL,
  filename = NULL,
  filemode = NULL,
  obsp = NULL,
  varp = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{X}}{A #observations × #variables data matrix. A view of the data is used if the data type matches, otherwise, a copy is made.}

\item{\code{obs}}{Key-indexed one-dimensional observations annotation of length #observations.}

\item{\code{var}}{Key-indexed one-dimensional variables annotation of length #variables.}

\item{\code{uns}}{Key-indexed unstructured annotation.}

\item{\code{obsm}}{Key-indexed multi-dimensional observations annotation of length #observations. If passing a \code{~numpy.ndarray}, it needs to have a structured datatype.}

\item{\code{varm}}{Key-indexed multi-dimensional variables annotation of length #variables. If passing a \code{~numpy.ndarray}, it needs to have a structured datatype.}

\item{\code{layers}}{Key-indexed multi-dimensional arrays aligned to dimensions of \code{X}.}

\item{\code{raw}}{Store raw version of \code{X} and \code{var} as \verb{$raw$X} and \verb{$raw$var}.}

\item{\code{dtype}}{Data type used for storage.}

\item{\code{shape}}{Shape list (#observations, #variables). Can only be provided if \code{X} is \code{NULL}.}

\item{\code{filename}}{Name of backing file. See \href{https://docs.h5py.org/en/latest/high/file.html#h5py.File}{h5py.File}.}

\item{\code{filemode}}{Open mode of backing file. See \href{https://docs.h5py.org/en/latest/high/file.html#h5py.File}{h5py.File}.}

\item{\code{obsp}}{Pairwise annotation of observations, a mutable mapping with array-like values.}

\item{\code{varp}}{Pairwise annotation of observations, a mutable mapping with array-like values.}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# use AnnData() instead of AnnDataR6$new()
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2"))
)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-obs_keys"></a>}}
\if{latex}{\out{\hypertarget{method-obs_keys}{}}}
\subsection{Method \code{obs_keys()}}{
List keys of observation annotation \code{obs}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$obs_keys()}\if{html}{\out{</div>}}
}

\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2"))
)
ad$obs_keys()
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-obs_names_make_unique"></a>}}
\if{latex}{\out{\hypertarget{method-obs_names_make_unique}{}}}
\subsection{Method \code{obs_names_make_unique()}}{
Makes the index unique by appending a number string to each duplicate index element: 1, 2, etc.

If a tentative name created by the algorithm already exists in the index, it tries the next integer in the sequence.

The first occurrence of a non-unique value is ignored.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$obs_names_make_unique(join = "-")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{join}}{The connecting string between name and integer (default: \code{"-"}).}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(rep(1, 6), nrow = 3),
  obs = data.frame(field = c(1, 2, 3))
)
ad$obs_names <- c("a", "a", "b")
ad$obs_names_make_unique()
ad$obs_names
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-obsm_keys"></a>}}
\if{latex}{\out{\hypertarget{method-obsm_keys}{}}}
\subsection{Method \code{obsm_keys()}}{
List keys of observation annotation \code{obsm}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$obsm_keys()}\if{html}{\out{</div>}}
}

\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  obsm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  )
)
ad$obs_keys()
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-var_keys"></a>}}
\if{latex}{\out{\hypertarget{method-var_keys}{}}}
\subsection{Method \code{var_keys()}}{
List keys of variable annotation \code{var}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$var_keys()}\if{html}{\out{</div>}}
}

\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2"))
)
ad$var_keys()
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-var_names_make_unique"></a>}}
\if{latex}{\out{\hypertarget{method-var_names_make_unique}{}}}
\subsection{Method \code{var_names_make_unique()}}{
Makes the index unique by appending a number string to each duplicate index element: 1, 2, etc.

If a tentative name created by the algorithm already exists in the index, it tries the next integer in the sequence.

The first occurrence of a non-unique value is ignored.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$var_names_make_unique(join = "-")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{join}}{The connecting string between name and integer (default: \code{"-"}).}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(rep(1, 6), nrow = 2),
  var = data.frame(field = c(1, 2, 3))
)
ad$var_names <- c("a", "a", "b")
ad$var_names_make_unique()
ad$var_names
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-varm_keys"></a>}}
\if{latex}{\out{\hypertarget{method-varm_keys}{}}}
\subsection{Method \code{varm_keys()}}{
List keys of variable annotation \code{varm}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$varm_keys()}\if{html}{\out{</div>}}
}

\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
  varm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  )
)
ad$varm_keys()
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-uns_keys"></a>}}
\if{latex}{\out{\hypertarget{method-uns_keys}{}}}
\subsection{Method \code{uns_keys()}}{
List keys of unstructured annotation \code{uns}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$uns_keys()}\if{html}{\out{</div>}}
}

\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
  uns = list(a = 1, b = 2, c = list(c.a = 3, c.b = 4))
)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-chunk_X"></a>}}
\if{latex}{\out{\hypertarget{method-chunk_X}{}}}
\subsection{Method \code{chunk_X()}}{
Return a chunk of the data matrix \code{X} with random or specified indices.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$chunk_X(select = 1000L, replace = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{select}}{Depending on the values:
\itemize{
\item 1 integer: A random chunk with select rows will be returned.
\item multiple integers: A chunk with these indices will be returned.
}}

\item{\code{replace}}{if \code{select} is an integer then \code{TRUE} means random sampling of indices with replacement,
\code{FALSE} without replacement.}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(runif(10000), nrow = 50)
)

ad$chunk_X(select = 10L) # 10 random samples
ad$chunk_X(select = 1:3) # first 3 samples
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-chunked_X"></a>}}
\if{latex}{\out{\hypertarget{method-chunked_X}{}}}
\subsection{Method \code{chunked_X()}}{
Return an iterator over the rows of the data matrix X.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$chunked_X(chunk_size = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{chunk_size}}{Row size of a single chunk.}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(runif(10000), nrow = 50)
)
ad$chunked_X(10)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-concatenate"></a>}}
\if{latex}{\out{\hypertarget{method-concatenate}{}}}
\subsection{Method \code{concatenate()}}{
Concatenate along the observations axis.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$concatenate(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{Deprecated}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-copy"></a>}}
\if{latex}{\out{\hypertarget{method-copy}{}}}
\subsection{Method \code{copy()}}{
Full copy, optionally on disk.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$copy(filename = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{filename}}{Path to filename (default: \code{NULL}).}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2)
)
ad$copy()
ad$copy("file.h5ad")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-rename_categories"></a>}}
\if{latex}{\out{\hypertarget{method-rename_categories}{}}}
\subsection{Method \code{rename_categories()}}{
Rename categories of annotation \code{key} in \code{obs}, \code{var}, and \code{uns}.
Only supports passing a list/array-like \code{categories} argument.
Besides calling \verb{self.obs[key].cat.categories = categories} –
similar for \code{var} - this also renames categories in unstructured
annotation that uses the categorical annotation \code{key}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$rename_categories(key, categories)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{key}}{Key for observations or variables annotation.}

\item{\code{categories}}{New categories, the same number as the old categories.}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2"))
)
ad$rename_categories("group", c(a = "A", b = "B")) # ??
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-strings_to_categoricals"></a>}}
\if{latex}{\out{\hypertarget{method-strings_to_categoricals}{}}}
\subsection{Method \code{strings_to_categoricals()}}{
Transform string annotations to categoricals.

Only affects string annotations that lead to less categories than the total number of observations.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$strings_to_categoricals(df = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{df}}{If \code{df} is \code{NULL}, modifies both \code{obs} and \code{var}, otherwise modifies \code{df} inplace.}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
)
ad$strings_to_categoricals() # ??
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-to_df"></a>}}
\if{latex}{\out{\hypertarget{method-to_df}{}}}
\subsection{Method \code{to_df()}}{
Generate shallow data frame.

The data matrix \code{X} is returned as data frame, where \code{obs_names} are the rownames, and \code{var_names} the columns names.

No annotations are maintained in the returned object.

The data matrix is densified in case it is sparse.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$to_df(layer = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{layer}}{Key for layers}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
  layers = list(
    spliced = matrix(c(4, 5, 6, 7), nrow = 2),
    unspliced = matrix(c(8, 9, 10, 11), nrow = 2)
  )
)

ad$to_df()
ad$to_df("unspliced")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-transpose"></a>}}
\if{latex}{\out{\hypertarget{method-transpose}{}}}
\subsection{Method \code{transpose()}}{
transpose Transpose whole object.

Data matrix is transposed, observations and variables are interchanged.

Ignores \code{.raw}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$transpose()}\if{html}{\out{</div>}}
}

\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2"))
)

ad$transpose()
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-write_csvs"></a>}}
\if{latex}{\out{\hypertarget{method-write_csvs}{}}}
\subsection{Method \code{write_csvs()}}{
Write annotation to .csv files.

It is not possible to recover the full AnnData from these files. Use \code{\link[=write_h5ad]{write_h5ad()}} for this.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$write_csvs(dirname, skip_data = TRUE, sep = ",")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{dirname}}{Name of the directory to which to export.}

\item{\code{skip_data}}{Skip the data matrix \code{X}.}

\item{\code{sep}}{Separator for the data}

\item{\code{anndata}}{An \code{\link[=AnnData]{AnnData()}} object}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
  varm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  ),
  uns = list(a = 1, b = 2, c = list(c.a = 3, c.b = 4))
)

ad$to_write_csvs("output")

unlink("output", recursive = TRUE)
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-write_h5ad"></a>}}
\if{latex}{\out{\hypertarget{method-write_h5ad}{}}}
\subsection{Method \code{write_h5ad()}}{
Write .h5ad-formatted hdf5 file.

Generally, if you have sparse data that are stored as a dense matrix, you can
dramatically improve performance and reduce disk space by converting to a csr_matrix:
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$write_h5ad(
  filename,
  compression = NULL,
  compression_opts = NULL,
  as_dense = list()
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{filename}}{Filename of data file. Defaults to backing file.}

\item{\code{compression}}{See the h5py \href{http://docs.h5py.org/en/latest/high/dataset.html#dataset-compression}{filter pipeline}.
Options are \code{"gzip"}, \code{"lzf"} or \code{NULL}.}

\item{\code{compression_opts}}{See the h5py \href{http://docs.h5py.org/en/latest/high/dataset.html#dataset-compression}{filter pipeline}.}

\item{\code{as_dense}}{Sparse in AnnData object to write as dense. Currently only supports \code{"X"} and \code{"raw/X"}.}

\item{\code{anndata}}{An \code{\link[=AnnData]{AnnData()}} object}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
  varm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  ),
  uns = list(a = 1, b = 2, c = list(c.a = 3, c.b = 4))
)

ad$write_h5ad("output.h5ad")

file.remove("output.h5ad")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-write_loom"></a>}}
\if{latex}{\out{\hypertarget{method-write_loom}{}}}
\subsection{Method \code{write_loom()}}{
Write .loom-formatted hdf5 file.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$write_loom(filename, write_obsm_varm = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{filename}}{The filename.}

\item{\code{write_obsm_varm}}{Whether or not to also write the varm and obsm.}

\item{\code{anndata}}{An \code{\link[=AnnData]{AnnData()}} object}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
  varm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  ),
  uns = list(a = 1, b = 2, c = list(c.a = 3, c.b = 4))
)

ad$write_loom("output.loom")

file.remove("output.loom")
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-print"></a>}}
\if{latex}{\out{\hypertarget{method-print}{}}}
\subsection{Method \code{print()}}{
Print AnnData object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AnnDataR6$print(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{optional arguments to print method.}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
ad <- AnnData(
  X = matrix(c(0, 1, 2, 3), nrow = 2),
  obs = data.frame(group = c("a", "b"), row.names = c("s1", "s2")),
  var = data.frame(type = c(1L, 2L), row.names = c("var1", "var2")),
  layers = list(
    spliced = matrix(c(4, 5, 6, 7), nrow = 2),
    unspliced = matrix(c(8, 9, 10, 11), nrow = 2)
  ),
  obsm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  ),
  varm = list(
    ones = matrix(rep(1L, 10), nrow = 2),
    rand = matrix(rnorm(6), nrow = 2),
    zeros = matrix(rep(0L, 10), nrow = 2)
  ),
  uns = list(a = 1, b = 2, c = list(c.a = 3, c.b = 4))
)

ad$print()
print(ad)
}
}
\if{html}{\out{</div>}}

}

}
}
