% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayescopulaglm.R
\name{bayescopulaglm}
\alias{bayescopulaglm}
\title{Sample from Bayesian copula GLM}
\usage{
bayescopulaglm(
  formula.list,
  family.list,
  data,
  histdata = NULL,
  b0 = NULL,
  c0 = NULL,
  alpha0 = NULL,
  gamma0 = NULL,
  Gamma0 = NULL,
  S0beta = NULL,
  sigma0logphi = NULL,
  v0 = NULL,
  V0 = NULL,
  beta0 = NULL,
  phi0 = NULL,
  M = 10000,
  burnin = 2000,
  thin = 1,
  adaptive = TRUE
)
}
\arguments{
\item{formula.list}{A \eqn{J}-dimensional list of formulas giving how the endpoints are related to the covariates}

\item{family.list}{A \eqn{J}-dimensional list of families giving how each endpoint is distributed. See \code{help(family)}}

\item{data}{A \code{data frame} containing all response variables and covariates. Variables must be named.}

\item{histdata}{\emph{Optional} historical data set for power prior on \eqn{\beta, \phi}}

\item{b0}{\emph{Optional} power prior hyperparameter. Ignored if \code{is.null(histdata)}. Must be a number between \eqn{(0, 1]} if \code{histdata} is not \code{NULL}}

\item{c0}{A \eqn{J}-dimensional vector for \eqn{\beta \mid \phi} prior covariance. If \code{NULL}, sets \eqn{c_0 = 10000} for each endpoint}

\item{alpha0}{A \eqn{J}-dimensional vector giving the shape hyperparameter for each dispersion parameter on the prior on \eqn{\phi}. If \code{NULL} sets \eqn{\alpha_0 = .01} for each dispersion parameter}

\item{gamma0}{A \eqn{J}-dimensional vector giving the rate hyperparameter for each dispersion parameter on the prior on \eqn{\phi}. If \code{NULL} sets \eqn{\alpha_0 = .01} for each dispersion parameter}

\item{Gamma0}{Initial value for correlation matrix. If \code{NULL} defaults to the correlation matrix from the responses.}

\item{S0beta}{A \eqn{J}-dimensional \code{list} for the covariance matrix for random walk metropolis on beta. Each matrix must have the same dimension as the corresponding regression coefficient. If \code{NULL}, uses \code{solve(crossprod(X))}}

\item{sigma0logphi}{A \eqn{J}-dimensional \code{vector} giving the standard deviation on \eqn{\log(\phi)} for random walk metropolis. If \code{NULL} defaults to \code{0.1}}

\item{v0}{An integer scalar giving degrees of freedom for Inverse Wishart prior. If \code{NULL} defaults to J + 2}

\item{V0}{An integer giving inverse scale parameter for Inverse Wishart prior. If \code{NULL} defaults to \code{diag(.001, J)}}

\item{beta0}{A \eqn{J}-dimensional \code{list} giving starting values for random walk Metropolis on the regression coefficients. If \code{NULL}, defaults to the GLM MLE}

\item{phi0}{A \eqn{J}-dimensional \code{vector} giving initial values for dispersion parameters. If \code{NULL}. Dispersion parameters will always return 1 for binomial and Poisson models}

\item{M}{Number of desired posterior samples after burn-in and thinning}

\item{burnin}{burn-in parameter}

\item{thin}{post burn-in thinning parameter}

\item{adaptive}{logical indicating whether to use adaptive random walk MCMC to estimate parameters. This takes longer, but generally has a better acceptance rate}
}
\value{
A named list. 
\code{["betasample"]} gives a \eqn{J}-dimensional list of sampled coefficients as matrices. 
\code{["phisample"]} gives a \eqn{M \times J} matrix of sampled dispersion parameters. 
\code{["Gammasample"]} gives a \eqn{J \times J \times M} array of sampled correlation matrices. 
\code{["betaaccept"]} gives a \eqn{M \times J} matrix where each row indicates whether the proposal for the regression coefficient was accepted.
\code{["phiaccept"]} gives a \eqn{M \times J} matrix where each row indicates whether the proposal for the dispersion parameter was accepted
}
\description{
Sample from a GLM via Bayesian copula regression model.
Uses random-walk Metropolis to update regression coefficients and dispersion parameters.
Assumes Inverse Wishart prior on augmented data.
}
\examples{
set.seed(1234)
n <- 100
M <- 100

x <- runif(n, 1, 2)
y1 <- 0.25 * x + rnorm(100)
y2 <- rpois(n, exp(0.25 * x))

formula.list <- list(y1 ~ 0 + x, y2 ~ 0 + x)
family.list <- list(gaussian(), poisson())
data = data.frame(y1, y2, x)

## Perform copula regression sampling with default
## (noninformative) priors
sample <- bayescopulaglm(
  formula.list, family.list, data, M = M, burnin = 0, adaptive = F
)
## Regression coefficients
summary(do.call(cbind, sample$betasample))

## Dispersion parameters
summary(sample$phisample)

## Posterior mean correlation matrix
apply(sample$Gammasample, c(1,2), mean)

## Fraction of accepted betas
colMeans(sample$betaaccept)

## Fraction of accepted dispersion parameters
colMeans(sample$phiaccept)
}
