#' @title WWARN Parasite Clearance Estimator (PCE)
#' 
#' @description 
#' This function is a wrapper function of the WWARN PCE method to calculate the parasite clearance rates of 
#' a given data set of patient profiles. The function returns the output data frame and it also saves more comprehensive 
#' outputs under a folder named "PceEstimates" under the current working directory.
#' 
#' @param data a data frame containing the profiles of patients.
#' This data frame must contain \code{id}, \code{time}, and \code{count} columns, 
#' in that order. The first column represents the IDs of patients. 
#' The second and third columns contain parasite measurements (per microliter) in different times.
#' @param detect.limit detection limit of the parasite density in blood
#' @param outlier.detect indicator of whether or not to use Flegg's outlier detection method
#' @param ... additional parameters.
#' 
#' @export
#' 
#' @return All results are saved under a folder named "PceEstimates" under the current working directory.
#' \item{output}{Output data frame. If \code{outlier.detect = TRUE}, the cleaned data frame will be returned.}
#' 
#' @details 
#' This function gives users a way to calculate the parasite clearance rates by using the method in Flegg et al. (2011). 
#' Users can compare the results with that given by our method of Bayesian heierarchical model. The output is saved under 
#' a folder named "PceEstimates" under the current working directory. \code{data} should be a data frame in the form of the example data 
#' \code{pursat} provided in this package. \code{detect.limit} is the detection limit of the parasite density in blood. The default
#' value is set to be 15. \code{outlier.detect} is an indicator users can turn off if the dataset has already been cleaned. Otherwise,
#' it is always recommended to set \code{outlier.detect = TRUE} to let the program automatically detect outliers in the dataset.
#'  
#' @author Colin B. Fogarty <cfogarty@mit.edu>, Saeed Sharifi-Malvajerdi <saeedsh@wharton.upenn.edu>, Feiyu Zhu <feiyuzhu@sas.upenn.edu>
#' 
#' @references Flegg, J. A., Guerin, P. J., White, N. J., & Stepniewska, K. (2011). 
#' Standardizing the measurement of parasite clearance in falciparum malaria: the parasite clearance estimator. 
#' Malaria journal, 10(1), 339.
#' 
#' @examples
#' \dontshow{
#' data("pursat")
#' data = pursat[pursat["id"] <= 80 & pursat["id"] > 70,]
#' output <- calculatePCE(data = data, detect.limit = 15, outlier.detect = TRUE)
#' }
#' \donttest{
#' data("pursat")
#' output <- calculatePCE(data = pursat, detect.limit = 15, outlier.detect = TRUE)
#' }

#require(AER)
#require(MASS)

calculatePCE <- function(data, detect.limit = 15, outlier.detect = TRUE, ...) {
  
  dir <- getwd()
  if(is.null(data$Predicted))
  {
    cat("\n Calculating WWARN PCE Estimates... \n\n")
    if (!dir.exists(paste(dir, "/PceEstimates", sep="")))
    {
      dir.create(paste(dir,"/PceEstimates", sep=""))
    }
    setwd(paste(dir,"/PceEstimates", sep=""))
    suppressWarnings(
        suppressMessages(calculateParasiteClearance(Name = "pceestimates", minpara = detect.limit, data1 = data))
    )
    data = read.csv("pceestimates_Estimates.csv")
    data = as.data.frame(data)
    
    # If users want to use Flegg's method to detect outliers (outlier.detect == TRUE), then
    # after running the PCE method, we detect and exclude points with outlier status 1 and 2
    # outlier status 3 appears in a tail phase, we don't remove that
    if (outlier.detect == TRUE) {
      data = data[(data$Outlier!=1 & data$Outlier!=2), ]
    }
    
    # If users don't want to detect outliers in this case (i.e. data doesn't have the Predicted column)
    # a warning message should be reported
    if (outlier.detect == FALSE) {
      warning("You may forget to detect outliers.", 
              "\nIf outliers before the tail phase were removed from the input data by yourself, please ignore this warning.",
              "\nOtherwise, please re-run the function by setting outlier.detect = TRUE and the outliers will be automatically removed for you.")
    }
  }
  setwd(dir)
  
  # If data contains predicted PCE (generated by users or by our program) but the type 1 or type 2 outliers are not eliminated, 
  # then the program ends and produces an error message.
  if(!is.null(data$Precited) & (is.element(1, data$Outlier) | is.element(2, data$Outlier))) {
    stop("There are still outliers before the tail phase, please remove them and then predict Flegg's PCE.",
         "\nYou can choose to delete the old predicted PCE column and then run the program with outlier.detect = TRUE,",
         "\nwhich will automatically clean the data set and generate a predicated PCE column by Flegg's method.")
  }
  
  # we here only return the cleaned data object
  # All other results are saved under the folder ./PceEstimates
  cat("\n\n") # starting a new line
  print("More outputs are saved under ./PceEstimates.")
  #print("The data frame will be returned by calculatePCE. If outlier.detect = TRUE has been specified, then the returned data frame will be cleaned automatically.")
  return(data)
  
}
