% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run_mcmc.R
\name{run_mcmc.nongaussian}
\alias{run_mcmc.nongaussian}
\title{Bayesian Inference of Non-Gaussian State Space Models}
\usage{
\method{run_mcmc}{nongaussian}(
  model,
  iter,
  particles,
  output_type = "full",
  mcmc_type = "is2",
  sampling_method = "psi",
  burnin = floor(iter/2),
  thin = 1,
  gamma = 2/3,
  target_acceptance = 0.234,
  S,
  end_adaptive_phase = FALSE,
  local_approx = TRUE,
  threads = 1,
  seed = sample(.Machine$integer.max, size = 1),
  max_iter = 100,
  conv_tol = 1e-08,
  ...
)
}
\arguments{
\item{model}{Model model.}

\item{iter}{Number of MCMC iterations.}

\item{particles}{Number of state samples per MCMC iteration.
Ignored if \code{mcmc_type} is \code{"approx"}.}

\item{output_type}{Either \code{"full"} 
(default, returns posterior samples of states alpha and hyperparameters theta), 
\code{"theta"} (for marginal posterior of theta), 
or \code{"summary"} (return the mean and variance estimates of the states 
and posterior samples of theta).  In case of \code{"summary"}, means and 
covariances are computed using the full output of particle filter 
instead of sampling one of these as in case of \code{output_type = "full"}.}

\item{mcmc_type}{What MCMC algorithm to use? Possible choices are
\code{"pm"} for pseudo-marginal MCMC,
\code{"da"} for delayed acceptance version of PMCMC , 
\code{"approx"} for approximate inference based on the Gaussian approximation of the model,
or one of the three importance sampling type weighting schemes:
\code{"is3"} for simple importance sampling (weight is computed for each MCMC iteration independently),
\code{"is2"} for jump chain importance sampling type weighting (default), or
\code{"is1"} for importance sampling type weighting where the number of particles used for
weight computations is proportional to the length of the jump chain block.}

\item{sampling_method}{If \code{"psi"}, \eqn{\psi}-APF is used for state sampling
(default). If \code{"spdk"}, non-sequential importance sampling based
on Gaussian approximation is used. If \code{"bsf"}, bootstrap filter
is used.}

\item{burnin}{Length of the burn-in period which is disregarded from the
results. Defaults to \code{iter / 2}.}

\item{thin}{Thinning rate. Defaults to 1. Increase for large models in
order to save memory. For IS-corrected methods, larger
value can also be statistically more effective. 
Note: With \code{output_type = "summary"}, the thinning does not affect the computations 
of the summary statistics in case of pseudo-marginal methods.}

\item{gamma}{Tuning parameter for the adaptation of RAM algorithm. Must be
between 0 and 1 (not checked).}

\item{target_acceptance}{Target acceptance rate for MCMC. Defaults to 0.234. 
For DA-MCMC, this corresponds to first stage acceptance rate, i.e., the total acceptance 
rate will be smaller.}

\item{S}{Initial value for the lower triangular matrix of RAM
algorithm, so that the covariance matrix of the Gaussian proposal
distribution is \eqn{SS'}. Note that for some parameters 
(currently the standard deviation and dispersion parameters of bsm_ng models) the sampling
is done for transformed parameters with internal_theta = log(theta).}

\item{end_adaptive_phase}{If \code{TRUE}, S is held fixed after the burnin period. Default is \code{FALSE}.}

\item{local_approx}{If \code{TRUE} (default), Gaussian approximation needed for
importance sampling is performed at each iteration. If \code{FALSE}, approximation is updated only
once at the start of the MCMC using the initial model.}

\item{threads}{Number of threads for state simulation. The default is 1. 
Note that parallel computing is only used in the post-correction phase of IS-MCMC 
and when sampling the states in case of approximate models.}

\item{seed}{Seed for the random number generator.}

\item{max_iter}{Maximum number of iterations used in Gaussian approximation.}

\item{conv_tol}{Tolerance parameter used in Gaussian approximation.}

\item{...}{Ignored.}
}
\description{
Methods for posterior inference of states and parameters.
}
\examples{
set.seed(1)
n <- 50 
slope <- cumsum(c(0, rnorm(n - 1, sd = 0.001)))
level <- cumsum(slope + c(0, rnorm(n - 1, sd = 0.2)))
y <- rpois(n, exp(level))
poisson_model <- bsm_ng(y, 
  sd_level = halfnormal(0.01, 1), 
  sd_slope = halfnormal(0.01, 0.1), 
  P1 = diag(c(10, 0.1)), distribution = "poisson")
  
# Note small number of iterations for CRAN checks
mcmc_out <- run_mcmc(poisson_model, iter = 1000, particles = 10, 
  mcmc_type = "da")
summary(mcmc_out, what = "theta", return_se = TRUE)

set.seed(123)
n <- 50
sd_level <- 0.1
drift <- 0.01
beta <- -0.9
phi <- 5

level <- cumsum(c(5, drift + rnorm(n - 1, sd = sd_level)))
x <- 3 + (1:n) * drift + sin(1:n + runif(n, -1, 1))
y <- rnbinom(n, size = phi, mu = exp(beta * x + level))

model <- bsm_ng(y, xreg = x,
  beta = normal(0, 0, 10),
  phi = halfnormal(1, 10),
  sd_level = halfnormal(0.1, 1), 
  sd_slope = halfnormal(0.01, 0.1),
  a1 = c(0, 0), P1 = diag(c(10, 0.1)^2), 
  distribution = "negative binomial")

# run IS-MCMC
# Note small number of iterations for CRAN checks
# so no real use of parallelization
fit <- run_mcmc(model, iter = 5000,
  particles = 10, mcmc_type = "is2", seed = 1, threads = 2)

# extract states   
d_states <- as.data.frame(fit, variable = "states", time = 1:n)

library("dplyr")
library("ggplot2")

 # compute summary statistics
level_sumr <- d_states \%>\% 
  filter(variable == "level") \%>\%
  group_by(time) \%>\%
  summarise(mean = Hmisc::wtd.mean(value, weight, normwt = TRUE), 
    lwr = Hmisc::wtd.quantile(value, weight, 
      0.025, normwt = TRUE), 
    upr = Hmisc::wtd.quantile(value, weight, 
      0.975, normwt = TRUE))

# visualize
level_sumr \%>\% ggplot(aes(x = time, y = mean)) + 
  geom_line() +
  geom_line(aes(y = lwr), linetype = "dashed", na.rm = TRUE) +
  geom_line(aes(y = upr), linetype = "dashed", na.rm = TRUE) +
  theme_bw() + 
  theme(legend.title = element_blank()) + 
  xlab("Time") + ylab("Level")

# Bivariate Poisson model:

set.seed(1)
x <- cumsum(c(3, rnorm(19, sd = 0.5)))
y <- cbind(
  rpois(20, exp(x)), 
  rpois(20, exp(x)))

prior_fn <- function(theta) {
  # half-normal prior using transformation
  dnorm(exp(theta), 0, 1, log = TRUE) + theta # plus jacobian term
}

update_fn <- function(theta) {
  list(R = array(exp(theta), c(1, 1, 1)))
}

model <- ssm_mng(y = y, Z = matrix(1,2,1), T = 1, 
  R = 0.1, P1 = 1, distribution = "poisson",
  init_theta = log(0.1), 
  prior_fn = prior_fn, update_fn = update_fn)
  
# Note small number of iterations for CRAN checks
out <- run_mcmc(model, iter = 5000, mcmc_type = "approx")

sumr <- as.data.frame(out, variable = "states") \%>\% 
  group_by(time) \%>\% mutate(value = exp(value)) \%>\%
  summarise(mean = mean(value), 
    ymin = quantile(value, 0.05), ymax = quantile(value, 0.95))
ggplot(sumr, aes(time, mean)) + 
geom_ribbon(aes(ymin = ymin, ymax = ymax),alpha = 0.25) + 
geom_line() + 
geom_line(data = data.frame(mean = y[, 1], time = 1:20), colour = "tomato") + 
geom_line(data = data.frame(mean = y[, 2], time = 1:20), colour = "tomato") +
theme_bw()

}
