% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{kalman_dk}
\alias{kalman_dk}
\title{Durbin and Koopman Simulation Smoother}
\usage{
kalman_dk(y, z, sigma_u, sigma_v, B, a_init, P_init)
}
\arguments{
\item{y}{a \eqn{K \times T} matrix of endogenous variables.}

\item{z}{a \eqn{KT \times M} matrix of explanatory variables.}

\item{sigma_u}{the inverse of the constant \eqn{K \times K} error variance-covariance matrix.
For time varying variance-covariance matrices a \eqn{KT \times K} can be specified.}

\item{sigma_v}{the inverse of the constant \eqn{M \times M} coefficient variance-covariance matrix.
For time varying variance-covariance matrices a \eqn{MT \times M} can be specified.}

\item{B}{an \eqn{M \times M} autocorrelation matrix of the transition equation.}

\item{a_init}{an M-dimensional vector of initial states.}

\item{P_init}{an \eqn{M \times M} variance-covariance matrix of the initial states.}
}
\value{
A \eqn{M \times T+1} matrix of state vector draws.
}
\description{
An implementation of the Kalman filter and backward smoothing
algorithm proposed by Durbin and Koopman (2002).
}
\details{
The function uses algorithm 2 from Durbin and Koopman (2002) to produce
a draw of the state vector \eqn{a_t} for \eqn{t = 1,...,T} for a state space model
with measurement equation
\deqn{y_t = Z_t a_t + u_t}
and transition equation 
\deqn{a_{t + 1} = B_t a_{t} + v_t,}
where \eqn{u_t \sim N(0, \Sigma_{u,t})} and \eqn{v_t \sim N(0, \Sigma_{v,t})}.
\eqn{y_t} is a K-dimensional vector of endogenous variables and
\eqn{Z_t = z_t^{\prime} \otimes I_K} is a \eqn{K \times M} matrix of regressors with
\eqn{z_t} as a vector of regressors.

The algorithm takes into account Jarociński (2015), where a possible missunderstanding
in the implementation of the algorithm of Durbin and Koopman (2002) is pointed out. Following
that note the function sets the mean of the initial state to zero in the first step of the algorithm.
}
\examples{

# Prepare data
data("e1")
data <- diff(log(e1))
temp <- gen_var(data, p = 2, deterministic = "const")
y <- temp$Y
x <- temp$Z
k <- nrow(y)
z <- kronecker(t(x), diag(1, k))
t <- ncol(y)
m <- k * nrow(x)

# Priors
a_mu_prior <- matrix(0, m)
a_v_i_prior <- diag(0.1, m)

a_Q <- diag(.0001, m)

# Initial value of Sigma
sigma <- tcrossprod(y) / t
sigma_i <- solve(sigma)

# Initial values for Kalman filter
y_init <- y * 0
a_filter <- matrix(0, m, t + 1)

# Initialise the Kalman filter
for (i in 1:t) {
  y_init[, i] <- y[, i] - z[(i - 1) * k + 1:k,] \%*\% a_filter[, i]
}
a_init <- post_normal_sur(y = y_init, z = z, sigma_i = sigma_i,
                          a_prior = a_mu_prior, v_i_prior = a_v_i_prior)
y_filter <- y - matrix(a_init, k) \%*\% x

# Kalman filter and backward smoother
a_filter <- kalman_dk(y = y_filter, z = z, sigma_u = sigma,
                      sigma_v = a_Q, B = diag(1, m),
                      a_init = matrix(0, m), P_init = a_Q)
                      
a <- a_filter + matrix(a_init, m, t + 1)

}
\references{
Durbin, J., & Koopman, S. J. (2002). A simple and efficient simulation smoother for
state space time series analysis. \emph{Biometrika, 89}(3), 603--615.

Jarociński, M. (2015). A note on implementing the Durbin and Koopman simulation
smoother. \emph{Computational Statistics and Data Analysis, 91}, 1--3.
\url{https://doi.org/10.1016/j.csda.2015.05.001}
}
