% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calib_pv.R
\name{calib_pv}
\alias{calib_pv}
\title{Estimate calibration curves for a multistate model using pseudo-values.}
\usage{
calib_pv(
  data.mstate,
  data.raw,
  j,
  s,
  t,
  tp.pred,
  curve.type = "rcs",
  rcs.nk = 3,
  loess.span = 0.75,
  loess.degree = 2,
  group.vars = NULL,
  n.pctls = NULL,
  CI = FALSE,
  CI.type = "parametric",
  CI.R.boot = NULL,
  data.pred.plot = NULL,
  transitions.out = NULL
)
}
\arguments{
\item{data.mstate}{Validation data in \code{msdata} format}

\item{data.raw}{Validation data in \code{data.frame} (one row per individual)}

\item{j}{Landmark state at which predictions were made}

\item{s}{Landmark time at which predictions were made}

\item{t}{Follow up time at which calibration is to be assessed}

\item{tp.pred}{Matrix of predicted transition probabilities at time t, if in state j at time s. There must be a seperate column for the predicted transition probabilities into every state, even if these predicted transition probabilities are 0.}

\item{curve.type}{Whether calibration curves are estimated using restricted cubic splines ('rcs') or loess smoothers ('loess')}

\item{rcs.nk}{Number of knots when curves are estimated using restricted cubic splines}

\item{loess.span}{Span when curves are estimated using loess smoothers}

\item{loess.degree}{Degree when curves are estimated. using loess smoothers}

\item{group.vars}{Baseline variables to define groups within which to estimate pseudo-values}

\item{n.pctls}{Number of percentiles to group individuals by with respect to predicted transition probabilities when estimating pseudo-values}

\item{CI}{Size of confidence intervals as a \%}

\item{CI.type}{Method for estimating confidence interval (\code{bootstrap} or \code{parametric})}

\item{CI.R.boot}{Number of bootstrap replicates when estimating the confidence interval for the calibration curve using bootstrapping}

\item{data.pred.plot}{Data frame or matrix of predicted risks for each possible transition over which to plot the calibration curves. Must have one column for every possible transition.}

\item{transitions.out}{Transitions for which to calculate calibration curves. Will do all possible transitions if left as NULL.}
}
\value{
\code{\link{calib_pv}} returns a list containing two elements:
\code{plotdata} and \code{metadata}. The \code{plotdata} element contains the
data for the calibration curves. This will itself be a list with each element
containing calibration plot data for the transition probabilities into each of the possible
states. Each list element contains patient ids (\code{id}) from \code{data.raw}, the predicted
transition probabilities (\code{pred}) and the estimated observed event
probabilities (\code{obs}). If a confidence interval is requested, upper (\code{obs.upper})
and lower (\code{obs.lower}) bounds for the observed event probabilities are also returned.
If data.pred.plot is defined manually, column (\code{id}) is not returned.
The \code{metadata} element contains metadata including: a vector of the possible transitions,
a vector of which transitions calibration curves have been estimated for, the
size of the confidence interval, the method for estimating the calibration curve
and other user specified information.
}
\description{
Creates the underlying data for the calibration curves. \code{calib_pv}
estimates the
observed event probabilities for a given set of predicted transition probabilities
in a cohort of interest. This is done using techniques for assessing calibration of binary logistic regression models,
in combination with inverse probability of censoring weights and landmarking.
}
\details{
Observed event probabilities at time \code{t} are estimated for predicted
transition probabilities \code{tp.pred} out of state \code{j} at time \code{s}.
\code{calib_pv} estimates the observed event probabilities using pseudo-values (Andersen PK, Pohar Perme M, 2010)
calculated using the Aalen-Johansen estimator (Aalen OO, Johansen S, 1978)
Calibration curves are generated by regressing the pseudo-values on the predicted transition probabilities.
Currently calibration curves can be produced using loess smoothers or restricted cubic splines.
This will be updated to include restricted cubic splines. Landmarking (van Houwelingen HC, 2007) is applied to only assess calibration
in individuals who are uncensored and in state \code{j} at time \code{s}.

Two datasets for the same cohort of inidividuals must be provided. Firstly \code{data.mstate} must be a dataset of class \code{msdata},
generated using the \code{[mstate]} package. This dataset is used to apply the landmarking. Secondly, \code{data.raw} must be
a \code{data.frame} with one row per individual, containing the desired variables for
calculating pseudo-values within (no baseline variables required if \code{group.vars = NULL}).
Confidence intervals for the calibration curves can be estimated using bootstrapping.

The calibration curves can be plotted using \code{\link{plot.calib_pv}}.
}
\examples{
# Using competing risks data out of initial state.
# See vignette: comparison-with-graphical-calibration-curves-in-competing-risk-setting.
# Estimate pseudo-value calibration curves for the predicted transition
# probabilities at time t = 1826, when predictions were made at time
# s = 0 in state j = 1. These predicted transition probabilities are stored in tp.cmprsk.j0.

# To minimise example time we reduce the datasets to 50 individuals.
# Extract the predicted transition probabilities out of state j = 1 for first 50 individuals
tp.pred <- tp.cmprsk.j0 |>
 dplyr::filter(id \%in\% 1:50) |>
 dplyr::select(any_of(paste("pstate", 1:6, sep = "")))
# Reduce ebmtcal to first 50 individuals
ebmtcal <- ebmtcal |> dplyr::filter(id \%in\% 1:50)
# Reduce msebmtcal.cmprsk to first 50 individuals
msebmtcal.cmprsk <- msebmtcal.cmprsk |> dplyr::filter(id \%in\% 1:50)

# Now estimate the observed event probabilities for each possible transition.
dat.calib.pv <- calib_pv(data.mstate = msebmtcal.cmprsk,
  data.raw = ebmtcal,
  j = 1,
  s = 0,
  t = 1826,
  tp.pred = tp.pred,
  curve.type = "loess",
  loess.span = 1,
  loess.degree = 1)

# The data for each calibration curve are stored in the "plotdata" list
# element.
str(dat.calib.pv)

}
\references{
Aalen OO, Johansen S. An Empirical Transition Matrix for Non-Homogeneous Markov Chains Based on Censored Observations.
\emph{Scand J Stat}. 1978;5(3):141-150.

Andersen PK, Pohar Perme M. Pseudo-observations in survival analysis.
\emph{Stat Methods Med Res}. 2010;19(1):71-99. doi:10.1177/0962280209105020

van Houwelingen HC (2007). “Dynamic Prediction by Landmarking in Event History Analysis.”
\emph{Scandinavian Journal of Statistics}, 34(1), 70–85.
}
