\name{simulate_mk_model}
\alias{simulate_mk_model}
\title{
Simulate an Mk model for discrete trait evolution.
}
\description{
Given a rooted phylogenetic tree, a fixed-rates continuous-time Markov model for the evolution of a discrete trait ("Mk model", described by a transition matrix) and a probability vector for the root, simulate random outcomes of the model on all nodes and/or tips of the tree. The function traverses nodes from root to tips and randomly assigns a state to each node or tip based on its parent's previously assigned state and the specified transition rates between states. The generated states have joint distributions consistent with the Markov model. Optionally, multiple independent simulations can be performed using the same model.
}
\usage{
simulate_mk_model(tree, Q, root_probabilities="stationary",
                  include_tips=TRUE, include_nodes=TRUE, 
                  Nsimulations=1, drop_dims=TRUE)
}
\arguments{
\item{tree}{
A rooted tree of class "phylo". The root is assumed to be the unique node with no incoming edge.
}
\item{Q}{
A numeric matrix of size Nstates x Nstates, storing the transition rates between states. In particular, every row must sum up to zero.
}
\item{root_probabilities}{
Probabilities of the different states at the root.
Either a character vector with value "stationary" or "flat", or a numeric vector of length Nstates, where Nstates is the number of possible states of the trait. In the later case, \code{root_probabilities} must be a valid probability vector, i.e. with non-negative values summing up to 1. "stationary" sets the probabilities at the root to the stationary distribution of \code{Q} (see \code{\link{get_stationary_distribution}}), while "flat" means that each state is equally probable at the root.
}
\item{include_tips}{
Include random states for the tips. If \code{FALSE}, no states will be returned for tips.
}
\item{include_nodes}{
Include random states for the nodes. If \code{FALSE}, no states will be returned for nodes.
}
\item{Nsimulations}{
Number of random independent simulations to perform. For each node and/or tip, there will be \code{Nsimulations} random states generated.
}
\item{drop_dims}{
Logical, specifying whether the returned \code{tip_states} and \code{node_states} (see below) should be vectors, if \code{Nsimulations==1}. If \code{drop_dims==FALSE}, then \code{tip_states} and \code{tip_nodes} will always be 2D matrices.
}
}


\details{
For this function, the trait's states must be represented by integers within 1,..,Nstates, where Nstates is the total number of possible states. If the states are originally in some other format (e.g. characters or factors), you should map them to a set of integers 1,..,Nstates. These integers should correspond to row & column indices in the transition matrix \code{Q}. You can easily map any set of discrete states to integers using the function \code{\link{map_to_state_space}}.

If \code{tree$edge.length} is missing, each edge in the tree is assumed to have length 1. The tree may include multi-furcations (i.e. nodes with more than 2 children) as well as mono-furcations (i.e. nodes with only one child). The time required per simulation decreases with the total number of requested simulations.
}


\value{
A list with the following elements:
\item{tip_states}{
Either \code{NULL} (if \code{include_tips==FALSE}), or a 2D integer matrix of size Nsimulations x Ntips with values in 1,..,Nstates, where Ntips is the number of tips in the tree and Nstates is the number of possible states of the trait. The [r,c]-th entry of this matrix will be the state of tip c generated by the r-th simulation. If \code{drop_dims==TRUE} and \code{Nsimulations==1}, then \code{tip_states} will be a vector.
}
\item{node_states}{
Either \code{NULL} (if \code{include_nodes==FALSE}), or a 2D integer matrix of size Nsimulations x Nnodes with values in 1,..,Nstates, where Nnodes is the number of nodes in the tree. The [r,c]-th entry of this matrix will be the state of node c generated by the r-th simulation. If \code{drop_dims==TRUE} and \code{Nsimulations==1}, then \code{node_states} will be a vector.
}
}

\author{Stilianos Louca}

%\references{
%}

\seealso{
\code{\link{exponentiate_matrix}}, \code{\link{get_stationary_distribution}}, 
\code{\link{simulate_bm_model}}, \code{\link{simulate_ou_model}}, \code{\link{simulate_rou_model}}
}

\examples{\dontrun{
# generate a random tree
tree = generate_random_tree(list(birth_rate_intercept=1),max_tips=1000)$tree

# simulate discrete trait evolution on the tree (5 states)
Nstates = 5
Q = get_random_mk_transition_matrix(Nstates, rate_model="ARD", max_rate=0.1)
tip_states = simulate_mk_model(tree, Q)$tip_states

# plot histogram of simulated tip states
barplot(table(tip_states)/length(tip_states), xlab="state")
}}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{Mk model}
\keyword{random}
\keyword{simulation}