% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rainfall.R
\name{rainfall}
\alias{rainfall}
\title{Rainfall indices}
\usage{
rainfall(
  object,
  day.one = NULL,
  span = 150,
  timeseries = FALSE,
  intervals = 5,
  ...
)
}
\arguments{
\item{object}{a numeric vector of geographic coordinates (lonlat) or
a matrix containing the precipitation data.
When lonlat is used, the function makes a call to
\code{nasapower::get_power} to fetch and concatenate environmental 
data from NASA POWER (\url{https://power.larc.nasa.gov/})
for the parameter PRECTOT (precipitation)}

\item{day.one}{a vector of class \code{Date} for the starting date to 
capture the environmental data (YYYY-MM-DD)}

\item{span}{an integer or a vector with integers for the duration 
of the timespan to be captured}

\item{timeseries}{logical, \code{FALSE} for a single point time series
observation or \code{TRUE} for a time series based on \var{intervals}}

\item{intervals}{integer no lower than 5, for the days intervals when
\var{timeseries} = \code{TRUE}}

\item{...}{additional arguments passed to 
\code{\link[nasapower]{get_power}} methods}
}
\value{
A dataframe with rainfall indices:
\item{MLDS}{maximum length of consecutive dry day, rain < 1 mm (days)}
\item{MLWS}{maximum length of consecutive wet days, rain >= 1 mm (days)}
\item{R10mm}{number of heavy precipitation days 10 >= rain < 20 mm (days)}
\item{R20mm}{number of very heavy precipitation days rain >= 20 (days)}
\item{Rx1day}{maximum 1-day precipitation (mm)}
\item{Rx5day}{maximum 5-day precipitation (mm)}
\item{R95p}{total precipitation when rain > 95th percentile (mm)}
\item{R99p}{total precipitation when rain > 99th percentile (mm)}
\item{Rtotal}{total precipitation (mm) in wet days, rain >= 1 (mm)}
\item{SDII}{simple daily intensity index, total precipitation divided by the
 number of wet days (mm/days)}
 
 When \var{timeseries} = \code{TRUE}, an id is created, 
 which is the index for the rownames of the provided \var{object}.
}
\description{
Compute rainfall indices over a timespan
}
\examples{
# Using local sources
data("chirp", package = "climatrends")

day <- as.Date("2013-10-28", format = "\%Y-\%m-\%d")

rainfall(chirp,
         day.one = day,
         span = 11)

\donttest{
# Using NASA POWER
library("nasapower")

# random points within bbox(11, 12, 55, 58)
set.seed(123)
lonlat <- data.frame(lon = runif(3, 11, 12),
                     lat = runif(3, 55, 58))

# random dates within 2018-05-15 and 2018-05-20
set.seed(321)
dates <- as.integer(runif(3, 17666, 17670))
dates <- as.Date(dates, origin = "1970-01-01")

# calculate rainfall for the first 50 days after day.one
rainfall(lonlat,
         day.one = dates,
         span = 50)


# include the first 15 days before day.one (residual precipitation)
rainfall(lonlat,
         day.one = dates,
         span = 50,
         days.before = 15)

# rainfall indices over a time series          
rainfall(lonlat,
         day.one = dates,
         span = 50,
         timeseries = TRUE,
         intervals = 7)
}       
         
}
\references{
Aguilar E., et al. (2005). Journal of Geophysical Research, 
110(D23), D23107. \cr\url{https://doi.org/10.1029/2005JD006119}

Sparks A. H. (2018). Journal of Open Source Software, 3(30), 1035. 
\cr\url{https://doi.org/10.21105/joss.01035}
}
\seealso{
Other climatology functions: 
\code{\link{ETo}()},
\code{\link{GDD}()},
\code{\link{temperature}()}
}
\concept{climatology functions}
