#' Recursively collapse a distance matrix by amalgamating populations
#'
#' This script generates a fixed difference matrix from a genlight object \{adegenet\} and 
#' amalgamates populations with a fixed difference count less than or equal to a specified 
#' threshold, tpop. The parameter tpop is used to generate amalgamations based on absolute absence
#' of fixed differences (tpop=0), corroborated fixed differences (tpop=1, recommended), or a 
#' higher level of corroboration after examining the distribution of fixed differences across 
#' the dataset.
#' 
#' The distance matricies are generated by gl.fixed.diff(). The script gl.collapse() examines
#' the distance matrix of fixed differences and amalgamates populations with fixed differences
#' of zero. This is done iteratively until no further amalgamations are possible. If
#' tpop is greater than zero, the script then steps up to tpop=1, that is, examines
#' the distance matrix of fixed differences and amalgamates populations with fixed differences
#' of one or less until no further amalgamations are possible. And so on, until the user specified
#' value of tpop is reached.
#'
#' A final table is generated with no fixed differences less than or equal to the specified 
#' threshold, tpop, and the genlight object is recoded with the new population assignments 
#' arising from the amalgamations. 
#' 
#' @param x -- name of the genlight object from which the distance matricies are to be calculated [required]
#' @param tloc -- threshold defining a fixed difference (e.g. 0.05 implies 95:5 vs 5:95 is fixed) [default 0]
#' @param tpop -- max number of fixed differences allowed in amalgamating populations [default 0]
#' @param verbose -- verbosity: 0, silent or fatal errors; 1, begin and end; 2, progress log ; 3, progress and results summary; 5, full report [default 2 or as specified using gl.set.verbosity]
#' @return A list containing the final gl object and the following square matricies
#'         [[1]] $gl -- the input genlight object;
#'         [[2]] $fd -- raw fixed differences;
#'         [[3]] $pcfd -- percent fixed differences;
#'         [[4]] $nobs -- mean no. of individuals used in each comparison;
#'         [[5]] $nloc -- total number of loci used in each comparison;
#'         [[6]] $expfpos -- NA's, populated if required by gl.fixed.diff setting test=T [by simulation];
#'         [[7]] $sdfpos -- NA's, populated if required by gl.fixed.diff setting test=T [by simulation];
#'         [[8]] $prob -- NA's, populated by gl.fixed.diff setting test=T [by simulation]
#' @export
#' @author Arthur Georges (Post to \url{https://groups.google.com/d/forum/dartr})
#' @examples
#' fd <- gl.collapse.recursive(testset.gl, tloc=0, tpop=2, verbose=3)


gl.collapse.recursive <- function(x,tloc=0,tpop=1,verbose=NULL) {
  
# TRAP COMMAND, SET VERSION
  
  funname <- match.call()[[1]]
  build <- "Jacob"
  
# SET VERBOSITY
  
  if (is.null(verbose)){ 
    if(!is.null(x@other$verbose)){ 
      verbose <- x@other$verbose
    } else { 
      verbose <- 2
    }
  } 
  
  if (verbose < 0 | verbose > 5){
    cat(paste("  Warning: Parameter 'verbose' must be an integer between 0 [silent] and 5 [full report], set to 2\n"))
    verbose <- 2
  }
  
# FLAG SCRIPT START
  
  if (verbose >= 1){
    if(verbose==5){
      cat("Starting",funname,"[ Build =",build,"]\n")
    } else {
      cat("Starting",funname,"\n")
    }
  }
  
# STANDARD ERROR CHECKING
  
  if(class(x)!="genlight") {
    stop("Fatal Error: genlight object required!\n")
  }
  
  if (all(x@ploidy == 1)){
    if (verbose >= 2){cat("  Processing  Presence/Absence (SilicoDArT) data\n")}
    data.type <- "SilicoDArT"
  } else if (all(x@ploidy == 2)){
    if (verbose >= 2){cat("  Processing a SNP dataset\n")}
    data.type <- "SNP"
  } else {
    stop("Fatal Error: Ploidy must be universally 1 (fragment P/A data) or 2 (SNP data)")
  }
  
# SCRIPT SPECIFIC ERROR CHECKING
  
  tpop <- as.integer(tpop)
  if (tpop < 0 || tpop > nPop(x)) {
    stop("  Fatal Error: Parameter tpop must be between 0 and",nPop(x),"typically small (e.g. 3)\n")
  }

  if (tloc < 0 || tloc > 1){
    stop("  Fatal Error: Parameter tloc must be between 0 and 1, typically small (e.g. 0.05)\n")
  }
  verbose <- as.integer(verbose)
  if (verbose < 0 || verbose > 5){
    stop("  Fatal Error: Parameter verbose must be between 0 and 5\n")
  }
  
# DO THE JOB
  
  if (verbose >= 2) {
    cat("  Recursively amalgamating populations with",tpop,"or fewer fixed differences\n")
    cat("  Calculating initial fixed difference matrix\n\n")
  }
  
# DO FOR EACH VALUE OF TPOP
  
  # Create the initial distance matrix
  fd <- gl.fixed.diff(x, test=FALSE, tloc=tloc, verbose=0)
  
  # Store the length of the fd matrix
  fd.hold <- dim(fd$fd)[1]
  
  for (i in 0:tpop){
    
   if( verbose >= 2 ){
      cat(paste("\nSetting tpop to", i,"\n"))
    }
  
# Set the iteration counter
  count <- 0

# Repeat until no change to the fixed difference matrix
  repeat {
    count <- count + 1
    if( verbose >= 2 ){
      cat(paste("Iteration:", count,"\n"))
    }
    # if (verbose >= 2) {
    #   cat("  Collapsing the last supplied fixed difference matrix\n\n")
    # }
    
    # Collapse the matrix
      fdcoll <- gl.collapse(fd, tpop=i, tloc=tloc, verbose=verbose)

      if (nPop(x) == 1)  {
        cat("Warning: All populations amalgamated to one on iteration",count,"\n")
        break
      }
      
    # If not different in dimensions from previous, break
      if (dim(fdcoll$fd)[1] == fd.hold) {
        if(verbose >= 2) {cat(paste("\n  No further amalgamation of populations on iteration",count,"\n"))}
        break
      }
      
    # Hold the dimensions of the new fixed difference matrix
      fd.hold <- dim(fdcoll$fd)[1]
      fd <- fdcoll
      
  } #end repeat
  } #end major for loop, by tpop
  
  if (verbose >= 3) {
    if (tloc == 0 ){
      cat("    Using absolute fixed differences\n")
    } else {  
      cat("    Using fixed differences defined with tolerance",tloc,"\n")
    }   
    cat("    Number of fixed differences allowing population amalgamation fd <=",tpop,"(",round((tpop*100/nLoc(x)),4),"%)\n")
  }
  
  # Final populations
  fd <- fdcoll
  table(pop(fdcoll$gl))

  l <- list(gl=fd$gl,fd=fd$fd,pcfd=fd$pcfd,nobs=fd$nobs,nloc=fd$nloc,expfpos=fd$expfpos,sdfpos=fd$sdfpos,pval=fd$pval)
  class(l) <- "fd"
  
  # Return the matricies
  if (verbose >= 4) {
    cat("Returning a list of class 'fd' containing the following square matricies:\n",
        "         [[1]] $gl -- input genlight object;\n",
        "         [[2]] $fd -- raw fixed differences;\n",
        "         [[3]] $pcfd -- percent fixed differences;\n",
        "         [[4]] $nobs -- mean no. of individuals used in each comparison;\n",
        "         [[5]] $nloc -- total number of loci used in each comparison;\n",
        "         [[6]] $expfpos -- if test=TRUE, the expected count of false positives for each comparison [by simulation]\n",
        "         [[6]] $sdfpos -- if test=TRUE, the standard deviation of the count of false positives for each comparison [by simulation]\n",
        "         [[7]] $prob -- if test=TRUE, the significance of the observed count of fixed differences [by simulation]\n")
  }
  
# FLAG SCRIPT END
  
  if (verbose > 0) {
    cat("\nCompleted:",funname,"\n")
  }
  
  return(l)
}

