#' Request creation of a transferable model
#'
#' Requests generation of an transferable model file for use in an on-premise
#' DataRobot standalone prediction environment. This function can only be used
#' if model export is enabled, and will only be useful if you have an on-premise
#' environment in which to import it.
#'
#' This function does not download the exported file. Use
#' `DownloadTransferableModel` for that.
#' @inheritParams DeleteProject
#' @param modelId numeric. Unique alphanumeric identifier for the model of interest.
#' @param predictionIntervalSize integer. Optional. Added in 2.19. For supervised
#'   time series projects, this is the desired prediction interval size for the
#'   exported model. A prediction interval is the range of values DataRobot expects
#'   actual values of the target to fall within 0 to 100 (inclusive).
#' @return jobId
#' @examples
#' \dontrun{
#'   projectId <- "59a5af20c80891534e3c2bde"
#'   modelId <- "5996f820af07fc605e81ead4"
#'   jobId <- RequestTransferableModel(projectId,
#'                                     modelId,
#'                                     50)
#'   WaitForJobToComplete(projectId, jobId)
#'   file <- file.path(tempdir(), "model.drmodel")
#'   DownloadTransferableModel(projectObject, modelId, file)
#' }
#' @family Transferable Model functions
#' @md
#' @export
RequestTransferableModel <- function(project, modelId, predictionIntervalSize = NULL) {
  projectId <- ValidateProject(project)
  routeString <- "modelExports"
  body <- list(projectId = projectId,
               modelId = modelId)
  body$percentile <- as.integer(predictionIntervalSize)
  postResponse <- DataRobotPOST(routeString, body = body, returnRawResponse = TRUE)
  routeString <- UrlJoin("projects", projectId, "jobs", JobIdFromResponse(postResponse))
  jobsResponse <- DataRobotGET(routeString, simplifyDataFrame = FALSE)
  return(jobsResponse$id)
}


#' Download an transferable model file for use in an on-premise DataRobot standalone
#' prediction environment.
#'
#' This function can only be used if model export is enabled, and will only be useful
#' if you have an on-premise environment in which to import it.
#' @inheritParams DeleteProject
#' @param modelId numeric. Unique alphanumeric identifier for the model of interest.
#' @param modelFile character. File name to be use for transferable model
#' @return NULL
#' @examples
#' \dontrun{
#'   projectId <- "59a5af20c80891534e3c2bde"
#'   modelId <- "5996f820af07fc605e81ead4"
#'   file <- file.path(tempdir(), "model.drmodel")
#'   DownloadTransferableModel(projectId, modelId, file)
#' }
#' @family Transferable Model functions
#' @export
DownloadTransferableModel <- function(project, modelId, modelFile) {
  projectId <- ValidateProject(project)
  routeString <- UrlJoin("projects", projectId, "models", modelId, "export")
  response <- DataRobotGET(routeString, as = "file", filename = modelFile)
  invisible(NULL)
}


#' Import a previously exported model for predictions.
#' @param  modelFile character. Path to binary transferable model file.
#' @param maxWait integer. Specifies how many seconds to wait for upload to finish.
#' @return A list describing uploaded transferable model
#' with the following components:
#' \itemize{
#'   \item note. Character string Manually added node about this imported model.
#'   \item datasetName. Character string Filename of the dataset used to create the project the
#'     model belonged to.
#'   \item modelName. Character string Model type describing the model generated by DataRobot.
#'   \item displayName. Character string Manually specified human-readable name of the imported
#'     model.
#'   \item target. Character string The target of the project the model belonged to prior to export.
#'   \item projectName. Character string Name of the project the model belonged to prior to export.
#'   \item importedByUsername. Character string Username of the user who imported the model.
#'   \item importedAt. Character string The time the model was imported.
#'   \item version. Numeric Project version of the project the model belonged to.
#'   \item projectId. Character id of the project the model belonged to prior to export.
#'   \item featurelistName. Character string Name of the featurelist used to train the model.
#'   \item createdByUsername. Character string Username of the user who created the model prior to
#'     export.
#'   \item importedById. Character string id of the user who imported the model.
#'   \item id. Character string id of the import.
#'   \item createdById. Character string id of the user who created the model prior to export.
#'   \item modelId. Character string original id of the model prior to export.
#'   \item originUrl. Character string URL.
#' }
#' @examples
#' \dontrun{
#'   UploadTransferableModel("model.drmodel")
#' }
#' @family Transferable Model functions
#' @export
UploadTransferableModel <- function(modelFile, maxWait = 600) {
  dataList <- list(file = httr::upload_file(modelFile), name = basename(modelFile))
  routeString <- "importedModels/"
  postResponse <- DataRobotPOST(routeString, body = dataList,
                                returnRawResponse = TRUE, timeout = maxWait)
  modelInfo <- WaitForAsyncReturn(GetRedirectFromResponse(postResponse),
                                  addUrl = FALSE,
                                  maxWait = maxWait,
                                  failureStatuses = "ERROR")
  as.dataRobotTransferableModel(modelInfo)
}


#' Retrieve imported model info using import id
#' @param importId character. Id of the import.
#' @return A list describing uploaded transferable model
#' with the following components:
#' \itemize{
#'   \item note. Character string Manually added node about this imported model.
#'   \item datasetName. Character string Filename of the dataset used to create the project the
#'     model belonged to.
#'   \item modelName. Character string Model type describing the model generated by DataRobot.
#'   \item displayName. Character string Manually specified human-readable name of the imported
#'     model.
#'   \item target. Character string The target of the project the model belonged to prior to export.
#'   \item projectName. Character string Name of the project the model belonged to prior to export.
#'   \item importedByUsername. Character string Username of the user who imported the model.
#'   \item importedAt. Character string The time the model was imported.
#'   \item version. Numeric Project version of the project the model belonged to.
#'   \item projectId. Character id of the project the model belonged to prior to export.
#'   \item featurelistName. Character string Name of the featurelist used to train the model.
#'   \item createdByUsername. Character string Username of the user who created the model prior to
#'     export.
#'   \item importedById. Character string id of the user who imported the model.
#'   \item id. Character string id of the import.
#'   \item createdById. Character string id of the user who created the model prior to export.
#'   \item modelId. Character string original id of the model prior to export.
#'   \item originUrl. Character string URL.
#' }
#' @examples
#' \dontrun{
#'   id <- UploadTransferableModel("model.drmodel")
#'   GetTransferableModel(id)
#' }
#' @family Transferable Model functions
#' @export
GetTransferableModel <- function(importId) {
  routeString <- UrlJoin("importedModels", importId)
  modelInfo <- DataRobotGET(routeString)
  as.dataRobotTransferableModel(modelInfo)
}


#' Retrieve information about all imported models
#' This function returns a data.frame that describes all imported models
#' @param limit integer. The number of records to return. The server will use a (possibly finite)
#'   default if not specified.
#' @param offset integer. The number of records to skip.
#' @return A data.frame describing uploaded transferable model
#' with the following components:
#' \itemize{
#'   \item note. Character string Manually added node about this imported model.
#'   \item datasetName. Character string Filename of the dataset used to create the project the
#'     model belonged to.
#'   \item modelName. Character string Model type describing the model generated by DataRobot.
#'   \item displayName. Character string Manually specified human-readable name of the imported
#'     model.
#'   \item target. Character string The target of the project the model belonged to prior to export.
#'   \item projectName. Character string Name of the project the model belonged to prior to export.
#'   \item importedByUsername. Character string Username of the user who imported the model.
#'   \item importedAt. Character string The time the model was imported.
#'   \item version. Numeric Project version of the project the model belonged to.
#'   \item projectId. Character id of the project the model belonged to prior to export.
#'   \item featurelistName. Character string Name of the featurelist used to train the model.
#'   \item createdByUsername. Character string Username of the user who created the model prior to
#'     export.
#'   \item importedById. Character string id of the user who imported the model.
#'   \item id. Character string id of the import.
#'   \item createdById. Character string id of the user who created the model prior to export.
#'   \item modelId. Character string original id of the model prior to export.
#'   \item originUrl. Character string URL.
#' }
#' @examples
#' \dontrun{
#'   ListTransferableModels()
#' }
#' @family Transferable Model functions
#' @export
ListTransferableModels <- function(limit = NULL, offset = NULL) {
  routeString <- "importedModels/"
  body <- list(limit = limit, offset = offset)
  modelsInfo <- GetServerDataInRows(DataRobotGET(routeString, body = body))
  as.dataRobotTransferableModel(modelsInfo)
}


#' Update the display name or note for an imported model.
#'
#' @inheritParams GetTransferableModel
#' @param displayName character. The new display name.
#' @param note character. The new note.
#' @return A list describing uploaded transferable model
#' with the following components:
#' \itemize{
#'   \item note. Character string Manually added node about this imported model.
#'   \item datasetName. Character string Filename of the dataset used to create the project the
#'     model belonged to.
#'   \item modelName. Character string Model type describing the model generated by DataRobot.
#'   \item displayName. Character string Manually specified human-readable name of the imported
#'     model.
#'   \item target. Character string The target of the project the model belonged to prior to export.
#'   \item projectName. Character string Name of the project the model belonged to prior to export.
#'   \item importedByUsername. Character string Username of the user who imported the model.
#'   \item importedAt. Character string The time the model was imported.
#'   \item version. Numeric Project version of the project the model belonged to.
#'   \item projectId. Character id of the project the model belonged to prior to export.
#'   \item featurelistName. Character string Name of the featurelist used to train the model.
#'   \item createdByUsername. Character string Username of the user who created the model prior to
#'     export.
#'   \item importedById. Character string id of the user who imported the model.
#'   \item id. Character string id of the import.
#'   \item createdById. Character string id of the user who created the model prior to export.
#'   \item modelId. Character string original id of the model prior to export.
#'   \item originUrl. Character string URL.
#' }
#' @examples
#' \dontrun{
#'   id <- UploadTransferableModel("model.drmodel")
#'   UpdateTransferableModel(id, displayName = "NewName", note = "This is my note.")
#' }
#' @family Transferable Model functions
#' @export
UpdateTransferableModel <- function(importId, displayName = NULL, note = NULL) {
  if (!is.null(displayName) || !is.null(note)) {
    routeString <- UrlJoin("importedModels",  importId)
    body <- list(displayName = displayName, note = note)
    DataRobotPATCH(routeString, body = body)
  }
  GetTransferableModel(importId)
}


as.dataRobotTransferableModel <- function(inList) {
  elements <- c("note",
                "datasetName",
                "modelName",
                "displayName",
                "target",
                "projectName",
                "importedByUsername",
                "importedAt",
                "version",
                "projectId",
                "featurelistName",
                "createdByUsername",
                "importedById",
                "id",
                "createdById",
                "modelId",
                "originUrl")
  ApplySchema(inList, elements)
}


#' Delete this imported model.
#'
#' @inheritParams GetTransferableModel
#' @return NULL
#' @examples
#' \dontrun{
#'   id <- UploadTransferableModel("model.drmodel")
#'   DeleteTransferableModel(id)
#' }
#' @family Transferable Model functions
#' @export
DeleteTransferableModel <- function(importId) {
  model <- GetTransferableModel(importId)
  print(model)
  routeString <- UrlJoin("importedModels",  importId)
  response <- DataRobotDELETE(routeString)
  message(paste("Model", model$modelName,
                "(import Id = ", model$id, ") deleted from prediction server"))
  invisible(NULL)
}
