#' Put metadata
#'
#' Write versioned dataset metadata to a destination.
#'
#' @param metadata a metadata list generated by generate_metadata
#' @param destination a local directory path or an arrow SubTreeFileSystem
#' @param verbose TRUE /FALSE should the function be chatty?
#'
#' @return TRUE
#' @importFrom purrr map
#' @importFrom arrow write_csv_arrow
#' @export
#'
#' @examples
#' temp_dir <- tempfile()
#' dir.create(temp_dir, recursive = TRUE)
#' df <- data.frame(a = 1:5, b = letters[1:5])
#' md <- generate_metadata(df, temp_dir)
#'
#' put_metadata(md, temp_dir)
#' unlink(temp_dir)
#'
put_metadata <- function(metadata,
                         destination,
                         verbose = FALSE) {
  put_path <- fix_path("metadata.csv", destination, verbose = verbose)

  make_prefix(destination)

  schema_check <-
    all(c("schema_ver", "base_path", "diffed", "backup_count") %in% names(unlist(metadata)))

  if (!(schema_check)) {
    stop(
      "metadata schema check failed. Please pass a valid metadata list, as generated by generate_metadata()"
    )
  }

  metadata_string <- purrr::map(metadata, paste, collapse = "|")
  metadata_df <- as.data.frame(metadata_string)
  arrow::write_csv_arrow(metadata_df, put_path, include_header = TRUE)
  return(TRUE)
}
