% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dwplot.R
\name{dwplot}
\alias{dwplot}
\alias{dw_plot}
\title{Dot-and-Whisker Plots of Regression Results}
\usage{
dwplot(
  x,
  ci = 0.95,
  dodge_size = 0.4,
  order_vars = NULL,
  show_intercept = FALSE,
  margins = FALSE,
  model_name = "model",
  style = c("dotwhisker", "distribution"),
  by_2sd = FALSE,
  vline = NULL,
  dot_args = list(size = 1.2),
  whisker_args = list(size = 0.5),
  dist_args = list(alpha = 0.5),
  line_args = list(alpha = 0.75, size = 1),
  ...
)

dw_plot(
  x,
  ci = 0.95,
  dodge_size = 0.4,
  order_vars = NULL,
  show_intercept = FALSE,
  margins = FALSE,
  model_name = "model",
  style = c("dotwhisker", "distribution"),
  by_2sd = FALSE,
  vline = NULL,
  dot_args = list(size = 1.2),
  whisker_args = list(size = 0.5),
  dist_args = list(alpha = 0.5),
  line_args = list(alpha = 0.75, size = 1),
  ...
)
}
\arguments{
\item{x}{Either a model object to be tidied with \code{\link[broomExtra]{tidy_parameters}}, or a list of such model objects, or a tidy data frame of regression results (see 'Details').}

\item{ci}{A number indicating the level of confidence intervals; the default is .95.}

\item{dodge_size}{A number indicating how much vertical separation should be between different models' coefficients when multiple models are graphed in a single plot.  Lower values tend to look better when the number of independent variables is small, while a higher value may be helpful when many models appear on the same plot; the default is 0.4.}

\item{order_vars}{A vector of variable names that specifies the order in which the variables are to appear along the y-axis of the plot.}

\item{show_intercept}{A logical constant indicating whether the coefficient of the intercept term should be plotted.}

\item{margins}{A logical value indicating whether presenting the average marginal effects of the estimates. See the Details for more information.}

\item{model_name}{The name of a variable that distinguishes separate models within a tidy data frame.}

\item{style}{Either \code{"dotwhisker"} or \code{"distribution"}. \code{"dotwhisker"}, the default, shows the regression coefficients' point estimates as dots with confidence interval whiskers.  \code{"distribution"} shows the normal distribution with mean equal to the point estimate and standard deviation equal to the standard error, underscored with a confidence interval whisker.}

\item{by_2sd}{When x is model object or list of model objects, should the coefficients for predictors that are not binary be rescaled by twice the standard deviation of these variables in the dataset analyzed, per Gelman (2008)?  Defaults to \code{FALSE}.  Note that when x is a tidy data frame, one can use \code{\link[dotwhisker]{by_2sd}} to rescale similarly.}

\item{vline}{A \code{geom_vline()} object, typically with \code{xintercept = 0}, to be drawn behind the coefficients.}

\item{dot_args}{When \code{style} is "dotwhisker", a list of arguments specifying the appearance of the dots representing mean estimates.  For supported arguments, see \code{\link[ggplot2]{geom_point}}.}

\item{whisker_args}{When \code{style} is "dotwhisker", a list of arguments specifying the appearance of the whiskers representing the confidence intervals.  For supported arguments, see \code{\link[ggstance]{geom_linerangeh}}.}

\item{dist_args}{When \code{style} is "distribution", a list of arguments specifying the appearance of normally distributed regression estimates.  For supported arguments, see \code{\link[ggplot2]{geom_polygon}}.}

\item{line_args}{When \code{style} is "distribution", a list of arguments specifying the appearance of the line marking the confidence interval beneath the normal distribution.  For supported arguments, see \code{\link[ggstance]{geom_linerangeh}}.}

\item{\dots}{Extra arguments to pass to \code{\link[broomExtra]{tidy_parameters}}.}
}
\value{
The function returns a \code{ggplot} object.
}
\description{
\code{dwplot} is a function for quickly and easily generating dot-and-whisker plots of regression models saved in tidy data frames.
}
\details{
\code{dwplot} visualizes regression model objects or regression results saved in tidy data frames by, e.g., \code{\link[broomExtra]{tidy_parameters}} as dot-and-whisker plots generated by \code{\link[ggplot2]{ggplot}}.

Tidy data frames to be plotted should include the variables \code{term} (names of predictors), \code{estimate} (corresponding estimates of coefficients or other quantities of interest), \code{std.error} (corresponding standard errors), and optionally \code{model} (when multiple models are desired on a single plot; a different name for this last variable may be specified using the model_name argument).
In place of \code{std.error} one may substitute \code{conf.low} (the lower bounds of the confidence intervals of each estimate) and \code{conf.high} (the corresponding upper bounds).

For convenience, \code{dwplot} also accepts as input those model objects that can be tidied by \code{\link[broomExtra]{tidy_parameters}}, or a list of such model objects.

By default, the plot will display 95-percent confidence intervals.  To display a different interval when passing a model object or objects, specify a \code{ci} argument to pass to \code{\link[broomExtra]{tidy_parameters}}.  When passing a data frame of results, include the variables \code{conf.low} and \code{conf.high} describing the bounds of the desired interval.

Because the function can take a data frame as input, it is easily employed for a wide range of models, including those not supported by \code{\link[broomExtra]{tidy_parameters}}.
And because the output is a \code{ggplot} object, it can easily be further customized with any additional arguments and layers supported by \code{ggplot2}.
Together, these two features make \code{dwplot} extremely flexible.

\code{dwplot} provides an option to present the average marginal effect directly based on \code{\link[margins]{margins}}. Users can alter the confidence intervals of the margins through the \code{ci} argument. See the full list of supported functions in the document of the package \code{\link{margins}}. The `margins` argument also works for \code{small_multiple} and \code{secret_weapon}.
}
\examples{
library(dplyr)
# Plot regression coefficients from a single model object
data(mtcars)
m1 <- lm(mpg ~ wt + cyl + disp, data = mtcars)
dwplot(m1, vline = geom_vline(xintercept = 0, colour = "grey50", linetype = 2)) +
    xlab("Coefficient")
# using 99\% confidence interval
dwplot(m1, ci = .99)
# Plot regression coefficients from multiple models
m2 <- update(m1, . ~ . - disp)
dwplot(list(full = m1, nodisp = m2))
# Change the appearance of dots and whiskers
dwplot(m1, dot_args = list(size = 3, pch = 21, fill = "white"))
# Plot regression coefficients from multiple models on the fly
mtcars \%>\%
    split(.$am) \%>\%
    purrr::map(~ lm(mpg ~ wt + cyl + disp, data = .x)) \%>\%
    dwplot() \%>\%
    relabel_predictors(c(wt = "Weight", cyl = "Cylinders", disp = "Displacement")) +
    theme_bw() + xlab("Coefficient") + ylab("") +
    geom_vline(xintercept = 0, colour = "grey60", linetype = 2) +
    ggtitle("Predicting Gas Mileage, OLS Estimates") +
    theme(plot.title = element_text(face = "bold"),
          legend.position = c(.995, .99),
          legend.justification = c(1, 1),
          legend.background = element_rect(colour="grey80"),
          legend.title.align = .5) +
    scale_colour_grey(start = .4, end = .8,
                      name = "Transmission",
                      breaks = c("Model 0", "Model 1"),
                      labels = c("Automatic", "Manual"))

}
\references{
Kastellec, Jonathan P. and Leoni, Eduardo L. 2007. "Using Graphs Instead of Tables in Political Science." *Perspectives on Politics*, 5(4):755-771.

Gelman, Andrew. 2008. "Scaling Regression Inputs by Dividing by Two Standard Deviations." *Statistics in Medicine*, 27:2865-2873.
}
