% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ELHMC.R
\name{ELHMC}
\alias{ELHMC}
\title{Empirical Likelihood Hamiltonian Monte Carlo Sampling}
\usage{
ELHMC(initial, data, fun, dfun, prior, dprior, n.samples = 100,
  lf.steps = 10, epsilon = 0.05, tol = 10^-5, detailed = FALSE)
}
\arguments{
\item{initial}{A vector containing the initial values of the parameter}

\item{data}{A matrix containing the data}

\item{fun}{The estimating function \eqn{g}. It takes in a parameter vector
\code{params} as the first argument and a data point vector \code{x} as the
 second parameter. This function returns a vector.}

\item{dfun}{A function that calculates the gradient of the estimating function
\eqn{g}. It takes in a parameter vector \code{params} as the first argument
 and a data point vector \code{x} as the second argument. This function
 returns a matrix.}

\item{prior}{A function with one argument \code{x} that returns
the prior density of the parameter of interest}

\item{dprior}{A function with one argument \code{x} that returns
the gradient of the density of the parameter of interest}

\item{n.samples}{Number of samples to draw}

\item{lf.steps}{Number of leap frog steps in each Hamiltonian Monte Carlo
update}

\item{epsilon}{The leap frog step size}

\item{tol}{EL tolerance}

\item{detailed}{If this is set to \code{TRUE}, the function will return a list
with extra information.}
}
\value{
The function returns a list with the following elements:
 \item{\code{samples}}{A matrix containing the parameter samples}
 \item{\code{acceptance.rate}}{The acceptance rate (numeric)}

 If \code{detailed = TRUE}, the list contains these extra elements:
 \item{\code{proposed}}{A matrix containing the proposed values at
   \code{n.samaples - 1} Hamiltonian Monte Carlo updates}
 \item{\code{acceptance}}{A vector of \code{TRUE}/\code{FALSE} values
   indicates whether each proposed value is accepted}
 \item{\code{trajectory}}{A list with 2 elements \code{trajectory.q} and
   \code{trajectory.p}. These are lists of matrices contraining position and
   momentum values along trajectory in each Hamiltonian Monte Carlo update.}
}
\description{
This function draws samples from a Empirical Likelihood Bayesian posterior
distribution of parameters using Hamiltonian Monte Carlo.
}
\details{
Suppose there are data \eqn{x = (x_1, x_2, ..., x_n)} where \eqn{x_i}
 takes values in \eqn{R^p} and follow probability distribution \eqn{F}.
 Also, \eqn{F} comes from a family of distributions that depends on
 a parameter \eqn{\theta = (\theta_1, ..., \theta_d)} and there is
 a smooth function
 \eqn{g(x_i, \theta) = (g_1(x_i, \theta), ...,g_q(x_i, \theta))^T} that
 satisfies \eqn{E_F[g(x_i, \theta)] = 0} for \eqn{i = 1, ...,n}.

 \code{ELHMC} draws samples from a Empirical Likelihood Bayesian
 posterior distribution of the parameter \eqn{\theta}, given the data \eqn{x}
 as \code{data}, the smoothing function \eqn{g} as \code{fun},
 and the gradient of \eqn{g} as \code{dfun}.
}
\examples{
\dontrun{
## Suppose there are four data points (1, 1), (1, -1), (-1, -1), (-1, 1)
x = rbind(c(1, 1), c(1, -1), c(-1, -1), c(-1, 1))
## If the parameter of interest is the mean, the smoothing function and
## its gradient would be
f <- function(params, x) {
 x - params
}
df <- function(params, x) {
 rbind(c(-1, 0), c(0, -1))
}
## Draw 50 samples from the Empirical Likelihood Bayesian posterior distribution
## of the mean, using initial values (0.96, 0.97) and a standard normal distribution
## as the prior:
normal_prior <- function(x) {
 exp(-0.5 * x ^ 2) / sqrt(2 * pi)
}
normal_prior_gradient <- function(x) {
 -x * exp(-0.5 * x ^ 2) / sqrt(2 * pi)
}
set.seed(1234)
mean.samples <- ELHMC(initial = c(0.96, 0.97), data = x, fun = f, dfun = df,
                     n.samples = 50, prior = normal_prior,
                     dprior = normal_prior_gradient)
plot(mean.samples$samples, type = "l", xlab = "", ylab = "")
}
}
\references{
Chaudhuri, S., Mondal, D. and Yin, T. (2015)
 Hamiltonian Monte Carlo sampling in Bayesian empirical likelihood
 computation.
 \emph{Journal of the Royal Statistical Society: Series B}.

 Neal, R. (2011) MCMC for using Hamiltonian dynamics.
 \emph{Handbook of Markov Chain Monte Carlo}
 (eds S. Brooks, A.Gelman, G. L.Jones and X.-L. Meng), pp. 113-162.
 New York: Taylor and Francis.
}

