% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/w_clim.R
\name{w_clim}
\alias{w_clim}
\title{Temperature trend obtained from WorldClim.}
\usage{
w_clim(
  y_ini = c(N = 400, N = 400, N = 400),
  temp_cmin = rep(18, 3),
  temp_cmax = c(25, 28, 32),
  ro = rep(0.7, 3),
  lambda = rep(5e-05, 3),
  lat = rep(-33, 3),
  lon = rep(-71, 3),
  s = 10,
  res = 5,
  time_start = 2000,
  time_end = 2070,
  leap = 1/12
)
}
\arguments{
\item{y_ini}{Initial population values (must be written with its name: N).}

\item{temp_cmin}{Minimum critical temperature.}

\item{temp_cmax}{Maximum critical temperature.}

\item{ro}{Population growth rate at optimum temperature.}

\item{lambda}{Marginal loss by non-thermodependent intraspecific competition.}

\item{lat, lon}{Geographical coordinates of some place of interest of study, in decimal degrees.}

\item{s}{Bioclimatic variable.}

\item{res}{Spatial resolution.}

\item{time_start}{Start of time sequence.}

\item{time_end}{End of time sequence.}

\item{leap}{Sequence increase.}
}
\value{
(1) A data.frame with columns having the simulated trends.

(2) A two-panel figure in which (a) shows the population abundance curves represented by solid lines
           and the corresponding carrying capacities are represented by shaded areas. In (b) the temperature
           trend is shown. The three simultaneous simulations are depicted by different colors, i.e. 1st brown,
           2nd green and 3rd blue.
}
\description{
This function allows simulating the effect of temperature trends on the
             abundance of ectotherm populations in different geographic locations.
             Temperature data is obtained from WorldClim.
}
\details{
Three populations and/or scenarios can be simulated simultaneously. The temperature trends
        are obtained by data extracted from WorldClim for the years 2000, 2050 and 2070 at a specific
        location. The function internally calls the function getData of the raster package (Hijmans, 2020) to
        obtain the bioclimatic variable of interest a given spatial resolution. An exponential
        expression is fitted using the nls function. In each input vector, the parameters for
        the three simulations must be specified (finite numbers for the initial population abundance).
        The simulations are obtained by a model that incorporates the effects of temperature over time,
        which leads to a non-autonomous ODE approach. This is function uses the ODE solver implemented
        in the package deSolve (Soetaert et al., 2010). In the first three examples, three geographic locations are
        considered for Macrolophus pygmaeus as reported in Sánchez et al. (2012).
}
\examples{
\dontrun{
#######################################################################
  #Example 1: Different initial population abundances.
#######################################################################

w_clim(y_ini = c(N = 100, N = 200, N = 400),
      temp_cmin = rep(18,3),
      temp_cmax = rep(30,3),
      ro = rep(0.7,3),
      lambda = rep(0.00005,3),
      lat = rep(-33,3),
      lon = rep(-71,3),
      s = 5,
      res = 5,
      time_start = 2000,
      time_end = 2070,
      leap = 1/12)


#######################################################################
  #Example 2: Different thermal tolerance ranges.
#######################################################################

temp_cmin3 <- 18
temp_cmin2 <- 10/9*temp_cmin3
temp_cmin1 <- 10/9*temp_cmin2

temp_cmax1 <- 32.4
temp_cmax2 <- 10/9*temp_cmax1
temp_cmax3 <- 10/9*temp_cmax2

w_clim(y_ini = c(N = 100, N = 100, N = 100),
      temp_cmin = c(temp_cmin1,temp_cmin2,temp_cmin3),
      temp_cmax = c(temp_cmax1,temp_cmax2,temp_cmax3),
      ro = rep(0.7,3),
      lambda = rep(0.00005,3),
      lat = rep(-33,3),
      lon = rep(-71,3),
      s = 5,
      res = 5,
      time_start = 2000,
      time_end = 2070,
      leap = 1/12)

#######################################################################
  #Example 3: Different latitudes.
#######################################################################

lat1 <- -10
lat2 <- -33
lat3 <- -42

w_clim(y_ini = c(N = 100, N = 100, N = 100),
      temp_cmin = rep(18,3),
      temp_cmax = rep(40,3),
      ro = rep(0.7,3),
      lambda = rep(0.00005,3),
      lat = c(lat1,lat2,lat3),
      lon = rep(-71,3),
      s = 5,
      res = 5,
      time_start = 2000,
      time_end = 2070,
      leap = 1/12)

#######################################################################
  #Example 4: Different marginal losses by a non-thermodependent
  #           component of intraspecific competition.
#######################################################################

lambda3 <- 0.01
lambda2 <- 1/2*lambda3
lambda1 <- 1/2*lambda2

w_clim(y_ini = c(N = 100, N = 100, N = 100),
       temp_cmin = rep(18,3),
       temp_cmax = rep(30,3),
       ro = rep(0.7,3),
       lambda = c(lambda1,lambda2,lambda3),
       lat = rep(-33,3),
       lon = rep(-71,3),
       s = 5,
       res = 5,
       time_start = 2000,
       time_end = 2070,
       leap = 1/12)


#######################################################################
  #Application example I: Bioclimatic variable
  #                       (Annual Mean Temperature).
#######################################################################

#We consider a population of Macrolophus pygmaeus in three different
#locations, and its intrinsic growth rate is adjusted to data obtained
#from Rezende and Bozinovic (2019).


github_link <- "https://github.com/Victor-Saldana/epcc/raw/main/M_pygmaeus.xlsx"
library(httr)
temp_file <- tempfile(fileext = ".xlsx")
req <- GET(github_link,
          authenticate(Sys.getenv("GITHUB_PAT"), ""),
          write_disk(path = temp_file))
M_pygmaeus <- readxl::read_excel(temp_file)

TPC <- rate_adjustment(data = M_pygmaeus)

#locality 1
lat1 <- 38.1827778
lon1 <- -1.7380555

#locality 2
lat2 <- 41.01384
lon2 <- 28.94966

#locality 3
lat3 <- 39.7213889
lon3 <- 21.63416638888889

w_clim(y_ini = c(N = 100, N = 100, N = 100),
       temp_cmin = rep(TPC$temp_cmin,3),
       temp_cmax = rep(TPC$temp_cmax,3),
       ro = rep(TPC$ro,3),
       lambda = rep(0.00005,3),
       lat = c(lat1,lat2,lat3),
       lon = c(lon1,lon2,lon3),
       s = 1,
       res = 5,
       time_start = 2000,
       time_end = 2070,
       leap = 1/12)

#######################################################################
  #Application example II: Bioclimatic variable
  #                        (Max Temperature of Warmest Month).
#######################################################################

#We consider a population of Macrolophus pygmaeus in three different
#locations, and its intrinsic growth rate is adjusted to data obtained
#from Rezende and Bozinovic (2019).


github_link <- "https://github.com/Victor-Saldana/epcc/raw/main/M_pygmaeus.xlsx"
library(httr)
temp_file <- tempfile(fileext = ".xlsx")
req <- GET(github_link,
          authenticate(Sys.getenv("GITHUB_PAT"), ""),
          write_disk(path = temp_file))
M_pygmaeus <- readxl::read_excel(temp_file)

TPC <- rate_adjustment(data = M_pygmaeus)

#locality 1
lat1 <- 38.1827778
lon1 <- -1.7380555

#locality 2
lat2 <- 41.01384
lon2 <- 28.94966

#locality 3
lat3 <- 39.7213889
lon3 <- 21.63416638888889

w_clim(y_ini = c(N = 100, N = 100, N = 100),
       temp_cmin = rep(TPC$temp_cmin,3),
       temp_cmax = rep(TPC$temp_cmax,3),
       ro = rep(TPC$ro,3),
       lambda = rep(0.00005,3),
       lat = c(lat1,lat2,lat3),
       lon = c(lon1,lon2,lon3),
       s = 5,
       res = 5,
       time_start = 2000,
       time_end = 2070,
       leap = 1/12)

#######################################################################
  #Application example III: Bioclimatic variable
  #                         (Mean Temperature of Warmest Quarter).
#######################################################################

#We consider a population of Macrolophus pygmaeus in three different
#locations, and its intrinsic growth rate is adjusted to data obtained
#from Rezende and Bozinovic (2019).


github_link <- "https://github.com/Victor-Saldana/epcc/raw/main/M_pygmaeus.xlsx"
library(httr)
temp_file <- tempfile(fileext = ".xlsx")
req <- GET(github_link,
          authenticate(Sys.getenv("GITHUB_PAT"), ""),
          write_disk(path = temp_file))
M_pygmaeus <- readxl::read_excel(temp_file)

TPC <- rate_adjustment(data = M_pygmaeus)

#locality 1
lat1 <- 38.1827778
lon1 <- -1.7380555

#locality 2
lat2 <- 41.01384
lon2 <- 28.94966

#locality 3
lat3 <- 39.7213889
lon3 <- 21.63416638888889

w_clim(y_ini = c(N = 100, N = 100, N = 100),
       temp_cmin = rep(TPC$temp_cmin,3),
       temp_cmax = rep(TPC$temp_cmax,3),
       ro = rep(TPC$ro,3),
       lambda = rep(0.00005,3),
       lat = c(lat1,lat2,lat3),
       lon = c(lon1,lon2,lon3),
       s = 10,
       res = 5,
       time_start = 2000,
       time_end = 2070,
       leap = 1/12)
}
}
\references{
Hijmans, R.J. (2020). Package `raster' (Version 3.3-13). pp. 1-249.

Rezende, E. L., & Bozinovic, F. (2019). Thermal performance across levels of biological
           organization. Philosophical Transactions of the Royal Society B: Biological Sciences,
           374(1778), 20180549.doi:10.1098/rstb.2018.0549

Sanchez, J. A., Spina, M. L., & Perera, O. P. (2012). Analysis of the population
           structure of Macrolophus pygmaeus (Rambur) (Hemiptera: Miridae) in the Palaearctic
           region using microsatellite markers. Ecology and Evolution, 2(12), 3145-3159.
           doi:10.1002/ece3.420

Soetaert, K., Petzoldt, T., & Setzer, R. (2010). Solving Differential Equations in R:
           Package deSolve. Journal of Statistical Software, 33(9), 1 - 25.
           doi:http://dx.doi.org/10.18637/jss.v033.i09
}
