##' Graphical representation of indivduals (rows) of a multivariate analysis
##' 
##' This function displays a graphical representation of the individuals (rows) of a multivariate analysis.
##'
##' @param obj a multivariate analysis results object. Currently only MCA is supported
##' @param ... arguments passed to other methods
##' @import ggplot2
##' @export

ggind <- function(obj, ...) {
    UseMethod("ggind")
  }


##' Graphical representation of individuals (rows) of a multiple correspondance analysis
##' 
##' This function displays a graphical representation of the individuals
##' (rows) of a multiple correspondence analysis generated by the \code{MCA}
##' function of the \code{FactoMineR} package.
##'
##' @rdname ggind
##' @param xax number of the x axis
##' @param yax number of the y axis
##' @param fac an optional factor by which points are colored, and confidence ellipses drawn
##' @param label legend title
##' @param alpha points opacity
##' @param palette palette for points coloring, if \code{fac} is not \code{NULL}
##' @export


ggind.MCA <- function(obj, xax = 1, yax = 2, fac = NA, label = NULL, alpha = 0.5, palette = "Set1", ...) {
  .tmp <- data.frame(x = obj$ind$coord[,xax],
                     y = obj$ind$coord[,yax],
                     fac = fac)
  g <- ggplot(data = .tmp, aes_string(x = "x", y = "y")) + 
    geom_vline(xintercept = 0) +
    geom_hline(yintercept = 0) +
    coord_fixed(ratio = 1) +
    scale_x_continuous(paste0("Dim.",xax)) +
    scale_y_continuous(paste0("Dim.",yax))
  if (all(is.na(fac))) {
    g <- g +  geom_point(alpha = alpha)
  }
  else {
    g <- g + geom_point(aes(col = factor(fac)), alpha = alpha) + 
      stat_ellipse(aes(col = factor(fac)),level = 0.95) +
      scale_color_brewer(label, palette = palette)
  }
  g
}



##' Graphical representation of the variables (columnss) of a multivariate analysis
##' 
##' This function displays a graphical representation of the variables (columns) of a multivariate analysis.
##'
##' @param obj a multivariate analysis results object. Currently only MCA is supported
##' @param ... arguments passed to other methods
##' @author Julien Barnier <julien.barnier@@ens.lyon.fr>
##' @seealso \code{\link[FactoMineR]{MCA}}
##' @import ggplot2
##' @export

ggvar <- function(obj, ...) {
    UseMethod("ggvar")
  }




##' Graphical representation of variables (columns) of a multiple correspondance analysis
##' 
##' This function displays a graphical representation of the variables
##' (columns) of a multiple correspondence analysis generated by the \code{MCA}
##' function of the \code{FactoMineR} package.
##'
##' @rdname ggvar
##' @param xax number of the x axis
##' @param yax number of the y axis
##' @param alpha points opacity
##' @param size text size
##' @param palette palette for variables coloring
##' @export 

ggvar.MCA <- function(obj, xax = 1, yax = 2, size = 4, alpha = 0.5, palette = "Set1", ...) {
  vars <- data.frame(obj$var$coord)
  varnames <- sapply(obj$call$X[,obj$call$quali], nlevels)
  vars$varnames <- rep(names(varnames),varnames)
  vars$modnames <- rownames(vars)
  x <- paste0("Dim.",xax)
  y <- paste0("Dim.",yax)  
  
  g <- ggplot(data = vars) +
    geom_vline(xintercept = 0) +
    geom_hline(yintercept = 0) +
    ##geom_point(aes_string(x=x,y=y,colour="varnames")) +
    geom_text(aes_string(x = x, y = y, colour = "varnames", label = "modnames"), size = size) +
    scale_color_brewer(palette = palette) +
    coord_fixed(ratio = 1)
  
  g  
}
