// Algorithm ported from Eigen, a lightweight C++ template library
// for linear algebra.
//
// Copyright (C) 2013 Gauthier Brun <brun.gauthier@gmail.com>
// Copyright (C) 2013 Nicolas Carre <nicolas.carre@ensimag.fr>
// Copyright (C) 2013 Jean Ceccato <jean.ceccato@ensimag.fr>
// Copyright (C) 2013 Pierre Zoppitelli <pierre.zoppitelli@ensimag.fr>
// Copyright (C) 2013 Jitse Niesen <jitse@maths.leeds.ac.uk>
// Copyright (C) 2014-2017 Gael Guennebaud <gael.guennebaud@inria.fr>
//
// Source Code Form is subject to the terms of the Mozilla
// Public License v. 2.0. If a copy of the MPL was not distributed
// with this file, You can obtain one at http://mozilla.org/MPL/2.0/.

use super::jacobi::{jacobi_svd, JacobiRotation, Skip};
use crate::{
    assert,
    linalg::{temp_mat_req, temp_mat_uninit, temp_mat_zeroed},
    unzipped,
    utils::{simd::SimdFor, thread::join_raw},
    zipped, ColMut, ColRef, MatMut, Parallelism,
};
use coe::Coerce;
use core::{iter::zip, mem::swap};
use dyn_stack::{PodStack, SizeOverflow, StackReq};
use faer_entity::*;
use reborrow::*;

#[allow(dead_code)]
fn bidiag_to_mat<E: RealField>(diag: &[E], subdiag: &[E]) -> crate::Mat<E> {
    let mut mat = crate::Mat::<E>::zeros(diag.len() + 1, diag.len());

    for (i, d) in diag.iter().enumerate() {
        mat.write(i, i, *d);
    }
    for (i, d) in subdiag.iter().enumerate() {
        mat.write(i + 1, i, *d);
    }

    mat
}

fn compute_svd_of_m<E: RealField>(
    mut um: Option<MatMut<'_, E>>,
    mut vm: Option<MatMut<'_, E>>,
    diag: &mut [E],
    col0: &[E],
    outer_perm: &[usize],
    epsilon: E,
    _consider_zero_threshold: E,
    stack: PodStack<'_>,
) {
    let n = diag.len();

    diag[0] = E::faer_zero();
    let mut actual_n = n;
    while actual_n > 1 && diag[actual_n - 1] == E::faer_zero() {
        actual_n -= 1;
        assert!(col0[actual_n] == E::faer_zero());
    }

    let (perm, stack) = stack.collect(
        col0.iter()
            .take(actual_n)
            .map(|x| x.faer_abs())
            .enumerate()
            .filter(|(_, x)| *x != E::faer_zero())
            .map(|(i, _)| i),
    );
    let perm = &*perm;
    let (col0_perm, stack) = stack.collect(perm.iter().map(|&p| col0[p]));
    let (diag_perm, stack) = stack.collect(perm.iter().map(|&p| diag[p]));

    let (mut shifts, stack) = temp_mat_uninit::<E>(n, 1, stack);
    let shifts = shifts.as_mut();
    let (mut mus, stack) = temp_mat_uninit::<E>(n, 1, stack);
    let mus = mus.as_mut();
    let (mut singular_vals, stack) = temp_mat_uninit::<E>(n, 1, stack);
    let singular_vals = singular_vals.as_mut();
    let (mut zhat, stack) = temp_mat_uninit::<E>(n, 1, stack);
    let zhat = zhat.as_mut();

    let mut shifts = shifts.col_mut(0);
    let mut mus = mus.col_mut(0);
    let mut s = singular_vals.col_mut(0);
    let mut zhat = zhat.col_mut(0);

    compute_singular_values(
        shifts.rb_mut(),
        mus.rb_mut(),
        s.rb_mut(),
        diag,
        diag_perm,
        col0,
        col0_perm,
        epsilon,
    );
    perturb_col0(
        zhat.rb_mut(),
        col0,
        diag,
        perm,
        s.rb(),
        shifts.rb(),
        mus.rb(),
    );

    let (col_perm, stack) = stack.make_with(actual_n, |i| i);
    let (col_perm_inv, _) = stack.make_with(actual_n, |i| i);

    for i in 0..actual_n - 1 {
        if s.read(i) > s.read(i + 1) {
            let si = s.read(i);
            let sj = s.read(i + 1);
            s.write(i, sj);
            s.write(i + 1, si);

            col_perm.swap(i, i + 1);
        }
    }
    for (i, p) in col_perm.iter().copied().enumerate() {
        col_perm_inv[p] = i;
    }

    compute_singular_vectors(
        um.rb_mut(),
        vm.rb_mut(),
        zhat.rb(),
        diag,
        perm,
        outer_perm,
        col_perm_inv,
        actual_n,
        shifts.rb(),
        mus.rb(),
    );

    for (idx, diag) in diag[..actual_n].iter_mut().enumerate() {
        *diag = s.read(actual_n - idx - 1);
    }

    for (idx, diag) in diag[actual_n..n].iter_mut().enumerate() {
        *diag = s.read(actual_n + idx);
    }
}

#[inline(never)]
fn compute_singular_vectors<E: RealField>(
    mut um: Option<MatMut<E>>,
    mut vm: Option<MatMut<E>>,
    zhat: ColRef<E>,
    diag: &[E],
    perm: &[usize],
    outer_perm: &[usize],
    col_perm_inv: &[usize],
    actual_n: usize,
    shifts: ColRef<E>,
    mus: ColRef<E>,
) {
    let n = diag.len();

    for k in 0..n {
        let actual_k = if k >= actual_n {
            k
        } else {
            actual_n - col_perm_inv[k] - 1
        };
        let mut u = um.rb_mut().map(|u| u.col_mut(actual_k));
        let mut v = vm.rb_mut().map(|v| v.col_mut(actual_k));

        if zhat.read(k) == E::faer_zero() {
            if let Some(mut u) = u.rb_mut() {
                u.write(outer_perm[k], E::faer_one());
            }
            if let Some(mut v) = v.rb_mut() {
                v.write(outer_perm[k], E::faer_one());
            }
            continue;
        }

        let mu = mus.read(k);
        let shift = shifts.read(k);

        assert_eq!(zhat.row_stride(), 1);

        if let Some(mut u) = u.rb_mut() {
            assert_eq!(u.row_stride(), 1);
            for &i in perm {
                u.write(
                    outer_perm[i],
                    zhat.read(i)
                        .faer_div(diag[i].faer_sub(shift).faer_sub(mu))
                        .faer_div(diag[i].faer_add(shift.faer_add(mu))),
                );
            }
            u.write(n, E::faer_zero());
            let norm_inv = u.norm_l2().faer_inv();
            zipped!(u.rb_mut()).for_each(|unzipped!(mut x)| x.write(x.read().faer_mul(norm_inv)));
        }

        if let Some(mut v) = v {
            assert_eq!(v.row_stride(), 1);
            for &i in &perm[1..] {
                v.write(
                    outer_perm[i],
                    diag[i]
                        .faer_mul(zhat.read(i))
                        .faer_div(diag[i].faer_sub(shift).faer_sub(mu))
                        .faer_div(diag[i].faer_add(shift.faer_add(mu))),
                );
            }
            v.write(outer_perm[0], E::faer_one().faer_neg());
            let norm_inv = v.norm_l2().faer_inv();
            zipped!(v.rb_mut()).for_each(|unzipped!(mut x)| x.write(x.read().faer_mul(norm_inv)));
        }
    }
    if let Some(mut um) = um {
        um.write(n, n, E::faer_one());
    }
}

fn perturb_col0<E: RealField>(
    mut zhat: ColMut<E>,
    col0: &[E],
    diag: &[E],
    perm: &[usize],
    s: ColRef<E>,
    shifts: ColRef<E>,
    mus: ColRef<E>,
) {
    let n = diag.len();
    let m = perm.len();
    if m == 0 {
        zipped!(zhat).for_each(|unzipped!(mut x)| x.write(E::faer_zero()));
        return;
    }

    let last_idx = perm[m - 1];
    for k in 0..n {
        if col0[k] == E::faer_zero() {
            zhat.write(k, E::faer_zero());
            continue;
        }

        let dk = diag[k];
        let mut prod = (s.read(last_idx).faer_add(dk)).faer_mul(
            mus.read(last_idx)
                .faer_add(shifts.read(last_idx).faer_sub(dk)),
        );

        for l in 0..m {
            let i = perm[l];
            if i == k {
                continue;
            }
            if i >= k && l == 0 {
                prod = E::faer_zero();
                break;
            }
            let j = if i < k {
                i
            } else if l > 0 {
                perm[l - 1]
            } else {
                i
            };

            let term = ((s.read(j).faer_add(dk)).faer_div(diag[i].faer_add(dk))).faer_mul(
                (mus.read(j).faer_add(shifts.read(j).faer_sub(dk))).faer_div(diag[i].faer_sub(dk)),
            );
            prod = prod.faer_mul(term);
        }

        let tmp = prod.faer_sqrt();
        if col0[k] > E::faer_zero() {
            zhat.write(k, tmp);
        } else {
            zhat.write(k, tmp.faer_neg());
        }
    }
}

fn compute_singular_values<E: RealField>(
    shifts: ColMut<E>,
    mus: ColMut<E>,
    s: ColMut<E>,
    diag: &[E],
    diag_perm: &[E],
    col0: &[E],
    col0_perm: &[E],
    epsilon: E,
) {
    if E::IS_F64 {
        struct ImplF64<'a> {
            shifts: ColMut<'a, f64>,
            mus: ColMut<'a, f64>,
            s: ColMut<'a, f64>,
            diag: &'a [f64],
            diag_perm: &'a [f64],
            col0: &'a [f64],
            col0_perm: &'a [f64],
            epsilon: f64,
        }
        impl pulp::WithSimd for ImplF64<'_> {
            type Output = ();

            #[inline(always)]
            fn with_simd<S: pulp::Simd>(self, simd: S) -> Self::Output {
                let Self {
                    shifts,
                    mus,
                    s,
                    diag,
                    diag_perm,
                    col0,
                    col0_perm,
                    epsilon,
                } = self;
                compute_singular_values_generic::<f64>(
                    simd, shifts, mus, s, diag, diag_perm, col0, col0_perm, epsilon,
                )
            }
        }

        <f64 as ComplexField>::Simd::default().dispatch(ImplF64 {
            shifts: shifts.coerce(),
            mus: mus.coerce(),
            s: s.coerce(),
            diag: diag.coerce(),
            diag_perm: diag_perm.coerce(),
            col0: col0.coerce(),
            col0_perm: col0_perm.coerce(),
            epsilon: coe::coerce_static(epsilon),
        });
    } else if E::IS_F32 {
        struct ImplF32<'a> {
            shifts: ColMut<'a, f32>,
            mus: ColMut<'a, f32>,
            s: ColMut<'a, f32>,
            diag: &'a [f32],
            diag_perm: &'a [f32],
            col0: &'a [f32],
            col0_perm: &'a [f32],
            epsilon: f32,
        }
        impl pulp::WithSimd for ImplF32<'_> {
            type Output = ();

            #[inline(always)]
            fn with_simd<S: pulp::Simd>(self, simd: S) -> Self::Output {
                let Self {
                    shifts,
                    mus,
                    s,
                    diag,
                    diag_perm,
                    col0,
                    col0_perm,
                    epsilon,
                } = self;
                compute_singular_values_generic::<f32>(
                    simd, shifts, mus, s, diag, diag_perm, col0, col0_perm, epsilon,
                )
            }
        }

        <f64 as ComplexField>::Simd::default().dispatch(ImplF32 {
            shifts: shifts.coerce(),
            mus: mus.coerce(),
            s: s.coerce(),
            diag: diag.coerce(),
            diag_perm: diag_perm.coerce(),
            col0: col0.coerce(),
            col0_perm: col0_perm.coerce(),
            epsilon: coe::coerce_static(epsilon),
        });
    } else {
        compute_singular_values_generic(
            pulp::Scalar::new(),
            shifts,
            mus,
            s,
            diag,
            diag_perm,
            col0,
            col0_perm,
            epsilon,
        );
    }
}

#[inline(always)]
fn compute_singular_values_generic<E: RealField>(
    simd: impl pulp::Simd,
    mut shifts: ColMut<E>,
    mut mus: ColMut<E>,
    mut s: ColMut<E>,
    diag: &[E],
    diag_perm: &[E],
    col0: &[E],
    col0_perm: &[E],
    epsilon: E,
) {
    simd.vectorize(
        #[inline(always)]
        || {
            let n = diag.len();
            let mut actual_n = n;
            while actual_n > 1 && col0[actual_n - 1] == E::faer_zero() {
                actual_n -= 1;
            }
            let actual_n = actual_n;

            let two = E::faer_one().faer_add(E::faer_one());
            let eight = two
                .faer_scale_power_of_two(two)
                .faer_scale_power_of_two(two);
            let one_half = two.faer_inv();

            'kth_iter: for k in 0..n {
                s.write(k, E::faer_zero());
                shifts.write(k, E::faer_zero());
                mus.write(k, E::faer_zero());

                if col0[k] == E::faer_zero() || actual_n == 1 {
                    s.write(k, if k == 0 { col0[0] } else { diag[k] });
                    shifts.write(k, s.read(k));
                    mus.write(k, E::faer_zero());
                    continue 'kth_iter;
                }

                let last_k = k == actual_n - 1;
                let left = diag[k];
                let right = if last_k {
                    let mut norm2 = E::faer_zero();
                    for &x in col0 {
                        norm2 = norm2.faer_add(x.faer_mul(x));
                    }
                    diag[actual_n - 1].faer_add(norm2.faer_sqrt())
                } else {
                    let mut l = k + 1;
                    while col0[l] == E::faer_zero() {
                        l += 1;
                    }
                    diag[l]
                };

                let mid = left.faer_add(right.faer_sub(left).faer_scale_power_of_two(one_half));
                let [mut f_mid, f_max, f_mid_left_shift, f_mid_right_shift] = secular_eq_multi_fast(
                    [
                        mid,
                        if last_k {
                            right.faer_sub(left)
                        } else {
                            (right.faer_sub(left)).faer_scale_power_of_two(one_half)
                        },
                        one_half.faer_mul(right.faer_sub(left)),
                        one_half.faer_mul(right.faer_sub(left)).faer_neg(),
                    ],
                    col0_perm,
                    diag_perm,
                    [E::faer_zero(), left, left, right],
                );

                let (mut shift, mu) = if last_k || f_mid > E::faer_zero() {
                    (left, one_half.faer_mul(right.faer_sub(left)))
                } else {
                    (right, one_half.faer_mul(right.faer_sub(left)).faer_neg())
                };

                if f_mid_left_shift <= E::faer_zero() && f_mid_right_shift >= E::faer_zero() {
                    s.write(k, shift.faer_add(mu));
                    shifts.write(k, shift);
                    mus.write(k, mu);
                    continue 'kth_iter;
                }

                if !last_k {
                    if shift == left {
                        if f_mid_left_shift < E::faer_zero() {
                            shift = right;
                            f_mid = f_mid_right_shift;
                        }
                    } else if f_mid_right_shift > E::faer_zero() {
                        shift = left;
                        f_mid = f_mid_left_shift;
                    }
                }

                enum SecantError {
                    OutOfBounds,
                    PrecisionLimitReached,
                }

                let secant = {
                    #[inline(always)]
                    |mut mu_cur: E, mut mu_prev: E, mut f_cur: E, mut f_prev: E| {
                        if f_prev.faer_abs() < f_cur.faer_abs() {
                            swap(&mut f_prev, &mut f_cur);
                            swap(&mut mu_prev, &mut mu_cur);
                        }

                        let mut left_candidate = None;
                        let mut right_candidate = None;

                        let mut use_bisection = false;
                        let same_sign = f_prev.faer_mul(f_cur) > E::faer_zero();
                        if !same_sign {
                            let (min, max) = if mu_cur < mu_prev {
                                (mu_cur, mu_prev)
                            } else {
                                (mu_prev, mu_cur)
                            };
                            left_candidate = Some(min);
                            right_candidate = Some(max);
                        }

                        let mut err = SecantError::PrecisionLimitReached;

                        while f_cur != E::faer_zero()
                            && ((mu_cur.faer_sub(mu_prev)).faer_abs()
                                > eight.faer_mul(epsilon).faer_mul(
                                    if mu_cur.faer_abs() > mu_prev.faer_abs() {
                                        mu_cur.faer_abs()
                                    } else {
                                        mu_prev.faer_abs()
                                    },
                                ))
                            && ((f_cur.faer_sub(f_prev)).faer_abs() > epsilon)
                            && !use_bisection
                        {
                            // rational interpolation: fit a function of the form a / mu + b through
                            // the two previous iterates and use its
                            // zero to compute the next iterate
                            let a = (f_cur.faer_sub(f_prev))
                                .faer_mul(mu_prev.faer_mul(mu_cur))
                                .faer_div(mu_prev.faer_sub(mu_cur));
                            let b = f_cur.faer_sub(a.faer_div(mu_cur));
                            let mu_zero = a.faer_div(b).faer_neg();
                            let f_zero = secular_eq(mu_zero, col0_perm, diag_perm, shift);

                            if f_zero < E::faer_zero() {
                                left_candidate = Some(mu_zero);
                            } else {
                                right_candidate = Some(mu_zero);
                            }

                            mu_prev = mu_cur;
                            f_prev = f_cur;
                            mu_cur = mu_zero;
                            f_cur = f_zero;

                            if shift == left
                                && (mu_cur < E::faer_zero() || mu_cur > (right.faer_sub(left)))
                            {
                                err = SecantError::OutOfBounds;
                                use_bisection = true;
                            }
                            if shift == right
                                && (mu_cur < (right.faer_sub(left)).faer_neg()
                                    || mu_cur > E::faer_zero())
                            {
                                err = SecantError::OutOfBounds;
                                use_bisection = true;
                            }
                            if f_cur.faer_abs() > f_prev.faer_abs() {
                                // find mu such that a / mu + b = -k * f_zero
                                // a / mu = -f_zero - b
                                // mu = -a / (f_zero + b)
                                let mut k = E::faer_one();
                                for _ in 0..4 {
                                    let mu_opposite =
                                        a.faer_neg().faer_div(k.faer_mul(f_zero).faer_add(b));
                                    let f_opposite =
                                        secular_eq(mu_opposite, col0_perm, diag_perm, shift);
                                    if f_zero < E::faer_zero() && f_opposite >= E::faer_zero() {
                                        // this will be our right candidate
                                        right_candidate = Some(mu_opposite);
                                        break;
                                    } else if f_zero > E::faer_zero()
                                        && f_opposite <= E::faer_zero()
                                    {
                                        // this will be our left candidate
                                        left_candidate = Some(mu_opposite);
                                        break;
                                    }
                                    k = k.faer_scale_power_of_two(two);
                                }
                                use_bisection = true;
                            }
                        }
                        (use_bisection, mu_cur, left_candidate, right_candidate, err)
                    }
                };

                let (mut left_shifted, mut f_left, mut right_shifted, mut f_right) =
                    if shift == left {
                        (
                            E::faer_zero(),
                            E::faer_zero().faer_inv().faer_neg(),
                            if last_k {
                                right.faer_sub(left)
                            } else {
                                (right.faer_sub(left)).faer_mul(one_half)
                            },
                            if last_k { f_max } else { f_mid },
                        )
                    } else {
                        (
                            (right.faer_sub(left))
                                .faer_neg()
                                .faer_scale_power_of_two(one_half),
                            f_mid,
                            E::faer_zero(),
                            E::faer_zero().faer_inv(),
                        )
                    };

                assert!(
                    PartialOrd::partial_cmp(&f_left, &E::faer_zero())
                        != Some(core::cmp::Ordering::Greater)
                );
                assert!(
                    PartialOrd::partial_cmp(&f_right, &E::faer_zero())
                        != Some(core::cmp::Ordering::Less)
                );

                let mut iteration_count = 0;
                let mut f_prev = f_mid;
                // try to find non zero starting bounds

                let half0 = one_half;
                let half1 = half0.faer_scale_power_of_two(half0);
                let half2 = half1.faer_scale_power_of_two(half1);
                let half3 = half2.faer_scale_power_of_two(half2);
                let half4 = half3.faer_scale_power_of_two(half3);
                let half5 = half4.faer_scale_power_of_two(half4);
                let half6 = half5.faer_scale_power_of_two(half5);
                let half7 = half6.faer_scale_power_of_two(half6);

                let mu_values = if shift == left {
                    [
                        right_shifted.faer_scale_power_of_two(half7),
                        right_shifted.faer_scale_power_of_two(half6),
                        right_shifted.faer_scale_power_of_two(half5),
                        right_shifted.faer_scale_power_of_two(half4),
                        right_shifted.faer_scale_power_of_two(half3),
                        right_shifted.faer_scale_power_of_two(half2),
                        right_shifted.faer_scale_power_of_two(half1),
                        right_shifted.faer_scale_power_of_two(half0),
                    ]
                } else {
                    [
                        left_shifted.faer_scale_power_of_two(half7),
                        left_shifted.faer_scale_power_of_two(half6),
                        left_shifted.faer_scale_power_of_two(half5),
                        left_shifted.faer_scale_power_of_two(half4),
                        left_shifted.faer_scale_power_of_two(half3),
                        left_shifted.faer_scale_power_of_two(half2),
                        left_shifted.faer_scale_power_of_two(half1),
                        left_shifted.faer_scale_power_of_two(half0),
                    ]
                };
                let f_values =
                    secular_eq_multi_fast(mu_values, col0_perm, diag_perm, [(); 8].map(|_| shift));

                if shift == left {
                    let mut i = 0;
                    for (idx, (mu, f)) in zip(mu_values, f_values).enumerate() {
                        if f < E::faer_zero() {
                            left_shifted = mu;
                            f_left = f;
                            i = idx + 1;
                        }
                    }
                    if i < f_values.len() {
                        right_shifted = mu_values[i];
                        f_right = f_values[i];
                    }
                } else {
                    let mut i = 0;
                    for (idx, (mu, f)) in zip(mu_values, f_values).enumerate() {
                        if f > E::faer_zero() {
                            right_shifted = mu;
                            f_right = f;
                            i = idx + 1;
                        }
                    }
                    if i < f_values.len() {
                        left_shifted = mu_values[i];
                        f_left = f_values[i];
                    }
                }

                // try bisection just to get a good guess for secant
                while right_shifted.faer_sub(left_shifted)
                    > two.faer_mul(epsilon).faer_mul(
                        if left_shifted.faer_abs() > right_shifted.faer_abs() {
                            left_shifted.faer_abs()
                        } else {
                            right_shifted.faer_abs()
                        },
                    )
                {
                    let mid_shifted_arithmetic =
                        (left_shifted.faer_add(right_shifted)).faer_scale_power_of_two(one_half);
                    let mut mid_shifted_geometric = left_shifted
                        .faer_abs()
                        .faer_sqrt()
                        .faer_mul(right_shifted.faer_abs().faer_sqrt());
                    if left_shifted < E::faer_zero() {
                        mid_shifted_geometric = mid_shifted_geometric.faer_neg();
                    }
                    let mid_shifted = if mid_shifted_geometric == E::faer_zero() {
                        mid_shifted_arithmetic
                    } else {
                        mid_shifted_geometric
                    };
                    let f_mid = secular_eq(mid_shifted, col0_perm, diag_perm, shift);

                    if f_mid == E::faer_zero() {
                        s.write(k, shift.faer_add(mid_shifted));
                        shifts.write(k, shift);
                        mus.write(k, mid_shifted);
                        continue 'kth_iter;
                    } else if f_mid > E::faer_zero() {
                        right_shifted = mid_shifted;
                        f_prev = f_right;
                        f_right = f_mid;
                    } else {
                        left_shifted = mid_shifted;
                        f_prev = f_left;
                        f_left = f_mid;
                    }

                    if iteration_count == 4 {
                        break;
                    }

                    iteration_count += 1;
                }

                // try secant with the guess from bisection
                let args = if left_shifted == E::faer_zero() {
                    (
                        right_shifted.faer_add(right_shifted),
                        right_shifted,
                        f_prev,
                        f_right,
                    )
                } else if right_shifted == E::faer_zero() {
                    (
                        left_shifted.faer_add(left_shifted),
                        left_shifted,
                        f_prev,
                        f_left,
                    )
                } else {
                    (left_shifted, right_shifted, f_left, f_right)
                };

                let (use_bisection, mut mu_cur, left_candidate, right_candidate, _err) =
                    secant(args.0, args.1, args.2, args.3);

                match (left_candidate, right_candidate) {
                    (Some(left), Some(right)) if left < right => {
                        if left > left_shifted {
                            left_shifted = left;
                        }
                        if right < right_shifted {
                            right_shifted = right;
                        }
                    }
                    _ => (),
                }

                // secant failed, use bisection again
                if use_bisection {
                    while (right_shifted.faer_sub(left_shifted))
                        > two.faer_mul(epsilon).faer_mul(
                            if left_shifted.faer_abs() > right_shifted.faer_abs() {
                                left_shifted.faer_abs()
                            } else {
                                right_shifted.faer_abs()
                            },
                        )
                    {
                        let mid_shifted = (left_shifted.faer_add(right_shifted))
                            .faer_scale_power_of_two(one_half);
                        let f_mid = secular_eq(mid_shifted, col0_perm, diag_perm, shift);

                        if f_mid == E::faer_zero() {
                            break;
                        } else if f_mid > E::faer_zero() {
                            right_shifted = mid_shifted;
                        } else {
                            left_shifted = mid_shifted;
                        }
                    }

                    mu_cur = (left_shifted.faer_add(right_shifted)).faer_mul(one_half);
                }

                s.write(k, shift.faer_add(mu_cur));
                shifts.write(k, shift);
                mus.write(k, mu_cur);
            }
        },
    );
}

#[inline(always)]
fn secular_eq_multi_fast<const N: usize, E: RealField>(
    mu: [E; N],
    col0_perm: &[E],
    diag_perm: &[E],
    shift: [E; N],
) -> [E; N] {
    let mut res0 = [(); N].map(|_| E::faer_one());
    for (c0, d0) in col0_perm.iter().cloned().zip(diag_perm.iter().cloned()) {
        for ((res0, mu), shift) in res0
            .iter_mut()
            .zip(mu.iter().cloned())
            .zip(shift.iter().cloned())
        {
            *res0 = (*res0).faer_add((c0.faer_mul(c0)).faer_div(
                (d0.faer_sub(shift).faer_sub(mu)).faer_mul(d0.faer_add(shift).faer_add(mu)),
            ));
        }
    }
    res0
}

#[inline(always)]
fn secular_eq<E: RealField>(mu: E, col0_perm: &[E], diag_perm: &[E], shift: E) -> E {
    let mut res0 = E::faer_one();
    let mut res1 = E::faer_zero();
    let mut res2 = E::faer_zero();
    let mut res3 = E::faer_zero();
    let mut res4 = E::faer_zero();
    let mut res5 = E::faer_zero();
    let mut res6 = E::faer_zero();
    let mut res7 = E::faer_zero();

    let (col0_head, col0_perm) = pulp::as_arrays::<8, _>(col0_perm);
    let (diag_head, diag_perm) = pulp::as_arrays::<8, _>(diag_perm);
    for ([c0, c1, c2, c3, c4, c5, c6, c7], [d0, d1, d2, d3, d4, d5, d6, d7]) in
        col0_head.iter().zip(diag_head)
    {
        res0 = res0.faer_add(
            (c0.faer_div(d0.faer_sub(shift).faer_sub(mu)))
                .faer_mul(c0.faer_div(d0.faer_add(shift).faer_add(mu))),
        );
        res1 = res1.faer_add(
            (c1.faer_div(d1.faer_sub(shift).faer_sub(mu)))
                .faer_mul(c1.faer_div(d1.faer_add(shift).faer_add(mu))),
        );
        res2 = res2.faer_add(
            (c2.faer_div(d2.faer_sub(shift).faer_sub(mu)))
                .faer_mul(c2.faer_div(d2.faer_add(shift).faer_add(mu))),
        );
        res3 = res3.faer_add(
            (c3.faer_div(d3.faer_sub(shift).faer_sub(mu)))
                .faer_mul(c3.faer_div(d3.faer_add(shift).faer_add(mu))),
        );
        res4 = res4.faer_add(
            (c4.faer_div(d4.faer_sub(shift).faer_sub(mu)))
                .faer_mul(c4.faer_div(d4.faer_add(shift).faer_add(mu))),
        );
        res5 = res5.faer_add(
            (c5.faer_div(d5.faer_sub(shift).faer_sub(mu)))
                .faer_mul(c5.faer_div(d5.faer_add(shift).faer_add(mu))),
        );
        res6 = res6.faer_add(
            (c6.faer_div(d6.faer_sub(shift).faer_sub(mu)))
                .faer_mul(c6.faer_div(d6.faer_add(shift).faer_add(mu))),
        );
        res7 = res7.faer_add(
            (c7.faer_div(d7.faer_sub(shift).faer_sub(mu)))
                .faer_mul(c7.faer_div(d7.faer_add(shift).faer_add(mu))),
        );
    }

    let (col0_head, col0_perm) = pulp::as_arrays::<4, _>(col0_perm);
    let (diag_head, diag_perm) = pulp::as_arrays::<4, _>(diag_perm);
    for ([c0, c1, c2, c3], [d0, d1, d2, d3]) in col0_head.iter().zip(diag_head) {
        res0 = res0.faer_add(
            (c0.faer_div(d0.faer_sub(shift).faer_sub(mu)))
                .faer_mul(c0.faer_div(d0.faer_add(shift).faer_add(mu))),
        );
        res1 = res1.faer_add(
            (c1.faer_div(d1.faer_sub(shift).faer_sub(mu)))
                .faer_mul(c1.faer_div(d1.faer_add(shift).faer_add(mu))),
        );
        res2 = res2.faer_add(
            (c2.faer_div(d2.faer_sub(shift).faer_sub(mu)))
                .faer_mul(c2.faer_div(d2.faer_add(shift).faer_add(mu))),
        );
        res3 = res3.faer_add(
            (c3.faer_div(d3.faer_sub(shift).faer_sub(mu)))
                .faer_mul(c3.faer_div(d3.faer_add(shift).faer_add(mu))),
        );
    }

    let (col0_head, col0_perm) = pulp::as_arrays::<2, _>(col0_perm);
    let (diag_head, diag_perm) = pulp::as_arrays::<2, _>(diag_perm);
    for ([c0, c1], [d0, d1]) in col0_head.iter().zip(diag_head) {
        res0 = res0.faer_add(
            (c0.faer_div(d0.faer_sub(shift).faer_sub(mu)))
                .faer_mul(c0.faer_div(d0.faer_add(shift).faer_add(mu))),
        );
        res1 = res1.faer_add(
            (c1.faer_div(d1.faer_sub(shift).faer_sub(mu)))
                .faer_mul(c1.faer_div(d1.faer_add(shift).faer_add(mu))),
        );
    }

    for (c0, d0) in col0_perm.iter().zip(diag_perm) {
        res0 = res0.faer_add(
            (c0.faer_div(d0.faer_sub(shift).faer_sub(mu)))
                .faer_mul(c0.faer_div(d0.faer_add(shift).faer_add(mu))),
        );
    }

    ((res0.faer_add(res1)).faer_add(res2.faer_add(res3)))
        .faer_add((res4.faer_add(res5)).faer_add(res6.faer_add(res7)))
}

fn deflate<E: RealField>(
    diag: &mut [E],
    col0: &mut [E],
    jacobi_coeffs: &mut [JacobiRotation<E>],
    jacobi_indices: &mut [usize],
    transpositions: &mut [usize],
    perm: &mut [usize],
    k: usize,
    epsilon: E,
    consider_zero_threshold: E,
    stack: PodStack<'_>,
) -> (usize, usize) {
    let n = diag.len();
    let mut jacobi_0i = 0;
    let mut jacobi_ij = 0;

    let mut max_diag = E::faer_zero();
    let mut max_col0 = E::faer_zero();
    for d in diag[1..].iter() {
        max_diag = if d.faer_abs() > max_diag {
            d.faer_abs()
        } else {
            max_diag
        };
    }
    for d in col0.iter() {
        max_col0 = if d.faer_abs() > max_col0 {
            d.faer_abs()
        } else {
            max_col0
        };
    }

    let epsilon_strict = epsilon.faer_mul(max_diag);
    let epsilon_strict = if epsilon_strict > consider_zero_threshold {
        &epsilon_strict
    } else {
        &consider_zero_threshold
    };

    let two = E::faer_one().faer_add(E::faer_one());
    let eight = two
        .faer_scale_power_of_two(two)
        .faer_scale_power_of_two(two);
    let epsilon_coarse = eight.faer_mul(epsilon).faer_mul(if max_diag > max_col0 {
        max_diag
    } else {
        max_col0
    });

    // condition 4.1
    if diag[0] < epsilon_coarse {
        diag[0] = epsilon_coarse;
        col0[0] = epsilon_coarse;
    }

    // condition 4.2
    for x in &mut col0[1..] {
        if x.faer_abs() < *epsilon_strict {
            *x = E::faer_zero();
        }
    }

    // condition 4.3
    for i in 1..n {
        if diag[i] < epsilon_coarse {
            if let Some(rot) = deflation43(diag, col0, i) {
                jacobi_coeffs[jacobi_0i] = rot;
                jacobi_indices[jacobi_0i] = i;
                jacobi_0i += 1;
            }
        }
    }

    let mut total_deflation = true;
    for c in col0[1..].iter() {
        if PartialOrd::partial_cmp(&c.faer_abs(), &consider_zero_threshold)
            != Some(core::cmp::Ordering::Less)
        {
            total_deflation = false;
            break;
        }
    }

    perm[0] = 0;
    let mut p = 1;

    for (d, i) in diag[1..].iter().zip(1..n) {
        if d.faer_abs() < consider_zero_threshold {
            perm[p] = i;
            p += 1;
        }
    }

    let mut i = 1;
    let mut j = k + 1;

    for p in &mut perm[p..] {
        if i >= k + 1 {
            *p = j;
            j += 1;
        } else if j >= n {
            *p = i;
            i += 1;
        } else if diag[i] < diag[j] {
            *p = j;
            j += 1;
        } else {
            *p = i;
            i += 1;
        }
    }

    if total_deflation {
        for i in 1..n {
            let pi = perm[i];
            if diag[pi].faer_abs() < consider_zero_threshold || diag[pi] > diag[0] {
                perm[i - 1] = perm[i];
            } else {
                perm[i - 1] = 0;
                break;
            }
        }
    }

    let (real_ind, stack) = stack.make_with(n, |i| i);
    let (real_col, _) = stack.make_with(n, |i| i);

    for i in (if total_deflation { 0 } else { 1 })..n {
        let pi = perm[n - (if total_deflation { i + 1 } else { i })];
        let j = real_col[pi];

        diag.swap(i, j);

        if i != 0 && j != 0 {
            col0.swap(i, j);
        }

        transpositions[i] = j;

        let real_i = real_ind[i];
        real_col[real_i] = j;
        real_col[pi] = i;
        real_ind[j] = real_i;
        real_ind[i] = pi;
    }
    col0[0] = diag[0];
    for (i, p) in perm.iter_mut().enumerate() {
        *p = i;
    }
    for (i, j) in transpositions.iter().copied().enumerate() {
        perm.swap(i, j);
    }

    // condition 4.4
    let mut i = n - 1;
    while i > 0
        && (diag[i].faer_abs() < consider_zero_threshold
            || col0[i].faer_abs() < consider_zero_threshold)
    {
        i -= 1;
    }
    while i > 1 {
        if diag[i].faer_sub(diag[i - 1]) < *epsilon_strict {
            if let Some(rot) = deflation44(diag, col0, i - 1, i) {
                jacobi_coeffs[jacobi_0i + jacobi_ij] = rot;
                jacobi_indices[jacobi_0i + jacobi_ij] = i;
                jacobi_ij += 1;
            }
        }
        i -= 1;
    }

    (jacobi_0i, jacobi_ij)
}

fn deflation43<E: RealField>(
    diag: &mut [E],
    col0: &mut [E],
    i: usize,
) -> Option<JacobiRotation<E>> {
    let p = col0[0];
    let q = col0[i];
    if p == E::faer_zero() && q == E::faer_zero() {
        diag[i] = E::faer_zero();
        return None;
    }
    let rot = JacobiRotation::make_givens(p, q);

    let r = rot.c * p - rot.s * q;

    col0[0] = r;
    diag[0] = r;
    col0[i] = E::faer_zero();
    diag[i] = E::faer_zero();

    Some(rot)
}

fn deflation44<E: RealField>(
    diag: &mut [E],
    col0: &mut [E],
    i: usize,
    j: usize,
) -> Option<JacobiRotation<E>> {
    let p = col0[i];
    let q = col0[j];
    if p == E::faer_zero() && q == E::faer_zero() {
        diag[i] = diag[j];
        return None;
    }
    let rot = JacobiRotation::make_givens(p, q);
    let r = rot.c * p - rot.s * q;

    col0[i] = r;
    col0[j] = E::faer_zero();
    diag[i] = diag[j];

    Some(rot)
}

fn bidiag_svd_qr_algorithm_impl<E: RealField>(
    diag: &mut [E],
    subdiag: &mut [E],
    mut u: Option<MatMut<'_, E>>,
    mut v: Option<MatMut<'_, E>>,
    epsilon: E,
    consider_zero_threshold: E,
) {
    let n = diag.len();
    let max_iter = 30usize.saturating_mul(n).saturating_mul(n);

    let epsilon = epsilon.faer_scale_real(E::faer_from_f64(128.0));

    if let Some(mut u) = u.rb_mut() {
        u.fill_zero();
        u.diagonal_mut().column_vector_mut().fill(E::faer_one());
    }
    if let Some(mut v) = v.rb_mut() {
        v.fill_zero();
        v.diagonal_mut().column_vector_mut().fill(E::faer_one());
    }

    u = u.map(|u| u.submatrix_mut(0, 0, n, n));
    v = v.map(|v| v.submatrix_mut(0, 0, n, n));

    let mut max_val = E::faer_zero();

    for x in &*diag {
        let val = x.faer_abs();
        if val > max_val {
            max_val = val;
        }
    }
    for x in &*subdiag {
        let val = x.faer_abs();
        if val > max_val {
            max_val = val;
        }
    }

    if max_val == E::faer_zero() {
        return;
    }

    for x in &mut *diag {
        *x = (*x).faer_div(max_val);
    }
    for x in &mut *subdiag {
        *x = (*x).faer_div(max_val);
    }

    struct Impl<'a, E: Entity> {
        epsilon: E,
        consider_zero_threshold: E,
        max_iter: usize,
        diag: &'a mut [E],
        subdiag: &'a mut [E],
        u: Option<MatMut<'a, E>>,
        v: Option<MatMut<'a, E>>,
    }

    impl<E: RealField> pulp::WithSimd for Impl<'_, E> {
        type Output = ();

        #[inline(always)]
        fn with_simd<S: pulp::Simd>(self, simd: S) -> Self::Output {
            let Self {
                epsilon,
                consider_zero_threshold,
                max_iter,
                diag,
                subdiag,
                mut u,
                mut v,
            } = self;
            let n = diag.len();
            let arch = E::Simd::default();
            let epsilon2 = E::faer_mul(epsilon, epsilon);

            for iter in 0..max_iter {
                let _ = iter;
                for i in 0..n - 1 {
                    if subdiag[i].faer_abs2()
                        <= epsilon2
                            .faer_mul(E::faer_mul(diag[i].faer_abs(), diag[i + 1].faer_abs()))
                            + consider_zero_threshold
                    {
                        subdiag[i] = E::faer_zero();
                    }
                }
                for i in 0..n {
                    if diag[i].faer_abs() <= epsilon {
                        diag[i] = E::faer_zero();
                    }
                }

                let mut end = n;
                while end > 1 && subdiag[end - 2].faer_abs2() <= consider_zero_threshold {
                    end -= 1;
                }

                if end == 1 {
                    break;
                }

                let mut start = end - 1;
                while start > 0 && subdiag[start - 1] != E::faer_zero() {
                    start -= 1;
                }

                let mut found_zero_diag = false;
                for i in start..end - 1 {
                    if diag[i] == E::faer_zero() {
                        found_zero_diag = true;
                        let mut val = subdiag[i];
                        subdiag[i] = E::faer_zero();
                        for j in i + 1..end {
                            let rot = JacobiRotation::make_givens(diag[j], val);
                            diag[j] = rot
                                .c
                                .faer_mul(diag[j])
                                .faer_sub(rot.s.faer_mul(val))
                                .faer_abs();

                            if j < end - 1 {
                                (val, subdiag[j]) = (
                                    rot.s.faer_mul(subdiag[j]).faer_neg(),
                                    rot.c.faer_mul(subdiag[j]),
                                );
                            }

                            if let Some(v) = v.rb_mut() {
                                unsafe {
                                    rot.apply_on_the_right_in_place_arch(
                                        arch,
                                        v.rb().col(i).const_cast(),
                                        v.rb().col(j).const_cast(),
                                    );
                                }
                            }
                        }
                    }
                }

                if found_zero_diag {
                    continue;
                }

                let t00 = if end - start == 2 {
                    diag[end - 2].faer_abs2()
                } else {
                    diag[end - 2]
                        .faer_abs2()
                        .faer_add(subdiag[end - 3].faer_abs2())
                };
                let t11 = diag[end - 1]
                    .faer_abs2()
                    .faer_add(subdiag[end - 2].faer_abs2());
                let t01 = diag[end - 2].faer_mul(subdiag[end - 2]);

                let mu;
                if false {
                    let delta = E::faer_sub(
                        t00.faer_add(t11).faer_abs2(),
                        t00.faer_mul(t11)
                            .faer_sub(t01.faer_abs2())
                            .faer_scale_power_of_two(E::faer_from_f64(4.0)),
                    );

                    mu = if delta > E::faer_zero() {
                        let lambda0 = t00
                            .faer_add(t11)
                            .faer_add(delta.faer_sqrt())
                            .faer_scale_power_of_two(E::faer_from_f64(0.5));
                        let lambda1 = t00
                            .faer_add(t11)
                            .faer_sub(delta.faer_sqrt())
                            .faer_scale_power_of_two(E::faer_from_f64(0.5));

                        if lambda0.faer_sub(t11).faer_abs() < lambda1.faer_sub(t11).faer_abs() {
                            lambda0
                        } else {
                            lambda1
                        }
                    } else {
                        t11
                    };
                } else {
                    let t01_2 = t01.faer_abs2();
                    if t01_2 > consider_zero_threshold {
                        let d = (t00.faer_sub(t11)).faer_mul(E::faer_from_f64(0.5));
                        let mut delta = d.faer_abs2().faer_add(t01_2).faer_sqrt();
                        if d < E::faer_zero() {
                            delta = delta.faer_neg();
                        }

                        mu = t11.faer_sub(t01_2.faer_div(d.faer_add(delta)));
                    } else {
                        mu = t11
                    }
                }

                let mut y = diag[start].faer_abs2().faer_sub(mu);
                let mut z = diag[start].faer_mul(subdiag[start]);

                let simde = SimdFor::<E, S>::new(simd);
                let u_offset = simde.align_offset_ptr(
                    u.rb()
                        .map(|mat| mat.as_ptr())
                        .unwrap_or(E::faer_map(E::UNIT, |()| core::ptr::null())),
                    diag.len(),
                );
                let v_offset = simde.align_offset_ptr(
                    v.rb()
                        .map(|mat| mat.as_ptr())
                        .unwrap_or(E::faer_map(E::UNIT, |()| core::ptr::null())),
                    diag.len(),
                );

                for k in start..end - 1 {
                    let rot = JacobiRotation::make_givens(y, z);
                    if k > start {
                        subdiag[k - 1] = rot.c.faer_mul(y).faer_sub(rot.s.faer_mul(z)).faer_abs();
                    }

                    let mut diag_k = diag[k];

                    (diag_k, subdiag[k]) = (
                        simde.scalar_mul_add_e(
                            rot.c,
                            diag_k,
                            simde.scalar_mul(rot.s.faer_neg(), subdiag[k]),
                        ),
                        simde.scalar_mul_add_e(rot.s, diag_k, simde.scalar_mul(rot.c, subdiag[k])),
                    );

                    y = diag_k;
                    (z, diag[k + 1]) = (
                        simde.scalar_mul(rot.s.faer_neg(), diag[k + 1]),
                        simde.scalar_mul(rot.c, diag[k + 1]),
                    );

                    if let Some(u) = u.rb_mut() {
                        unsafe {
                            rot.apply_on_the_right_in_place_with_simd_and_offset(
                                simd,
                                u_offset,
                                u.rb().col(k).const_cast(),
                                u.rb().col(k + 1).const_cast(),
                            );
                        }
                    }

                    let rot = JacobiRotation::make_givens(y, z);

                    diag_k = rot.c.faer_mul(y).faer_sub(rot.s.faer_mul(z)).faer_abs();
                    diag[k] = diag_k;
                    (subdiag[k], diag[k + 1]) = (
                        simde.scalar_mul_add_e(
                            rot.c,
                            subdiag[k],
                            simde.scalar_mul(rot.s.faer_neg(), diag[k + 1]),
                        ),
                        simde.scalar_mul_add_e(
                            rot.s,
                            subdiag[k],
                            simde.scalar_mul(rot.c, diag[k + 1]),
                        ),
                    );

                    if k < end - 2 {
                        y = subdiag[k];
                        (z, subdiag[k + 1]) = (
                            simde.scalar_mul(rot.s.faer_neg(), subdiag[k + 1]),
                            simde.scalar_mul(rot.c, subdiag[k + 1]),
                        );
                    }

                    if let Some(v) = v.rb_mut() {
                        unsafe {
                            rot.apply_on_the_right_in_place_with_simd_and_offset(
                                simd,
                                v_offset,
                                v.rb().col(k).const_cast(),
                                v.rb().col(k + 1).const_cast(),
                            );
                        }
                    }
                }
            }
        }
    }

    E::Simd::default().dispatch(Impl {
        epsilon,
        consider_zero_threshold,
        max_iter,
        diag,
        subdiag,
        u: u.rb_mut(),
        v: v.rb_mut(),
    });

    for (j, d) in diag.iter_mut().enumerate() {
        if *d < E::faer_zero() {
            *d = d.faer_neg();
            if let Some(mut v) = v.rb_mut() {
                for i in 0..n {
                    v.write(i, j, v.read(i, j).faer_neg());
                }
            }
        }
    }

    for k in 0..n {
        let mut max = E::faer_zero();
        let mut max_idx = k;
        for kk in k..n {
            if diag[kk] > max {
                max = diag[kk];
                max_idx = kk;
            }
        }

        if k != max_idx {
            diag.swap(k, max_idx);
            if let Some(u) = u.rb_mut() {
                crate::perm::swap_cols_idx(u, k, max_idx);
            }
            if let Some(v) = v.rb_mut() {
                crate::perm::swap_cols_idx(v, k, max_idx);
            }
        }
    }

    for x in &mut *diag {
        *x = (*x).faer_mul(max_val);
    }
}

/// svd of bidiagonal lower matrix of shape (n + 1, n), with the last row being all zeros
pub fn compute_bidiag_real_svd<E: RealField>(
    diag: &mut [E],
    subdiag: &mut [E],
    mut u: Option<MatMut<'_, E>>,
    v: Option<MatMut<'_, E>>,
    jacobi_fallback_threshold: usize,
    bidiag_qr_fallback_threshold: usize,
    epsilon: E,
    consider_zero_threshold: E,
    parallelism: Parallelism,
    stack: PodStack<'_>,
) {
    let n = diag.len();

    if n <= jacobi_fallback_threshold {
        let (mut s, _) = temp_mat_zeroed::<E>(n, n, stack);
        let mut s = s.as_mut();

        for i in 0..n {
            s.write(i, i, diag[i]);
            if i + 1 < n {
                s.write(i + 1, i, subdiag[i]);
            }
        }

        jacobi_svd(
            s.rb_mut(),
            u.rb_mut().map(|u| u.submatrix_mut(0, 0, n, n)),
            v,
            Skip::None,
            epsilon,
            consider_zero_threshold,
        );

        for (i, diag) in diag.iter_mut().enumerate() {
            *diag = s.read(i, i);
        }
        if let Some(mut u) = u {
            zipped!(u.rb_mut().row_mut(n)).for_each(|unzipped!(mut x)| x.write(E::faer_zero()));
            zipped!(u.rb_mut().col_mut(n)).for_each(|unzipped!(mut x)| x.write(E::faer_zero()));
            u.write(n, n, E::faer_one());
        }
    } else if n <= bidiag_qr_fallback_threshold {
        bidiag_svd_qr_algorithm_impl(diag, subdiag, u, v, epsilon, consider_zero_threshold);
    } else {
        match u {
            Some(u) => bidiag_svd_impl(
                diag,
                subdiag,
                u,
                v,
                true,
                jacobi_fallback_threshold,
                epsilon,
                consider_zero_threshold,
                parallelism,
                stack,
            ),
            None => {
                let (mut u, stack) = temp_mat_uninit::<E>(2, n + 1, stack);
                let u = u.as_mut();
                bidiag_svd_impl(
                    diag,
                    subdiag,
                    u,
                    v,
                    false,
                    jacobi_fallback_threshold,
                    epsilon,
                    consider_zero_threshold,
                    parallelism,
                    stack,
                );
            }
        }
    }
}

/// svd of bidiagonal lower matrix
fn bidiag_svd_impl<E: RealField>(
    diag: &mut [E],
    subdiag: &mut [E],
    mut u: MatMut<'_, E>,
    mut v: Option<MatMut<'_, E>>,
    fill_u: bool,
    jacobi_fallback_threshold: usize,
    epsilon: E,
    consider_zero_threshold: E,
    parallelism: Parallelism,
    mut stack: PodStack<'_>,
) {
    let n = diag.len();

    let mut max_val = E::faer_zero();

    for x in &*diag {
        let val = x.faer_abs();
        if val > max_val {
            max_val = val;
        }
    }
    for x in &*subdiag {
        let val = x.faer_abs();
        if val > max_val {
            max_val = val;
        }
    }

    if max_val == E::faer_zero() {
        u.fill_zero();
        if u.nrows() == n + 1 {
            u.diagonal_mut().column_vector_mut().fill(E::faer_one());
        } else {
            u.write(0, 0, E::faer_one());
            u.write(1, n, E::faer_one());
        }
        if let Some(mut v) = v {
            v.fill_zero();
            v.diagonal_mut().column_vector_mut().fill(E::faer_one());
        };
        return;
    }

    for x in &mut *diag {
        *x = (*x).faer_div(max_val);
    }
    for x in &mut *subdiag {
        *x = (*x).faer_div(max_val);
    }

    assert!(subdiag.len() == n);
    assert!(n > jacobi_fallback_threshold);

    let k = n / 2;
    let rem = n - 1 - k;

    let (d1, alpha_d2) = diag.split_at_mut(k);
    let (sub_d1, beta_sub_d2) = subdiag.split_at_mut(k);
    let (alpha, d2) = alpha_d2.split_first_mut().unwrap();
    let (beta, sub_d2) = beta_sub_d2.split_first_mut().unwrap();
    let alpha = *alpha;
    let beta = *beta;

    let compact_u = (u.nrows() != n + 1) as usize;

    if k <= jacobi_fallback_threshold || rem <= jacobi_fallback_threshold {
        let (mut u1_alloc, stack) =
            temp_mat_uninit::<E>(k + 1, compact_u * (k + 1), stack.rb_mut());
        let mut u1_alloc = u1_alloc.as_mut();
        let (mut u2_alloc, stack) = temp_mat_uninit::<E>(rem + 1, compact_u * (rem + 1), stack);
        let mut u2_alloc = u2_alloc.as_mut();

        let (_u0, mut u1, mut u2) = if compact_u == 0 {
            let (u1, u2) = u.rb_mut().split_at_row_mut(k + 1);
            let (u0, u1) = u1.split_at_col_mut(1);
            (
                u0,
                u1.submatrix_mut(0, 0, k + 1, k + 1),
                u2.submatrix_mut(0, k, rem + 1, rem + 1),
            )
        } else {
            (
                u.rb_mut().col_mut(0).as_2d_mut(),
                u1_alloc.rb_mut(),
                u2_alloc.rb_mut(),
            )
        };

        let (mut v1, mut v2) = match v.rb_mut() {
            Some(v) => {
                let (v1, v2) = v.split_at_row_mut(k);
                (
                    Some(v1.submatrix_mut(0, 1, k, k + 1)),
                    Some(v2.submatrix_mut(1, k, rem, rem + 1)),
                )
            }
            None => (None, None),
        };

        let (mut matrix1, stack) = temp_mat_zeroed::<E>(k + 1, k + 1, stack);
        let (mut matrix2, _) = temp_mat_zeroed::<E>(rem + 1, rem + 1, stack);
        let mut matrix1 = matrix1.as_mut();
        let mut matrix2 = matrix2.as_mut();

        for j in 0..k {
            matrix1.write(j, j, d1[j]);
            matrix1.write(j + 1, j, sub_d1[j]);
        }
        for j in 0..rem {
            matrix2.write(j, j + 1, d2[j]);
            matrix2.write(j + 1, j + 1, sub_d2[j]);
        }

        jacobi_svd(
            matrix1.rb_mut(),
            Some(u1.rb_mut()),
            v1.rb_mut(),
            Skip::Last,
            epsilon,
            consider_zero_threshold,
        );
        for j in 0..matrix1.ncols() {
            for i in 0..matrix1.nrows() {
                if i != j {
                    matrix1.write(i, j, E::faer_zero());
                }
            }
        }
        jacobi_svd(
            matrix2.rb_mut(),
            Some(u2.rb_mut()),
            v2.rb_mut(),
            Skip::First,
            epsilon,
            consider_zero_threshold,
        );
        for j in 0..matrix2.ncols() {
            for i in 0..matrix1.nrows() {
                if i != j {
                    matrix1.write(i, j, E::faer_zero());
                }
            }
        }

        if cfg!(debug_assertions) {
            if let Some(v1) = v1 {
                zipped!(v1.col_mut(k)).for_each(|unzipped!(mut x)| x.write(E::faer_nan()));
            }
            if let Some(v2) = v2 {
                zipped!(v2.col_mut(0)).for_each(|unzipped!(mut x)| x.write(E::faer_nan()));
            }
        }

        for j in 0..k {
            diag[j + 1] = matrix1.read(j, j);
        }
        for j in 0..rem {
            diag[j + k + 1] = matrix2.read(j + 1, j + 1);
        }

        if compact_u == 1 {
            // need to copy the first and last rows
            //
            // NOTE: we handle the rotation of (Q1, q1) here, so no need to handle it later when
            // compact_u == 1
            for (row, row1, row2) in [(0, 0, 0), (1, k, rem)] {
                zipped!(
                    u.rb_mut().row_mut(row).subcols_mut(1, k),
                    u1_alloc.rb().row(row1).subcols(0, k),
                )
                .for_each(|unzipped!(mut dst, src)| dst.write(src.read()));
                u.write(row, 0, u1_alloc.read(row1, k));

                zipped!(
                    u.rb_mut().row_mut(row).subcols_mut(k + 1, rem),
                    u2_alloc.rb().row(row2).subcols(1, rem),
                )
                .for_each(|unzipped!(mut dst, src)| dst.write(src.read()));
                u.write(row, n, u2_alloc.read(row2, 0));
            }
        } else {
            let (_, u2) = if compact_u == 0 {
                let (u1, u2) = u.rb_mut().split_at_row_mut(k + 1);
                (u1, u2)
            } else {
                let (u1, u2) = u.rb_mut().split_at_row_mut(1);
                (u1, u2)
            };

            let (left, right) = u2.split_at_col_mut(k + 1);
            let left = left.col_mut(k);
            let right = right.col_mut(rem);
            zipped!(right, left).for_each(|unzipped!(mut right, mut left)| {
                right.write(left.read());

                if cfg!(debug_assertions) {
                    left.write(E::faer_nan());
                }
            });
        }
    } else {
        let (mut u1, mut u2) = if compact_u == 0 {
            let (u1, u2) = u.rb_mut().split_at_row_mut(k + 1);
            (
                u1.submatrix_mut(0, 1, k + 1, k + 1),
                u2.submatrix_mut(0, k + 1, rem + 1, rem + 1),
            )
        } else {
            // NOTE: need to handle rotation of Q1, q1
            let (u1, u2) = u.rb_mut().split_at_col_mut(k + 1);
            (u1, u2)
        };

        let (mut v1, mut v2) = match v.rb_mut() {
            Some(v) => {
                let (v1, v2) = v.split_at_row_mut(k);
                (
                    Some(v1.submatrix_mut(0, 1, k, k)),
                    Some(v2.submatrix_mut(1, k + 1, rem, rem)),
                )
            }
            None => (None, None),
        };

        let stack_bytes = stack.len_bytes();
        let (mem1, stack2) = stack.rb_mut().make_raw::<u8>(stack_bytes / 2);
        let stack1 = PodStack::new(mem1);

        join_raw(
            |parallelism| {
                bidiag_svd_impl(
                    d1,
                    sub_d1,
                    u1.rb_mut(),
                    v1.rb_mut(),
                    true,
                    jacobi_fallback_threshold,
                    epsilon,
                    consider_zero_threshold,
                    parallelism,
                    stack1,
                );
            },
            |parallelism| {
                bidiag_svd_impl(
                    d2,
                    sub_d2,
                    u2.rb_mut(),
                    v2.rb_mut(),
                    true,
                    jacobi_fallback_threshold,
                    epsilon,
                    consider_zero_threshold,
                    parallelism,
                    stack2,
                );
            },
            parallelism,
        );

        if compact_u == 1 {
            // handle rotation of Q1, q1
            for i in (0..k).rev() {
                crate::perm::swap_cols_idx(u1.rb_mut(), i, i + 1);
            }
        }

        for i in (0..k).rev() {
            diag[i + 1] = diag[i];
        }
    }

    if let Some(mut v) = v.rb_mut() {
        v.write(k, 0, E::faer_one());
    };

    let lambda = if compact_u == 0 {
        u.read(k, k + 1)
    } else {
        // we already rotated u
        u.read(1, 0)
    };
    let phi = if compact_u == 0 {
        u.read(k + 1, n)
    } else {
        u.read(0, n)
    };

    let al = alpha.faer_mul(lambda);
    let bp = beta.faer_mul(phi);

    let r0 = ((al.faer_mul(al)).faer_add(bp.faer_mul(bp))).faer_sqrt();
    let (c0, s0) = if r0 == E::faer_zero() {
        (E::faer_one(), E::faer_zero())
    } else {
        (al.faer_div(r0), bp.faer_div(r0))
    };

    let col0 = subdiag;
    diag[0] = r0;
    col0[0] = r0;

    if compact_u == 0 {
        let (u1, u2) = if compact_u == 0 {
            let (u1, u2) = u.rb_mut().split_at_row_mut(k + 1);
            (u1, u2)
        } else {
            let (u1, u2) = u.rb_mut().split_at_row_mut(1);
            (u1, u2)
        };

        let (mut u0_top, u1) = u1.split_at_col_mut(1);
        let (u1, mut un_top) = u1.split_at_col_mut(n - 1);
        let (mut u0_bot, u2) = u2.split_at_col_mut(1);
        let (u2, mut un_bot) = u2.split_at_col_mut(n - 1);

        for j in 0..k {
            col0[j + 1] = alpha.faer_mul(u1.read(k, j));
        }
        for j in 0..rem {
            col0[j + 1 + k] = beta.faer_mul(u2.read(0, j + k));
        }

        zipped!(
            u0_top.rb_mut().col_mut(0),
            un_top.rb_mut().col_mut(0),
            u1.col_mut(k),
        )
        .for_each(|unzipped!(mut dst0, mut dstn, mut src)| {
            let src_ = src.read();
            dst0.write(c0.faer_mul(src_));
            dstn.write(s0.faer_neg().faer_mul(src_));
            if cfg!(debug_assertions) {
                src.write(E::faer_nan());
            }
        });

        zipped!(u0_bot.rb_mut().col_mut(0), un_bot.rb_mut().col_mut(0),).for_each(
            |unzipped!(mut dst0, mut dstn)| {
                let src_ = dstn.read();
                dst0.write(s0.faer_mul(src_));
                dstn.write(c0.faer_mul(src_));
            },
        );
    } else {
        for j in 0..k {
            col0[j + 1] = alpha.faer_mul(u.read(1, j + 1));
            u.write(1, j + 1, E::faer_zero());
        }
        for j in 0..rem {
            col0[j + 1 + k] = beta.faer_mul(u.read(0, j + k + 1));
            u.write(0, j + k + 1, E::faer_zero());
        }

        let q10 = u.read(0, 0);
        let q21 = u.read(1, n);

        u.write(0, 0, c0.faer_mul(q10));
        u.write(0, n, s0.faer_neg().faer_mul(q10));
        u.write(1, 0, s0.faer_mul(q21));
        u.write(1, n, c0.faer_mul(q21));
    }

    let (perm, stack) = stack.rb_mut().make_with(n, |_| 0usize);
    let (jacobi_coeffs, stack) = stack.make_with(n, |_| JacobiRotation {
        c: E::faer_zero(),
        s: E::faer_zero(),
    });
    let (jacobi_indices, mut stack) = stack.make_with(n, |_| 0);

    let (jacobi_0i, jacobi_ij) = {
        let (transpositions, stack) = stack.rb_mut().make_with(n, |_| 0usize);
        deflate(
            diag,
            col0,
            jacobi_coeffs,
            jacobi_indices,
            transpositions,
            perm,
            k,
            epsilon,
            consider_zero_threshold,
            stack,
        )
    };

    let allocate_vm = v.is_some() as usize;
    let allocate_um = fill_u as usize;
    let (mut um, stack) = temp_mat_zeroed::<E>(n + 1, allocate_um * (n + 1), stack);
    let (mut vm, mut stack) = temp_mat_zeroed::<E>(n, allocate_vm * n, stack);
    let mut um = um.as_mut();
    let mut vm = vm.as_mut();

    compute_svd_of_m(
        fill_u.then_some(um.rb_mut()),
        v.is_some().then_some(vm.rb_mut()),
        diag,
        col0,
        perm,
        epsilon,
        consider_zero_threshold,
        stack.rb_mut(),
    );

    for (rot, &i) in jacobi_coeffs[jacobi_0i..][..jacobi_ij]
        .iter()
        .zip(&jacobi_indices[jacobi_0i..][..jacobi_ij])
        .rev()
    {
        let (i, j) = (i - 1, i);
        let actual_i = perm[i];
        let actual_j = perm[j];

        if fill_u {
            let (row_i, row_j) = um.rb_mut().two_rows_mut(actual_i, actual_j);
            rot.apply_on_the_left_in_place(row_i, row_j);
        }

        if v.is_some() {
            let (row_i, row_j) = vm.rb_mut().two_rows_mut(actual_i, actual_j);
            rot.apply_on_the_left_in_place(row_i, row_j);
        }
    }

    if fill_u {
        for (rot, &i) in jacobi_coeffs[..jacobi_0i]
            .iter()
            .zip(&jacobi_indices[..jacobi_0i])
            .rev()
        {
            let (row_0, row_i) = um.rb_mut().two_rows_mut(0, i);
            rot.apply_on_the_left_in_place(row_0, row_i);
        }
    }

    let _v_is_none = v.is_none();

    let mut update_v = |parallelism: Parallelism<'_>, stack: PodStack<'_>| {
        let (mut combined_v, _) = temp_mat_uninit::<E>(n, allocate_vm * n, stack);
        let mut combined_v = combined_v.as_mut();
        let v_rhs = vm.rb();

        if let Some(mut v) = v.rb_mut() {
            let mut combined_v = combined_v.rb_mut();
            let (mut combined_v1, combined_v2) = combined_v.rb_mut().split_at_row_mut(k);
            let mut combined_v2 = combined_v2.submatrix_mut(1, 0, rem, n);

            let v_lhs = v.rb();
            let v_lhs1 = v_lhs.submatrix(0, 1, k, k);
            let v_lhs2 = v_lhs.submatrix(k + 1, k + 1, rem, rem);
            let (v_rhs1, v_rhs2) = v_rhs.split_at_row(1).1.split_at_row(k);

            join_raw(
                |parallelism| {
                    crate::linalg::matmul::matmul(
                        combined_v1.rb_mut(),
                        v_lhs1,
                        v_rhs1,
                        None,
                        E::faer_one(),
                        parallelism,
                    )
                },
                |parallelism| {
                    crate::linalg::matmul::matmul(
                        combined_v2.rb_mut(),
                        v_lhs2,
                        v_rhs2,
                        None,
                        E::faer_one(),
                        parallelism,
                    )
                },
                parallelism,
            );

            crate::linalg::matmul::matmul(
                combined_v.rb_mut().submatrix_mut(k, 0, 1, n),
                v_lhs.submatrix(k, 0, 1, 1),
                v_rhs.submatrix(0, 0, 1, n),
                None,
                E::faer_one(),
                parallelism,
            );

            zipped!(v.rb_mut(), combined_v.rb())
                .for_each(|unzipped!(mut dst, src)| dst.write(src.read()));
        }
    };

    let mut update_u = |parallelism: Parallelism<'_>, stack: PodStack<'_>| {
        let (mut combined_u, _) = temp_mat_uninit::<E>(n + 1, allocate_um * (n + 1), stack);
        let mut combined_u = combined_u.as_mut();

        if fill_u {
            let (mut combined_u1, mut combined_u2) = combined_u.rb_mut().split_at_row_mut(k + 1);
            let u_lhs = u.rb();
            let u_rhs = um.rb();
            let (u_lhs1, u_lhs2) = (
                u_lhs.submatrix(0, 0, k + 1, k + 1),
                u_lhs.submatrix(k + 1, k + 1, rem + 1, rem + 1),
            );
            let (u_rhs1, u_rhs2) = u_rhs.split_at_row(k + 1);

            join_raw(
                |parallelism| {
                    // matrix matrix
                    crate::linalg::matmul::matmul(
                        combined_u1.rb_mut(),
                        u_lhs1,
                        u_rhs1,
                        None,
                        E::faer_one(),
                        parallelism,
                    );
                    // rank 1 update
                    crate::linalg::matmul::matmul(
                        combined_u1.rb_mut(),
                        u_lhs.col(n).subrows(0, k + 1).as_2d(),
                        u_rhs2.row(rem).as_2d(),
                        Some(E::faer_one()),
                        E::faer_one(),
                        parallelism,
                    );
                },
                |parallelism| {
                    // matrix matrix
                    crate::linalg::matmul::matmul(
                        combined_u2.rb_mut(),
                        u_lhs2,
                        u_rhs2,
                        None,
                        E::faer_one(),
                        parallelism,
                    );
                    // rank 1 update
                    crate::linalg::matmul::matmul(
                        combined_u2.rb_mut(),
                        u_lhs.col(0).subrows(k + 1, rem + 1).as_2d(),
                        u_rhs1.row(0).as_2d(),
                        Some(E::faer_one()),
                        E::faer_one(),
                        parallelism,
                    );
                },
                parallelism,
            );

            zipped!(u.rb_mut(), combined_u.rb())
                .for_each(|unzipped!(mut dst, src)| dst.write(src.read()));
        }
    };

    if compact_u == 1 {
        update_v(parallelism, stack.rb_mut());
        if fill_u {
            let (mut combined_u, _) = temp_mat_uninit::<E>(2, n + 1, stack);
            let mut combined_u = combined_u.as_mut();
            crate::linalg::matmul::matmul(
                combined_u.rb_mut(),
                u.rb(),
                um.rb(),
                None,
                E::faer_one(),
                parallelism,
            );
            zipped!(u.rb_mut(), combined_u.rb())
                .for_each(|unzipped!(mut dst, src)| dst.write(src.read()));
        }
    } else {
        match parallelism {
            #[cfg(feature = "rayon")]
            Parallelism::Rayon(_) if !_v_is_none => {
                let req_v = crate::linalg::temp_mat_req::<E>(n, n).unwrap();
                let (mem_v, stack_u) =
                    stack.make_aligned_raw::<u8>(req_v.size_bytes(), req_v.align_bytes());
                let stack_v = PodStack::new(mem_v);
                crate::utils::thread::join_raw(
                    move |parallelism| update_v(parallelism, stack_v),
                    move |parallelism| update_u(parallelism, stack_u),
                    parallelism,
                );
            }
            _ => {
                update_v(parallelism, stack.rb_mut());
                update_u(parallelism, stack);
            }
        }
    }

    for x in &mut *diag {
        *x = (*x).faer_mul(max_val);
    }
}

pub fn bidiag_real_svd_req<E: Entity>(
    n: usize,
    jacobi_fallback_threshold: usize,
    compute_u: bool,
    compute_v: bool,
    parallelism: Parallelism,
) -> Result<StackReq, SizeOverflow> {
    if n <= jacobi_fallback_threshold {
        temp_mat_req::<E>(n, n)
    } else {
        let _ = parallelism;
        let perm = StackReq::try_new::<usize>(n)?;
        let jacobi_coeffs = StackReq::try_new::<JacobiRotation<E>>(n)?;
        let jacobi_indices = perm;
        let transpositions = perm;
        let real_ind = perm;
        let real_col = perm;

        let um = temp_mat_req::<E>(n + 1, n + 1)?;
        let vm = temp_mat_req::<E>(n, if compute_v { n } else { 0 })?;

        let combined_u = temp_mat_req::<E>(if compute_u { n + 1 } else { 2 }, n + 1)?;
        let combined_v = vm;

        let prologue = StackReq::try_all_of([perm, jacobi_coeffs, jacobi_indices])?;

        StackReq::try_all_of([
            prologue,
            um,
            vm,
            combined_u,
            combined_v,
            transpositions,
            real_ind,
            real_col,
        ])
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::{assert, Mat};
    use assert_approx_eq::assert_approx_eq;

    macro_rules! make_stack {
        ($req: expr) => {
            ::dyn_stack::PodStack::new(&mut ::dyn_stack::GlobalPodBuffer::new($req.unwrap()))
        };
    }

    // to avoid overflowing the stack
    macro_rules! vec_static {
    ($($x:expr),+ $(,)?) => (
        {
            static ARRAY: &[f64] = &[$($x),+];
            ARRAY.to_vec()
        }
    );
    }

    #[test]
    fn test_svd_n() {
        for n in [9, 16, 32, 64, 128, 256, 512, 1024] {
            let diag = (0..n).map(|_| rand::random::<f64>()).collect::<Vec<_>>();
            let subdiag = (0..n).map(|_| rand::random::<f64>()).collect::<Vec<_>>();
            dbg!(&diag, &subdiag);

            let n = diag.len();
            let mut u = Mat::from_fn(n + 1, n + 1, |_, _| f64::NAN);
            let mut v = Mat::from_fn(n, n, |_, _| f64::NAN);
            let s = {
                let mut diag = diag.clone();
                let mut subdiag = subdiag.clone();
                compute_bidiag_real_svd(
                    &mut diag,
                    &mut subdiag,
                    Some(u.as_mut()),
                    Some(v.as_mut()),
                    5,
                    0,
                    f64::EPSILON,
                    f64::MIN_POSITIVE,
                    Parallelism::None,
                    make_stack!(bidiag_real_svd_req::<f64>(
                        n,
                        5,
                        true,
                        true,
                        Parallelism::None
                    )),
                );
                Mat::from_fn(n + 1, n, |i, j| if i == j { diag[i] } else { 0.0 })
            };

            let reconstructed = &u * &s * v.transpose();
            for j in 0..n {
                for i in 0..n + 1 {
                    let target = if i == j {
                        diag[j]
                    } else if i == j + 1 {
                        subdiag[j]
                    } else {
                        0.0
                    };

                    assert_approx_eq!(reconstructed.read(i, j), target, 1e-10);
                }
            }
        }
    }

    #[test]
    fn test_svd_4() {
        let diag = vec_static![1.0, 2.0, 3.0, 4.0];
        let subdiag = vec_static![1.0, 1.0, 1.0];

        let n = diag.len();
        let mut u = Mat::from_fn(n, n, |_, _| f64::NAN);
        let mut v = Mat::from_fn(n, n, |_, _| f64::NAN);
        let s = {
            let mut diag = diag.clone();
            let mut subdiag = subdiag.clone();
            bidiag_svd_qr_algorithm_impl(
                &mut diag,
                &mut subdiag,
                Some(u.as_mut()),
                Some(v.as_mut()),
                f64::EPSILON,
                f64::MIN_POSITIVE,
            );
            Mat::from_fn(n, n, |i, j| if i == j { diag[i] } else { 0.0 })
        };

        let reconstructed = &u * &s * v.transpose();
        for j in 0..n {
            for i in 0..n {
                let target = if i == j {
                    diag[j]
                } else if i == j + 1 {
                    subdiag[j]
                } else {
                    0.0
                };

                assert_approx_eq!(reconstructed.read(i, j), target, 1e-10);
            }
        }
    }

    #[test]
    fn test_svd_64() {
        let diag = vec_static![
            0.5488135039273248,
            0.6027633760716439,
            0.4236547993389047,
            0.4375872112626925,
            0.9636627605010293,
            0.7917250380826646,
            0.5680445610939323,
            0.07103605819788694,
            0.02021839744032572,
            0.7781567509498505,
            0.978618342232764,
            0.46147936225293185,
            0.11827442586893322,
            0.1433532874090464,
            0.5218483217500717,
            0.26455561210462697,
            0.45615033221654855,
            0.018789800436355142,
            0.6120957227224214,
            0.9437480785146242,
            0.359507900573786,
            0.6976311959272649,
            0.6667667154456677,
            0.2103825610738409,
            0.31542835092418386,
            0.5701967704178796,
            0.9883738380592262,
            0.2088767560948347,
            0.6531083254653984,
            0.4663107728563063,
            0.15896958364551972,
            0.6563295894652734,
            0.1965823616800535,
            0.8209932298479351,
            0.8379449074988039,
            0.9764594650133958,
            0.9767610881903371,
            0.7392635793983017,
            0.2828069625764096,
            0.29614019752214493,
            0.317983179393976,
            0.06414749634878436,
            0.5666014542065752,
            0.5232480534666997,
            0.5759464955561793,
            0.31856895245132366,
            0.13179786240439217,
            0.2894060929472011,
            0.5865129348100832,
            0.8289400292173631,
            0.6778165367962301,
            0.7351940221225949,
            0.24875314351995803,
            0.592041931271839,
            0.2230816326406183,
            0.44712537861762736,
            0.6994792753175043,
            0.8137978197024772,
            0.8811031971111616,
            0.8817353618548528,
            0.7252542798196405,
            0.9560836347232239,
            0.4238550485581797,
            0.019193198309333526,
        ];
        let subdiag = vec_static![
            0.7151893663724195,
            0.5448831829968969,
            0.6458941130666561,
            0.8917730007820798,
            0.3834415188257777,
            0.5288949197529045,
            0.925596638292661,
            0.08712929970154071,
            0.832619845547938,
            0.8700121482468192,
            0.7991585642167236,
            0.7805291762864555,
            0.6399210213275238,
            0.9446689170495839,
            0.4146619399905236,
            0.7742336894342167,
            0.5684339488686485,
            0.6176354970758771,
            0.6169339968747569,
            0.6818202991034834,
            0.43703195379934145,
            0.06022547162926983,
            0.6706378696181594,
            0.1289262976548533,
            0.3637107709426226,
            0.43860151346232035,
            0.10204481074802807,
            0.16130951788499626,
            0.2532916025397821,
            0.24442559200160274,
            0.11037514116430513,
            0.1381829513486138,
            0.3687251706609641,
            0.09710127579306127,
            0.09609840789396307,
            0.4686512016477016,
            0.604845519745046,
            0.039187792254320675,
            0.1201965612131689,
            0.11872771895424405,
            0.41426299451466997,
            0.6924721193700198,
            0.2653894909394454,
            0.09394051075844168,
            0.9292961975762141,
            0.6674103799636817,
            0.7163272041185655,
            0.18319136200711683,
            0.020107546187493552,
            0.004695476192547066,
            0.27000797319216485,
            0.9621885451174382,
            0.5761573344178369,
            0.5722519057908734,
            0.952749011516985,
            0.8464086724711278,
            0.29743695085513366,
            0.39650574084698464,
            0.5812728726358587,
            0.6925315900777659,
            0.5013243819267023,
            0.6439901992296374,
            0.6063932141279244,
            0.30157481667454933,
        ];

        let n = diag.len();
        let mut u = Mat::from_fn(n + 1, n + 1, |_, _| f64::NAN);
        let mut v = Mat::from_fn(n, n, |_, _| f64::NAN);
        let s = {
            let mut diag = diag.clone();
            let mut subdiag = subdiag.clone();
            compute_bidiag_real_svd(
                &mut diag,
                &mut subdiag,
                Some(u.as_mut()),
                Some(v.as_mut()),
                15,
                0,
                f64::EPSILON,
                f64::MIN_POSITIVE,
                Parallelism::None,
                make_stack!(bidiag_real_svd_req::<f64>(
                    n,
                    15,
                    true,
                    true,
                    Parallelism::None
                )),
            );
            Mat::from_fn(n + 1, n, |i, j| if i == j { diag[i] } else { 0.0 })
        };

        let reconstructed = &u * &s * v.transpose();
        for j in 0..n {
            for i in 0..n + 1 {
                let target = if i == j {
                    diag[j]
                } else if i == j + 1 {
                    subdiag[j]
                } else {
                    0.0
                };

                assert_approx_eq!(reconstructed.read(i, j), target, 1e-10);
            }
        }
    }

    #[test]
    fn test_svd_128() {
        let diag = vec_static![
            0.21604803719303378,
            0.3911685871373043,
            0.4717353802588816,
            0.5258511967180588,
            0.3495587375007383,
            0.29956620660149913,
            0.9617586737752963,
            0.1358382160501187,
            0.7927594027427639,
            0.7002703649721469,
            0.5011867621846828,
            0.5508360458872776,
            0.6671001529243077,
            0.6182640702775855,
            0.537113258218727,
            0.6494319004775305,
            0.22394544080467793,
            0.48963764985534675,
            0.4960750561790864,
            0.6762779313777806,
            0.4942507487962028,
            0.30598289084328734,
            0.7477750830615565,
            0.4134175601075717,
            0.16210479706508774,
            0.8554869501826141,
            0.9633922725373281,
            0.5178447186808554,
            0.4808128823427542,
            0.21235530384938095,
            0.34390969950363515,
            0.5222397627933266,
            0.8078540262388403,
            0.3084527067162488,
            0.8510243010197533,
            0.7492658574080864,
            0.2971760706318315,
            0.5821109217348188,
            0.9355688927263782,
            0.6568884170143395,
            0.7143623902994362,
            0.8745547764908594,
            0.3166725157072694,
            0.06280104609776738,
            0.9988219571244557,
            0.3034566490500038,
            0.6043605519679998,
            0.5327046414132618,
            0.8160784550544813,
            0.33220426954591,
            0.3160884461616036,
            0.08177180318496124,
            0.5859174456552851,
            0.225028943377522,
            0.6862486995947498,
            0.3697307197174694,
            0.7873879339970076,
            0.21908989285489933,
            0.5410943047067103,
            0.6302243946164361,
            0.17747192668740164,
            0.6281781273604742,
            0.5854835895783808,
            0.6512696242357562,
            0.559113383282545,
            0.7596325146050337,
            0.09312745837133651,
            0.501703867727036,
            0.949275885265856,
            0.620974047588036,
            0.3874150582755552,
            0.7083379430913563,
            0.75288059477905,
            0.1270527976228708,
            0.23126586686009443,
            0.12024520920717885,
            0.5679798123160427,
            0.17978590193238875,
            0.6968486822029739,
            0.7157516389948776,
            0.863508815070862,
            0.15864367506453403,
            0.11417600568460162,
            0.9651132813595767,
            0.10920826282790252,
            0.28700997153205676,
            0.7054115856120382,
            0.3490250121285702,
            0.29128696537701393,
            0.9304161241740285,
            0.2268455711369768,
            0.7658439715371834,
            0.06071820836196018,
            0.027168754664580574,
            0.4433866338712438,
            0.8175541779071445,
            0.1195115235309906,
            0.5543104624561522,
            0.3831276253977298,
            0.4944969243226346,
            0.5069163526882893,
            0.2519761931522101,
            0.3802289988930322,
            0.12792877754948118,
            0.964418293878996,
            0.5028833771104924,
            0.7140891912929843,
            0.929920514299548,
            0.9622470160446642,
            0.9165762824392009,
            0.957409262046926,
            0.046890401426795014,
            0.9559558333706967,
            0.10165169845100896,
            0.4030729711821963,
            0.7457966905965205,
            0.45506350389528505,
            0.22855385350793034,
            0.5774367409801651,
            0.3395031602763888,
            0.8750661230154188,
            0.5436696130661226,
            0.14750222902451415,
            0.4702601766248026,
            0.380398914581512,
            0.9870908933390458,
            0.46972043263478913,
            0.7629347676594994,
        ];
        let subdiag = vec_static![
            0.5656024571426115,
            0.45845513013153916,
            0.8464475246274293,
            0.5477997098985157,
            0.8505749005789477,
            0.036655381821360744,
            0.2164923701172936,
            0.992162216073592,
            0.2442305366823595,
            0.5417621610202344,
            0.4608606260638025,
            0.41227354070140787,
            0.9159226592102876,
            0.41719392867697913,
            0.30450224568165174,
            0.913124938919393,
            0.8998452924705163,
            0.8615757311218621,
            0.2152284123127688,
            0.4290329466601026,
            0.5684735244446122,
            0.5679483074313831,
            0.34457343811812624,
            0.5415296298206568,
            0.8356244784272918,
            0.8166459785498866,
            0.5080772280859633,
            0.8956149463267449,
            0.04644596806209644,
            0.3433783039423306,
            0.6431291583487241,
            0.4639223708461163,
            0.8505923626098933,
            0.04989146706988801,
            0.7831842489006767,
            0.27932627070175853,
            0.4802138742827945,
            0.5760948972572593,
            0.5545322682351106,
            0.21873418294084268,
            0.431998289203515,
            0.28960427805813527,
            0.20805283252719742,
            0.4598338664038314,
            0.03434413796223135,
            0.8739945690639832,
            0.5815729275918338,
            0.19359623734135956,
            0.7141431528881316,
            0.5483053379185088,
            0.3859904909506836,
            0.32779561877996766,
            0.7473245156367854,
            0.7527840401770349,
            0.5332180778014928,
            0.18845765841689788,
            0.8863719798459543,
            0.04292531786719711,
            0.7455031922487241,
            0.46581152947672433,
            0.2528295777598255,
            0.3175198547141883,
            0.45895268958636226,
            0.5910400449400208,
            0.5907671590866751,
            0.7200711220634236,
            0.06754760808821003,
            0.4622760636221317,
            0.6725272498965258,
            0.842299904498038,
            0.675399181783893,
            0.8815937503757029,
            0.4030870020741989,
            0.7417783045865814,
            0.27829985877736574,
            0.16223373545195352,
            0.047472863984402536,
            0.9762183713220033,
            0.850015796049705,
            0.9883602721062648,
            0.7183825826458256,
            0.6776325302074241,
            0.5606755750903751,
            0.4901171002893656,
            0.726732063504903,
            0.5776852831317164,
            0.7123622484466453,
            0.755897963752805,
            0.5446524557651221,
            0.3337687893180128,
            0.3075795178050936,
            0.8001257236535856,
            0.18895675106770227,
            0.3579844168353461,
            0.3527072586559228,
            0.0817878204567436,
            0.9969511917959174,
            0.6404417022308447,
            0.28418361844714657,
            0.6511969684943811,
            0.2866537500197578,
            0.33561205627307966,
            0.2534861786545628,
            0.9960188356208826,
            0.10107370019257966,
            0.6295541630401397,
            0.5638169140807354,
            0.7619672332144566,
            0.19599633482758116,
            0.004262801281641138,
            0.5637409510062904,
            0.15132931034408448,
            0.6357856412777408,
            0.3570943637285525,
            0.8986379725558856,
            0.6123637576833882,
            0.21915528413252194,
            0.35614983876651884,
            0.9830902701738703,
            0.2656035472280712,
            0.8894933827255144,
            0.5619386899890909,
            0.019131462959508294,
            0.36198104450872315,
            0.8387140376274692,
            0.2967905230261907,
            0.009493153610368066,
            0.4451542437520205,
        ];

        let n = diag.len();
        let mut u = Mat::from_fn(n + 1, n + 1, |_, _| f64::NAN);
        let mut v = Mat::from_fn(n, n, |_, _| f64::NAN);
        let s = {
            let mut diag = diag.clone();
            let mut subdiag = subdiag.clone();
            compute_bidiag_real_svd(
                &mut diag,
                &mut subdiag,
                Some(u.as_mut()),
                Some(v.as_mut()),
                40,
                0,
                f64::EPSILON,
                f64::MIN_POSITIVE,
                Parallelism::None,
                make_stack!(bidiag_real_svd_req::<f64>(
                    n,
                    40,
                    true,
                    true,
                    Parallelism::None
                )),
            );
            Mat::from_fn(n + 1, n, |i, j| if i == j { diag[i] } else { 0.0 })
        };

        let reconstructed = &u * &s * v.transpose();
        for j in 0..n {
            for i in 0..n + 1 {
                let target = if i == j {
                    diag[j]
                } else if i == j + 1 {
                    subdiag[j]
                } else {
                    0.0
                };

                assert_approx_eq!(reconstructed.read(i, j), target, 1e-10);
            }
        }
    }

    #[test]
    fn test_svd_1024_0() {
        let diag = vec_static![
            0.8845251118613418,
            0.34455256888844044,
            0.210674711024898,
            0.19415197496797754,
            0.9667549946932977,
            0.9929941756848952,
            0.3449032796124365,
            0.013043707957282269,
            0.5379750826661898,
            0.9878349516485647,
            0.7840176804531493,
            0.05421416657132472,
            0.6276152195489152,
            0.5302903207188766,
            0.1489678571817551,
            0.5910099656870687,
            0.8125771508983507,
            0.019854461222473585,
            0.23300742519619422,
            0.9261309512615169,
            0.5099296602111684,
            0.396690210400782,
            0.2657945708665065,
            0.04886273313636724,
            0.5138205614876258,
            0.12184534545958348,
            0.8914471736391029,
            0.9265260573331591,
            0.4878431401362272,
            0.7697237381965547,
            0.009936509018894535,
            0.8800411633128924,
            0.41970310146014045,
            0.053141783595033076,
            0.6082362328591278,
            0.4297917877465598,
            0.9264338860742358,
            0.20337132094924437,
            0.6186986895951906,
            0.514494342538388,
            0.36647591921360345,
            0.7909607320635065,
            0.11648430104115892,
            0.9981247894173411,
            0.4214625077906733,
            0.27873435601711005,
            0.06275412441803119,
            0.17994040410520007,
            0.5767826619151486,
            0.8276287407761077,
            0.4856049420452119,
            0.1824024117042553,
            0.380547967315335,
            0.18277861527693784,
            0.8560272319230807,
            0.7352358350367258,
            0.37244824553668243,
            0.08858898352613309,
            0.15670394303205137,
            0.9549608259831481,
            0.4609226155156112,
            0.2752940722916162,
            0.14648689328571252,
            0.24702880747653244,
            0.26413558185873487,
            0.25963973131499496,
            0.7874760238719776,
            0.5392390398518286,
            0.43108494111567286,
            0.9602150966834309,
            0.263919286112432,
            0.5519733682457418,
            0.27087432517628174,
            0.41373002164894046,
            0.4961242345741296,
            0.8786873455034356,
            0.22141198367945658,
            0.12815828684009156,
            0.24793784677162356,
            0.3242928455356374,
            0.5232575470210502,
            0.781790771620908,
            0.2378803144741315,
            0.9224957075362035,
            0.8654377128841579,
            0.22767383234003058,
            0.6367731312601166,
            0.8505361623040123,
            0.020146188482329075,
            0.5206013694815037,
            0.36776570341882464,
            0.37256832562995335,
            0.2371097865141898,
            0.32354779308058046,
            0.3840014267954045,
            0.9273526050829554,
            0.6974410601518757,
            0.4684376632291788,
            0.33199244231586855,
            0.1592208015718991,
            0.8457916545164874,
            0.7594540340761741,
            0.5977160139755071,
            0.2448529282667069,
            0.7422321131631072,
            0.4491255167076257,
            0.877793952642107,
            0.2061115436687152,
            0.5955960979297951,
            0.3641242521958884,
            0.06830447764964187,
            0.29548562339469486,
            0.14656535751600874,
            0.9347902426607325,
            0.551454375693908,
            0.05755666167494633,
            0.1182662384866503,
            0.9969493917667304,
            0.7774163485337313,
            0.7232409678464244,
            0.953755416217937,
            0.13987247560464577,
            0.8056655147304023,
            0.8381653208805445,
            0.5208871733781028,
            0.7353011028495107,
            0.7522163928333057,
            0.10541375581323387,
            0.09041155597045147,
            0.8667408478176504,
            0.9654448739439632,
            0.005792304462705622,
            0.2335819251562139,
            0.5369003975588765,
            0.03737234922010857,
            0.4588058730077743,
            0.884749406352933,
            0.07830951962815003,
            0.05162426233203987,
            0.1968400879118768,
            0.8007876035442365,
            0.7968477315552086,
            0.8047714077241233,
            0.2765763666146831,
            0.19054608072462764,
            0.5407650387375886,
            0.44575925860601684,
            0.69988681364929,
            0.02921195559473666,
            0.9519063242299393,
            0.2292637344597196,
            0.57168345100491,
            0.6119802248741711,
            0.9061002127200055,
            0.835234531347637,
            0.21775743544975845,
            0.598015663069716,
            0.7246168958089019,
            0.18660869930659219,
            0.6860807800890482,
            0.6207537425828924,
            0.04036114082282971,
            0.2596034256687044,
            0.9263145409044506,
            0.12006520113520502,
            0.06526114830309226,
            0.6060749180627996,
            0.830726692759335,
            0.8819438566592785,
            0.2823300181080588,
            0.004762366303322829,
            0.15705700354656182,
            0.872273157439614,
            0.05161458570595012,
            0.19404590278273093,
            0.673337106135467,
            0.5276143405427874,
            0.15032516518663774,
            0.7269693382522674,
            0.6496354105288265,
            0.8783772718768026,
            0.7230285777369317,
            0.8006872911221266,
            0.6166519065388856,
            0.9526515838852074,
            0.13932837641394247,
            0.27769520707366524,
            0.6915062763055476,
            0.512026353250563,
            0.2632486782448227,
            0.5995502982365921,
            0.8976384724135182,
            0.43965952491907645,
            0.05856887773794872,
            0.04886342450006653,
            0.46804720854588455,
            0.7525286155087892,
            0.9556104870431635,
            0.9135617349595734,
            0.10507903894086212,
            0.3874922350178007,
            0.9433755059296061,
            0.40312789184461495,
            0.7281420809216822,
            0.7473556564126743,
            0.13580390254853258,
            0.4793108553831614,
            0.6077475752583249,
            0.09916537750409427,
            0.984284070196559,
            0.8563424060832624,
            0.5371224391731257,
            0.6848273225152729,
            0.6507685185025187,
            0.5547937322274868,
            0.4056063327295283,
            0.5804860496295368,
            0.8124064239085033,
            0.1602324786734518,
            0.09880451576259175,
            0.09935758772113779,
            0.8971081077362497,
            0.11222279366053156,
            0.42060122955982093,
            0.6114566179885966,
            0.6453598215339088,
            0.4912286655584044,
            0.8837577596587839,
            0.601795323666988,
            0.7152818307776255,
            0.5926042612262687,
            0.36686793218606273,
            0.0313842872598018,
            0.36981194752406976,
            0.007013645381377498,
            0.6233497518351521,
            0.19247812929961905,
            0.3366789389253526,
            0.4837733261985061,
            0.7060286649954945,
            0.6560485260353782,
            0.30678422327474575,
            0.05424227380794244,
            0.9809566991181687,
            0.9679876980114167,
            0.37060417403309087,
            0.8876909232855882,
            0.9266175828719014,
            0.6157004519300252,
            0.4742621496185583,
            0.4716225437585564,
            0.42558581920979377,
            0.6870915543189346,
            0.2106157507697909,
            0.7148685882645731,
            0.29741609982987893,
            0.05585922166871293,
            0.16842926321869744,
            0.269368257543357,
            0.09856436162547,
            0.6318795405201129,
            0.06554775449804462,
            0.7513407348457041,
            0.2836215405898581,
            0.1328047454725776,
            0.09368346629167423,
            0.5376447868406243,
            0.3562742880762211,
            0.6976956411522769,
            0.8652060945219386,
            0.6927438310591169,
            0.8055640024374024,
            0.7116858599057524,
            0.6436185630532363,
            0.00622643683652091,
            0.45442170067986876,
            0.1372780376144771,
            0.9439243200885419,
            0.5291647990839722,
            0.5027492197791616,
            0.9617660041030734,
            0.7804120429985603,
            0.27125510499764616,
            0.8643033678059027,
            0.5619503443692003,
            0.8188456709963385,
            0.4012996542042663,
            0.23310987710285913,
            0.8899985323819251,
            0.4664831476280643,
            0.6729125114401061,
            0.9664025202400076,
            0.43486799439819446,
            0.8974277439810396,
            0.4808449065701321,
            0.11506094057610716,
            0.25379791153831577,
            0.03964053497879083,
            0.36909125512271623,
            0.4029973377655117,
            0.009001224984161449,
            0.9758190049357144,
            0.39023786547431394,
            0.9112427690611551,
            0.779990667772811,
            0.9036018460133876,
            0.18512984695811086,
            0.9178143533632862,
            0.4278781144480831,
            0.661581165339597,
            0.6343292826215128,
            0.31166415713361884,
            0.856217051781153,
            0.3196697246058282,
            0.1245222224484781,
            0.7728908759067107,
            0.9636719737186232,
            0.9340817362172681,
            0.3221972107257043,
            0.31519271142473704,
            0.8976309840829237,
            0.7086399432314598,
            0.04136495166094478,
            0.07658868727707802,
            0.2363447085818272,
            0.8878358508273737,
            0.9174891002547204,
            0.781061898467373,
            0.5988437757616014,
            0.29070843935762014,
            0.44093957477344725,
            0.11435842604864599,
            0.38771810414159247,
            0.009518348153462197,
            0.06806110689805611,
            0.62129101334924,
            0.11035187903178978,
            0.45351609034461293,
            0.47733641219479117,
            0.059036119775712326,
            0.28778565060683126,
            0.7395520480546994,
            0.6364194028524534,
            0.33159486012856376,
            0.13146467242925763,
            0.20926575331965835,
            0.5914783076243179,
            0.14538250921489848,
            0.3285367514346902,
            0.14111259631818374,
            0.9271206639868662,
            0.4791305932719009,
            0.42484371918265673,
            0.9243113101525169,
            0.04657014131556869,
            0.47012270169625714,
            0.017578423681492317,
            0.41192041951803826,
            0.5435010948082887,
            0.830019400684995,
            0.9838262050532701,
            0.31155683385462063,
            0.3395810989977315,
            0.008514249063874657,
            0.36804963249184464,
            0.39065717416407375,
            0.48060664877288317,
            0.9177524734572041,
            0.9963808703554067,
            0.46762091468546574,
            0.2190669248616377,
            0.7546402963954308,
            0.3826675586492012,
            0.4519670427156247,
            0.10034147317999753,
            0.9339045232941123,
            0.6861661261352751,
            0.01108362938610885,
            0.42186561885257856,
            0.6049633961733282,
            0.6665693539826331,
            0.43832298084278354,
            0.28759644404552964,
            0.42589599699235514,
            0.6215685157259329,
            0.5819897901940292,
            0.654993364802773,
            0.8849572516939255,
            0.056661097249976033,
            0.6252616876482108,
            0.12912956119956975,
            0.5937043009630114,
            0.3567979519308234,
            0.6651293721946168,
            0.4162450059551198,
            0.7866799944868544,
            0.33559766465195273,
            0.16910220850833024,
            0.2557155444695407,
            0.1371622094265289,
            0.5169211909209518,
            0.27000552870068484,
            0.8070007223783799,
            0.35499549142147346,
            0.014302822401554782,
            0.22722151291096326,
            0.07998829912888339,
            0.8194348476014353,
            0.88446073640522,
            0.6619719078328444,
            0.9367053935121534,
            0.23570771605208773,
            0.844467854244812,
            0.08641849711080207,
            0.1608054804669894,
            0.1752041524603587,
            0.6092665138366469,
            0.33373985941726614,
            0.9113357896883022,
            0.11595639850509243,
            0.7068757086199514,
            0.7865607768581245,
            0.9346872146042198,
            0.05564406010499112,
            0.2699085433214231,
            0.05591730860133559,
            0.6608433401410738,
            0.530444670200964,
            0.8319757527334114,
            0.486955173829685,
            0.41444998950490164,
            0.7846459652439971,
            0.07682909713484198,
            0.028896143688622145,
            0.7735911432210456,
            0.8561347987430765,
            0.6833766669122127,
            0.732403990897978,
            0.07393737509067311,
            0.6859173206119581,
            0.10353213742451339,
            0.9490710297122695,
            0.34603734460546876,
            0.7042465786381801,
            0.6744746690394707,
            0.734932043801607,
            0.41656957539578376,
            0.5227623335281744,
            0.32679798434601914,
            0.713091728672727,
            0.10090676539403032,
            0.08333177896943456,
            0.43136813553231124,
            0.5553735253983637,
            0.9938059660537949,
            0.2038100855893512,
            0.7904924245747698,
            0.7654582607613682,
            0.2632496613842946,
            0.5032987983538753,
            0.10059506853686151,
            0.05392849355937801,
            0.27997229159761305,
            0.05738118418904803,
            0.5888198410031323,
            0.5496228609661271,
            0.18124563158960938,
            0.9279548341660727,
            0.22356900241713673,
            0.5074439639227654,
            0.8465638446831127,
            0.9939222841611288,
            0.009203738044384568,
            0.9035625463960713,
            0.5478498401870193,
            0.1436740504780365,
            0.11030013507966019,
            0.49435562218779827,
            0.36474996817626115,
            0.8690729143610514,
            0.27512438505007153,
            0.05519547164147387,
            0.19233743278188775,
            0.0664409503721326,
            0.6489080576699797,
            0.019063712733404903,
            0.1534430886662448,
            0.24480545379252816,
            0.01051499963138458,
            0.6856116940528466,
            0.7773097044715064,
            0.3677858272104545,
            0.5607469272924686,
            0.08438163910055485,
            0.4927293124318882,
            0.3758536484533026,
            0.9852448384568894,
            0.38038214870756404,
            0.4768862738938858,
            0.42504128314102363,
            0.884574088023932,
            0.7447948103427381,
            0.6281952560439542,
            0.8422943316973285,
            0.11117974951930387,
            0.6788820387560264,
            0.9748171029633533,
            0.3112345837082654,
            0.19692145192748078,
            0.1898685764540009,
            0.9232274745660699,
            0.17044419712823644,
            0.4851270278696377,
            0.5536690025965553,
            0.209418220241673,
            0.6259250634518518,
            0.7543334303125396,
            0.2937849309999855,
            0.3729476658139417,
            0.39313457462054435,
            0.06974764236154585,
            0.10344310048189886,
            0.507518560753304,
            0.43774470454608494,
            0.8364146679204518,
            0.9070492787267092,
            0.9567774773222549,
            0.10500970235931761,
            0.3998498929261717,
            0.29717649355166853,
            0.7615834186247338,
            0.6342900101100667,
            0.2849863073120511,
            0.037515223183286706,
            0.1435650512414116,
            0.4946558413658533,
            0.24838964588562307,
            0.06953183293100962,
            0.010979908802799532,
            0.6470620854580164,
            0.1489505156426364,
            0.9318992263165846,
            0.11352771228732728,
            0.16700430653862086,
            0.4766929353339845,
            0.5097455503317575,
            0.4326982921969347,
            0.7741042678051568,
            0.08165168787991572,
            0.5578748265687337,
            0.2308499062220727,
            0.5123779157582458,
            0.24100763785021273,
            0.10886023061825767,
            0.3463078209397997,
            0.4346935062407594,
            0.3541824862033963,
            0.8506208375314257,
            0.11865040517795522,
            0.8787835948303524,
            0.7514204500151836,
            0.3537273898842549,
            0.18495105215340069,
            0.8794538168214154,
            0.43846955371546736,
            0.6223808367304243,
            0.2532422174248372,
            0.0173945729559013,
            0.825120152220085,
            0.8343521374210742,
            0.24514667775364773,
            0.5864829268587433,
            0.06114537616802951,
            0.08947759755430962,
            0.2670467129935481,
            0.6218659288801367,
            0.5701826218390645,
            0.30123481898013516,
            0.3555024041627374,
            0.5942989396022218,
            0.5850733554368567,
            0.3977819861337537,
            0.26570347091195734,
            0.25740366564861905,
            0.46423759145504806,
            0.40930616359772365,
            0.8642930629057376,
            0.8548459173798607,
            0.5610185499351202,
            0.9079259267908577,
            0.9540217218606191,
            0.9877136190157554,
            0.25638467812786225,
            0.45228748162692256,
            0.23021853631358136,
            0.7606298373681132,
            0.11809446396524093,
            0.5464203016842142,
            0.82014799173527,
            0.5673485070159224,
            0.28189744630354674,
            0.5728473340310175,
            0.12745779045010586,
            0.2093600651192693,
            0.040409181249943193,
            0.5437097498036081,
            0.4698843713650859,
            0.3744100758683092,
            0.8820853651881632,
            0.09661682517079428,
            0.44008282207016947,
            0.7661341654608439,
            0.5002899778280783,
            0.939935188435343,
            0.4037845646767523,
            0.4754687371351335,
            0.12348699298976351,
            0.8328535547922692,
            0.7550678974094668,
            0.13420599429716162,
            0.37226957043440323,
            0.35897133577902696,
            0.2839857243007452,
            0.30151359234377895,
            0.4873323691626037,
            0.09644463427460526,
            0.3068651283752245,
            0.19457042965184612,
            0.13193683664769307,
            0.15402117137314475,
            0.12060810096037711,
            0.47588374471922557,
            0.9825405359602971,
            0.09506011601995101,
            0.0951473299180351,
            0.17552464191224793,
            0.5446585979359402,
            0.3933775183583844,
            0.5313822262288094,
            0.638956815956248,
            0.9173221237559014,
            0.4995138447644756,
            0.2610913790829166,
            0.4833107536687732,
            0.9971758070471496,
            0.3421854083206227,
            0.5080486727041559,
            0.4256910136975586,
            0.32446998261012305,
            0.08767191717339928,
            0.4374180631006207,
            0.26125518794943137,
            0.44389514759815984,
            0.9388266039769766,
            0.8201374509465024,
            0.7438169768604541,
            0.8032628491632073,
            0.09098688736942062,
            0.8547640562601011,
            0.5552333141514629,
            0.36260676800032277,
            0.5034045469637766,
            0.15186149958437956,
            0.3944636144675695,
            0.3721326486252522,
            0.923337028171655,
            0.41430211287983776,
            0.2602515627476548,
            0.9397086532157933,
            0.5230207413511277,
            0.475892714054884,
            0.694683555642716,
            0.5118255773056839,
            0.3113994477099723,
            0.5547390653920204,
            0.5893906407999145,
            0.7659093103586729,
            0.5075122380840992,
            0.18283934901868948,
            0.5698607238462075,
            0.13449423615296285,
            0.045459918245799424,
            0.08625964304367006,
            0.417395859295686,
            0.261062158102505,
            0.8075298740340944,
            0.8883665580333145,
            0.4889019655098171,
            0.2221188930219562,
            0.2816496152946476,
            0.15494833191304713,
            0.21418617334287815,
            0.3237557989949549,
            0.9196198742987929,
            0.18572851649416433,
            0.5285212603645132,
            0.9355753348529846,
            0.6891075139555204,
            0.257586325634688,
            0.43689292209889663,
            0.8657427762269848,
            0.20275374629883136,
            0.928494262842214,
            0.48760667502157895,
            0.40064262508174053,
            0.2925932318415707,
            0.2143069075286741,
            0.40430984322007,
            0.4463752172022737,
            0.9123971917156188,
            0.7471055164332462,
            0.12019130373697817,
            0.6346911798610109,
            0.404160753722707,
            0.38544851846509864,
            0.7365095934005528,
            0.6581005716429144,
            0.27835749174429714,
            0.8608667648331466,
            0.10716728595973024,
            0.7642064541269226,
            0.7732208264521794,
            0.24547528427024534,
            0.8093161349811911,
            0.281236029404172,
            0.5399086933128905,
            0.9555409254750834,
            0.8728156769659005,
            0.2392265872077518,
            0.41583479392446654,
            0.26231913777793525,
            0.882082615484338,
            0.7285497107647825,
            0.5363502438082618,
            0.21407252245857544,
            0.9790657697354228,
            0.34832348843681604,
            0.8995015876461342,
            0.5399994071733517,
            0.627367538005929,
            0.8936327303225186,
            0.1458309386700306,
            0.39889901760514235,
            0.021967589354523587,
            0.27632921178999303,
            0.44994401565091635,
            0.4526171030273327,
            0.7152832299491756,
            0.5370716005441079,
            0.9290413356376639,
            0.7638540725626427,
            0.29043023254730993,
            0.028544561435507876,
            0.2597221997555347,
            0.028028354812694056,
            0.016281249462922864,
            0.920307786946952,
            0.10851409325535322,
            0.37557097156207275,
            0.7966291035142514,
            0.2150703311155887,
            0.12780106678926428,
            0.017508933064456667,
            0.8615446835117896,
            0.5377076451210898,
            0.8495505289919821,
            0.45096310987004407,
            0.5250602002063562,
            0.07124839465430677,
            0.6185298746083393,
            0.2814465744642637,
            0.8171841827449374,
            0.7584862446805867,
            0.12567788435821203,
            0.5930540147639154,
            0.8849415301031589,
            0.17427330180330625,
            0.2010378264968402,
            0.39174436702226,
            0.9012408312896981,
            0.1386741748649597,
            0.38639748547112107,
            0.09006322004054756,
            0.9082962363057114,
            0.811784624155095,
            0.6743784350882154,
            0.09906487768907013,
            0.9309469870173297,
            0.8897197263370551,
            0.12541115209796438,
            0.23312512510466665,
            0.5132385782782324,
            0.7275904539321562,
            0.9057213452366281,
            0.7297530279941175,
            0.791978695353606,
            0.37092185205852835,
            0.8483398896367347,
            0.30955294767063535,
            0.771189644766232,
            0.2906290667202338,
            0.7453198057550258,
            0.21653899571012458,
            0.9775513518255299,
            0.7254220951183088,
            0.8668113703090181,
            0.6194065009415832,
            0.2863451172116679,
            0.5667190680021205,
            0.925653625989403,
            0.033560039914012574,
            0.8973876639754387,
            0.7224153371687022,
            0.8064677351942471,
            0.5780794168031764,
            0.7660235251315161,
            0.2620511545407739,
            0.5841291237263148,
            0.40459159694220515,
            0.12814289255326505,
            0.4588594921757614,
            0.20444429900397298,
            0.943652543350375,
            0.16833992985205037,
            0.7340254790523432,
            0.941407765783495,
            0.8125595396371769,
            0.5615635814574278,
            0.023829005699015915,
            0.0894218297915016,
            0.9726621750060844,
            0.27371882433441985,
            0.820099804114962,
            0.6495996966118213,
            0.16101762786768825,
            0.1423893361655213,
            0.5309658651157899,
            0.7697814537672145,
            0.7580719157068739,
            0.9399829889011113,
            0.3639016293263453,
            0.8838453513063597,
            0.39192310465767566,
            0.8617173246541459,
            0.805017619875162,
            0.6524173955629243,
            0.4596275958450332,
            0.7026014717755514,
            0.15674099767732896,
            0.9306233380687109,
            0.03954219378811974,
            0.5429358106690747,
            0.9574396141996595,
            0.3423698971057674,
            0.2447146653935679,
            0.09921039521270503,
            0.7378442271637177,
            0.12563985902014574,
            0.5326702755764166,
            0.045217673715322926,
            0.38389292763866045,
            0.5354917794116903,
            0.5845898831760369,
            0.16812202883973026,
            0.5485428102270978,
            0.8416052158014439,
            0.6206682880766721,
            0.4435313057311998,
            0.151713060737244,
            0.48157407311325273,
            0.6986899281158176,
            0.2210887015121874,
            0.07005386341944864,
            0.014868415754305087,
            0.5194135227188144,
            0.39088624580450126,
            0.7605832049265179,
            0.6762673208469522,
            0.8291607514571048,
            0.5767977185364629,
            0.1476360889153545,
            0.8713989228781874,
            0.4201968970228106,
            0.27597036603247105,
            0.20622555712119583,
            0.23377214116657952,
            0.6378667544453603,
            0.4808971835768795,
            0.6942176290429053,
            0.7514562431659588,
            0.3254218376076441,
            0.7759065993020333,
            0.8688364664055,
            0.058653408092502635,
            0.49549567998888433,
            0.03657833451630399,
            0.16917910139727643,
            0.42411804891524285,
            0.9161646353617822,
            0.5736335260984191,
            0.8560891920908282,
            0.8295971825773533,
            0.6198112877891949,
            0.9533525740629326,
            0.2228519083099909,
            0.12958204310042476,
            0.02964863024908193,
            0.5827534617648027,
            0.4744264395380964,
            0.29056315277638267,
            0.6576898201570917,
            0.942398882416274,
            0.6366537862128913,
            0.7155637078822991,
            0.3371336756466209,
            0.9127030992249006,
            0.6100868652627647,
            0.5419675813942905,
            0.4314235263828414,
            0.2229787266376665,
            0.5751203571398641,
            0.5377920941894427,
            0.6184651925697617,
            0.10166459142369644,
            0.5006884922345262,
            0.8473281521128757,
            0.354027469370917,
            0.989332514582519,
            0.6564861907905184,
            0.571402636513833,
            0.6317054209816227,
            0.3694852621662059,
            0.7574513830542177,
            0.45679332668735106,
            0.1271953777164424,
            0.5910815545784368,
            0.813904952296986,
            0.02310514525211138,
            0.7424699129535078,
            0.3915098149952104,
            0.01155938751400587,
            0.4031433809049668,
            0.678027647415598,
            0.7082527232743558,
            0.9353783729655919,
            0.8801376399629393,
            0.5387820724455229,
            0.2722569005205504,
            0.1470331398956941,
            0.8260710776157071,
            0.713566911335019,
            0.9170469271968411,
            0.13936763708492894,
            0.42161327076184363,
            0.28424349125683035,
            0.3671033730334905,
            0.13246773458329553,
            0.8973331016642114,
            0.6665503100026733,
            0.757804611455464,
            0.5896758029668179,
            0.19844119526434956,
            0.07843991721429922,
            0.9603382105509987,
            0.6974678498027387,
            0.9912625862864402,
            0.781246847308964,
            0.07999498724724818,
            0.8913156249997439,
            0.3145980902194806,
            0.12647260044256647,
            0.7946548793044863,
            0.033871301015447,
            0.5504501814363837,
            0.4511052584536204,
            0.774098664423315,
            0.2487470001472769,
            0.6067861889279945,
            0.6685445735264952,
            0.38709013510050394,
            0.9169539240621843,
            0.9500984625411802,
            0.8067182843492807,
            0.03180693064936568,
            0.6709304220148694,
            0.18076652905884238,
            0.250368676753443,
            0.2860975092729513,
            0.8710391195602349,
            0.167497840100998,
            0.7353235414942768,
            0.9352292778787517,
            0.4511306842770142,
            0.3827758461191245,
            0.281066928172492,
            0.6980469279053299,
            0.9201458951964375,
            0.016736962594040006,
            0.18537591560445787,
            0.6421454251873325,
            0.14953390974062974,
            0.3545167086497891,
            0.8357720236520648,
            0.6067021461866138,
            0.3807258166781119,
            0.0950091742072432,
            0.6519302579483973,
            0.1447369400465809,
            0.3768570579783762,
            0.9376829093113643,
            0.06614007922558363,
            0.38431596584393857,
            0.8842261159619674,
            0.5613192921444363,
            0.8808932748628524,
            0.9700610004017022,
            0.6144252347744887,
            0.32795571909065,
            0.2077968628811766,
            0.4474606065976239,
            0.9370658217651426,
            9.666962222321107e-5,
            0.1523470538193925,
            0.2122919170895673,
        ];
        let subdiag = vec_static![
            0.3056447441500063,
            0.7456252987130297,
            0.18795399969254234,
            0.10580210815412516,
            0.3667790559520604,
            0.11312018616553432,
            0.43130959590408824,
            0.12733946348946112,
            0.7754200015422843,
            0.11462492748669817,
            0.1646228849693424,
            0.5498181705657144,
            0.8215402007359489,
            0.5549515047858895,
            0.6747472950692731,
            0.9944413348382201,
            0.9009360373643603,
            0.3080754436431906,
            0.12964878635305754,
            0.3831765057928852,
            0.8174053443325175,
            0.917121275562522,
            0.6238095863626737,
            0.1930541259971955,
            0.1083599360191293,
            0.6664213913848369,
            0.9142140073621671,
            0.8317624912354198,
            0.5792497764455579,
            0.9528256828095099,
            0.7216235869434947,
            0.7580767812895445,
            0.8659742281381297,
            0.1860180109069518,
            0.7141567274413423,
            0.9929140205251573,
            0.01480728764882,
            0.7181791146645335,
            0.6989258997781399,
            0.7444815566895207,
            0.5935818769457846,
            0.42163998690313864,
            0.9477114391316797,
            0.34502405257829605,
            0.17545845562326523,
            0.059405897876968594,
            0.1726715671891963,
            0.6980682853819796,
            0.015667055926685935,
            0.7442929546620496,
            0.09489347192295339,
            0.26294454450087634,
            0.14678124983505092,
            0.9766379153111909,
            0.5446124500851669,
            0.6951033374681663,
            0.39424735336141103,
            0.3638362437695718,
            0.8268894263436269,
            0.03569331911488183,
            0.18008375334466242,
            0.21093348750655783,
            0.6403166330072856,
            0.24151644542863315,
            0.8007941708457903,
            0.9546164335175866,
            0.564693254286534,
            0.5939417115372281,
            0.848858061717452,
            0.678305051442029,
            0.34145082438373064,
            0.018996526002347913,
            0.5350645128642897,
            0.16380607335737096,
            0.9541311617136843,
            0.8972390112132765,
            0.20620851212196378,
            0.4064444449496657,
            0.22853398038768846,
            0.5625510649455169,
            0.3292394396387699,
            0.9499985301262932,
            0.46961796034565706,
            0.8550401342925872,
            0.11331071995283704,
            0.5370408275887198,
            0.4225061626572716,
            0.7113564007341124,
            0.5440695613095045,
            0.9429119229264217,
            0.9418192314199187,
            0.4595018133979456,
            0.11997303237263768,
            0.5050075121654448,
            0.8464315483316138,
            0.6001500262169397,
            0.014170583876986775,
            0.5891290090330017,
            0.8269559647969559,
            0.727008606346369,
            0.40308085040130714,
            0.4769274878744385,
            0.5031017174901194,
            0.9653308905352374,
            0.05820643913386647,
            0.5839503387605185,
            0.12402624718791366,
            0.9808307159271583,
            0.31183256753334077,
            0.2968868830122854,
            0.46023047923805205,
            0.14833982530111567,
            0.65765248704152,
            0.7931977034792722,
            0.10422377898659152,
            0.3237615937340822,
            0.44036120748135754,
            0.8152082481801677,
            0.5657751815100746,
            0.12247213033062843,
            0.12411466645776581,
            0.8111678326060824,
            0.8659650943526952,
            0.6401522697786636,
            0.8305130254791445,
            0.6155701616514582,
            0.976003503734365,
            0.5426197331282824,
            0.3930558684294502,
            0.415333251350124,
            0.6737291989761925,
            0.9073237675548902,
            0.5576736295634749,
            0.8829065362399116,
            0.8401069643961545,
            0.44484143164409407,
            0.8945051891278987,
            0.8324315271287057,
            0.37572810811171,
            0.5311645816253128,
            0.5410183184009483,
            0.19744166096326343,
            0.19842870677648106,
            0.5879941475040675,
            0.9786993854633442,
            0.5968620552818462,
            0.5135705169514763,
            0.9090223217449162,
            0.44535947745956805,
            0.3300597048965267,
            0.715628098634161,
            0.9203855120442683,
            0.4774988986047791,
            0.5589992822372473,
            0.186736834799486,
            0.4008272532964271,
            0.3954667759754853,
            0.6375743442561205,
            0.7570893839877972,
            0.3132382034582841,
            0.939759448592276,
            0.13880950488085908,
            0.8326450074085596,
            0.5435488869084204,
            0.8871764967336385,
            0.5104803787523359,
            0.8183761212207235,
            0.09841598011855401,
            0.7787772646287051,
            0.025782098684299593,
            0.1582336677185774,
            0.47189096656959695,
            0.6411639191498966,
            0.9121043592216188,
            0.8836178727290614,
            0.8619350562353174,
            0.35930466893572144,
            0.037447833449754,
            0.6479060568978083,
            0.11479409296465548,
            0.7151813234442181,
            0.8774288509162476,
            0.5088504695085865,
            0.30086979283431325,
            0.6783177195786527,
            0.8487535314833418,
            0.3917061519411893,
            0.1529037805915775,
            0.2794436368037698,
            0.3867049568857387,
            0.95028935609001,
            0.668297605120564,
            0.4091293928635532,
            0.826787369452402,
            0.49502893536128045,
            0.9204313334180478,
            0.04227152077501617,
            0.8591242352702294,
            0.08034515572168355,
            0.4016398475609876,
            0.552956667354517,
            0.995434784708499,
            0.7916465831808283,
            0.4431013751392342,
            0.6671168551178643,
            0.18963684541621384,
            0.9408021548483484,
            0.39524245248420486,
            0.4398351316404564,
            0.09865745210171994,
            0.34529364209770197,
            0.7896598088250412,
            0.620626100417164,
            0.8944054172862875,
            0.08461446884599422,
            0.5738108783945065,
            0.9234462345123426,
            0.4666998582752204,
            0.3511458700743658,
            0.687981064083925,
            0.5857159194448827,
            0.7796140288522504,
            0.7053391405697281,
            0.28374733052199586,
            0.2077966875285202,
            0.47248703407139336,
            0.10889214016591153,
            0.9907974152807177,
            0.5277388644238672,
            0.25542631824829676,
            0.6096386554262448,
            0.4853875369586189,
            0.7758278847547917,
            0.6503482206015344,
            0.7872521250255423,
            0.5835673769403026,
            0.054438190571200584,
            0.07841186389369081,
            0.42435351797029797,
            0.15148954511964963,
            0.013217228657095292,
            0.20321355812331443,
            0.41134865019643196,
            0.893762124482854,
            0.04578706985005354,
            0.7631748707976872,
            0.057836476103902634,
            0.7696513764038294,
            0.514646143350958,
            0.3699021639343091,
            0.2740557035261182,
            0.08325942766122896,
            0.21511166645435398,
            0.5173831302175944,
            0.37558859246961807,
            0.9425750468539488,
            0.14280581796726732,
            0.551969939510297,
            0.01932819087738935,
            0.5728385856388669,
            0.8283348728460406,
            0.07934738138747399,
            0.9705733034919944,
            0.30600584387572205,
            0.20438950881156082,
            0.6570611977784478,
            0.431894419621217,
            0.21644201229949733,
            0.2357002275663601,
            0.010031862472789643,
            0.7852407949785581,
            0.47591278164194006,
            0.017083534404552236,
            0.70441920699062,
            0.9879201237833972,
            0.2661143719211483,
            0.2933730098258043,
            0.913911512160399,
            0.5924086218087098,
            0.15018476931849045,
            0.7090500894817965,
            0.1578831976360242,
            0.37364308551885306,
            0.47221324235540074,
            0.2080799934617963,
            0.5960336468592128,
            0.0024262366029996763,
            0.9686573315244363,
            0.007790105364738564,
            0.7973099957337009,
            0.07772308814111217,
            0.28058789167890674,
            0.16115336259960567,
            0.27509642154779956,
            0.6947525070182677,
            0.5369825146778103,
            0.978827946469132,
            0.659858378865912,
            0.6527252000429293,
            0.7041139810749213,
            0.2869627484454058,
            0.33308842879697975,
            0.44159455295268457,
            0.27216835488560753,
            0.3393119948713913,
            0.7648329706153236,
            0.6966637044522928,
            0.2025159847212873,
            0.6374963882353294,
            0.4530678605483741,
            0.01198799668408701,
            0.8501904802162441,
            0.31602528613237324,
            0.5743196328856983,
            0.2041484874611329,
            0.31698102697078223,
            0.7251656050864485,
            0.18523041092012338,
            0.6475193739002286,
            0.4539029450472052,
            0.9443982414985209,
            0.3590332826624424,
            0.9670507123723026,
            0.48731607627716933,
            0.5297020372301194,
            0.5214213268459551,
            0.5455630655556016,
            0.28708268593677544,
            0.9311414177391263,
            0.8950938351965099,
            0.16123011234222318,
            0.22098266484499085,
            0.7493059992396844,
            0.5712912493855609,
            0.7748724242424189,
            0.1739137089632864,
            0.3759998878743801,
            0.43548864712056934,
            0.355953178481756,
            0.2934045169930304,
            0.8731884218145788,
            0.03932032739156133,
            0.730978570503939,
            0.8963555774338611,
            0.9235650437208576,
            0.019397306498324163,
            0.490413428295972,
            0.2280813710984807,
            0.3186296331862041,
            0.8588266145518306,
            0.827949675823181,
            0.04632864897129574,
            0.04514427058718651,
            0.6937002384358558,
            0.7248161565262558,
            0.48925517507535177,
            0.4218869798777849,
            0.05192442176785239,
            0.34126462267735147,
            0.07526076288545036,
            0.6615484694091124,
            0.3858483293934476,
            0.3040038827852344,
            0.45193192552327277,
            0.9725483191206692,
            0.9658066630850666,
            0.663806767413649,
            0.8982898058299352,
            0.47055757592740133,
            0.9177236633247238,
            0.3121548993263339,
            0.3278465102083806,
            0.4559364970566179,
            0.9335005702379894,
            0.8765771154233144,
            0.7446961762075289,
            0.5226080548332727,
            0.5239841658992287,
            0.7053040960804962,
            0.9747877890656634,
            0.30076239044021935,
            0.3635336573288519,
            0.25239032264109484,
            0.851046363997222,
            0.4102002699352153,
            0.5376447442499298,
            0.29695939161672513,
            0.7692080830680088,
            0.8077862697669561,
            0.8231780782958322,
            0.14925121737806113,
            0.09895355062061273,
            0.04238351423906839,
            0.6795101515783823,
            0.1836252581270097,
            0.5572449717764562,
            0.3448496904665723,
            0.8941305356130093,
            0.7754476123893801,
            0.4437958505318783,
            0.8131328674103127,
            0.3414329616729884,
            0.11087369270133707,
            0.17316839004945794,
            0.20096665035614536,
            0.25528470534727865,
            0.6186892404700086,
            0.2776263570822872,
            0.3917028828828043,
            0.6853848690727818,
            0.38794287559658314,
            0.36733234794384995,
            0.030467194704015377,
            0.13556571798486705,
            0.114452983143975,
            0.21227875846915634,
            0.7440182663283812,
            0.9553443480741944,
            0.8754730300219125,
            0.7362659679257295,
            0.5168804142498504,
            0.9278764580227046,
            0.05667135817563429,
            0.7977067629053805,
            0.6366757931184828,
            0.11550601919616832,
            0.7260010059519495,
            0.4258311238966114,
            0.14344953690709206,
            0.6854867267306801,
            0.5597128337100352,
            0.04519280943731574,
            0.3051578732289518,
            0.6941951391481741,
            0.7335129606264628,
            0.45287668627794686,
            0.8023458563642463,
            0.3355089550382604,
            0.23690241421025737,
            0.5865628088550219,
            0.8070294117926576,
            0.7124386404922848,
            0.8508405528573492,
            0.38998403890301847,
            0.815015129748834,
            0.4835569836615241,
            0.3794810774642573,
            0.4301823167622735,
            0.5777686495434117,
            0.0003715282384250118,
            0.7751840931504775,
            0.6848118511089216,
            0.3321502477454171,
            0.5726173363219021,
            0.1102768963346471,
            0.2050268517435817,
            0.8411717246701556,
            0.4359633005450527,
            0.06980411443387724,
            0.2276405768217764,
            0.4242564784378763,
            0.11031511244150671,
            0.35550185167373227,
            0.15961222186795354,
            0.8973558153349923,
            0.901699256812279,
            0.6109656303963225,
            0.15705655447271927,
            0.6039864598974551,
            0.4803097140654581,
            0.6886961751362133,
            0.5145445724508422,
            0.3885304600460471,
            0.06320845270286535,
            0.7926903801117495,
            0.17244112950216717,
            0.37968586799778037,
            0.5285337631120802,
            0.5581469584064788,
            0.7606546883505797,
            0.9801350487476163,
            0.17601211683642592,
            0.23016998819143686,
            0.9988735128922861,
            0.9964709310114106,
            0.3003878806236101,
            0.38715325920437826,
            0.8933941630898731,
            0.04044891940291784,
            0.2912809677735896,
            0.2048054109429659,
            0.8974107873594714,
            0.9015098002883342,
            0.3127354083014837,
            0.12022010780426284,
            0.779603168492849,
            0.6226056705611006,
            0.9501761800681467,
            0.282754714138722,
            0.3587714367899729,
            0.11307089195455344,
            0.6846512481308641,
            0.481035669443018,
            0.47720857165909913,
            0.2989436500983975,
            0.04195958303853875,
            0.943795202215035,
            0.7404344642271719,
            0.9200687466415679,
            0.9530475209590279,
            0.051576771787694375,
            0.37071948953905287,
            0.3023620352339571,
            0.8547931843849371,
            0.08280654010989841,
            0.25224156676090626,
            0.04676323990591791,
            0.9514986278924559,
            0.8873782411977091,
            0.34813698418421946,
            0.6888427139542841,
            0.9795211279872262,
            0.47563598498364135,
            0.8510891916037336,
            0.9957574580691713,
            0.8789164335285607,
            0.13899143629846944,
            0.33114356066954376,
            0.9804934223443615,
            0.5127871503741248,
            0.5491200087939495,
            0.12851835578042703,
            0.39361881630527806,
            0.407189498993495,
            0.4115818275499319,
            0.8534685193082066,
            0.3367047428929839,
            0.3875080369444178,
            0.3471177549950232,
            0.5410567225309034,
            0.016101658822052722,
            0.15343556374742873,
            0.6038258644435197,
            0.848910592498697,
            0.5440969764396003,
            0.7036065942941163,
            0.9363074855883752,
            0.690762068697277,
            0.810502996960463,
            0.9021388042924308,
            0.41061513647136283,
            0.3760313352802984,
            0.6534842342139258,
            0.17245604214887256,
            0.6545305414026276,
            0.09828309819907222,
            0.021347300043353834,
            0.06625066534939339,
            0.5811737328445331,
            0.2221369839526659,
            0.03386026217665872,
            0.5120593901028011,
            0.21604187959069587,
            0.6568365943735496,
            0.9228807070509343,
            0.9692443403037638,
            0.5039428835203609,
            0.473312973525324,
            0.8989304510416462,
            0.2893830738769403,
            0.5400512220559429,
            0.8317662543115151,
            0.3623927368486186,
            0.05592363514496557,
            0.2013084863718314,
            0.3511849575792956,
            0.04681970618106457,
            0.7438759357097672,
            0.40428289351850066,
            0.24886656470306912,
            0.41530975264763426,
            0.27360365999610137,
            0.18264151057750744,
            0.12633567080262964,
            0.14720341595628206,
            0.5963412693934422,
            0.7451023325387626,
            0.8403064154125609,
            0.9113802628195199,
            0.7925361242043578,
            0.4700198681208524,
            0.1624131121674216,
            0.1842174547458587,
            0.27103286978547636,
            0.036913276711137644,
            0.794120002199789,
            0.7847477377353473,
            0.9847126875015699,
            0.0604246807941794,
            0.1020765263813116,
            0.4001432174450992,
            0.9029570047752941,
            0.9935574220017901,
            0.08496185993123218,
            0.35193678925047556,
            0.6650825896044992,
            0.633123113949017,
            0.12042379186706065,
            0.7267519470749007,
            0.04960213723225171,
            0.8003428563837104,
            0.7105055279962602,
            0.6389089164725457,
            0.9327742233980036,
            0.5161833987704538,
            0.4119267138012994,
            0.7355326371619856,
            0.9258725256400323,
            0.2714885046018005,
            0.3078641242003942,
            0.4641400556798959,
            0.8967994682075197,
            0.09741890119083885,
            0.6961235966679272,
            0.641309602331906,
            0.7589672431976652,
            0.9460042050819939,
            0.16457408744224433,
            0.5905166066735666,
            0.0645854262660579,
            0.9225895487625081,
            0.8453244898116592,
            0.5383869429327891,
            0.8884890274777821,
            0.4197319265036752,
            0.7500156983166194,
            0.9891042589881829,
            0.01985050325526294,
            0.7134241950971687,
            0.3587169082881143,
            0.7525893263163326,
            0.6953390708586171,
            0.03232330711058795,
            0.6353086545865292,
            0.45124341761979414,
            0.9769930600025036,
            0.5669232572468578,
            0.938668658106906,
            0.4338164100168044,
            0.11222921731792379,
            0.03797720384967085,
            0.09844857763256376,
            0.11518910692815554,
            0.49466900134027825,
            0.3983581902084776,
            0.8316429889064332,
            0.47312999251333,
            0.4048903290459912,
            0.7884840506956975,
            0.39816934657167413,
            0.83333182620336,
            0.023453054038727217,
            0.3375802902232228,
            0.32676656023392314,
            0.8232734767141012,
            0.25256746203294367,
            0.7700913406909243,
            0.5389695844493064,
            0.5681167784974525,
            0.22419003095004353,
            0.9035474024664304,
            0.7863241515251058,
            0.3626483309717664,
            0.02491911392573798,
            0.55146177634034,
            0.21720842632939563,
            0.3041317541664137,
            0.9531364351621072,
            0.8973796968943464,
            0.229000671250574,
            0.8402370193406613,
            0.022037895183599443,
            0.836737917765931,
            0.6546192476595766,
            0.43845846524390697,
            0.18287290874651896,
            0.39722304854774637,
            0.3725183902897248,
            0.6359875693255428,
            0.761674061561912,
            0.9121528380976099,
            0.4037871092328117,
            0.07153474113947322,
            0.16417634442759,
            0.22179410578287384,
            0.6442255610401365,
            0.09656417127999872,
            0.9601509654146453,
            0.4733888258175044,
            0.7439657611528108,
            0.9675530630038234,
            0.1510564854296319,
            0.04369329308286296,
            0.309118725372204,
            0.623425445598849,
            0.29021374730693483,
            0.6766726069997415,
            0.8259349890902644,
            0.24871023553950133,
            0.5074934477683689,
            0.5856137161892475,
            0.4243963244550426,
            0.710578563910339,
            0.18834193797214582,
            0.6503852885476976,
            0.615751995056994,
            0.5014273566071713,
            0.6645690595539534,
            0.3405948211103986,
            0.19711784980950442,
            0.2998398002751148,
            0.9195616191263201,
            0.1505262293424886,
            0.8751730466670284,
            0.00834168424231696,
            0.025827354362135457,
            0.5824682659644557,
            0.15495960696894429,
            0.6559559366086971,
            0.6772137725181362,
            0.5636153937421962,
            0.7251286869551127,
            0.7726158652102284,
            0.3192164017737723,
            0.006218550100633324,
            0.8431317973283949,
            0.1425622398238403,
            0.0722025483794001,
            0.15911713561273344,
            0.3956011147755507,
            0.06941113474263982,
            0.12008503151088745,
            0.9082094438076037,
            0.6178024403175922,
            0.4235734206741463,
            0.15946541380135226,
            0.9290456481503413,
            0.7417571225789505,
            0.15009156352315944,
            0.6588396380864261,
            0.9080493138910737,
            0.1594600252406626,
            0.14932023883763978,
            0.5107192256238069,
            0.004524080201668057,
            0.1319270564403211,
            0.3453777437326653,
            0.8575218402523684,
            0.9720797451699371,
            0.721050556067865,
            0.719486529832955,
            0.3355247320271999,
            0.4320398768281565,
            0.1388342354348916,
            0.713952178822861,
            0.09473434985030837,
            0.16170333007911564,
            0.45399893003858427,
            0.4471522975096025,
            0.1620629777947441,
            0.555120679776045,
            0.9913221164835081,
            0.9916262640284876,
            0.6656790994905667,
            0.1584319326383905,
            0.7772315732793612,
            0.04950106833674606,
            0.6168568525923946,
            0.8850085432507142,
            0.6008569436635426,
            0.28434403407848496,
            0.5222091024862929,
            0.3289136157327879,
            0.8242517958509156,
            0.0014543544186828017,
            0.7055903162855025,
            0.6918733037053658,
            0.1215286623156362,
            0.606625560777267,
            0.38474228310690695,
            0.5561028312732113,
            0.6925434201360956,
            0.5030049415767984,
            0.4373757067462809,
            0.40262470465620004,
            0.29679518675170746,
            0.23443397779709685,
            0.9401971220147833,
            0.5022479785631987,
            0.8756315599830912,
            0.3566036956928199,
            0.17094319362349586,
            0.23332990304574608,
            0.3974251403805953,
            0.6744086868924933,
            0.8963282849932238,
            0.6278592405421798,
            0.03448142439565771,
            0.9835666101928536,
            0.4157584014025524,
            0.006801090241055574,
            0.2259553681297769,
            0.041359086533867884,
            0.17851980455078198,
            0.5069878837056208,
            0.387536883241537,
            0.2762301107628514,
            0.6649724905761829,
            0.12499462696151864,
            0.22684860045680122,
            0.41735033027464374,
            0.17412791593945343,
            0.6701391576784195,
            0.5307914849082684,
            0.40813851781982124,
            0.22390129513353285,
            0.005649833428942208,
            0.4883481964270028,
            0.8688437714062154,
            0.0935682881726475,
            0.12094810540039058,
            0.34318168456821574,
            0.8382723727680698,
            0.8198775509442384,
            0.7032040540890907,
            0.40973147740347793,
            0.8882182101776983,
            0.2593416896520184,
            0.7514215657589697,
            0.9891040983618512,
            0.5365707653429487,
            0.5337413102321141,
            0.06903019106136221,
            0.03681902345797561,
            0.3714071649975875,
            0.5895822360285221,
            0.27855398075340465,
            0.07192639886602326,
            0.8219230883311229,
            0.42371556956209966,
            0.9923543662319326,
            0.8757205988037184,
            0.2495530315044605,
            0.8918477804095697,
            0.8102897369370692,
            0.6608053584890721,
            0.6334165418357576,
            0.7346001037710248,
            0.3531776159961877,
            0.027802332060074764,
            0.29596932972525725,
            0.6611550391042507,
            0.05971419088778107,
            0.5336938252898652,
            0.6474078218411075,
            0.8733833124315973,
            0.27064285916942077,
            0.6800043429248648,
            0.7797586491845876,
            0.09633675865715385,
            0.15677008169215234,
            0.23394539969965789,
            0.7323421797485367,
            0.2695584209878511,
            0.8923611906456285,
            0.009384104149009809,
            0.20087989891307956,
            0.37650593207094696,
            0.9675672321088212,
            0.2460276680731922,
            0.8829436148568359,
            0.9011339590969748,
            0.9893348013640839,
            0.34402326744018086,
            0.2918830938824284,
            0.032443773270217524,
            0.20236048346275737,
            0.10644854768113488,
            0.39314962067289483,
            0.16844774292617437,
            0.0193640642109717,
            0.32131633915309843,
            0.9031289996826208,
            0.7996900072509187,
            0.9544912609903554,
            0.21238118940953343,
            0.5588780213505607,
            0.359423748193081,
            0.14306783265825895,
            0.03646714126807182,
            0.8108290011843635,
            0.5024978136603396,
            0.30684702940974795,
            0.08078866745769053,
            0.584846202681692,
            0.9731481619668385,
            0.015849082319753238,
            0.47318193572395295,
            0.45751317022352533,
            0.1347836174790057,
            0.07869729277442472,
            0.3730223469708691,
            0.8108036315360886,
            0.6191676048612971,
            0.3026033801925956,
            0.12952357665540415,
            0.9899082523324236,
            0.7427677782218289,
            0.5197759491989283,
            0.9967841038515952,
            0.5035569325914083,
            0.3834211729816368,
            0.3309587035330964,
            0.05340317428058894,
            0.6463414755064464,
            0.5887364268384743,
            0.1394604722694749,
            0.457951772941251,
            0.45288455141492356,
            0.08565262580204513,
            0.6084418050718142,
            0.8203820932095913,
            0.6461410253957783,
            0.2280002938711736,
            0.30887824900482863,
            0.9628221732310465,
            0.44209913122842126,
            0.35857995161780576,
            0.07182970907223107,
            0.171047907339957,
            0.24878787204659314,
            0.06756423474769113,
            0.8732787479160486,
            0.11342075749834801,
            0.9967087053746094,
            0.29132559183022533,
            0.7537365927002493,
            0.29658308252612886,
            0.8269346653596071,
            0.8714825210553282,
            0.07360104666475487,
            0.33337023074576533,
            0.9799778581582539,
            0.08375046754795701,
            0.8660403417189309,
            0.28286840561868565,
            0.8252295280561422,
            0.21889588864658394,
            0.003964186364546207,
            0.027000390024012777,
            0.4606990121656196,
            0.3297646362154407,
            0.8929719909234096,
            0.18414480606970096,
            0.5377843931338276,
            0.506843731844477,
            0.3338619626598003,
            0.300351599526833,
            0.1808568608406541,
            0.41477945200762745,
            0.5759119192571923,
            0.11284300062398211,
            0.749403315530038,
            0.3859482139006477,
            0.6051165122948982,
            0.971891394749644,
            0.6605478455153012,
            0.5861700067837108,
            0.07225848311996741,
            0.26352858091537257,
            0.8742611615172913,
            0.8420666125054983,
            0.30095829631010684,
            0.9035078353304578,
            0.6468305240161736,
            0.40631323088985216,
            0.26893323723422446,
            0.14348472814734614,
            0.8446063514723486,
            0.8841258110212956,
            0.259918014410066,
            0.6701006927979326,
            0.9864420189950698,
            0.7476246814792673,
            0.9505952494844244,
            0.3799973827243003,
            0.5547180102231944,
            0.7105616395309934,
            0.5383483239781777,
            0.31000439759468,
            0.7059163158915425,
            0.6856815369544714,
            0.39866345825125704,
            0.042513276582041515,
            0.337803158592229,
            0.39465691436662753,
            0.40306375626322977,
            0.8235182077717283,
            0.6090460629372503,
            0.6764030144841834,
            0.9635539428291812,
            0.7042014067502246,
            0.5127830179290652,
            0.9131677796050657,
            0.5311464655158742,
            0.8535060094374574,
            0.6649277719671861,
            0.8158423948021637,
            0.3003102858044917,
            0.9543186871563114,
            0.6693534602340823,
            0.9280256323955216,
            0.07378036482014472,
            0.7734809173963871,
        ];

        let n = diag.len();
        let mut u = Mat::from_fn(n + 1, n + 1, |_, _| f64::NAN);
        let mut v = Mat::from_fn(n, n, |_, _| f64::NAN);
        let s = {
            let mut diag = diag.clone();
            let mut subdiag = subdiag.clone();
            compute_bidiag_real_svd(
                &mut diag,
                &mut subdiag,
                Some(u.as_mut()),
                Some(v.as_mut()),
                40,
                0,
                f64::EPSILON,
                f64::MIN_POSITIVE,
                Parallelism::None,
                make_stack!(bidiag_real_svd_req::<f64>(
                    n,
                    40,
                    true,
                    true,
                    Parallelism::None
                )),
            );
            Mat::from_fn(n + 1, n, |i, j| if i == j { diag[i] } else { 0.0 })
        };

        let reconstructed = &u * &s * v.transpose();
        for j in 0..n {
            for i in 0..n + 1 {
                let target = if i == j {
                    diag[j]
                } else if i == j + 1 {
                    subdiag[j]
                } else {
                    0.0
                };

                assert_approx_eq!(reconstructed.read(i, j), target, 1e-10);
            }
        }
    }

    #[test]
    fn test_svd_1024_1() {
        let diag = vec_static![
            0.4841629363684429,
            0.09500211557292393,
            0.09054687338157585,
            0.9349389484044085,
            0.2263540442491473,
            0.9883453587159688,
            0.27611412994054574,
            0.6934210307229077,
            0.7122722731446822,
            0.5764931807281632,
            0.7220922855179277,
            0.7690715441721571,
            0.18151810699724935,
            0.9703738941665442,
            0.46756918888654386,
            0.13829262641884732,
            0.26785351679939384,
            0.6826209585171479,
            0.7450857828877535,
            0.9612828629996442,
            0.8597056636877132,
            0.3147189885973717,
            0.9873882525578956,
            0.6390222666607266,
            0.6720524846258185,
            0.5710278204457248,
            0.012892307681411919,
            0.7804378611928872,
            0.2207373153565294,
            0.6251662057647014,
            0.13965841961139402,
            0.2611627557457328,
            0.21717079350115864,
            0.43746631711094563,
            0.3551960866238878,
            0.6076465429515558,
            0.10918455630762303,
            0.45967703949396843,
            0.7809114058971589,
            0.9954329677023613,
            0.6578987749953844,
            0.4524030893215839,
            0.8561365938081589,
            0.2702994410284004,
            0.5910189554474038,
            0.9215907321467688,
            0.7637726604998021,
            0.9996866856048456,
            0.4508665111686113,
            0.8440519119195404,
            0.5636981910772803,
            0.19691331749849916,
            0.14415353894000804,
            0.5491145376147477,
            0.5355242003504422,
            0.3569416212159129,
            0.7716747444655886,
            0.6481040816879275,
            0.0861928588345302,
            0.5644989365621479,
            0.4595649555376554,
            0.9924614965479639,
            0.34457880027775034,
            0.2709464498548626,
            0.6175995628458563,
            0.06410776075791469,
            0.8668177074089747,
            0.5133288724007797,
            0.26197543298622306,
            0.636716245285106,
            0.07784169215536019,
            0.23103012381763322,
            0.820507398231268,
            0.023267897611993882,
            0.9922903501796004,
            0.28413038646415656,
            0.4902041468658134,
            0.5353728340280559,
            0.39068757308715574,
            0.24110407030206493,
            0.7982266250190029,
            0.7498583088545846,
            0.8054303600922323,
            0.727152156075704,
            0.4433406657279716,
            0.11813500917759767,
            0.19322091867414604,
            0.6073386848510687,
            0.1892620575417534,
            0.9276884465180404,
            0.07416066041374547,
            0.11055221860974573,
            0.02742841195102952,
            0.9046475741926477,
            0.33798444110426407,
            0.25146251279015375,
            0.32543843780189374,
            0.23502095326757122,
            0.8151869673056854,
            0.06550480813538595,
            0.27213001583974805,
            0.2645917156255321,
            0.5965380863001157,
            0.6850808677017469,
            0.5485690599538193,
            0.7758605807485486,
            0.2819101125296871,
            0.21744462405804543,
            0.3210524068951949,
            0.8219967960710458,
            0.7168902509705798,
            0.20198849159812726,
            0.886790290413851,
            0.8405592175465555,
            0.6418755287944,
            0.2717157722359368,
            0.8785193775058516,
            0.2883033828479482,
            0.10560195836536401,
            0.9457172912478532,
            0.24413132566350226,
            0.4798067689848272,
            0.9758153512537662,
            0.9461639347527152,
            0.5015824600544672,
            0.4743546024442775,
            0.9439300463479419,
            0.8210197093977561,
            0.9756341250452755,
            0.3034443196917138,
            0.800696521458751,
            0.2469174591716501,
            0.24844908962357637,
            0.6729189361923051,
            0.34976823478261343,
            0.4548480895382889,
            0.06654283936888206,
            0.0538242709114648,
            0.6604648475989987,
            0.3761972351630566,
            0.17973288886807337,
            0.7785638677716654,
            0.3078309007281408,
            0.34383089894111674,
            0.9740115657118916,
            0.4142242073362077,
            0.46788525339663745,
            0.5102644222807242,
            0.6821466796495953,
            0.9234699208966476,
            0.25792002693706884,
            0.4473866152881345,
            0.5316752542174584,
            0.6753097405426745,
            0.9875155323493832,
            0.4846734070202009,
            0.25690620257834906,
            0.17444353295771786,
            0.1459026770332964,
            0.03789979000177324,
            0.009118098382703943,
            0.3369152724463358,
            0.6194036398104508,
            0.8802810559789717,
            0.5818695975885244,
            0.26902985408430724,
            0.8102535632453027,
            0.14367013617397928,
            0.5154528225856342,
            0.5525644118916562,
            0.11068694771654464,
            0.06648831260453159,
            0.30229383062322257,
            0.11043335057841508,
            0.7879174452882572,
            0.6847400055584101,
            0.7093025409825697,
            0.5672680847728112,
            0.9220621299078362,
            0.2308678094100194,
            0.11095431934483901,
            0.5076880016394446,
            0.18043287756259263,
            0.3633969552240899,
            0.0074109593813797,
            0.09540490121094358,
            0.5571906415926369,
            0.43964175899243474,
            0.6761714080491062,
            0.4050321473554952,
            0.4602021774929316,
            0.07357681703454355,
            0.5647139088824532,
            0.6632368351901867,
            0.038928065259385525,
            0.44302015465469524,
            0.30679321194639764,
            0.7896327936646736,
            0.36808308055633976,
            0.9630604740173925,
            0.5525981053006727,
            0.4390151783304377,
            0.5986597671093334,
            0.7053735351729479,
            0.3744694351975014,
            0.06532632305974129,
            0.7215457382122921,
            0.008636006836127419,
            0.6280218545206359,
            0.33553959659277455,
            0.25002369784269374,
            0.2222248299935472,
            0.4568576034259546,
            0.4735434309364681,
            0.7560202412217341,
            0.2878333331705232,
            0.8562148011742544,
            0.9300668918750148,
            0.5038181128961342,
            0.8752815415361238,
            0.33283717808771585,
            0.0533181654462932,
            0.6436829043555116,
            0.6420080671552673,
            0.35854475824887166,
            0.27660688607425366,
            0.09673817671242757,
            0.7732513019446627,
            0.8411109662991648,
            0.749581543720349,
            0.9699796229928836,
            0.3521026232752017,
            0.7805969271105201,
            0.488079532732169,
            0.3153834803216925,
            0.07515270460745327,
            0.9732385585699602,
            0.4379249859226594,
            0.47941135538293433,
            0.13531367195050192,
            0.7146259058201841,
            0.4191544792989341,
            0.9918711123891871,
            0.30163052510779575,
            0.15398133758394816,
            0.8062176573881823,
            0.48038219879486577,
            0.3693644458963321,
            0.2627309145392913,
            0.7075668012609716,
            0.4565937211789599,
            0.5928922749230827,
            0.3384720695832677,
            0.7510771063489731,
            0.09656778467148652,
            0.6111109196994062,
            0.4147274155510545,
            0.0825902347020141,
            0.23342446109854698,
            0.8755256481911295,
            0.3410479507631834,
            0.49101534840303174,
            0.9696314869369961,
            0.510646800941389,
            0.21155499454128557,
            0.04077883012959882,
            0.38616102006555564,
            0.27156802107462563,
            0.06772016667809999,
            0.08411279644339542,
            0.8105121747384336,
            0.8226062374641675,
            0.44436768427765405,
            0.8725783369785883,
            0.779280645376425,
            0.3493455256878125,
            0.8904257905232991,
            0.17885649868819264,
            0.37091811243722606,
            0.9379307789143213,
            0.5731629435054184,
            0.24143821812058197,
            0.4049562680058758,
            0.42299735446198383,
            0.2527001487162306,
            0.9054521922102531,
            0.8277991529134413,
            0.8896621223418875,
            0.2839601147382753,
            0.46967983028760474,
            0.6504612353037362,
            0.34359028291120974,
            0.49029076200099064,
            0.14605930621786978,
            0.7971485298165085,
            0.5670859996593979,
            0.5839996800579184,
            0.6921622958466791,
            0.013697871099110137,
            0.5716412855888595,
            0.33906402404202585,
            0.22346066558462552,
            0.3951530817090858,
            0.3137771706646575,
            0.17589811380719156,
            0.6129863037725723,
            0.5529667540622067,
            0.12509740175859485,
            0.08164145825063607,
            0.6066900494595772,
            0.841451822666421,
            0.674802798921404,
            0.6185795283245251,
            0.9582067789561727,
            0.2569207105730952,
            0.6883723276402567,
            0.9437634272458615,
            0.6907567856518468,
            0.30993500261298346,
            0.35730588644284866,
            0.03729512159713966,
            0.6212835719526415,
            0.9352121927645822,
            0.003243518888762509,
            0.5186143509666988,
            0.6353445184976869,
            0.3617368415649064,
            0.0490224440259972,
            0.050915012421679506,
            0.2242457283292124,
            0.8147096772905736,
            0.6409998571108203,
            0.4429200379026843,
            0.9765405935181827,
            0.947474779514954,
            0.18982873752274343,
            0.9001090977751718,
            0.24092805773510084,
            0.32038670586129525,
            0.7243688544237985,
            0.16277724068208277,
            0.7593379678662593,
            0.03968818460330925,
            0.7244607120881216,
            0.5165074643832736,
            0.5509416401806246,
            0.27586276721428116,
            0.21355188867529862,
            0.6848900102063389,
            0.6036452778970051,
            0.24640329117263937,
            0.3724137096019503,
            0.6206648555568799,
            0.7958800469169095,
            0.4900216823225789,
            0.7667675482271813,
            0.7289915357163391,
            0.1869296356076564,
            0.4524808232640244,
            0.7127991694765413,
            0.8523560180647201,
            0.08452399270145938,
            0.08783929331698737,
            0.9187616736013637,
            0.8344792646462194,
            0.9271740035408323,
            0.6370044327004926,
            0.873942787396045,
            0.9326891534142052,
            0.583677916072454,
            0.5922397691297683,
            0.034166387868455494,
            0.49377929786225716,
            0.06968856674405388,
            0.3615198138197385,
            0.6147672290102518,
            0.7483870374797253,
            0.5383849201838203,
            0.20836389375701803,
            0.051848999025362885,
            0.7320297267131506,
            0.924153708258932,
            0.24940540066017358,
            0.7254900302438296,
            0.8398185409002303,
            0.777945701086813,
            0.6028385015259239,
            0.8381119231050299,
            0.27366756479468357,
            0.46285049762980857,
            0.007202203008645491,
            0.20677467863560528,
            0.28019168675955963,
            0.6564077343354984,
            0.8408712038064403,
            0.5349118002964522,
            0.43356705996070866,
            0.5525629375884271,
            0.8029332651737107,
            0.3789630801802527,
            0.17394534806832818,
            0.988718962514845,
            0.008255988120142943,
            0.015895226625594283,
            0.6206393567272559,
            0.9209847878321579,
            0.39110624455115695,
            0.7437450370797415,
            0.8941157743879106,
            0.011860557459840382,
            0.4065207233329732,
            0.23282699080782854,
            0.4136910892382927,
            0.7206841713816652,
            0.9464858269922742,
            0.9756111958533134,
            0.8903027302325448,
            0.7803498010740468,
            0.8488613163761867,
            0.07892513274645052,
            0.30522172054483054,
            0.4993261339902775,
            0.1307441671901306,
            0.6724515744593036,
            0.4119707684405064,
            0.38835171094008847,
            0.2895036626226579,
            0.718862164114217,
            0.7422150014781085,
            0.577025210006751,
            0.8438049461114684,
            0.04479570356686591,
            0.006571236569574479,
            0.17682956223658808,
            0.967370326521429,
            0.6761115997966827,
            0.896496576282121,
            0.42414571451691774,
            0.2643999567897386,
            0.7674469660502053,
            0.13691540715717643,
            0.6323850606911723,
            0.722449896798871,
            0.12800291638316297,
            0.5857947189453917,
            0.4163336920317575,
            0.7884813514972842,
            0.1508727072569006,
            0.19160043939953175,
            0.23301374790352558,
            0.9359396775007895,
            0.9774923318040876,
            0.36863638896238904,
            0.6610912680758803,
            0.46057923250460875,
            0.9590829904391315,
            0.1991152018103779,
            0.28851906877515865,
            0.6000178123986674,
            0.1718467848396532,
            0.7441859910583363,
            0.4622654532527547,
            0.5555247226894746,
            0.7677773119779349,
            0.6189662457058476,
            0.6988753748655162,
            0.4750032009040738,
            0.5857617813696513,
            0.3775093165711094,
            0.44160821335373923,
            0.12090759035787657,
            0.6841147475130123,
            0.03774648075440323,
            0.8954128692100448,
            0.5935413509246164,
            0.06702705653530572,
            0.28700587543321343,
            0.0697394877285672,
            0.9125516672429073,
            0.6719964366102898,
            0.9959123901824811,
            0.504950180087362,
            0.9477042666997653,
            0.09952647180930152,
            0.23616454829702027,
            0.8089820248976782,
            0.5723582906192601,
            0.45900223107602056,
            0.7190603249230394,
            0.7312224218219942,
            0.2440096836115907,
            0.3214679621992278,
            0.6436465041307681,
            0.8041091255879695,
            0.9295477549605089,
            0.22298283586216816,
            0.9521137131163172,
            0.9113445716477977,
            0.9091525207587764,
            0.98119554906578,
            0.9623632967357395,
            0.4176562224573084,
            0.43224238227876044,
            0.8594532789544238,
            0.16938535196509052,
            0.5192280607612347,
            0.9258356358139087,
            0.9614365095053088,
            0.9978293940686767,
            0.7113526084695919,
            0.12023427192990421,
            0.6130024834770222,
        ];
        let subdiag = vec_static![
            0.7089936750607991,
            0.4483258132447713,
            0.9981541517505396,
            0.4357762689203716,
            0.08785321714493932,
            0.5830593471239359,
            0.8920829832035405,
            0.41879264651643233,
            0.6111872756826352,
            0.8355133394363216,
            0.9793257049723074,
            0.3077792382592349,
            0.026969652678317857,
            0.4182365345593675,
            0.7165887422928039,
            0.32051113319939784,
            0.3040695673093029,
            0.12374943078010614,
            0.5760995454778959,
            0.8349949587005299,
            0.8659898386028743,
            0.7380424842517467,
            0.3393928531765691,
            0.7692042361831338,
            0.7796769522976207,
            0.03394625254827366,
            0.8571181864186348,
            0.6332478925233004,
            0.5184070284563695,
            0.019874111122098692,
            0.34934923133474194,
            0.8996435373419481,
            0.9380698449825945,
            0.42729146961334785,
            0.8503760643862777,
            0.02100272123108793,
            0.860005740282607,
            0.5326670023777708,
            0.5375140475812259,
            0.47919176360919014,
            0.7828043406116267,
            0.3794436036617802,
            0.9565916535824568,
            0.3980599768323815,
            0.8370299250814085,
            0.0849340864800805,
            0.7982582487709378,
            0.8809610530914053,
            0.36691575235983476,
            0.33283272535856,
            0.20354784446582086,
            0.5905114553771318,
            0.9047458258061869,
            0.0741901700696852,
            0.7621042915415703,
            0.9271099058676796,
            0.9611052970248738,
            0.21963155934129597,
            0.1558133881330961,
            0.19737060099724124,
            0.6639174149441336,
            0.8533931201297167,
            0.020945116530869057,
            0.6309209706848019,
            0.662729381326571,
            0.5035887768043217,
            0.1860476975995281,
            0.7977891924110831,
            0.6120596386351329,
            0.4508756288936743,
            0.4280982611244475,
            0.6565556625529626,
            0.05492439555509321,
            0.8468248520937989,
            0.20170352295274463,
            0.7393170393911367,
            0.48708726141266956,
            0.5460887373535718,
            0.656782452876117,
            0.568726037595071,
            0.026929796216858648,
            0.4245161336525607,
            0.9785776113872318,
            0.27746635862377156,
            0.14235497780854722,
            0.5841314393096457,
            0.9132559628467195,
            0.7254633296207957,
            0.3994675683815655,
            0.6542374938945021,
            0.1902512272011253,
            0.3277110299931433,
            0.33880621714171744,
            0.7594202382313123,
            0.0741024563031083,
            0.4449907034648557,
            0.17929774764796536,
            0.12210400948657041,
            0.6787006921534013,
            0.7281232868795051,
            0.7792629004817995,
            0.08064014445414724,
            0.9724962652917525,
            0.5641512037393203,
            0.9530186413776301,
            0.7883920219250815,
            0.450092735561583,
            0.5860790804560074,
            0.6807750490420452,
            0.987717614690177,
            0.21946639995166828,
            0.3756875030946536,
            0.0017060266273737357,
            0.3528908661861736,
            0.23522785730653006,
            0.6636060441904992,
            0.9407607887743468,
            0.7781934785793816,
            0.053397574393358016,
            0.5877392899593834,
            0.36009459733262517,
            0.3946933470154008,
            0.003646320901408595,
            0.597476378931147,
            0.43062018218454423,
            0.754668884154315,
            0.7355323365994135,
            0.5992070407324596,
            0.80570947364751,
            0.8795556637529897,
            0.8470422441081501,
            0.8946910951244957,
            0.6124018879965861,
            0.7088007839992834,
            0.9492294886442566,
            0.2251524815149637,
            0.4393504360847734,
            0.20876330443014623,
            0.6237404705834967,
            0.568434233274606,
            0.9168918462734527,
            0.3169042147373512,
            0.6276703450566922,
            0.08898882215107173,
            0.7321272744066755,
            0.8346756063313672,
            0.9565539131638344,
            0.003786651390763507,
            0.9521878269851143,
            0.8212172699917235,
            0.6238752560966074,
            0.09062018280410378,
            0.5532464578105598,
            0.4732513968021569,
            0.08469215363970584,
            0.6052403882500162,
            0.9941187028387413,
            0.7991286021147076,
            0.41618648415719917,
            0.22088579734088232,
            0.35506367821590523,
            0.3195577484429003,
            0.4312013001520172,
            0.34701617555070774,
            0.02232731969316082,
            0.6455567189539291,
            0.03371765698821416,
            0.9994490916392196,
            0.38346264435008326,
            0.3900943382330475,
            0.41645227148290087,
            0.4072897445006035,
            0.49980001378406624,
            0.9624024912309374,
            0.4243441122717946,
            0.10097127806713302,
            0.27318128155213284,
            0.01980432947685362,
            0.3919061611532205,
            0.5400372964620943,
            0.2837105264404538,
            0.5103853253037687,
            0.22502206457461582,
            0.42980394628690555,
            0.4255706600799779,
            0.05882680912038962,
            0.045980087565794414,
            0.4593492445739672,
            0.16748295392635637,
            0.5815760659541308,
            0.11804087865064283,
            0.9808592457277086,
            0.08447454175042679,
            0.14528681629095253,
            0.7966020134503727,
            0.39988958241786354,
            0.3966452292589152,
            0.7632471794259268,
            0.8364344797266556,
            0.05406706032400832,
            0.6196158413543642,
            0.8134083748522049,
            0.44315471096621983,
            0.7381281011866444,
            0.18178582271615507,
            0.39598457602595705,
            0.800755518965327,
            0.0996500182280885,
            0.6928794391981185,
            0.9271894035862747,
            0.420985543811215,
            0.8299212709536242,
            0.43361990762578595,
            0.8474588422074529,
            0.5384961461705045,
            0.04579216102837591,
            0.8144388324215048,
            0.7503331242965274,
            0.7443007483027947,
            0.8591172111741237,
            0.3129572042807881,
            0.43629364584579977,
            0.040775882175336786,
            0.5265987324893766,
            0.3278197764720532,
            0.9638233406714964,
            0.593089005448218,
            0.8531017811247185,
            0.3833716681860213,
            0.734770506472232,
            0.3961906892927213,
            0.7378952710775825,
            0.37054254273069753,
            0.25018676528013395,
            0.9155778839394266,
            0.5775373957855319,
            0.9299799078638226,
            0.1559513162598678,
            0.019987346908604264,
            0.35431474983114275,
            0.07348835806127052,
            0.05331421841012918,
            0.37541187674860843,
            0.3661350181166707,
            0.46649040220621407,
            0.5391607833956678,
            0.21034606925423682,
            0.058248534604802504,
            0.25723227398593673,
            0.3993796060279372,
            0.6796992856613469,
            0.08247673723983995,
            0.023451831694683567,
            0.13695099159078417,
            0.7384727200594983,
            0.6820765144990079,
            0.18829448527021186,
            0.41752496350142543,
            0.44795608890648975,
            0.05387150818596265,
            0.09342223756050061,
            0.36871457265674346,
            0.38200253190773514,
            0.9780514611196112,
            0.8456093037811834,
            0.1013726609210367,
            0.46219518711839835,
            0.6917545713401585,
            0.038618164075749584,
            0.40274635983558005,
            0.7018794912964974,
            0.6478854695078218,
            0.7477266217538235,
            0.6155100513086282,
            0.7365073270103238,
            0.12209169396041908,
            0.3190988421650174,
            0.8152293593221053,
            0.903108385570188,
            0.7992173115598106,
            0.7778936768007642,
            0.5419106038480274,
            0.41508926093445864,
            0.6245464965280143,
            0.21031078381218093,
            0.9157838433489752,
            0.2068179324203827,
            0.6403528464071976,
            0.22552934946201353,
            0.3498260607478918,
            0.5922724980687619,
            0.38056497907580034,
            0.016171926437021922,
            0.4563335453363586,
            0.8915640222527538,
            0.9009568027990429,
            0.39953462973284104,
            0.31682222212460576,
            0.8549860892263802,
            0.3874141470807463,
            0.7162930394888214,
            0.17862862530412082,
            0.7807735411730311,
            0.6005852379109198,
            0.5788955529636066,
            0.1998735921588788,
            0.0746258058981547,
            0.6755307862214068,
            0.959465854838282,
            0.7843160832503643,
            0.25173786054792646,
            0.8678783883274283,
            0.3354412471856949,
            0.298729267655819,
            0.7240785327926429,
            0.16587799922976687,
            0.7738582127496083,
            0.055824188508107664,
            0.8474459279899753,
            0.20327555814239462,
            0.6358785119454993,
            0.9923670488531944,
            0.3927013509323347,
            0.24821353641176047,
            0.7192887012700124,
            0.3391411570681473,
            0.8611950034309545,
            0.7322169437654343,
            0.6618650095669462,
            0.30558339768650067,
            0.7952130699383725,
            0.747194229022527,
            0.7877859162499721,
            0.2628035609220535,
            0.8908939970244619,
            0.1670668384252767,
            0.3197013127719599,
            0.4346388067636745,
            0.016611764360168424,
            0.3155599914778735,
            0.9711458254001295,
            0.5669388313504983,
            0.8225768349717703,
            0.10813802446203624,
            0.5150844011353757,
            0.5546695430534467,
            0.05753382687024333,
            0.3564421503451065,
            0.38259748941509086,
            0.4517495518340209,
            0.8539300200628299,
            0.6928383607839574,
            0.662790903000381,
            0.3728097457050529,
            0.07341040804273502,
            0.7135621444049983,
            0.31494535845890126,
            0.8463389381085897,
            0.3552370271076457,
            0.5334047271865571,
            0.44829450788280034,
            0.7392049863313708,
            0.9406772837478246,
            0.8205141626131973,
            0.4750865936497728,
            0.9879160960776046,
            0.4536776307377438,
            0.498044342363688,
            0.5982072579860978,
            0.47692296384000243,
            0.482346372060111,
            0.3520092892408174,
            0.6252355284088753,
            0.7908458992815688,
            0.4951939932647895,
            0.36334123906868776,
            0.6086985051092341,
            0.767302370706033,
            0.9117783574459476,
            0.4891381026342524,
            0.1379600694717562,
            0.8358452329194815,
            0.7878489151861279,
            0.0585353932698931,
            0.4137802559208035,
            0.4964892266394356,
            0.6340343086640661,
            0.8123945627818064,
            0.4276739901862191,
            0.46116608952790905,
            0.6117911541905424,
            0.06256859422580863,
            0.2897511039044913,
            0.9575747514837571,
            0.6260440704739371,
            0.28954243876188457,
            0.5730799397713243,
            0.5402210629491576,
            0.6899624495178694,
            0.3952333785839468,
            0.419967752739726,
            0.8817800062416639,
            0.9552369145214008,
            0.9404590446314967,
            0.8209434622238633,
            0.32398948251105153,
            0.23313715271019653,
            0.635024474315615,
            0.6206876961727517,
            0.09676694992655821,
            0.8247916367437644,
            0.6633543963118202,
            0.33219769457683146,
            0.6525608003946799,
            0.010264844795985328,
            0.20339365066885373,
            0.5293482755601088,
            0.6850695045131971,
            0.1837872683403884,
            0.19661655300335612,
            0.3215943227053656,
            0.4237431770336323,
            0.7069257266285722,
            0.00014216843975400906,
            0.12789839251588386,
            0.9870745527308604,
            0.3358551482188855,
            0.02649377917959217,
            0.6037087692892408,
            0.14468946436750985,
            0.8455686901890028,
            0.5489341054502427,
            0.6416105810992518,
            0.8716835798812409,
            0.9855863643288766,
            0.7621222238453107,
            0.10971708819658998,
            0.5992404260859467,
            0.0960184757175312,
            0.7478831288740773,
            0.21281437228159317,
            0.039989908116329964,
            0.6666889228105682,
            0.7082334278192032,
            0.4965891745189013,
            0.8448474491099335,
            0.51899221270943,
            0.053533044882776104,
            0.1832037916939313,
            0.7086558266051965,
            0.09232277641153386,
            0.6600839540684303,
            0.013429106260064105,
            0.755355951658492,
            0.1521493574747187,
            0.7359029926157673,
            0.31386400087456223,
            0.2786935794976002,
            0.8574892490208118,
            0.7048444522850063,
            0.11099956938971611,
            0.41377696534215747,
            0.4972042095004061,
            0.8123024006102493,
            0.019842136393627086,
            0.9110135852923356,
            0.7333099706166278,
            0.9354587752844313,
            0.9746672511177458,
            0.49312307327819005,
            0.009712103267392691,
            0.4179387564431378,
            0.4264449180491885,
            0.3178185085304487,
            0.4568848031303122,
            0.6746225129344577,
            0.612490689125883,
            0.43797541886896396,
            0.8206451670821743,
            0.5755039811379289,
            0.6969733354065955,
            0.5409265436547918,
            0.4319598200039658,
            0.2595330341771661,
            0.7496365412646532,
            0.44554171051435454,
            0.933953734036019,
            0.6394466211797196,
            0.6253318254844847,
            0.622598526858425,
            0.42819693534628167,
            0.49608801391604496,
            0.004078043936252662,
            0.31546465727721773,
            0.5620275115036629,
            0.24981547755110622,
            0.46638455050467276,
            0.8203192948951312,
            0.9531460405175395,
            0.08767861482470207,
            0.10182665848106964,
            0.41722139013286297,
            0.9050081587484748,
            0.7997471863258789,
            0.07654727069374845,
            0.015678303100936986,
            0.7531058353281517,
            0.11733320889337284,
            0.6678888964137968,
            0.25222433839055036,
            0.39593237313221374,
            0.2639734864670895,
        ];

        let n = diag.len();
        let mut u = Mat::from_fn(n + 1, n + 1, |_, _| f64::NAN);
        let mut v = Mat::from_fn(n, n, |_, _| f64::NAN);
        let s = {
            let mut diag = diag.clone();
            let mut subdiag = subdiag.clone();
            compute_bidiag_real_svd(
                &mut diag,
                &mut subdiag,
                Some(u.as_mut()),
                Some(v.as_mut()),
                40,
                0,
                f64::EPSILON,
                f64::MIN_POSITIVE,
                Parallelism::None,
                make_stack!(bidiag_real_svd_req::<f64>(
                    n,
                    40,
                    true,
                    true,
                    Parallelism::None
                )),
            );
            Mat::from_fn(n + 1, n, |i, j| if i == j { diag[i] } else { 0.0 })
        };

        let reconstructed = &u * &s * v.transpose();
        for j in 0..n {
            for i in 0..n + 1 {
                let target = if i == j {
                    diag[j]
                } else if i == j + 1 {
                    subdiag[j]
                } else {
                    0.0
                };

                assert_approx_eq!(reconstructed.read(i, j), target, 1e-10);
            }
        }
    }

    #[test]
    fn test_svd_1024_2() {
        let diag = vec_static![
            0.009879042533147642,
            0.25942766457357236,
            0.40170082173257604,
            0.32716377867657065,
            0.22784080238282056,
            0.8011220261693188,
            0.41034688531555497,
            0.6337482059687781,
            0.9268147331465462,
            0.9007255417991846,
            0.39500004088544705,
            0.03216097972007015,
            0.012917906438069893,
            0.7941462600953578,
            0.756551131637162,
            0.6277212943286069,
            0.4750538755815341,
            0.8050777538357257,
            0.6411193842462393,
            0.5964472181215076,
            0.7526694114715058,
            0.8570880824272364,
            0.17902227574406282,
            0.8109514743574001,
            0.5606370535254248,
            0.6598192805934875,
            0.43078033013590533,
            0.5825029672946985,
            0.6956259864573019,
            0.4651998256633355,
            0.09762364687238201,
            0.24074173129736165,
            0.7123654938926,
            0.935829517920511,
            0.11246136111668859,
            0.28663069616787795,
            0.052842680220333116,
            0.015474947998349697,
            0.7310762139439319,
            0.23933959701846774,
            0.8822425987482592,
            0.8685083697899169,
            0.652772891730832,
            0.9613456318501793,
            0.3562755742479843,
            0.3529317070096095,
            0.5203651320326466,
            0.9003075461626292,
            0.1264415938158331,
            0.08351000046655321,
            0.1753943981928473,
            0.6980433592474139,
            0.8578269024506149,
            0.6908011920858023,
            0.5674517380556917,
            0.01702871963976549,
            0.07839669995380683,
            0.47841583322083037,
            0.34526252196058815,
            0.8234110245951314,
            0.08003030493935503,
            0.8985662624131026,
            0.46734383275741376,
            0.7161188969879848,
            0.3815623699352576,
            0.9985517387444293,
            0.8492572583223449,
            0.8205777546331864,
            0.8770957313542075,
            0.9737076660054085,
            0.014235964068120666,
            0.8072818162682582,
            0.6886219904348015,
            0.4552576261137854,
            0.9974413848153643,
            0.663487101355474,
            0.12797047898911695,
            0.8730708596405391,
            0.7358872544204906,
            0.0024232499164960064,
            0.3123244065132025,
            0.9129677973115747,
            0.12381606964146508,
            0.6714861406297731,
            0.6330879630315885,
            0.15970950034649944,
            0.12990016300317808,
            0.20715056410837385,
            0.23864283500085526,
            0.9196384835185076,
            0.6786187887920997,
            0.971364503522331,
            0.4991613803578423,
            0.19656248169292956,
            0.36089468141027725,
            0.17351799213873953,
            0.34283715347954935,
            0.202274893550491,
            0.7573587290836572,
            0.28781764486060746,
            0.29286987330948355,
            0.18041818830387502,
            0.11085478108502878,
            0.9248101417660487,
            0.7196648633376221,
            0.7948623795256567,
            0.9618964043172259,
            0.5358432591346612,
            0.41078427228367975,
            0.7118268810392685,
            0.08930552972766292,
            0.3037261272140287,
            0.4873595668729337,
            0.2759135908426116,
            0.004931914769020551,
            0.9751958784750651,
            0.4803312130077346,
            0.19625162474739133,
            0.6338278104821975,
            0.2070709701863036,
            0.35289231926619125,
            0.8075051412594402,
            0.18922437321221275,
            0.8090676916035391,
            0.34692036473765697,
            0.5134105296786168,
            0.2537789123309152,
            0.31740425288639873,
            0.6914865672540939,
            0.6171757974608877,
            0.8067004627565533,
            0.5548379967230543,
            0.08923412688878118,
            0.7373239475246149,
            0.9973855333306278,
            0.7893809917271625,
            0.684211884371379,
            0.46218191430276034,
            0.17597080473258842,
            0.9632675715873064,
            0.5611078882167525,
            0.2939088572594233,
            0.1312840046608058,
            0.4074087197731001,
            0.24449749140257027,
            0.9584081457748832,
            0.598735572583611,
            0.8719134466881342,
            0.7473254819575135,
            0.43051500118847075,
            0.5680395392137619,
            0.9992003434168677,
            0.09697960853735677,
            0.7539212337070309,
            0.3135142583807121,
            0.3089605207933839,
            0.7247740236649471,
            0.4341641533676518,
            0.9738557480191191,
            0.7568112381061494,
            0.6988946241305319,
            0.8671069033150816,
            0.9199084451720417,
            0.2339808510845095,
            0.9120851645301419,
            0.52184538505289,
            0.8686042347324593,
            0.5825948463951326,
            0.3066180985985336,
            0.9893240124363446,
            0.9542060294495651,
            0.5433692697539214,
            0.07007746021702566,
            0.849506872522393,
            0.46513028331219763,
            0.5841677314662393,
            0.9388553738536585,
            0.4135859170203644,
            0.13279656262896178,
            0.9764168577593498,
            0.7121771180000449,
            0.41753476055643757,
            0.4694451918226963,
            0.5606108747640179,
            0.6921120653839485,
            0.10404989133518994,
            0.9096628373622678,
            0.6212496902578757,
            0.9039775825048112,
            0.2519785269022933,
            0.642636753005936,
            0.39313619189426474,
            0.5049087793125976,
            0.185123471096711,
            0.4738105908508532,
            0.40077831714296763,
            0.6610403139371497,
            0.16960410142423576,
            0.7720571606452117,
            0.47301034519076934,
            0.5671632092141498,
            0.058571919388965,
            0.1137817769029924,
            0.9293888916781323,
            0.7337496351796344,
            0.47872618860877636,
            0.049900525598919554,
            0.5899099081681471,
            0.04598868985641036,
            0.7118427779653944,
            0.22686684711973948,
            0.09467985694967407,
            0.828126484887492,
            0.38062807400811627,
            0.13432158358838764,
            0.8819366554870842,
            0.06076628659369543,
            0.5018549738258863,
            0.47726578024585364,
            0.6796227829227661,
            0.0307355495533701,
            0.7219689204239849,
            0.2665398161653749,
            0.7457946992965994,
            0.8014906918498865,
            0.7032050489593877,
            0.39913936032204245,
            0.344868641245594,
            0.8460367081641266,
            0.8611874056977944,
            0.28677011663765195,
            0.9236609706292132,
            0.592690189893941,
            0.24627686698958096,
            0.05327782694837213,
            0.31732511563500243,
            0.03522553387439098,
            0.5105357808439257,
            0.5300916241678767,
            0.9122917979999389,
            0.3896359534024416,
            0.8157532084708795,
            0.8165976967791265,
            0.6515888300611864,
            0.01682583396667725,
            0.8404708464497803,
            0.5019880976343682,
            0.5407503246795986,
            0.7360568473812241,
            0.6853540818970443,
            0.5247918359630233,
            0.7393887921536667,
            0.8131098769020638,
            0.04116878848575245,
            0.17884808631370075,
            0.795585260728776,
            0.8963137184542852,
            0.20078125672437008,
            0.6822915485500384,
            0.13695879926780696,
            0.2850697874999216,
            0.018404474915290847,
            0.6142587891665715,
            0.7426430268567997,
            0.5602787315688591,
            0.025757157656970397,
            0.6646633612347755,
            0.20212404501160852,
            0.08176186878334868,
            0.3472804854339161,
            0.9443673717488501,
            0.58684993396532,
            0.7981297693851774,
            0.7585078706062724,
            0.6032665234646059,
            0.0897210077358962,
            0.9947864491545249,
            0.2899798467013329,
            0.9361953260343231,
            0.2876360062375225,
            0.15022765092332824,
            0.0947873377448969,
            0.19800367758769943,
            0.08747917919372117,
            0.4949460545148765,
            0.011802102235748402,
            0.07848227267953467,
            0.7220657649901457,
            0.8143141134598415,
            0.2534052430910867,
            0.3001731119539667,
            0.12340729205709178,
            0.787317670027495,
            0.629669432731565,
            0.2880200853036611,
            0.16226451938719422,
            0.36046675429078934,
            0.1701666367778586,
            0.03453409120781836,
            0.6004559189388794,
            0.23581641485601612,
            0.5280669351043628,
            0.7548114921998968,
            0.2509906416696046,
            0.9386190839216705,
            0.5780293946541897,
            0.1309615885069001,
            0.20329455560733034,
            0.5028424085002999,
            0.2400295683326339,
            0.3143016923933426,
            0.969039206593452,
            0.367912630536846,
            0.5705945838187225,
            0.5248938705978503,
            0.29731376199703297,
            0.9502044873972788,
            0.6259073750825663,
            0.41444941678247393,
            0.28737013771372466,
            0.48705632242104335,
            0.6507400543562598,
            0.6592907477931843,
            0.4995334924295577,
            0.08724920185129703,
            0.046334012127957735,
            0.12789571448865977,
            0.7925758209411751,
            0.41177797999498333,
            0.7280349975614596,
            0.16678381826476318,
            0.6782940771926207,
            0.8241527879608631,
            0.2886575230739752,
            0.5496549642848929,
            0.20551094816275284,
            0.49464000644863804,
            0.39009807772712346,
            0.30220616569167813,
            0.7046371963064363,
            0.8947263781950788,
            0.5575751294561873,
            0.9322608164618286,
            0.741536364482585,
            0.1909643568050814,
            0.7851540896341616,
            0.5184542009678219,
            0.7893146472184174,
            0.23365714387028158,
            0.8852121789147809,
            0.14445654209664516,
            0.4588607222062475,
            0.24432012414481186,
            0.9612842447541281,
            0.45983047487849205,
            0.5523603190690298,
            0.3291724000017524,
            0.43385527333597307,
            0.3505744714212933,
            0.023087516727097412,
            0.5071839293492867,
            0.8551265527948764,
            0.41987334130223,
            0.7841528560802025,
            0.4472398086198166,
            0.6435783918893202,
            0.05825220493874805,
            0.41536222391490796,
            0.8746933502601629,
            0.2731947693472474,
            0.5305223903055022,
            0.4678374426676837,
            0.03756925534516398,
            0.2926013108528348,
            0.3084928970079136,
            0.6475766601345854,
            0.04212485331082927,
            0.5635803337549966,
            0.9650888196115347,
            0.7256947295966187,
            0.6033636731787934,
            0.03830387828268755,
            0.4822322188559294,
            0.9144517266362243,
            0.7313201755134171,
            0.030072503406533335,
            0.818875249918176,
            0.8143288731590852,
            0.8329629136784739,
            0.1944921757341337,
            0.05999952456212865,
            0.954791314590924,
            0.6968444446984309,
            0.8415438993281099,
            0.8104038752608848,
            0.9235846588764935,
            0.5071930898060761,
            0.981281045431366,
            0.01932415570284629,
            0.17232045920218053,
            0.5339798690507419,
            0.21671241307027111,
            0.9693469116453112,
            0.6804465936338698,
            0.6920768870899838,
            0.3685724389382682,
            0.6320342688659961,
            0.9485911813862122,
            0.21613777453897431,
            0.3490192603566519,
            0.4344451777213022,
            0.47322333392509885,
            0.7019421587493292,
            0.6284156144703128,
            0.02091274361028006,
            0.20753380828946955,
            0.05065441005923155,
            0.514343363311585,
            0.4382356537431309,
            0.8311461626025239,
            0.44464423311655177,
            0.07117653274382474,
            0.15834970227217948,
            0.4362212212517339,
            0.9145574773998393,
            0.35832540201183105,
            0.6792445083226095,
            0.9292244651766712,
            0.9066256192582868,
            0.7616847446103808,
            0.42116197156710145,
            0.4015473379915795,
            0.6460768041419113,
            0.2862748731816819,
            0.7020276191930521,
            0.6291602513924968,
            0.5232558433180349,
            0.8257727166559992,
            0.11513885209593933,
            0.6925901029622323,
            0.5375112983483977,
            0.5865353672412245,
            0.1444663737523756,
            0.6317046556717392,
            0.12083588293905656,
            0.7679086813222006,
            0.9467660510253647,
            0.7772848574161543,
            0.6546720940739391,
            0.09593875178740707,
            0.854691686793777,
            0.9269677437706112,
            0.6201262539631173,
            0.033532540194856075,
            0.48361037458788025,
            0.7499998154464882,
            0.9599138494945278,
            0.6373755929402666,
            0.67169969402632,
            0.3279703742416862,
            0.4952635435438998,
            0.6900526231740292,
            0.5105236806997386,
            0.1643138318654166,
            0.41587392439644844,
            0.7444604521060232,
            0.14403480963377757,
            0.830570382850683,
            0.5350012051469126,
            0.0760991476119226,
            0.7692125685549903,
            0.7130648584389353,
            0.9479603653126061,
            0.5304396418782369,
            0.20982837490877027,
            0.5718515733933942,
            0.6948247919391625,
            0.6507592130358502,
            0.6396306475898776,
            0.10592189604912228,
            0.5918255784804017,
            0.4079679176088701,
            0.15059104663989364,
            0.5671066668133875,
            0.7055056043665614,
            0.24494640874438212,
            0.25963553178303445,
            0.13255408754459386,
            0.8433198877908112,
            0.5998435267670393,
            0.9331163579073388,
            0.7473276773883499,
            0.914652860213476,
            0.5399820119328973,
            0.7255849342393966,
            0.24515202079667575,
            0.7240060457690117,
            0.6797790363172005,
            0.9636781230274964,
            0.9621076396127072,
            0.526453558327734,
            0.20461044049262478,
            0.221737143318534,
            0.3042132609158593,
            0.7166006793681836,
            0.7208533268745831,
            0.14111583611047418,
            0.16505397977974479,
            0.6930709356098063,
            0.3469557583864098,
            0.4028006795191459,
            0.3031107087201411,
            0.39025588145942935,
            0.23485551066905996,
            0.5449188856687478,
            0.8810702418577913,
            0.0650207940766595,
            0.12237389969036438,
            0.7730308336766517,
            0.8004635175819177,
            0.6048075860557162,
            0.06523207730761515,
            0.40485317304823243,
            0.08902021468999932,
            0.20730816080223702,
            0.21001234293356918,
            0.23199645465127916,
            0.8123002420509785,
            0.3499116098336068,
            0.7499122414990476,
            0.7866312792602179,
            0.12174694137861608,
            0.32405932418634065,
            0.8258813665400088,
            0.04656678385188917,
            0.34120591602043515,
            0.025432139703123524,
            0.7513611942051909,
            0.40921909742638385,
            0.8768715967745427,
            0.6233021962940871,
            0.321448374528979,
            0.22230905602996653,
            0.6042217477703352,
            0.22034890014792163,
            0.9012174395684431,
            0.3780544593364079,
            0.17182226098290831,
            0.9536597136299334,
            0.7013268968862395,
            0.7957401114548772,
            0.37762917847620403,
            0.6641353472252048,
            0.5990337788510948,
            0.8244425305058642,
            0.6336964023073468,
            0.9442533024008415,
            0.9856053064515053,
            0.23764177109429674,
            0.5153091466750802,
            0.22192209755800385,
            0.653677202419529,
            0.08566875119109962,
            0.25922411512320986,
            0.7756516972996109,
            0.6518894605239182,
            0.8983646890758403,
            0.7839468017430613,
            0.2928502179805963,
            0.858351123436328,
            0.41833268377774047,
            0.2314661073036962,
            0.759291174791694,
            0.2680111781356541,
            0.3488173733450649,
            0.6756344422792672,
            0.7115499536533283,
            0.43697837993273214,
            0.2296222477305303,
            0.9561418763126147,
            0.2360168383852771,
            0.5707091940070331,
            0.1223442272970009,
            0.7920142561811463,
            0.11604585925925615,
            0.9083026514927254,
            0.340155551667485,
            0.6112643982241555,
            0.2015453241965941,
            0.49085157267329715,
            0.24211104326232125,
            0.45282116527917005,
            0.7562240898603866,
            0.7309128543377432,
            0.07706898694016251,
            0.6087936043086346,
            0.10628692407945894,
            0.9562681558712472,
            0.1307744960951679,
            0.4684732006832625,
            0.31153597327978666,
            0.6428299893138236,
            0.3865890604057184,
            0.7005259891521617,
            0.5180895174480458,
            0.8733436608237053,
            0.11341584270240046,
            0.811670609222333,
            0.5948402781373923,
            0.6543611792928933,
            0.8203804051476661,
            0.9439613667301399,
            0.04885500824368505,
            0.2639193725053357,
            0.39181319459953656,
            0.6963701495177078,
            0.7984729628367462,
            0.08962455833811778,
            0.20513997094453396,
            0.8229728748865878,
            0.6826715373558596,
            0.8869606824313254,
            0.32075328872089903,
            0.2715256841091682,
            0.34871567943039405,
            0.5057068180254091,
            0.7108863566584402,
            0.017315503417683065,
            0.11129374732080222,
            0.2149408872089178,
            0.03972142104021714,
            0.7114201947815033,
            0.7735142558448584,
            0.08096205710211934,
            0.7264487468336878,
            0.7275229961863563,
            0.1701241597350034,
            0.4554828558908014,
            0.3674820156462085,
            0.10296088297118644,
            0.907807960856776,
            0.9315090425783622,
            0.02269088145021081,
            0.8768534746621365,
            0.5143117619574731,
            0.7623025158310304,
            0.8165144399320536,
            0.9478885317309997,
            0.776460718598049,
            0.2818285909214623,
            0.5313107639119145,
            0.5223934559866608,
            0.5908742994286429,
            0.15799529054304107,
            0.6957020996541937,
            0.14067939009805353,
            0.17976054718260315,
            0.8553325397169877,
            0.7751947200973635,
            0.6542489234703435,
            0.9359516544956115,
            0.18910644338747085,
            0.3060744915742655,
            0.07156522488893935,
            0.7858905172353533,
            0.30841426063537336,
            0.7961153602347614,
            0.5242480442051327,
            0.7608381696457592,
            0.7321043646809537,
            0.3160923005617713,
            0.6308133085131272,
            0.6404187406431489,
            0.0314570800078281,
            0.6381340941591219,
            0.28335780818324463,
            0.8080440852641214,
            0.6650322810432938,
            0.7382350607745667,
            0.9417050501423426,
            0.4616591796097078,
            0.35170317130919027,
            0.735261952464759,
            0.941824765484256,
            0.0704397787913904,
            0.8025735792631282,
            0.884345853448629,
            0.36856840140455305,
            0.5384999491079693,
            0.5088196392361701,
            0.3690437535318991,
            0.6893065321084926,
            0.4043065001820547,
            0.7321490028319614,
            0.9953331565039383,
            0.8708898957578411,
            0.5678444419468969,
            0.6948121027529817,
            0.13237429831154945,
            0.758791690392103,
            0.9000474656571474,
            0.4130616897175554,
            0.6048306050375922,
            0.690857261930525,
            0.17608744899984585,
            0.8144564217524697,
            0.008916716494768329,
            0.3728910723008364,
            0.14230574725308864,
            0.07032084742055866,
            0.13510122662840574,
            0.7110610697088638,
            0.3230680758079223,
            0.8971823950988784,
            0.7593906944269069,
            0.7180945813078534,
            0.40156447934212547,
            0.31446765260880194,
            0.590384621574585,
            0.45397604575440187,
            0.959643273040757,
            0.3643877880513411,
            0.43129660288653504,
            0.8882110225380889,
            0.721800188113398,
            0.3597821910241765,
            0.9117927009769277,
            0.3174172692380255,
            0.7337327946421679,
            0.4101950457979521,
            0.3966570730442849,
            0.9368223197108299,
            0.3461667670681947,
            0.5215603324247206,
            0.11744811566395297,
            0.636498465619982,
            0.2038693777421673,
            0.9303912197571679,
            0.816998163909247,
            0.9789272639916347,
            0.6892868829686003,
            0.6982463926962832,
            0.841847840269864,
            0.25713058055406,
            0.06088862527493233,
            0.245800983643233,
            0.5342357284707632,
            0.9952357763786429,
            0.2441059118185811,
            0.8036209652416689,
            0.687229655807877,
            0.8475116280256628,
            0.36684229689996317,
            0.42475107653902533,
            0.9140454655574699,
            0.2679255563124284,
            0.534492910235425,
            0.1461389965077352,
            0.6985612438516512,
            0.8781293966055371,
            0.8839097809041689,
            0.43478503161232085,
            0.496953743989921,
            0.9862174771277655,
            0.04159572606291473,
            0.07255688902459867,
            0.9498054639038871,
            0.570391827242604,
            0.49001305411912444,
            0.8530655079238382,
            0.7419750948446635,
            0.49794852153088176,
            0.3289905594740178,
            0.5264460549027793,
            0.35096788116253996,
            0.38039435732459825,
            0.7528001382362867,
            0.7955875745990474,
            0.8484075180960026,
            0.5718070080412917,
            0.664181837639004,
            0.5930016381927362,
            0.7722245984189531,
            0.3454918499428087,
            0.598053446867222,
            0.9121501991469333,
            0.507109495465786,
            0.5620786175441324,
            0.8731547356388709,
            0.6666694634544433,
            0.5270620554009692,
            0.21624156898941072,
            0.44549942626809236,
            0.9034774952953463,
            0.6282940981219879,
            0.10859408808142523,
            0.07679682374546248,
            0.7015771280590647,
            0.4940827467634742,
            0.6153747570296487,
            0.1897752420621278,
            0.7028657997775055,
            0.5448505061986961,
            0.9921027559867645,
            0.41950910409868847,
            0.13178954908140816,
            0.2383518904066938,
            0.7815615667973201,
            0.16895292701555942,
            0.23169578068378394,
            0.3124439808679126,
            0.6233459461330343,
            0.4349122823170706,
            0.39269732832045945,
            0.08541285410083688,
            0.18466437935111302,
            0.719077270753475,
            0.7289340738984056,
            0.5193818534486477,
            0.354789781075149,
            0.6030125564046028,
            0.9734844474127914,
            0.5788586904350241,
            0.8896532902937736,
            0.3971867904276313,
            0.8112369010323465,
            0.48111285523424907,
            0.14542610042311843,
            0.9279678895807261,
            0.5356954396762273,
            0.1852933106306257,
            0.3561778722680313,
            0.31317097177836395,
            0.3551128084171603,
            0.08163166610234718,
            0.3156656187869643,
            0.9227454628213471,
            0.8362487440447994,
            0.39868405125484874,
            0.25902201552505455,
            0.2332786851666846,
            0.13660977228406157,
            0.06455443947244532,
            0.8168118066925686,
            0.7645494702824027,
            0.34009410584127797,
            0.47355152908799425,
            0.6852613137806506,
            0.368704208265222,
            0.9205666526517542,
            0.7954839177184126,
            0.9238677728627528,
            0.5541829847437505,
            0.8340909169505104,
            0.44100901165142437,
            0.19118634047301042,
            0.5663865981345195,
            0.3949474219220366,
            0.6983211472251544,
            0.51050268058169,
            0.554548371059581,
            0.28516342269217076,
            0.3959973967500061,
            0.09362588841780872,
            0.14196882945885458,
            0.2590132572403807,
            0.21249760169673348,
            0.24510066104064332,
            0.8808342100052384,
            0.5885744306658329,
            0.6046618092366636,
            0.8049617843116875,
            0.7213516051413413,
            0.16189442021410227,
            0.13991712115866184,
            0.6847675452857435,
            0.5968706224800094,
            0.6782643285889133,
            0.6898580897254263,
            0.5662067190549015,
            0.9583244345802936,
            0.2912642684755796,
            0.7630161101088999,
            0.03281787135301295,
            0.6971232011889607,
            0.4513833334496118,
            0.7160717719914459,
            0.6744556167554911,
            0.7012838246074843,
            0.09747407973728395,
            0.39103018910292175,
            0.6401692990540451,
            0.8732274653813095,
            0.1990875759939147,
            0.992779916472509,
            0.17438406058704536,
            0.8965750031118914,
            0.5548778214789011,
            0.09041896734415455,
            0.8202251402395749,
            0.9217434077259461,
            0.12848714981921638,
            0.4423636010983777,
            0.6245854605562289,
            0.8479875577813115,
            0.21833062484336352,
            0.08978365634540908,
            0.3971703136588606,
            0.270921228050496,
            0.2082154959111303,
            0.30839569718340565,
            0.7813961974128387,
            0.4284726179385906,
            0.11102333241822904,
            0.9916589209720094,
            0.9423505009705155,
            0.659033078754542,
            0.11169527470064744,
            0.8508594981950214,
            0.7756263095253668,
            0.01958367233718139,
            0.6671403237915123,
            0.9931912176940958,
            0.1828227445986681,
            0.671083610894634,
            0.3120957376876963,
            0.05823283339664698,
            0.4539694522719664,
            0.3827425186513863,
            0.8161555016670786,
            0.6909702044024906,
            0.5697887820353031,
            0.47684073212447675,
            0.1174826727347773,
            0.9199399451192102,
            0.5341426981168305,
            0.3843053536441867,
            0.28744865946458,
            0.847420343974517,
            0.2090150894636853,
            0.3749965328313567,
            0.8057213252426457,
            0.5005627406838957,
            0.5143571962319217,
            0.7657339905459002,
            0.7329828664873212,
            0.2440568382361218,
            0.8502214304598523,
            0.6260033891681143,
            0.06626999120022115,
            0.058801167239939334,
            0.25972930664267024,
            0.6451029221057365,
            0.6260012237257966,
            0.5846441474757913,
            0.98815366379417,
            0.05783009574274389,
            0.022349918869454588,
            0.1600834646879955,
            0.08953514758686276,
            0.8732084555182431,
            0.8115969955208467,
            0.23373497446989266,
            0.10840903372449262,
            0.3171564540123466,
            0.6658826853300668,
            0.43676485701893675,
            0.4344333657895355,
            0.07897530154578136,
            0.016469919896786256,
            0.5675235011316633,
            0.5808433804516457,
            0.867716441356151,
            0.2065739667627342,
            0.25440188873038416,
            0.3422924158271836,
            0.3953463754311852,
            0.5749685502945107,
            0.1488782862289172,
            0.7394424780124986,
            0.3390228551044585,
            0.28553557726203915,
            0.8153568215504137,
            0.024964426512648186,
            0.10278618583088106,
            0.8669568659754601,
            0.7881946366992756,
            0.658616381755007,
            0.11159208601537995,
            0.10096754637870087,
            0.6410275673490674,
            0.5332073715199348,
            0.8536682385385184,
            0.3629141869892666,
            0.4737316407585195,
            0.33920893131595764,
            0.5363296458774595,
            0.7991161654752915,
            0.8448716812778778,
            0.9529552780758653,
            0.3430740253249046,
            0.5199790758371671,
            0.5085226165222254,
            0.9476254029397022,
            0.3888290071361509,
            0.8298957295600268,
            0.015354702895921113,
            0.026795420102779577,
            0.5623317424080893,
            0.44155098929951186,
            0.0216140477018405,
            0.9153672471307774,
            0.7322390118784943,
            0.35474773974775453,
            0.8456112753342347,
            0.12427812853910869,
            0.8178060502734751,
            0.09769257732793735,
            0.2352335200687209,
            0.5219444775119129,
            0.18953776590190818,
            0.7089965698272449,
            0.9271773068939287,
            0.26661820468540565,
        ];
        let subdiag = vec_static![
            0.37671885831556107,
            0.5914902345047548,
            0.4573673949813909,
            0.278061312576772,
            0.45512214158576403,
            0.3112899705911143,
            0.4880214441872541,
            0.9582807681021962,
            0.48963048000783027,
            0.3250457425435612,
            0.07352275648645279,
            0.67413401097283,
            0.3100466148803773,
            0.562276992540448,
            0.3892394671456678,
            0.749936908252996,
            0.3694048046552584,
            0.7083582626246234,
            0.6251830717953774,
            0.7884290671362195,
            0.08925884337373147,
            0.2613537478548059,
            0.46587313139923503,
            0.701485596521086,
            0.33649429784277063,
            0.6911793664914623,
            0.9948554052777019,
            0.17308968749086218,
            0.14674255849462148,
            0.8732244248197987,
            0.9456441706261545,
            0.640728931177753,
            0.7986202000553076,
            0.7023010978504587,
            0.11704483536140065,
            0.884159157005428,
            0.38767049994077984,
            0.33752430428623426,
            0.3799123019580549,
            0.00314019945840871,
            0.6190706208888044,
            0.5102826797204365,
            0.6797651848612186,
            0.9399355917564993,
            0.5581123097220246,
            0.12682711067698937,
            0.6538602970641864,
            0.9045235067178719,
            0.6892314551830656,
            0.9104493575898686,
            0.3174388259005446,
            0.07255503694667453,
            0.8195907836202596,
            0.2065308345615483,
            0.6563964950866789,
            0.8285736443349354,
            0.2807382142533439,
            0.888192042410335,
            0.7281834856246764,
            0.4213180062854599,
            0.3891228421856179,
            0.9700914470984294,
            0.45160015526230735,
            0.581090670815197,
            0.5935074179617855,
            0.7314200085933598,
            0.7668608047743345,
            0.9645838183585224,
            0.5595954893259129,
            0.501403728329053,
            0.7252189956851528,
            0.3877102331948553,
            0.7312171132795519,
            0.45921662392553675,
            0.03569997105246314,
            0.11035683929716755,
            0.23537156497955436,
            0.7453083988700915,
            0.16039524701959074,
            0.38914641522433013,
            0.6488288881008611,
            0.7802459577505126,
            0.7855394167175755,
            0.9525845711062543,
            0.3835654723986076,
            0.6540539835774286,
            0.5411639500064723,
            0.4824262094927344,
            0.9023430874410875,
            0.559384552682,
            0.5198552240452828,
            0.08131268459802798,
            0.9648278171342273,
            0.18547718903819543,
            0.018187371199786972,
            0.3548694940703081,
            0.5769353673932345,
            0.8277144778427715,
            0.41156560741241444,
            0.30580641025730293,
            0.4994004898094244,
            0.6543879049826437,
            0.03768458013740572,
            0.8847452861945968,
            0.17246126984418086,
            0.8021618206146044,
            0.6792215520070621,
            0.1494249976088271,
            0.04323242654960879,
            0.46497519643274443,
            0.7311303757347277,
            0.6907293989179726,
            0.5561071159235972,
            0.8664847074180113,
            0.10656582443043194,
            0.8943531293401422,
            0.5810938327611845,
            0.4079086398883156,
            0.4376953742835751,
            0.5150452019463021,
            0.4654673204490244,
            0.6181269596797001,
            0.4972856911548612,
            0.8791344234134285,
            0.48448853677476866,
            0.5271322135896357,
            0.9187839153765355,
            0.6432254405537613,
            0.1395032787202216,
            0.4617483512605155,
            0.7439775577234952,
            0.20688487763651586,
            0.6837970698781676,
            0.6284010284439561,
            0.9877247981721398,
            0.2593161005225657,
            0.523627762922784,
            0.22137969255505463,
            0.7131606420460196,
            0.2852462931306611,
            0.5124483971307822,
            0.9367014111119283,
            0.32731755693626097,
            0.964236792324476,
            0.7493694578953238,
            0.07744815323017573,
            0.12949889017954386,
            0.2641939259367596,
            0.20324028762179547,
            0.713464930463833,
            0.7854027536448535,
            0.6805436303675616,
            0.8942998010145956,
            0.6086916036361911,
            0.67377363814928,
            0.6969042644434692,
            0.45367268191922705,
            0.02611844492272608,
            0.4700011168748164,
            0.006154223920945445,
            0.49785132681634103,
            0.9801180571483392,
            0.23142606315597036,
            0.6857808368058479,
            0.13658722793896216,
            0.2253508839222368,
            0.24281067691841418,
            0.8526910646423107,
            0.4902538927391915,
            0.22738048994855875,
            0.7705487908544527,
            0.6240777789061516,
            0.2755720074547112,
            0.9203432076130207,
            0.5971832484284051,
            0.5435267728965103,
            0.6279113755719903,
            0.32112529775845144,
            0.22720242153795212,
            0.6868459996576608,
            0.3458489741208203,
            0.4386189341171709,
            0.5662864682421069,
            0.0915222517019979,
            0.9327725500558302,
            0.8388716532891217,
            0.8354865519804079,
            0.6436987359219395,
            0.9536675701446755,
            0.5023348488433156,
            0.03744165598145255,
            0.691465268394428,
            0.3471961034273211,
            0.9487485590405136,
            0.08059004207151443,
            0.9130473718156124,
            0.8711760574531869,
            0.8984710118710725,
            0.38433446373471525,
            0.6047403061075196,
            0.40230752044977547,
            0.8999549061609351,
            0.3045400555681539,
            0.18733188204216666,
            0.47881738310505884,
            0.09948824662632694,
            0.8107201856317277,
            0.15303290808660974,
            0.09469168130866801,
            0.6229102667450735,
            0.17724775681237315,
            0.39906288328249606,
            0.949347149715389,
            0.43036938994682195,
            0.0026488356345022446,
            0.8051683567235493,
            0.7142222915323341,
            0.6565513212405722,
            0.8186110241383198,
            0.1075867852301704,
            0.00651363620150236,
            0.381315358116759,
            0.8340802896496835,
            0.20799848101982932,
            0.8380203793470061,
            0.551697397258092,
            0.9741153378014853,
            0.6716747126650318,
            0.5405891135029178,
            0.6401015262720249,
            0.960909912158254,
            0.7755110312141643,
            0.825478905278364,
            0.5064634220126577,
            0.18812888182280796,
            0.07283708755624119,
            0.8220237358112901,
            0.47238295701015254,
            0.6869923481614207,
            0.9163118520579482,
            0.22074610201161338,
            0.9919447342875314,
            0.6222215472277868,
            0.05033016452819428,
            0.9254287027213731,
            0.24236628317521214,
            0.2968826172357997,
            0.1346880107440669,
            0.1049489403202527,
            0.46459571678534284,
            0.24948187054913806,
            0.41911978648494297,
            0.09618836090256044,
            0.05268542651235997,
            0.3316174971449575,
            0.04388767991512044,
            0.9807739468209036,
            0.5532017827965474,
            0.5092525272651334,
            0.8587394080322006,
            0.23648035753216734,
            0.17429808586148532,
            0.5447600177654469,
            0.1476302487546165,
            0.904924948067489,
            0.8467065400997184,
            0.5588587037416475,
            0.46532532965862294,
            0.4149944587994253,
            0.23699178409866384,
            0.8286742504857455,
            0.14964344260285367,
            0.9419164055233096,
            0.9931685556787592,
            0.4726614335735072,
            0.946096426095138,
            0.9665484700176549,
            0.07045472429632438,
            0.7287290285529795,
            0.8288758892449009,
            0.7345986156299534,
            0.47027165222078315,
            0.7751737277463969,
            0.9166110660109572,
            0.6205218974962594,
            0.3932245033802777,
            0.8655119836283389,
            0.8156558903134565,
            0.8276658886810031,
            0.2769252733692703,
            0.36745540867630144,
            0.26391095661736186,
            0.3870335115679614,
            0.18833085664599503,
            0.015739548377707413,
            0.6947652147498935,
            0.2621931039418285,
            0.4154570640135682,
            0.18354694016920148,
            0.6002442189687933,
            0.21341866623144667,
            0.014316938347873842,
            0.08793627028714246,
            0.3392283957313005,
            0.13925243425867062,
            0.8548106737856321,
            0.023344909551561766,
            0.3025207610425611,
            0.945332983928621,
            0.7773281993770145,
            0.15438666764019215,
            0.5686998114809476,
            0.425263658454611,
            0.5698154725148552,
            0.31995833736537793,
            0.5130449716086514,
            0.5415413773729916,
            0.5070504977672251,
            0.7314785852161422,
            0.9164413704713618,
            0.47868307722193637,
            0.2797797423204419,
            0.36067713495419584,
            0.31363319118611555,
            0.584081118667626,
            0.9605866755599066,
            0.029448776145766686,
            0.010061835212571646,
            0.22825920384517806,
            0.4407086297444909,
            0.10409337117967787,
            0.4281760291107445,
            0.5764734569225264,
            0.499293343590413,
            0.01156525855674373,
            0.6298199613506629,
            0.49898816079666986,
            0.3276462012959164,
            0.10406410400424715,
            0.21298925796192814,
            0.7143447965846307,
            0.14330778019390145,
            0.5220594982174991,
            0.1612667007268822,
            0.6588380169227596,
            0.4461368189323778,
            0.7678796507689396,
            0.666856979318627,
            0.014515023557269857,
            0.06127676491411305,
            0.597597408831207,
            0.37768829596205644,
            0.6219969152775721,
            0.9583181166230411,
            0.6332556393902555,
            0.4259492962772,
            0.07414642967222973,
            0.3986185525248154,
            0.9951481013998621,
            0.22221987650085284,
            0.25699804840117135,
            0.6179696994779834,
            0.8873487040207438,
            0.22843263145778092,
            0.6136308661271643,
            0.20955478035522423,
            0.3625123515888926,
            0.24938304785805532,
            0.8221863612689626,
            0.5369326805849128,
            0.3074126328324287,
            0.5063245676951496,
            0.4159538708900693,
            0.9826556534402682,
            0.4998560106432337,
            0.6554358467129536,
            0.2983117434500435,
            0.8151056082766394,
            0.5617812186813114,
            0.6034086492730604,
            0.7672580488648616,
            0.8394903728876504,
            0.8952815873470424,
            0.20341397101519865,
            0.806254091478268,
            0.34267625270062285,
            0.9476407294662152,
            0.2064797236007565,
            0.7119644394035383,
            0.5827323992955744,
            0.6333999682656258,
            0.6269038390180789,
            0.7306530101971123,
            0.5882072413692148,
            0.37411987116797396,
            0.7394626084285107,
            0.6034979112638256,
            0.9497021206516011,
            0.5288288204083893,
            0.2145851268473622,
            0.4127087781332467,
            0.7919434793274045,
            0.032275117628955186,
            0.7981343755742937,
            0.4236652909693067,
            0.6346653999042923,
            0.22605950058719504,
            0.23353327390338197,
            0.6454368269955231,
            0.6903283666721761,
            0.050371688763550804,
            0.3029110335301539,
            0.7411095709216069,
            0.646547812318539,
            0.9947965405449165,
            0.2069106723312889,
            0.37728669576698826,
            0.6167675225215548,
            0.00037982409956549557,
            0.13918512269609817,
            0.657012324268671,
            0.6996147873421552,
            0.9884209510126968,
            0.10207112600773072,
            0.019589969842475186,
            0.336964372718644,
            0.649511178652959,
            0.873933177063334,
            0.8072005961299136,
            0.24126519965699433,
            0.08087090399306507,
            0.3117059701347713,
            0.08713456880487291,
            0.9112616635359201,
            0.7724985234587572,
            0.1420125872236404,
            0.4067531007183758,
            0.4565126887800255,
            0.9900905198977834,
            0.9539849962805971,
            0.30592877425803555,
            0.37379672981349643,
            0.14814228940159513,
            0.30108786205609817,
            0.28475225344120836,
            0.6949194401093525,
            0.06133195651122192,
            0.1266008554617476,
            0.4678260487872101,
            0.13067096758710472,
            0.9966025134228441,
            0.8412390691232315,
            0.33677990591679996,
            0.3202667802084005,
            0.822367534698531,
            0.7484547922817696,
            0.580971290578795,
            0.731768755265186,
            0.656774887520502,
            0.7562236849426837,
            0.7324615664579461,
            0.02605317808061014,
            0.6252775532655153,
            0.4880586770226836,
            0.15830320642227313,
            0.1577722612303587,
            0.9399377238705633,
            0.09199236365438479,
            0.5141857949052452,
            0.20520174663841795,
            0.7312832640957163,
            0.8512997175885758,
            0.2429844189801832,
            0.6962563154025011,
            0.2911701833779504,
            0.39475481571321036,
            0.7029050839702605,
            0.6484825002668674,
            0.84235527303145,
            0.6778076937926241,
            0.6545263168073979,
            0.5140095648203321,
            0.26849315235961557,
            0.39907949272234455,
            0.98699167127683,
            0.21175821385418037,
            0.2727480230175171,
            0.9242951744683571,
            0.9544291666832672,
            0.3799670059501187,
            0.6075396673376375,
            0.3689661320555503,
            0.533480654790047,
            0.3948759679148285,
            0.9792483973292584,
            0.4545600614965364,
            0.25104389201596145,
            0.3016669883944194,
            0.5778596873800093,
            0.8821438126300893,
            0.2394120582161572,
            0.8514555613182432,
            0.02728197763223017,
            0.95618909135417,
            0.988317490479576,
            0.4656200643386421,
            0.9164866947655901,
            0.2425520118547264,
            0.2786013763922671,
            0.6409225165618465,
            0.4322896551380161,
            0.896283621442633,
            0.3528279710987984,
            0.3458776140371883,
            0.3926615598298674,
            0.6073580809091094,
            0.8978424554512642,
            0.6136497663097625,
            0.8015478709326808,
            0.3000791080727212,
            0.2612365927588616,
            0.49266658420487863,
            0.6204823307612688,
            0.7081517751636764,
            0.9873668343127493,
            0.7974268715138637,
            0.8552507847007327,
            0.6241304039507437,
            0.4493355957117916,
            0.4997017785448291,
            0.5423048132829118,
            0.7977727346207493,
            0.9471571348619552,
            0.10614004964510082,
            0.4668591435509476,
            0.06427964817407728,
            0.17870813686089104,
            0.40245064946196807,
            0.7014738544339131,
            0.5898819217035093,
            0.4060535075147119,
            0.5965737853917953,
            0.40369116418676576,
            0.8402342577916865,
            0.5023421457456148,
            0.5592856881877423,
            0.9876780795149107,
            0.38095436627099755,
            0.5418480772861584,
            0.5201475180818953,
            0.11077651762226226,
            0.04891875740038698,
            0.4321120103913775,
            0.3313372742768437,
            0.9524091629081568,
            0.7351481295656412,
            0.9802285170280726,
            0.03256205399455114,
            0.29622581219576605,
            0.7504473724310984,
            0.023610198380436653,
            0.35225229074093023,
            0.20623013094606446,
            0.18395329863493248,
            0.21162417115461296,
            0.266258371833647,
            0.7910894135194491,
            0.2948732772325322,
            0.45193405824630484,
            0.7827640072063997,
            0.3272464693835584,
            0.3590703348938932,
            0.9053887182392713,
            0.260734491433541,
            0.9335424084074295,
            0.4895282182065558,
            0.6093493355254171,
            0.827229023701996,
            0.5260395735713417,
            0.7530094768179988,
            0.8344653425327245,
            0.0175433083881974,
            0.47663052519822213,
            0.49355625329240416,
            0.6760088943526494,
            0.9183319182023756,
            0.48031034432991215,
            0.033144339236412024,
            0.9579611864275008,
            0.9355921241628944,
            0.6100038886322742,
            0.059582063986656,
            0.9304901072202086,
            0.6490490199389266,
            0.8133501581536393,
            0.5443970163208959,
            0.7895508898069663,
            0.008234208936151788,
            0.19448701666375245,
            0.5857552231262603,
            0.13161503863434043,
            0.8430986706515197,
            0.3595573006222008,
            0.7965981295941231,
            0.7622607721891685,
            0.8564553956523567,
            0.590949180036414,
            0.08410767390646223,
            0.3392872183125214,
            0.9667007736840229,
            0.6828970072620655,
            0.10869524768152206,
            0.9054388671728684,
            0.2510120984459946,
            0.72683154583824,
            0.9065014251550254,
            0.8240588017372226,
            0.39832881136411735,
            0.29586566076513443,
            0.27231653061202776,
            0.2225041595115479,
            0.7531107061947561,
            0.13623618600686382,
            0.962132738206387,
            0.09202790330977118,
            0.48912293025621323,
            0.4662671532771704,
            0.5724524802462818,
            0.10862916922174215,
            0.5097504645705111,
            0.4734157974868579,
            0.5936398000748225,
            0.5706760165639232,
            0.47877146129242354,
            0.3569484728678505,
            0.43750524080573117,
            0.22836690846107333,
            0.05014670163853774,
            0.9342739907641169,
            0.9123235129953162,
            0.7393837398896341,
            0.9614384354013058,
            0.2632619151416744,
            0.6015720930528762,
            0.2915538253232436,
            0.39651861916564457,
            0.7251375049603032,
            0.05867306112415971,
            0.16352602214241463,
            0.9770475402469034,
            0.5284184240147222,
            0.5762730072134777,
            0.4495121401495401,
            0.9765338268274393,
            0.365207870200173,
            0.12708090128554583,
            0.6027956048763583,
            0.5454357893582741,
            0.7111229935420789,
            0.9453585140521719,
            0.6393982206310176,
            0.0044447432736431924,
            0.39558040065831745,
            0.022294500695522745,
            0.7880293225490371,
            0.035977054535484165,
            0.4356643142262643,
            0.34794382350216524,
            0.4418759628658767,
            0.7548667280976449,
            0.2985768707213895,
            0.7887556919107688,
            0.024581552417021757,
            0.11479114310376104,
            0.6744517238457035,
            0.21380811554405432,
            0.7801353746945496,
            0.8945839576525288,
            0.02758538453796866,
            0.3035020839379008,
            0.9411808986768297,
            0.51154150401942,
            0.05196362821474476,
            0.7564913935174579,
            0.4945397921866519,
            0.07571769465829692,
            0.47176023514193377,
            0.9412222172978777,
            0.17821558159513373,
            0.970098583312266,
            0.049678001561435536,
            0.5282543131824963,
            0.9195636690845274,
            0.580707894992502,
            0.6436419240880916,
            0.8828698558116453,
            0.33456339231155086,
            0.7130944538737045,
            0.381334999001282,
            0.5283525925748707,
            0.05570331945995366,
            0.4275057645580982,
            0.10356525324389354,
            0.3448219306673921,
            0.7507575543143622,
            0.615387395817341,
            0.9538403086071486,
            0.49489369103001113,
            0.9342638137384386,
            0.9256850304480524,
            0.5880487343901908,
            0.17958373014335627,
            0.4091517089852903,
            0.385026519006998,
            0.45249038383133466,
            0.9858503372857962,
            0.00031198828893852504,
            0.3719068262043139,
            0.10411368790662046,
            0.44320908869970854,
            0.8951858582652665,
            0.19294289592225955,
            0.5232524924984412,
            0.6001668435571029,
            0.19734738436204435,
            0.37626328940149556,
            0.2897811525513808,
            0.6364073350780381,
            0.661498148086808,
            0.7813931421859311,
            0.6278265170462115,
            0.7583740975472366,
            0.6799175342481804,
            0.9066999579443696,
            0.3475828478605564,
            0.060343370769859406,
            0.202507435661069,
            0.8813568791886893,
            0.06942037583836713,
            0.3318909739561675,
            0.38693708277716576,
            0.3147812838735806,
            0.9083694614390199,
            0.163219723845336,
            0.7627845184211082,
            0.3966225506877895,
            0.3025087400461888,
            0.827305122508102,
            0.15974159228331752,
            0.5017386939774493,
            0.019212456406948353,
            0.1624888049671609,
            0.645504196486869,
            0.38307083501726735,
            0.40711015264386474,
            0.4528963731958805,
            0.11597437855385284,
            0.6381777994910081,
            0.3906872686622783,
            0.26976485437222464,
            0.26994263150216347,
            0.09705594563167386,
            0.4788574085815859,
            0.9775695003785172,
            0.6949308334368314,
            0.3463339983995992,
            0.6085143645378281,
            0.6286543736685208,
            0.6704200601265262,
            0.6444867982337624,
            0.7525131003940668,
            0.6670108559661567,
            0.3424460557095089,
            0.3631098600861672,
            0.060289091839881515,
            0.29590063473634287,
            0.6525899475679741,
            0.004016733044088228,
            0.6829409304109124,
            0.4207109732553651,
            0.9983817489619621,
            0.33796620630760343,
            0.21825058956525878,
            0.7145208077354601,
            0.3730746011421181,
            0.5148923528528624,
            0.359517615213803,
            0.5388128303438028,
            0.24612964336995868,
            0.8348211740910664,
            0.3680799354334652,
            0.5938777901835578,
            0.9693619278997123,
            0.63502699914433,
            0.2893253960008174,
            0.13509250325327038,
            0.36862188742042024,
            0.29663855954539364,
            0.014634597819981887,
            0.9838685067729482,
            0.40534103683651823,
            0.7403701388181022,
            0.16835583265547993,
            0.12700460220642362,
            0.7207691595571962,
            0.5895587933186803,
            0.13801375702961016,
            0.7365407266091945,
            0.895877103522504,
            0.8181127596953975,
            0.5512076632229491,
            0.43804704130036365,
            0.3047890197125347,
            0.05871850867556605,
            0.5664267566275901,
            0.6757612704343661,
            0.48814173731731725,
            0.9122164606471833,
            0.3953974538772719,
            0.8540596204791605,
            0.6624972009780908,
            0.1368244448223419,
            0.9002879813217955,
            0.03367267607122437,
            0.5859187152901306,
            0.5939521498370595,
            0.27070049883692604,
            0.3874703426725369,
            0.6316320569590927,
            0.2753649119788282,
            0.6983400164467722,
            0.436767193768297,
            0.7069482709119186,
            0.8570946674945635,
            0.2553948436480411,
            0.6216992746613719,
            0.6004845327072269,
            0.8422491143872637,
            0.9498636512812554,
            0.3866497784950864,
            0.28374422577522496,
            0.4484194706153507,
            0.15600873280876326,
            0.4110822407480742,
            0.6634369279220605,
            0.3320855993415712,
            0.1311963787482282,
            0.1016464738803563,
            0.6595722102106649,
            0.38410774470653086,
            0.9599050501610701,
            0.7171562328698321,
            0.5062967661500575,
            0.7461308001849676,
            0.4770988836615446,
            0.42913775832385126,
            0.1291400652996738,
            0.093764427689083,
            0.5267191753683139,
            0.49264539213347924,
            0.41107561151809036,
            0.022091368689010404,
            0.015325726632782755,
            0.8697551765369783,
            0.6837995372622451,
            0.7878106054084043,
            0.25831297384252294,
            0.8134685055868323,
            0.36202799069146974,
            0.9757682306946027,
            0.6473127267758995,
            0.2986449225504132,
            0.8881611353540096,
            0.19239744013030524,
            0.40733335986526575,
            0.5321704089004125,
            0.09898594195868682,
            0.9507441811197719,
            0.844880344564211,
            0.03731918415138358,
            0.3385622493858168,
            0.8504023048799197,
            0.6989467669923123,
            0.2549675349670987,
            0.48572089933797746,
            0.07288131474554993,
            0.09191220411594636,
            0.31033556092635084,
            0.30395736119114725,
            0.3846459319653658,
            0.392841391931931,
            0.24653075820003234,
            0.8544516886192999,
            0.8245326833288107,
            0.9958164191489396,
            0.19854830880772956,
            0.7766367108118657,
            0.09761322283985896,
            0.9417275796434791,
            0.7850343271348909,
            0.28609148479516555,
            0.5125689627715785,
            0.9258578542694729,
            0.275764740840005,
            0.6016164455627839,
            0.3462210019447792,
            0.3233452195422518,
            0.8409645559476775,
            0.7524922981717692,
            0.6339260467857786,
            0.9314935276749724,
            0.972677031160904,
            0.31299813790851694,
            0.8760082255657017,
            0.761333890179974,
            0.45938005370147283,
            0.520200318837958,
            0.6076441106575183,
            0.8724900021090676,
            0.8611036210839981,
            0.6552473684667536,
            0.6252833327380347,
            0.5185353691425693,
            0.9040227518479278,
            0.27970653915447574,
            0.9845390078155203,
            0.346673916502631,
            0.6623590611948782,
            0.006379473313236073,
            0.3734655087767689,
            0.7744014191561744,
            0.16961823466111714,
            0.3659953549414826,
            0.11118971782813591,
            0.759433495713853,
            0.1487394051434685,
            0.46865406255457365,
            0.8750582025359577,
            0.5073586413621533,
            0.41813721709898055,
            0.8391732370379675,
            0.5352984693712416,
            0.06008548992839924,
            0.350790921866179,
            0.21069799207917228,
            0.7949364947712025,
            0.39135641034895996,
            0.08709451091563836,
            0.8814193716323061,
            0.27314551566775214,
            0.9830695721780692,
            0.503725832798613,
            0.6084501117465266,
            0.47272365964120555,
            0.3892550561813739,
            0.09819293907928683,
            0.2384714035720421,
            0.08522948546776465,
            0.047686853879890134,
            0.5882039696597616,
            0.8672446036284184,
            0.27450605832508945,
            0.8613595906290894,
            0.8745126749842754,
            0.9851395552590442,
            0.4978119091583112,
            0.8993476815825009,
            0.7448498189517259,
            0.8369711602636397,
            0.8224174091515873,
            0.6822207719042813,
            0.1488219227059281,
            0.5778325840739624,
            0.4795873368483069,
            0.7090204945767113,
            0.15922992305806227,
            0.27438393598215605,
            0.2793210940950517,
            0.8884162176404946,
            0.6608015270385535,
            0.5282031642912364,
            0.9994618773829554,
            0.13146227355187012,
            0.14029156682207644,
            0.600131650519978,
            0.6390150442010555,
            0.31119136217286747,
            0.4016278174225866,
            0.07902734097141195,
            0.18629172890496581,
            0.8417267599628195,
            0.0009516721844141651,
            0.8728767092552249,
            0.5705298847912903,
            0.2329828050038638,
            0.1633535986935607,
            0.5443617553931892,
            0.9586315815409794,
            0.023573402817029132,
            0.630137650018288,
            0.3893839335340684,
            0.6032743106801581,
            0.6895868748541909,
            0.25749567302946685,
            0.9422588879367039,
            0.45962380488382426,
            0.13146747514986146,
            0.26518064000063035,
            0.7877056840494667,
            0.3289307264763762,
            0.6165600739898811,
            0.38624729302603034,
            0.8218929084592703,
            0.7273333375682015,
            0.6951739643262745,
            0.2620961872208437,
            0.867643751647387,
            0.5043461487128705,
            0.6162961579478152,
            0.9288355286850705,
            0.9298555904716704,
            0.019198104824075357,
            0.9225521021579641,
            0.0888584112444325,
        ];

        let n = diag.len();
        let mut u = Mat::from_fn(n + 1, n + 1, |_, _| f64::NAN);
        let mut v = Mat::from_fn(n, n, |_, _| f64::NAN);
        let s = {
            let mut diag = diag.clone();
            let mut subdiag = subdiag.clone();
            compute_bidiag_real_svd(
                &mut diag,
                &mut subdiag,
                Some(u.as_mut()),
                Some(v.as_mut()),
                40,
                0,
                f64::EPSILON,
                f64::MIN_POSITIVE,
                Parallelism::None,
                make_stack!(bidiag_real_svd_req::<f64>(
                    n,
                    40,
                    true,
                    true,
                    Parallelism::None
                )),
            );
            Mat::from_fn(n + 1, n, |i, j| if i == j { diag[i] } else { 0.0 })
        };

        let reconstructed = &u * &s * v.transpose();
        for j in 0..n {
            for i in 0..n + 1 {
                let target = if i == j {
                    diag[j]
                } else if i == j + 1 {
                    subdiag[j]
                } else {
                    0.0
                };

                assert_approx_eq!(reconstructed.read(i, j), target, 1e-10);
            }
        }
    }

    #[test]
    fn test_svd_1024_3() {
        let diag = vec_static![
            0.060466268434667514,
            0.24070433133590274,
            0.05184276387772846,
            0.9898738802383618,
            0.5867224416658019,
            0.8920795530759005,
            0.8213260193280432,
            0.1274585126399569,
            0.5137105305003337,
            0.8929508711344447,
            0.34644482421179423,
            0.15662711364303783,
            0.13204459060563245,
            0.2452181097021333,
            0.5529961084728586,
            0.6059885561225518,
            0.335661034135773,
            0.8178650650229435,
            0.6928232210699408,
            0.6993333722077023,
            0.3614916242067844,
            0.7887570130392116,
            0.14888470133705733,
            0.15853694539661844,
            0.5334912050093811,
            0.3612993548593889,
            0.6903211472739849,
            0.64703468631002,
            0.829497094858506,
            0.2522343254330478,
            0.9505881136261928,
            0.7340919699859023,
            0.4445327799200357,
            0.276725070716636,
            0.8813837074592618,
            0.8641241736265327,
            0.37601439289318583,
            0.7406106431867137,
            0.8187897453440819,
            0.44679682056462267,
            0.6866279118940135,
            0.6322556970967156,
            0.5177506730663544,
            0.9643854839179787,
            0.6465100827097693,
            0.339197465636223,
            0.7513292599815653,
            0.5602603062963017,
            0.4553307839097991,
            0.22844009431280932,
            0.6235416261588994,
            0.2262804849429565,
            0.9136387602980167,
            0.9915699182157715,
            0.7259256237506346,
            0.3929852244527421,
            0.2155371379405563,
            0.8086877453574028,
            0.8647281751947866,
            0.606292073442245,
            0.30240091664534696,
            0.9225948840817058,
            0.6810625155003264,
            0.05179968431711934,
            0.595659576422282,
            0.7564014939381208,
            0.13794493869812707,
            0.36285564872228715,
            0.32342613546406007,
            0.79124383797685,
            0.03550783493526544,
            0.7487151700009632,
            0.6765223339310117,
            0.5937152610233598,
            0.07403103674795586,
            0.7606782796859048,
            0.18578166030690013,
            0.7013672177585895,
            0.2033016281334249,
            0.36779130656982795,
            0.6624345349315481,
            0.17276643968296868,
            0.7326612329736877,
            0.39683700964818025,
            0.9249585764714129,
            0.9058620912528339,
            0.5273041580770942,
            0.25056807692037486,
            0.48806466395096515,
            0.06552243597979834,
            0.978315141677103,
            0.502782492170536,
            0.5975218887295574,
            0.8954320387435921,
            0.18424721934266852,
            0.7682726736917449,
            0.43241327339139235,
            0.9531509198549333,
            0.9046304333439057,
            0.209761648896709,
            0.8410391805702537,
            0.10657531625818184,
            0.05568939898497982,
            0.676138899758238,
            0.532991981445195,
            0.6550486984863335,
            0.5855669735264619,
            0.7507383900993289,
            0.6547493864972989,
            0.39773180941485786,
            0.21977443508356898,
            0.20152541150538528,
            0.23513555154348653,
            0.6260455323467065,
            0.5686434999835956,
            0.3645862855059864,
            0.5982150616652503,
            0.681662158287625,
            0.5023884533718146,
            0.08596469818006636,
            0.865840915557725,
            0.5096733454968728,
            0.7376731774487919,
            0.23500068740937674,
            0.7760973488757089,
            0.30075646064672434,
            0.9565535059308128,
            0.144637054217305,
            0.766299241844306,
            0.9182047949601909,
            0.8981438083995072,
            0.10243727574939943,
            0.6444340540568536,
            0.39948153880812554,
            0.22358680321222313,
            0.5940835708596859,
            0.3193867223358725,
            0.8098588935943911,
            0.010859229221052202,
            0.41213053744762584,
            0.5781544958693299,
            0.2559845296086064,
            0.7173914473935831,
            0.3618426918318993,
            0.6545691051178975,
            0.16389232580120328,
            0.029853796333192628,
            0.1112042501011069,
            0.4131986127015722,
            0.0675709858058694,
            0.47697288079451994,
            0.7579635090222712,
            0.939295939451755,
            0.06104837646390948,
            0.6746647049785921,
            0.41976327511961575,
            0.9033609580534747,
            0.2726505261765354,
            0.6810347096753436,
            0.5761914873814167,
            0.5605593648243876,
            0.2040106732498883,
            0.06559590462849507,
            0.19867228693975358,
            0.3795569308165738,
            0.3576230887086853,
            0.6646129207450863,
            0.19159996290089576,
            0.26289150671048334,
            0.22984353830414195,
            0.10998864141249054,
            0.14726051758768155,
            0.09891300931363878,
            0.20699396396845948,
            0.4396333193710559,
            0.66049328467152,
            0.7484210613887542,
            0.6214537395445267,
            0.19715195461724988,
            0.26926297891853945,
            0.013177250804681129,
            0.5401145872074694,
            0.8876912160030009,
            0.3958473022669018,
            0.8525317391364969,
            0.9646251671547991,
            0.7639431892415399,
            0.6176857025453736,
            0.2440159000876102,
            0.4338943641942402,
            0.6325069093330767,
            0.42747587311352664,
            0.611567551714215,
            0.6500639751807277,
            0.837380450954655,
            0.2998521440987161,
            0.17958444155475328,
            0.3889472877193627,
            0.6898239835481556,
            0.8370090796660907,
            0.6064193851083068,
            0.4274634932199487,
            0.6087718098881021,
            0.508256409689673,
            0.339079489379435,
            0.10509571352293445,
            0.005572687883438454,
            0.021981179575465415,
            0.29212956768147014,
            0.7867865404724902,
            0.1807384308077843,
            0.8665190424853524,
            0.6184357021120035,
            0.7536704913156237,
            0.7271598615669912,
            0.49475392709260113,
            0.728471317065983,
            0.3269837398505393,
            0.4704197531972808,
            0.5738584097539486,
            0.758731532875281,
            0.7260845747787328,
            0.6016048024552125,
            0.8010556210240337,
            0.004072879110130434,
            0.8437986241962506,
            0.5146240323523598,
            0.13904073839194253,
            0.6089836301203322,
            0.897844926432131,
            0.49057508349361,
            0.29005256571070226,
            0.15772152541113738,
            0.4738990343662315,
            0.5021527238202769,
            0.6937852389489244,
            0.009814520346564715,
            0.15675068544014148,
            0.44385575330248883,
            0.4485313346936677,
            0.4217206515389108,
            0.7577444148847231,
            0.7258868622262982,
            0.6193382669120356,
            0.039861150798117695,
            0.4349519716620509,
            0.5532646897637169,
            0.28283121796858735,
            0.0023393446168564758,
            0.54027399425744,
            0.5054603742880899,
            0.8414459593178738,
            0.8888400513837226,
            0.6477615598506755,
            0.5463852474548928,
            0.22949832244208968,
            0.31690470916831925,
            0.33430272710595077,
            0.0018161848198701147,
            0.49139883689044017,
            0.6797887885789651,
            0.5861639317229459,
            0.035282919686503766,
            0.10547619370542094,
            0.7271339305759082,
            0.5016067603875377,
            0.8157682787043935,
            0.4560885642814806,
            0.6929899653460532,
            0.5490310236928294,
            0.4530007155322374,
            0.44049548891776324,
            0.3073456560671243,
            0.7919866041386854,
            0.9399916206326354,
            0.7527419916944253,
            0.10112484431007052,
            0.2243815401473951,
            0.07039176683656823,
            0.6169715161346244,
            0.08123319473394963,
            0.33386307361579226,
            0.7833580512725923,
            0.2505341520737143,
            0.30782230756455664,
            0.7479846627003448,
            0.2317759693352216,
            0.933348786188565,
            0.2750733182757722,
            0.7252317550631081,
            0.06447834497830485,
            0.032088850870325425,
            0.8873228725504719,
            0.15311031466542857,
            0.2687565257110237,
            0.004116203327798051,
            0.9334773509255564,
            0.4411582352055272,
            0.915699640538774,
            0.6540205822944993,
            0.22129013965221733,
            0.9197389724854845,
            0.7413607462811231,
            0.33557913718762133,
            0.48682708653142126,
            0.23261276643023798,
            0.7298006313346186,
            0.05533463818311113,
            0.5593987881884823,
            0.014043027025292698,
            0.8466474369742201,
            0.476037950820811,
            0.8444410961569767,
            0.16171101653838926,
            0.42377692467080386,
            0.9780260328429565,
            0.2702756266464895,
            0.6443988832387432,
            0.8993272243844188,
            0.7327024724314705,
            0.6054206077823142,
            0.588745513467837,
            0.6099464369729185,
            0.5242293066667245,
            0.6367910856473106,
            0.95096174945863,
            0.7939574092983276,
            0.3852863174746787,
            0.20255106392741273,
            0.4245303071023879,
            0.9967055343008029,
            0.4458664645461715,
            0.9257682958149203,
            0.3415930338870309,
            0.19630239658931292,
            0.5492196374119545,
            0.15687901061103193,
            0.6952090496329576,
            0.3956802166028377,
            0.5576427063755885,
            0.2040497200375847,
            0.5924369085502799,
            0.31777150634999896,
            0.6188717500402732,
            0.10233206975978115,
            0.6843201135492385,
            0.97041224730265,
            0.2995654214923589,
            0.6445544485180777,
            0.8301737642729924,
            0.1087683656777727,
            0.20019953992353745,
            0.22780836215874622,
            0.1276194439987589,
            0.38899213955877954,
            0.5064763107596547,
            0.6461906101897172,
            0.29746881883163756,
            0.12374457000837158,
            0.8069582603483658,
            0.25061357630791214,
            0.271756743228987,
            0.5239738641031194,
            0.6553247838521854,
            0.683817735836212,
            0.06802304767960465,
            0.7418517643983699,
            0.07774073352161648,
            0.9620799864233502,
            0.7358697033778088,
            0.6321490186410873,
            0.6177412024364181,
            0.868224449638578,
            0.7861974420960143,
            0.9434823199966517,
            0.6471369630040992,
            0.4048133530399751,
            0.21870779533881557,
            0.04034528679441152,
            0.40085799212995166,
            0.7373559016434784,
            0.8017355475036948,
            0.8967226537967593,
            0.7176831089070658,
            0.2613162507363975,
            0.3295930554915326,
            0.9184791699454984,
            0.640401400967848,
            0.8506077837886161,
            0.7664496800476533,
            0.1840288042109116,
            0.19215292024269792,
            0.9621850056142566,
            0.7270739851985532,
            0.4000222354425286,
            0.7398694017088451,
            0.8102704435115223,
            0.04869313905506856,
            0.9793000884909765,
            0.1741974674289617,
            0.3761252028210257,
            0.5186546213017674,
            0.6544630055785722,
            0.035486878857104,
            0.8302105873193061,
            0.027163249430116165,
            0.22238221247779077,
            0.6629828858480685,
            0.28167067761823206,
            0.15507531405462183,
            0.5299310060229058,
            0.17615517303353367,
            0.9160813969780187,
            0.21545598837949875,
            0.12327465612788735,
            0.22341207675958197,
            0.7350951031975246,
            0.20212244417713343,
            0.10806891511769379,
            0.07983934140633298,
            0.5784829838569209,
            0.378211476865968,
            0.1600334499763354,
            0.005458695509232947,
            0.04659586838146379,
            0.16941413051975984,
            0.8176494023491518,
            0.8193983869596879,
            0.8108230208574312,
            0.6743040619295617,
            0.9291104455200995,
            0.4551862464547173,
            0.5081503416196177,
            0.5863629766904948,
            0.879725318736957,
            0.44780357058335174,
            0.08000528315519806,
            0.7156738239702364,
            0.5579713809884808,
            0.3849056408121806,
            0.5439488769415571,
            0.4231699491641453,
            0.28212369837803397,
            0.5234097014488315,
            0.8952558680115784,
            0.5907776503878093,
            0.20310821526826173,
            0.36655334752448054,
            0.8022062978517467,
            0.28033812768504074,
            0.6691226037291831,
            0.7733590567458623,
            0.3259616467610902,
            0.6486620415911235,
            0.9052615556593607,
            0.9541189951513983,
            0.7111093017245231,
            0.824177966913593,
            0.3118026811100194,
            0.51957010835145,
            0.6783040667606364,
            0.2548584383539012,
            0.775639794655331,
            0.34912344674321705,
            0.5113227047977591,
            0.38981951977369,
            0.6214892598250877,
            0.004845542093178001,
            0.004956807664176277,
            0.33672401396540286,
            0.6019483398625038,
            0.5470295540948279,
            0.16163727213383083,
            0.31074927912001327,
            0.7678958430795156,
            0.6317779345503789,
            0.9512327678975153,
            0.46061166841382717,
            0.10696108239436053,
            0.381467056434603,
            0.8552430463509642,
            0.06439561868365684,
            0.12451070106137119,
            0.6305588706791049,
            0.40516850182186237,
            0.003916288600096851,
            0.26809866573936025,
            0.5283523433835026,
            0.8916385072682825,
            0.5637932289017291,
            0.5345383280816733,
            0.07945741563095476,
            0.4677977232825933,
            0.2180420111723671,
            0.6573840430041248,
            0.18553753661109285,
            0.012482076643287265,
            0.685607665579137,
            0.026974059044606813,
            0.6536573357598158,
            0.5502019862922036,
            0.5546464002416955,
            0.5321062302737098,
            0.7223743474389412,
            0.9466031287158234,
            0.7835158042814866,
            0.9456723815823406,
            0.8602920602906611,
            0.8689901181761116,
            0.9960072752618597,
            0.008499563604653315,
            0.832298969481983,
            0.37385951105474646,
            0.2766630334670427,
            0.35414551853848253,
            0.10147091413171239,
            0.44593584268151776,
            0.8478741461548431,
            0.174711009992771,
            0.29931914487221467,
            0.7374595528588062,
            0.6062871995304973,
            0.05456237473928949,
            0.8095786284527302,
            0.6711379698820543,
            0.7780227416082675,
            0.5698575810039058,
            0.8374210414415328,
            0.3421536429195372,
            0.5416694501192495,
            0.4345414149053174,
            0.0041291448761129335,
            0.12106504325643708,
            0.9729478194191092,
            0.2957285912729133,
            0.6734063562326669,
            0.32956672575040535,
            0.6941989745259985,
            0.39087545739740936,
            0.6285237359484402,
            0.5464775357369276,
            0.04355009567600665,
            0.29659575970461016,
            0.5061376740223799,
            0.09041273927723548,
            0.7292889822993045,
            0.9631242356697804,
            0.49001636433847895,
            0.3729672006767415,
            0.0402186820818361,
            0.06709255594189967,
            0.12055672632871284,
            0.6958349898364417,
            0.5168344061425046,
            0.9723494562157396,
            0.2768020892836862,
            0.8610575915312655,
            0.7795389307661503,
            0.1643268251583806,
            0.412251275774113,
            0.4955849398845882,
            0.4498738620874162,
            0.9636214177620601,
            0.09621299324389432,
            0.23941877187780947,
            0.8142114016710098,
            0.04011767557202495,
            0.8448948522047415,
            0.06508688407594121,
            0.1164247528356549,
            0.48177931620517767,
            0.030447745332007226,
            0.712141309568534,
            0.8902785075389523,
            0.44991611450123314,
            0.06202794456734484,
            0.5672075413811598,
            0.3724815418357975,
            0.5988538990093506,
            0.9866407611428497,
            0.6707747684273065,
            0.12620016729013617,
            0.38787793031078255,
            0.7718898918841123,
            0.7497504867080291,
            0.37030721375666986,
            0.4216214156243707,
            0.7528013955672017,
            0.26375981042113605,
            0.5955910637988521,
            0.073722423286946,
            0.9062905993144147,
            0.43474707761722586,
            0.5188590494104617,
            0.34246642189260734,
            0.34146401922917746,
            0.7885189916945865,
            0.17355374722805195,
            0.32886168894915657,
            0.9986868630489248,
            0.5417750460652352,
            0.744462980469815,
            0.8548589409072936,
            0.9716442844863626,
            0.4974343077698695,
            0.7760922947042525,
            0.7944167596266641,
            0.4928778233861939,
            0.8714672280850508,
            0.7891406408650357,
            0.11847785463381755,
            0.040171363401182436,
            0.028299207378359337,
            0.5586780482622543,
            0.34886663795266004,
            0.7099344140963683,
            0.8278707100372183,
            0.40841680139205205,
            0.22717483353945367,
            0.024642126536398412,
            0.1999884989286147,
            0.7682993947910368,
            0.4467044441952054,
            0.24941543162963653,
            0.3186497336994355,
            0.42566616386915146,
            0.06394087636847845,
            0.41221892621264355,
            0.7239998149149116,
            0.10263700729951386,
            0.9547246834071182,
            0.3504771138967774,
            0.4652280096801351,
            0.893819877320391,
            0.849888410579933,
            0.3422029148752527,
            0.6756666171603669,
            0.7157645146155942,
            0.5608365577577554,
            0.8169342505195489,
            0.8718381947664786,
            0.24315192131144703,
            0.3167124330729658,
            0.4359695688746591,
            0.8652427501010571,
            0.007244731613245792,
            0.7170792894729603,
            0.6366906405192163,
            0.40636730975020696,
            0.7724049513343092,
            0.04140455800237475,
            0.5565739537948702,
            0.32626353298657196,
            0.031511958736421763,
            0.7643753998335007,
            0.4051384102734069,
            0.18506632215576713,
            0.20762432059272773,
            0.8449985558223492,
            0.11777461293647384,
            0.25036959722091445,
            0.5560865084668252,
            0.8999275714064795,
            0.3714033649319849,
            0.7131232729916241,
            0.4214041397785607,
            0.2868553632670624,
            0.7576367387439932,
            0.135169590926147,
            0.06681060010943962,
            0.5466903263659443,
            0.1176693227529878,
            0.2642373085121925,
            0.08272031716148143,
            0.17071942595720246,
            0.3933979275630807,
            0.725986982480158,
            0.6823115648016137,
            0.37448258553278446,
            0.9193929709707876,
            0.9253058562898909,
            0.6022093621721928,
            0.15242080609822006,
            0.28374219703922376,
            0.9753381587492779,
            0.04746481883728326,
            0.25664431695643397,
            0.15539021587341173,
            0.7862337203184313,
            0.27574131085352216,
            0.9125493475582659,
            0.4230557954471307,
            0.7868471237898292,
            0.7151805984378823,
            0.7558284163545362,
            0.6892190014608551,
            0.06942146888586542,
            0.10266718139933051,
            0.6584439531793709,
            0.6037903104572208,
            0.8070197306982111,
            0.9465443219138302,
            0.9624363391526284,
            0.27380020601043753,
            0.46071553394157294,
            0.7591095060843289,
            0.9519670757359452,
            0.3032361639665756,
            0.48877456852536816,
            0.23597884517457612,
            0.5107030646252234,
            0.20055459161258038,
            0.09388146557794119,
            0.6746870338805814,
            0.584734389609156,
            0.17223641386296107,
            0.6671529009880626,
            0.6782652738638101,
            0.9629082587670728,
            0.4453057757752613,
            0.8191854208325298,
            0.3155184484281153,
            0.24310357163460306,
            0.2677050840923676,
            0.8506695537257243,
            0.5568953368620309,
            0.404051974119802,
            0.6977410257168541,
            0.8655026435822981,
            0.6094346023419185,
            0.2971758141470803,
            0.9583171645243728,
            0.5485752370307014,
            0.46227742900340474,
            0.5935123026790572,
            0.3023653692266124,
            0.7075259003887401,
            0.5516307411953051,
            0.17245473487528018,
            0.985520967370092,
            0.15238613683361613,
            0.5884373032730832,
            0.6483816290378118,
            0.6201418469386697,
            0.1322361382943461,
            0.9934621639718728,
            0.9781355751886759,
            0.2358141672103008,
            0.9282631573809376,
            0.9433712252291138,
            0.19114365959486102,
            0.3789557238185065,
            0.7949475415804416,
            0.16656266849309387,
            0.2261051183163817,
            0.8506784198446528,
            0.06489027887765886,
            0.8653835416561619,
            0.08671353337358623,
            0.854043504677798,
            0.11813182777366338,
            0.6238952061942427,
            0.7689563050256499,
            0.2774346379980891,
            0.33068522837337777,
            0.4261665169559953,
            0.3099483646782023,
            0.6193186881829519,
            0.9739056868446708,
            0.11586525141234083,
            0.6398109412587539,
            0.7251085509952001,
            0.4677220480238611,
            0.1523212370477175,
            0.44766112997883833,
            0.3173429343510642,
            0.9425945825618068,
            0.8219018790724759,
            0.0645605164193992,
            0.09485431523703491,
            0.3894183522316814,
            0.6093011450600736,
            0.878984960737678,
            0.029343395573226738,
            0.3324060238167921,
            0.4886808734964564,
            0.5156854980153162,
            0.14218370847598005,
            0.5344594796528955,
            0.18768022835138842,
            0.6497327847300925,
            0.3684397075864396,
            0.9193626028147847,
            0.6522977895048481,
            0.08962724254791887,
            0.5922688193985055,
            0.7356970039838325,
            0.5824259203362957,
            0.5407058235265322,
            0.5711451683211365,
            0.49565959637601587,
            0.5977429115752058,
            0.10924570433840974,
            0.22480770459645172,
            0.2871058305915194,
            0.2266596204129937,
            0.7738819468480213,
            0.17605223928789138,
            0.38475472364931584,
            0.18629263259774886,
            0.1807323566515121,
            0.8170521024950523,
            0.12208008613018384,
            0.005467015504290074,
            0.10910273820449712,
            0.6060511051567308,
            0.3434570300901455,
            0.00042444932756435794,
            0.8452782644730006,
            0.9714023607029164,
            0.37583231162140673,
            0.11442158503702637,
            0.96867187984804,
            0.6184568216803141,
            0.3118038927254134,
            0.9438646917460232,
            0.7614637047418356,
            0.5403571615757983,
            0.8389639104963986,
            0.6474181216179613,
            0.6002145037813064,
            0.6881183722357025,
            0.047934266407373305,
            0.2501358423963378,
            0.5416590133894971,
            0.0017290255627294693,
            0.6642777653367096,
            0.540981209017387,
            0.5647677211241512,
            0.35899883177363723,
            0.38419491129487704,
            0.1706402044865839,
            0.5809668876996945,
            0.3531159350243369,
            0.07469854012290267,
            0.6246702892852438,
            0.9643022681914122,
            0.7261306179469745,
            0.4144560462391834,
            0.36244445415993687,
            0.2558861740953887,
            0.5695408223196087,
            0.49036727984389294,
            0.16847584290780449,
            0.8195450223938221,
            0.07089507494394409,
            0.5942090735800623,
            0.9195293910931344,
            0.3638411671978604,
            0.7620942124631933,
            0.753028025483766,
            0.7893800196806074,
            0.16155119783118355,
            0.15624280494307863,
            0.5884210171713868,
            0.7877299121605147,
            0.46736221856235216,
            0.1075177932212199,
            0.10943159980480244,
            0.7449189094923113,
            0.9536679155443701,
            0.2438288221302346,
            0.2948772532797116,
            0.2811340878298155,
            0.7900588490997729,
            0.2331033682917616,
            0.7856198274591133,
            0.8004759863798926,
            0.5975198549591832,
            0.4293998770749423,
            0.2261563303429156,
            0.4908675014741476,
            0.6516101907788727,
            0.9849042526829336,
            0.5096053125861685,
            0.2524687595398467,
            0.9851444981174782,
            0.40827194856378135,
            0.19817984578370884,
            0.0567899929553356,
            0.23187270743000887,
            0.26135901720792143,
            0.08418116464410874,
            0.0785658190220071,
            0.9209075924087098,
            0.597140852146377,
            0.7469386786418659,
            0.733290194898679,
            0.4997152469553926,
            0.8008279336390877,
            0.5083788664229297,
            0.1759501822664441,
            0.727822384427287,
            0.3107953887336189,
            0.3263461902698255,
            0.2602514070443085,
            0.5896784464321209,
            0.9483301974243472,
            0.8834965514066166,
            0.22232780026925225,
            0.5096825539156082,
            0.5352893212972208,
            0.18860766379968918,
            0.8094037239862744,
            0.1945477950005433,
            0.3346403145575815,
            0.39807032343898285,
            0.8014796253216447,
            0.3525327782740121,
            0.10204756706208229,
            0.6549680247117429,
            0.1931072702615666,
            0.4008562149257491,
            0.25985247444643433,
            0.6846273489478197,
            0.040128872723685705,
            0.810217419170049,
            0.4727426697704351,
            0.6503758044743186,
            0.47353437971881507,
            0.778937139300587,
            0.7110721606372872,
            0.4540262500587219,
            0.9742712248510724,
            0.8712133108702252,
            0.6874239178482641,
            0.5127367450795323,
            0.3549059571467863,
            0.9954711217104326,
            0.16329220375994358,
            0.22488682737865584,
            0.04238837235610171,
            0.4797553309064344,
            0.39065154498280175,
            0.6429885955630378,
            0.49568347415307357,
            0.022575523424778088,
            0.961194228717609,
            0.9613236030742949,
            0.9123276267620781,
            0.3017018564004411,
            0.21606041602015058,
            0.8520552011529079,
            0.8661346671513412,
            0.8974966130669128,
            0.9091192570171245,
            0.3977199981220819,
            0.407665179605855,
            0.5989973603256267,
            0.8950455178181695,
            0.19226975006208935,
            0.8877488866890373,
            0.3058880482205242,
            0.39430566699018443,
            0.7111942756699692,
            0.9717304353935692,
            0.7953174032548256,
            0.6411039190462657,
            0.4183770103618606,
            0.45472310569528984,
            0.6292905925127901,
            0.8237179638721334,
            0.5209522377910386,
            0.2444280522514548,
            0.8176983127801919,
            0.2725176067143923,
            0.9322298522883796,
            0.22275268702206463,
            0.513242497886465,
            0.08604118049226273,
            0.47521992113068645,
            0.5243709544234121,
            0.17208119423584156,
            0.6655622046848257,
            0.6218443959425871,
            0.5226727984184025,
            0.15026648590545966,
            0.15125953123114178,
            0.4461976200696216,
            0.7948473402753538,
            0.3481285828584283,
            0.05300674182014431,
            0.1502646084491771,
            0.08540063807203002,
            0.9725510293110878,
            0.07053657732254148,
            0.9419090652947318,
            0.13506577135812226,
            0.5204099044846535,
            0.653488689141511,
            0.7710005906520486,
            0.5543106542237832,
            0.3842205681780151,
            0.9212065802958664,
            0.9271653521751265,
            0.985937112031812,
            0.8052550387811409,
            0.9540875736578137,
            0.05302964512587549,
            0.709320652650226,
            0.48105952476102876,
            0.3848453304921482,
            0.887370590062141,
            0.7946318913319451,
            0.2067211921271186,
            0.24186179375964567,
            0.1753371261689105,
            0.3700353275190944,
            0.0004962975712109463,
            0.3024470353450792,
            0.9107571026498392,
            0.6176365464203079,
            0.903725980632377,
            0.6396323654545216,
        ];
        let subdiag = vec_static![
            0.03514759598732997,
            0.9166162329394264,
            0.5920073429746628,
            0.03737502282782612,
            0.27541296782662794,
            0.35757716121142913,
            0.6802416588616755,
            0.7155740678046522,
            0.9649012116262692,
            0.11672125251026266,
            0.9199392299400145,
            0.04817171621171579,
            0.9307293351081974,
            0.4760605890414521,
            0.8545545249806196,
            0.19139397234856892,
            0.8017765737079704,
            0.9769789093224842,
            0.5608589872755957,
            0.18330323024368844,
            0.7059108114098925,
            0.48011278224461784,
            0.8806095547117393,
            0.41943394511355314,
            0.37253038596905974,
            0.4313419335715023,
            0.7803145989384619,
            0.019417776658885333,
            0.3266616106392123,
            0.8648268658307395,
            0.281834063555935,
            0.9958657655364979,
            0.7132082295208675,
            0.0907894892925506,
            0.748076919810375,
            0.3976930251985855,
            0.27650571177384575,
            0.03956621553665818,
            0.870979758528376,
            0.02177900264583299,
            0.09696722421897186,
            0.7156762119010204,
            0.47255821795062647,
            0.28423526759639406,
            0.2362274799311106,
            0.8357206203449294,
            0.44117178225566733,
            0.7160926363896872,
            0.9324907248737948,
            0.5620567520065424,
            0.9204444617400935,
            0.6483125256827211,
            0.8388013670815763,
            0.11787659695285657,
            0.6330966970250446,
            0.15906857751212133,
            0.023688684061386356,
            0.03861804582118156,
            0.11120087954397706,
            0.7888367298801179,
            0.40081391746255934,
            0.49441010134626695,
            0.32634190993787904,
            0.25140319280159973,
            0.6801348242915457,
            0.6868281418370772,
            0.40611738472575964,
            0.07828447213451328,
            0.6749380673181273,
            0.08926316605641849,
            0.12043114953075107,
            0.013241903881996309,
            0.6314587819785878,
            0.2359252644095119,
            0.9845051905574118,
            0.29696107568866137,
            0.4106419477189991,
            0.15123381021597337,
            0.3709176678784105,
            0.18195819174032302,
            0.8533596504653861,
            0.7510672174999008,
            0.21626704312676748,
            0.9636364489347745,
            0.6128308055692433,
            0.953115514029098,
            0.8436888435919734,
            0.2996352171373927,
            0.6372446318265245,
            0.2746728984555966,
            0.1037273047268431,
            0.7978236676924543,
            0.9484254202265332,
            0.6624929480737886,
            0.1133937333565549,
            0.33829316318797487,
            0.9223632567091015,
            0.5851974854049407,
            0.10409325590177565,
            0.8740115390408132,
            0.9845352406551445,
            0.13902919473775033,
            0.5031422886935428,
            0.04688375456992533,
            0.5081286565974175,
            0.868093991897646,
            0.17490335099705945,
            0.6540328561893354,
            0.8398081465083584,
            0.21693015392629467,
            0.2469186553171181,
            0.08429394697815695,
            0.352751452137246,
            0.10758030077795577,
            0.22700681314093718,
            0.2347731841807721,
            0.9977616327533844,
            0.8359041699873175,
            0.006618522134742899,
            0.43664918463005764,
            0.8210277077119603,
            0.2638399939782796,
            0.17336929801988665,
            0.10285997583877982,
            0.4652625971704971,
            0.2733273522826204,
            0.8574149574522787,
            0.9806185589394525,
            0.0015017390133097441,
            0.033430903824059666,
            0.30159631467225945,
            0.8974993026788298,
            0.6400720771489617,
            0.8978507974188904,
            0.2520999469590316,
            0.45836130177468615,
            0.6328879304823644,
            0.7423569927089957,
            0.7647616636938184,
            0.5064526550497733,
            0.9850250775854433,
            0.16694222041216755,
            0.4455501188059532,
            0.5374287739797512,
            0.3189502630832104,
            0.7215562186523863,
            0.15916949628148003,
            0.46432140332822636,
            0.9345412518790319,
            0.9020418859218371,
            0.9063125261934585,
            0.8776716709019419,
            0.7372809118435587,
            0.13416655716832815,
            0.20863558478898614,
            0.9601527516411504,
            0.04287960921174683,
            0.37755333952553993,
            0.3417720230505361,
            0.018645823306778442,
            0.6056383320364142,
            0.6514231553072014,
            0.7636607888909127,
            0.9636169157716151,
            0.9766562186982161,
            0.35309414208504075,
            0.666500807200095,
            0.7314451558204479,
            0.32899361692172513,
            0.4147820422080323,
            0.010638598007814415,
            0.75815558581836,
            0.02169815428383548,
            0.8393399189992057,
            0.5944394006298711,
            0.22382973848925802,
            0.3397652600088771,
            0.7015409578708514,
            0.7786099639559989,
            0.9043538590746677,
            0.02128322827623863,
            0.5587072619140682,
            0.04155163711549137,
            0.5996987443084926,
            0.553583814129375,
            0.3554407110197826,
            0.2744460862519005,
            0.7128777538871974,
            0.7530751205335345,
            0.9009495860810993,
            0.8917721418615082,
            0.9798925601886752,
            0.19342242305623314,
            0.024017196131949636,
            0.5245663591541098,
            0.13727729379295928,
            0.5491421206018336,
            0.060808975234664886,
            0.38472819295119587,
            0.6378674218418567,
            0.5227205485068104,
            0.8445990640792207,
            0.012436340335015283,
            0.7408733967606148,
            0.07054771013209749,
            0.07155631798187301,
            0.3109309497032946,
            0.8672898374299146,
            0.3583852631021335,
            0.5245779103895792,
            0.18802706385725088,
            0.7140966740493614,
            0.5499514336069355,
            0.09955949956945287,
            0.7245623053513952,
            0.17377343762547004,
            0.6595068015219847,
            0.8349315746460483,
            0.8113518945228055,
            0.5483076047004339,
            0.2756110290420781,
            0.3054178348722053,
            0.06871776701231946,
            0.3719726135350194,
            0.8287237621631821,
            0.8857687441690538,
            0.7100048919244728,
            0.39966266759915614,
            0.5316956873267635,
            0.5275213833268538,
            0.3615522954432596,
            0.2571877574661159,
            0.3199700071612991,
            0.8551425371306557,
            0.32048877323539016,
            0.02585267594661278,
            0.6839778709817063,
            0.6419014019329597,
            0.9408387544324863,
            0.40948833652960426,
            0.4543886322254115,
            0.135552823291479,
            0.0573359887606284,
            0.7641878684368038,
            0.6400819212712041,
            0.4034058258248284,
            0.048389281308018006,
            0.5084248649672614,
            0.6353549066600827,
            0.8206578550882837,
            0.044157301972724716,
            0.9774796963860074,
            0.17613366930197827,
            0.7236733379068533,
            0.4395954155931402,
            0.53220451402432,
            0.7394251944824503,
            0.5099726221171791,
            0.9068096835994949,
            0.8567380773152629,
            0.3990114830989736,
            0.5789177208783894,
            0.21703260425390858,
            0.09801724009974588,
            0.8261383241604257,
            0.1720131914688977,
            0.1741672387512948,
            0.21573167586527808,
            0.363995620056507,
            0.3877594862249947,
            0.7627538721891199,
            0.3028001552657369,
            0.6925192592005605,
            0.9983574500441812,
            0.4402506075402062,
            0.9969490201849028,
            0.883564743735146,
            0.5745948407978116,
            0.039390447079829194,
            0.648982540750061,
            0.8121077463283781,
            0.3078661764303069,
            0.5112013516016634,
            0.45855507485752867,
            0.8680067321374374,
            0.5878616980657485,
            0.04647126546747615,
            0.3469028573493742,
            0.7098480342833867,
            0.10437610961707922,
            0.7645967501135319,
            0.3358450730062015,
            0.9897160552499019,
            0.7369318977715539,
            0.416174494159927,
            0.8739296858827521,
            0.7469703290976271,
            0.2688365437717575,
            0.629024514537825,
            0.049646037576477675,
            0.5808053589412941,
            0.3030673768920322,
            0.5560270535501435,
            0.6428769952770576,
            0.0668502266998201,
            0.7596493050830692,
            0.8443113846787801,
            0.11146396526586733,
            0.6318800710105476,
            0.600856525368827,
            0.6752109718053634,
            0.5673512968959964,
            0.4574054679020503,
            0.10359220949390213,
            0.8382928237132409,
            0.14043612573194442,
            0.089990326035021,
            0.9132118395578155,
            0.6141894909830402,
            0.10621386006678057,
            0.29232241722433105,
            0.07661754069758586,
            0.2324111815248312,
            0.12446678249890808,
            0.8128595794855592,
            0.4006073821901792,
            0.4734387523433071,
            0.887065723190606,
            0.26632779471001966,
            0.8714787815141146,
            0.8471389583861026,
            0.685316691195084,
            0.6707247582581257,
            0.928579547545599,
            0.02553828614773368,
            0.599534398559242,
            0.2872291636088161,
            0.5187303634053566,
            0.6853633041230819,
            0.9801566043794827,
            0.28991992195404803,
            0.6133172505587892,
            0.33597328743115185,
            0.8121184044422642,
            0.17083229872471262,
            0.9513644781388463,
            0.35546101658313944,
            0.6423236811634249,
            0.7468201274076843,
            0.647302785857128,
            0.010738149004345554,
            0.7183851270529749,
            0.09435605574406558,
            0.9530985532147299,
            0.06728057507618279,
            0.8034456703633842,
            0.7979147696402938,
            0.3822105215671153,
            0.2297974374183981,
            0.44640217719557884,
            0.1883600083067679,
            0.04750218394092687,
            0.22170690949746585,
            0.4840046643798931,
            0.22475138267285244,
            0.4198932328501168,
            0.7889157743393862,
            0.15281044858523352,
            0.14673384971080916,
            0.006993889316575963,
            0.6364919809167784,
            0.4172781587709603,
            0.4491972187640223,
            0.16182920016249625,
            0.007403967991580274,
            0.691461320209579,
            0.5816637213029577,
            0.7522606715818594,
            0.0918933405438338,
            0.35159417994924447,
            0.7602447419723402,
            0.7142902647900448,
            0.6382243601647429,
            0.8480143370848189,
            0.6314392835621164,
            0.8439084708835212,
            0.11071350237055422,
            0.05782456734970254,
            0.6617906034954492,
            0.9185891532719831,
            0.12143371274043513,
            0.492750786634106,
            0.7853426435898732,
            0.4497887225806262,
            0.18635877906252618,
            0.019811422063421902,
            0.5934761906380646,
            0.4455857188059149,
            0.8096388796203312,
            0.10275649974036594,
            0.6257368006353998,
            0.575750202333855,
            0.8573719070085712,
            0.8239032358075691,
            0.34403591200202155,
            0.9345733962318697,
            0.625369461070786,
            0.43915860018762687,
            0.5899961835162477,
            0.053995956233354736,
            0.38046944755554934,
            0.9491278288233935,
            0.033130977568233466,
            0.17607535381374018,
            0.05801732382059033,
            0.4538755877525351,
            0.11743529785925255,
            0.6791318581799634,
            0.08544473560430244,
            0.5187497921796217,
            0.5889708254267537,
            0.9665003452472861,
            0.10418296172193109,
            0.0672448602119764,
            0.2342804383991538,
            0.7534895198422364,
            0.441835359111311,
            0.8080281548555065,
            0.8371361091120421,
            0.8671078767817112,
            0.9719590542686816,
            0.07419467888313469,
            0.43012106218645596,
            0.8088664789679649,
            0.13528519103218006,
            0.0798446432975134,
            0.013418704157530992,
            0.9794852288563065,
            0.9289867971403145,
            0.5690022331612405,
            0.8945759232383244,
            0.7853938914133286,
            0.7310762075200202,
            0.9859379728103466,
            0.831092060494062,
            0.8217409245038338,
            0.8999605479073557,
            0.05261025832698352,
            0.33623061838562873,
            0.7648537258952935,
            0.7659562194183969,
            0.8719329057090859,
            0.08419227378203165,
            0.3242045288837502,
            0.5171860173551537,
            0.019912782586444133,
            0.9384089169173252,
            0.5965042752440841,
            0.48278375926195205,
            0.16392438276287424,
            0.187140389551246,
            0.16916393839881505,
            0.13726551357673022,
            0.8269110011590474,
            0.6118271679991382,
            0.8152732754679466,
            0.666745307831142,
            0.6476883393802874,
            0.6220901203388768,
            0.38040332376345787,
            0.34115141071142563,
            0.7873648179135119,
            0.6664534365764685,
            0.9058029740954557,
            0.6484503546054443,
            0.764668715148312,
            0.9612701627106187,
            0.08976716760112169,
            0.7774948110004088,
            0.5308908425168403,
            0.4139821948135307,
            0.15163369419930572,
            0.11394098215608839,
            0.5513266492622274,
            0.09339060427254409,
            0.6481276388246241,
            0.598444020910348,
            0.8298448230916241,
            0.6727176112811568,
            0.38333435133740557,
            0.6468125790238676,
            0.5242080194390628,
            0.16316592649881434,
            0.6664818704126,
            0.48373523732095913,
            0.9604058684829354,
            0.2692523672866294,
            0.9064712993729671,
            0.9749843431565036,
            0.8257295231523102,
            0.5450919843847578,
            0.44285319458843475,
            0.7186843265960554,
            0.6843538986222125,
            0.5770108070727499,
            0.1483563341866252,
            0.5585015160821443,
            0.03159869315859776,
            0.7381257467734162,
            0.6274473768451448,
            0.9267372329435319,
            0.37363099006440503,
            0.47446843813525197,
            0.032772479544093236,
            0.00379906646971917,
            0.42616877659431984,
            0.6742650531619625,
            0.9648053952942566,
            0.25717093456208484,
            0.6552045037647151,
            0.8239025264272717,
            0.02591609051696553,
            0.1584841211968424,
            0.16818874642525816,
            0.5294395390101027,
            0.1642264759590285,
            0.4126302404299753,
            0.08988362194895594,
            0.4360622948309868,
            0.32436882717641535,
            0.4418212764871775,
            0.4604755504888851,
            0.9575949410941967,
            0.7175436604175588,
            0.29733768335110844,
            0.5293559499976667,
            0.4894719200579313,
            0.36939247060990377,
            0.1374347603269518,
            0.4693336639964696,
            0.5689248255490967,
            0.1925430841115714,
            0.4859926698559668,
            0.8005623663500411,
            0.18377993510143098,
            0.5730598356703445,
            0.2466855650149451,
            0.3778086183932944,
            0.8534163797973425,
            0.3958760269463045,
            0.46918329030046146,
            0.5961718951885366,
            0.5465884435695455,
            0.1339872295984894,
            0.9518564736255202,
            0.7290358177713954,
            0.2667982590176554,
            0.20064816906965677,
            0.9446972449968342,
            0.7832565372283562,
            0.4405421873350379,
            0.530088946997536,
            0.4725951296759483,
            0.0014663069930448414,
            0.19875745639800735,
            0.271404502879651,
            0.87258408818107,
            0.8344271403507775,
            0.84651528914761,
            0.9086951930453072,
            0.4873133597394722,
            0.05325920734817391,
            0.3828635353633184,
            0.05557465353817326,
            0.3814625184907462,
            0.4377099617929453,
            0.359395411391342,
            0.9886628059125666,
            0.17836199487456805,
            0.2924825293118364,
            0.858037965799582,
            0.21500863080861088,
            0.7264625195993915,
            0.8779249670674611,
            0.2000216508377537,
            0.4744403113140292,
            0.6925783675074796,
            0.687828393934673,
            0.8811510107307556,
            0.5873693485344339,
            0.9303459418450954,
            0.21704663826709047,
            0.18697461691318107,
            0.6343883180747689,
            0.6871257614783475,
            0.07313853559569883,
            0.830409517034205,
            0.060234730841782214,
            0.6871971775554103,
            0.403541702084807,
            0.9498998068464556,
            0.15592557291498454,
            0.4474366789947387,
            0.11848948493959344,
            0.6183179532225159,
            0.44094171296357565,
            0.5652525720091016,
            0.4446466973485682,
            0.3214135275545623,
            0.6010916819162343,
            0.19370814650434287,
            0.544230928868775,
            0.6118615221676226,
            0.8709265224511328,
            0.8950870194895194,
            0.2666189124135384,
            0.8820390336892018,
            0.647566638297354,
            0.2930025106494676,
            0.9484960225387983,
            0.708510551160303,
            0.22259188191974133,
            0.2270012493372623,
            0.8037648950839286,
            0.21209306011852014,
            0.13449963174741786,
            0.47387981509402244,
            0.3326571690695004,
            0.6691445517280875,
            0.3635737148819327,
            0.8098649333350489,
            0.22432743987894221,
            0.7891010532901155,
            0.9563360125823266,
            0.6989854294930989,
            0.02451751771421673,
            0.5153057236378842,
            0.7229025058224616,
            0.5081865221113376,
            0.16971912562831382,
            0.026240211120949763,
            0.6617582805876826,
            0.48737130022690356,
            0.026103944368583987,
            0.44381744169400816,
            0.7045794291220656,
            0.3652351288035923,
            0.16424098114074281,
            0.028751483462400862,
            0.8234836956782872,
            0.38853757267105626,
            0.6052144940177917,
            0.4897247651936114,
            0.3255237344699222,
            0.5513622088246961,
            0.13619547466300863,
            0.805436638429507,
            0.1845243360457135,
            0.8558996888545993,
            0.1557897878083101,
            0.08355102809063064,
            0.8458651963255258,
            0.432182232913888,
            0.9895611219947696,
            0.47153443081989654,
            0.8869894093118946,
            0.15559614616337059,
            0.9567507674202934,
            0.43519693875717824,
            0.06255791293620105,
            0.8513013588071069,
            0.14104499152728056,
            0.06623265234705933,
            0.858336978274352,
            0.17023080453003303,
            0.985181529598539,
            0.9367912077465413,
            0.6850447921424999,
            0.569567925927151,
            0.3574863857003564,
            0.3929414369619496,
            0.7798264977989812,
            0.11140464613840195,
            0.030617196786866252,
            0.8864366716402711,
            0.5785898432176089,
            0.643081257353523,
            0.9284374847655709,
            0.7001706302177434,
            0.6083818989887125,
            0.040026996801102666,
            0.8024670482256436,
            0.5944368875936563,
            0.6463789437698947,
            0.3066367398015226,
            0.3506669019695924,
            0.63431885164807,
            0.592319454812138,
            0.15094573781557963,
            0.340436339744575,
            0.804882873472292,
            0.44914911469387553,
            0.048470473475964004,
            0.3028203515471283,
            0.8865808880058462,
            0.016996466120910436,
            0.3588247157827066,
            0.44381366627273955,
            0.6133130243948027,
            0.6793732857893435,
            0.5937880447464559,
            0.9236333717131933,
            0.10620265902358594,
            0.6781272277941961,
            0.46206077738190166,
            0.5837411523988725,
            0.6838737935191735,
            0.8888571729522181,
            0.1117486127397801,
            0.5271394727235267,
            0.841582357601115,
            0.055683160653552055,
            0.7378009917988021,
            0.1345067804099308,
            0.023603067965801028,
            0.7956056827623649,
            0.3861304971070073,
            0.6100668109183262,
            0.6402060812881678,
            0.3340736360358295,
            0.512650267083197,
            0.6943621535616193,
            0.9673799816609256,
            0.8107827181347624,
            0.49841780499148625,
            0.7518770184506263,
            0.5565745335500417,
            0.9808723356822058,
            0.8068635360397197,
            0.908681262855838,
            0.9152605982597944,
            0.44866095133404205,
            0.9599851394933173,
            0.2934678845913977,
            0.7592374249514868,
            0.5291313480256107,
            0.5188340513175598,
            0.3463210956043563,
            0.16675722612875354,
            0.811258684871908,
            0.4144522597135534,
            0.3000106018374561,
            0.6486806688272189,
            0.7150432102084934,
            0.5626534553029507,
            0.8490672525306053,
            0.5975992482391111,
            0.4836329711391879,
            0.29236983269309746,
            0.9122909824387251,
            0.24101012619440676,
            0.28606135589771586,
            0.214284944484199,
            0.6987746413236384,
            0.8606187625525669,
            0.5680627754034978,
            0.5577448653156638,
            0.27292102034886667,
            0.4723605997197269,
            0.4008560087475681,
            0.023646571277147088,
            0.5516376105896065,
            0.19226168342556682,
            0.7485035347212252,
            0.47271107867679585,
            0.879161456599194,
            0.672627747030352,
            0.5166390976270142,
            0.6463033535971618,
            0.20387169812973893,
            0.0016747812894766234,
            0.029763819218643017,
            0.9021832815409186,
            0.18205733315974348,
            0.44566220678138957,
            0.5086824535884374,
            0.8622052399579576,
            0.8778492418306816,
            0.04147755982688972,
            0.0003300822774104928,
            0.2300233308570272,
            0.11638097773203415,
            0.8877912606239582,
            0.8223947699484698,
            0.9016308879256799,
            0.4471068691530159,
            0.0967523806018944,
            0.22988634800429852,
            0.30641596253986436,
            0.24355542901947214,
            0.13632875085756924,
            0.24647742569058617,
            0.5307455040750163,
            0.9863212880422773,
            0.918718097173517,
            0.9531872942401909,
            0.9846706539836511,
            0.010832008901674683,
            0.7874704045680089,
            0.42240650181801453,
            0.6122554366093659,
            0.7583106953085872,
            0.7931774645387552,
            0.03876987809799448,
            0.9824863334518148,
            0.8849963293165498,
            0.6846563583535574,
            0.17077550911827233,
            0.4472189707111268,
            0.3076696890971775,
            0.10452915885027414,
            0.8542979719186184,
            0.35567916294185664,
            0.41745038137207335,
            0.6803947523885654,
            0.4865145832835298,
            0.8890112117760177,
            0.8896159115424894,
            0.35791747143755714,
            0.7103716929172195,
            0.33545214090580033,
            0.7853990720028072,
            0.6047114987242408,
            0.6940787385341096,
            0.7999811177869283,
            0.9545612318219266,
            0.7691751897986325,
            0.3534218003313171,
            0.7164867468747648,
            0.6603465707298116,
            0.8149035048784745,
            0.4181618221513588,
            0.7210019025474951,
            0.15630788288921538,
            0.6132290075987188,
            0.5517610367620278,
            0.6277113065039732,
            0.39238074042319493,
            0.28377003769183407,
            0.5417866411605511,
            0.4566967099927354,
            0.5564426470099018,
            0.060868401764593405,
            0.6094880119699536,
            0.33370549844498154,
            0.14478677661254435,
            0.47878414441966277,
            0.9676546051816021,
            0.9881664521925151,
            0.41477979611479965,
            0.7831259314522806,
            0.5162035582913124,
            0.8740989440663438,
            0.1210324686662908,
            0.8286384476864525,
            0.5506716069892422,
            0.32863744747641355,
            0.07781119225183342,
            0.9444840881154352,
            0.30654969285100087,
            0.9839963670150201,
            0.198995514233535,
            0.2424161430409465,
            0.2640876580224849,
            0.13899252436716902,
            0.5320212354787701,
            0.13847175480439033,
            0.48587889178834587,
            0.29283891441097065,
            0.8263256175727217,
            0.11797632499159805,
            0.1443416658543194,
            0.3366947202678495,
            0.6603118455550755,
            0.32027750485155937,
            0.8465086212476037,
            0.20340544885996203,
            0.7926302215934362,
            0.2173652194945127,
            0.061374187687189496,
            0.1245294114075205,
            0.6911079151785875,
            0.36100282685823837,
            0.027576518401000838,
            0.31007214948205597,
            0.5956472548709824,
            0.0044205399575183435,
            0.29886063705375854,
            0.282423820051234,
            0.7321569261537294,
            0.3331502022181124,
            0.5543204711831691,
            0.8864507895811109,
            0.22522697958493698,
            0.051068413606819796,
            0.49820708734739494,
            0.28789124913862363,
            0.3921078283891909,
            0.8361533457723219,
            0.8116903276671275,
            0.4734334100328207,
            0.5059761491293135,
            0.5869857748739118,
            0.45295550704035903,
            0.3451473426273679,
            0.023617158864443488,
            0.5929015022396065,
            0.9304971171549896,
            0.1116490388957384,
            0.839607069069392,
            0.1475612833358515,
            0.3737390146619419,
            0.5277748973360896,
            0.4729791060341131,
            0.41423844610339855,
            0.1497186509723214,
            0.185069743330749,
            0.2896006671075433,
            0.23278840390731026,
            0.7485279424409984,
            0.2997849445534869,
            0.8647359103770899,
            0.49903346741276366,
            0.8348563902271234,
            0.1276644908715967,
            0.7367268375405149,
            0.45047362044769734,
            0.21333824304464144,
            0.7565022526983096,
            0.26033032435280856,
            0.41280765538307396,
            0.9228548390643795,
            0.5317233491114747,
            0.43788136452804827,
            0.2021074308947174,
            0.37643647889997844,
            0.5267542097405596,
            0.47938912629766406,
            0.6740194794509481,
            0.40081010192193645,
            0.9467340520851908,
            0.9943528870146244,
            0.02899210686599385,
            0.437152961618406,
            0.8113106881671761,
            0.20138449221768684,
            0.19906319473670275,
            0.8669149132765681,
            0.8257651884567754,
            0.7217443875783902,
            0.4853989704912177,
            0.14265772574321023,
            0.3114446963372245,
            0.6615457478785746,
            0.40723657815724,
            0.36869508529588135,
            0.06301641746332409,
            0.9311605327509352,
            0.744853501022084,
            0.22129519196477387,
            0.5935016131468461,
            0.8640573358547035,
            0.14708304719541654,
            0.16776139755404318,
            0.4437816365500199,
            0.7650892905039962,
            0.7687509868619725,
            0.5728038805423521,
            0.7968836078421103,
            0.5971615964003105,
            0.2556522728720847,
            0.37212254820872304,
            0.45580878740895203,
            0.824462777838793,
            0.4163502393144646,
            0.9577862104707457,
            0.1362484194994512,
            0.232791236677228,
            0.7785535014656395,
            0.5250141255249315,
            0.07796582605476632,
            0.8180435687781734,
            0.04904493687852163,
            0.09975917105814036,
            0.19042305196179277,
            0.4681031570408515,
            0.4185813964290853,
            0.7563440800937025,
            0.6938130124882698,
            0.0009878846110014106,
            0.5598787194431837,
            0.1188486188064396,
            0.6618904140253677,
            0.44247380503712164,
            0.575894868467275,
            0.018541030366061584,
            0.8931733797688469,
            0.4823910525159628,
            0.16914099091747758,
            0.6902171222226382,
            0.7497785583990529,
            0.5660572587171601,
            0.7237032190678135,
            0.8884497001412884,
            0.401976818736702,
            0.061561328160198325,
            0.17345963950113263,
            0.21741074761777046,
            0.664765026825943,
            0.0003142523234960226,
            0.49790557550355774,
            0.7972941049881628,
            0.4354805939118921,
            0.6289399489190155,
        ];

        let n = diag.len();
        let mut u = Mat::from_fn(n + 1, n + 1, |_, _| f64::NAN);
        let mut v = Mat::from_fn(n, n, |_, _| f64::NAN);
        let s = {
            let mut diag = diag.clone();
            let mut subdiag = subdiag.clone();
            compute_bidiag_real_svd(
                &mut diag,
                &mut subdiag,
                Some(u.as_mut()),
                Some(v.as_mut()),
                40,
                0,
                f64::EPSILON,
                f64::MIN_POSITIVE,
                Parallelism::None,
                make_stack!(bidiag_real_svd_req::<f64>(
                    n,
                    40,
                    true,
                    true,
                    Parallelism::None
                )),
            );
            Mat::from_fn(n + 1, n, |i, j| if i == j { diag[i] } else { 0.0 })
        };

        let reconstructed = &u * &s * v.transpose();
        for j in 0..n {
            for i in 0..n + 1 {
                let target = if i == j {
                    diag[j]
                } else if i == j + 1 {
                    subdiag[j]
                } else {
                    0.0
                };

                assert_approx_eq!(reconstructed.read(i, j), target, 1e-10);
            }
        }
    }

    #[test]
    fn test_svd_josef() {
        if option_env!("CI") == Some("true") {
            // too big for CI
            return;
        }

        let diag = [
            -4982.29,
            -115933.375,
            -39290.402,
            -53520.258,
            -36040.266,
            -52991.027,
            -38354.395,
            -30938.463,
            -36932.613,
            37908.934,
            -31735.686,
            -29773.53,
            29128.258,
            31024.092,
            27043.496,
            -26642.004,
            -22770.88,
            -24615.9,
            -24646.432,
            -23344.803,
            -22306.146,
            -28313.412,
            -21041.406,
            -17843.906,
            25445.979,
            20521.588,
            -21393.717,
            -15734.208,
            14880.282,
            16752.787,
            -20087.693,
            -21010.648,
            -15918.599,
            16010.183,
            -14591.378,
            14626.2295,
            -14125.587,
            -11615.434,
            -14451.93,
            -14029.534,
            -12767.689,
            13280.041,
            -11597.155,
            -14924.39,
            11490.427,
            -13197.846,
            12606.662,
            -12919.53,
            -12347.471,
            -13494.115,
            -10735.693,
            -10490.857,
            10106.977,
            -10709.639,
            11113.807,
            12343.91,
            -11163.296,
            -10034.844,
            9765.604,
            10918.809,
            9634.743,
            10636.991,
            -10069.745,
            9689.102,
            -9762.144,
            10608.88,
            9972.194,
            9481.316,
            9865.239,
            9927.2295,
            8516.115,
            8664.686,
            -10337.319,
            9192.186,
            9132.25,
            -9149.819,
            -8773.606,
            -9152.119,
            9922.931,
            -8277.607,
            9442.621,
            9155.135,
            -8595.654,
            -8202.36,
            -8671.575,
            -7491.8667,
            7864.3877,
            8508.76,
            7739.8296,
            -8434.24,
            -7402.3574,
            7942.3765,
            7715.6265,
            -7921.2646,
            -7477.97,
            7341.47,
            7194.351,
            -7990.051,
            7017.965,
            -6690.6177,
            6796.9893,
            7272.8877,
            -7799.5747,
            6942.703,
            -6850.0103,
            6868.014,
            -7249.935,
            -7188.4233,
            7201.181,
            6573.888,
            6416.4756,
            6486.113,
            -7208.8447,
            -6969.6523,
            6449.074,
            6484.493,
            -5915.7197,
            6796.77,
            -6897.931,
            6361.6387,
            6621.5605,
            6615.3833,
            -6307.6597,
            6149.8765,
            6478.978,
            -5931.9146,
            -6124.658,
            6460.2114,
            6022.815,
            5926.3867,
            -6111.6206,
            6433.607,
            -5722.0503,
            5761.825,
            -6182.5923,
            5869.465,
            -6260.1387,
            -5653.4727,
            5423.5923,
            -5897.6274,
            5304.13,
            6012.202,
            5470.3345,
            5700.818,
            -5660.3174,
            -5455.411,
            5610.549,
            5906.4536,
            5579.6704,
            5572.493,
            5725.8047,
            5681.753,
            -5642.6704,
            -4988.65,
            -4954.8135,
            5814.944,
            5512.877,
            5402.008,
            4855.9854,
            4949.4185,
            4635.087,
            5033.229,
            -5388.7305,
            -5202.3257,
            -5170.0,
            5015.769,
            5213.643,
            -5086.537,
            5065.4917,
            -4940.5835,
            5231.5156,
            4711.358,
            -4401.794,
            4999.302,
            5218.9434,
            5172.243,
            -5185.432,
            -4836.812,
            4736.684,
            -4872.7637,
            4633.3525,
            -5288.4663,
            -4677.9067,
            -4820.9033,
            4794.8374,
            4476.8047,
            -4492.192,
            -4820.052,
            -4993.5684,
            -4733.087,
            4228.255,
            -4792.1235,
            4617.104,
            4630.1646,
            -4692.7725,
            4679.7793,
            -4397.909,
            -4641.8184,
            -4523.9727,
            3998.4229,
            4432.242,
            4135.061,
            -4513.782,
            4299.5366,
            -4811.2036,
            4185.509,
            -4392.3916,
            -4819.5273,
            -4236.3315,
            -4433.4414,
            -4233.224,
            -4087.5273,
            3932.2993,
            -3951.3496,
            4183.8345,
            4169.011,
            4389.5957,
            4089.5005,
            4139.6587,
            -3923.8982,
            4055.6272,
            4314.8228,
            -4271.077,
            3957.9656,
            4427.326,
            -3919.874,
            -3845.3503,
            -4218.895,
            4158.164,
            4137.1973,
            3703.1816,
            3989.9053,
            3843.97,
            3669.0989,
            3870.9905,
            3899.541,
            3872.0227,
            -3774.3938,
            -4024.2832,
            3707.6277,
            -3854.9895,
            -3820.6206,
            3729.776,
            -3753.0942,
            4067.5044,
            3970.0217,
            3708.129,
            3750.0105,
            3887.169,
            -3992.7395,
            3752.418,
            3830.3691,
            3693.4216,
            -3594.8606,
            -3746.2397,
            3856.282,
            -3805.605,
            3883.8767,
            3498.372,
            -3802.2925,
            -3826.8152,
            3525.9395,
            3625.7004,
            -3617.321,
            -3862.9395,
            3339.9507,
            3678.5796,
            3515.13,
            3668.3691,
            3411.4905,
            3644.8364,
            3576.9736,
            -3607.6377,
            3395.716,
            -3555.1956,
            3402.8337,
            -3283.8967,
            -3428.6,
            -3597.503,
            -3619.1682,
            3356.2195,
            -3393.647,
            3431.7737,
            -3424.581,
            3066.751,
            3329.8325,
            -3267.4973,
            3459.7947,
            -3215.3904,
            -3332.2493,
            3586.8923,
            3414.78,
            3286.9368,
            3138.4446,
            3281.3345,
            -3200.3909,
            -3071.909,
            3389.9585,
            3236.008,
            3099.273,
            -3358.9856,
            3187.115,
            3221.6233,
            2974.988,
            -3166.289,
            -3105.8499,
            3329.7512,
            3206.7979,
            3190.7366,
            -3125.4917,
            -3168.5208,
            3248.1594,
            -3367.7104,
            -3043.5234,
            2946.306,
            -3266.0342,
            3062.026,
            3038.9165,
            -2973.2454,
            2935.5823,
            3104.637,
            2851.8098,
            -3192.922,
            -3007.9465,
            -3052.6748,
            2862.3652,
            2945.448,
            -3060.7654,
            2818.8655,
            -2793.471,
            -2752.508,
            2957.5132,
            2952.8528,
            2877.513,
            2862.35,
            2968.8633,
            -2980.1116,
            -2861.4932,
            2981.417,
            2911.4785,
            -3020.8245,
            -3052.7864,
            2736.9287,
            -2910.8767,
            3096.745,
            2970.5806,
            2773.7732,
            -2761.3765,
            2848.9063,
            2839.8674,
            -2687.5583,
            2866.1584,
            2755.2302,
            -2922.6636,
            2856.3538,
            -2787.0964,
            2997.8486,
            -2815.6057,
            2788.6094,
            2831.562,
            -2798.6238,
            -2742.732,
            2791.0784,
            -2701.462,
            -2654.2546,
            2818.6216,
            -2770.773,
            -2764.571,
            2654.076,
            -2568.2878,
            2754.331,
            -2607.611,
            2748.403,
            2620.354,
            2607.279,
            -2715.9624,
            2804.4758,
            2682.6628,
            2801.1475,
            -2792.3674,
            2659.9988,
            2559.9126,
            -2574.8054,
            2527.3965,
            2500.822,
            2573.422,
            -2655.8818,
            2482.9438,
            2637.9949,
            -2605.4722,
            2527.9106,
            2672.2786,
            2454.0564,
            2605.2224,
            2557.1516,
            -2491.6772,
            2648.446,
            2677.1328,
            -2534.0022,
            2490.6577,
            2621.6685,
            -2562.9365,
            -2503.2751,
            -2378.8179,
            -2583.1316,
            2467.8577,
            -2650.6985,
            -2502.8555,
            -2510.6296,
            2538.1626,
            -2548.566,
            -2609.3499,
            2483.5457,
            -2343.392,
            -2528.2317,
            -2402.8135,
            2407.9846,
            -2432.9636,
            2500.7588,
            -2463.4111,
            2613.5027,
            2621.8367,
            2535.5637,
            -2426.555,
            -2423.1538,
            2389.643,
            2491.0537,
            -2433.601,
            2243.0864,
            -2336.732,
            -2420.6333,
            2406.7131,
            2339.6067,
            -2341.1868,
            2320.0015,
            -2402.124,
            2349.8958,
            2291.1594,
            2431.9812,
            -2497.6008,
            -2358.7617,
            -2388.5554,
            2528.1172,
            -2363.5532,
            2421.4587,
            2362.6301,
            2324.895,
            2158.8047,
            2364.5686,
            2265.3398,
            2328.6145,
            2347.8743,
            2450.6047,
            -2275.4858,
            2249.0637,
            2275.181,
            2291.7605,
            2392.3835,
            2237.9243,
            2209.922,
            -2292.3228,
            2380.4546,
            -2303.7944,
            -2167.108,
            2217.6807,
            2341.2327,
            2274.9668,
            2212.3845,
            2162.894,
            2201.2314,
            -2259.2473,
            -2322.041,
            2161.0703,
            -2241.2964,
            2309.0347,
            2162.639,
            2133.585,
            2146.2742,
            -2219.4265,
            -2175.6277,
            -2131.5322,
            -2131.8984,
            -2210.1812,
            -2300.942,
            -2241.9749,
            -2243.779,
            2114.229,
            2172.203,
            2256.575,
            -2223.5083,
            2307.975,
            2266.5183,
            2132.6555,
            2138.8027,
            -2236.682,
            -2137.2383,
            -2167.881,
            -2120.135,
            -2173.858,
            2058.052,
            2242.881,
            -2143.3452,
            2047.1777,
            -2104.636,
            -2127.63,
            2085.836,
            2115.2517,
            2146.8203,
            2173.694,
            2092.4836,
            2121.7131,
            -2076.198,
            2045.5801,
            2068.5835,
            2089.7546,
            -2052.718,
            -2041.2389,
            2018.041,
            -1983.5045,
            -2191.864,
            2103.4622,
            -2034.5123,
            -2107.5308,
            2160.5508,
            2034.564,
            1976.581,
            -2071.0168,
            2018.0691,
            -2090.426,
            -2038.4648,
            -2031.5236,
            -2089.6526,
            1964.0378,
            -1981.9153,
            -2031.7898,
            1938.8636,
            -2051.538,
            2003.0314,
            2064.282,
            -1916.4147,
            1995.8485,
            1972.2367,
            -2031.3832,
            -2034.8833,
            1872.2113,
            -1960.8624,
            -1979.7122,
            2008.9146,
            1979.7423,
            1947.9535,
            -1995.3767,
            2062.5483,
            1917.1696,
            2018.1704,
            -1946.1768,
            1890.543,
            -1926.5361,
            2068.7007,
            1916.911,
            1962.296,
            -1903.1604,
            1913.7609,
            -1912.232,
            -1898.6573,
            1878.8776,
            1870.654,
            1940.3196,
            1907.1228,
            1890.8152,
            -1932.6548,
            1933.2946,
            -1929.7119,
            1940.2715,
            -1908.7405,
            1954.9514,
            2010.8771,
            1857.9669,
            -1973.3938,
            1938.807,
            -1913.4695,
            1974.0297,
            -1892.9646,
            1898.053,
            -2013.7845,
            1863.1199,
            1857.9475,
            -1849.4503,
            -1907.4922,
            -1859.8801,
            1900.619,
            1920.1183,
            1856.8063,
            1810.7302,
            -1931.161,
            -1851.9366,
            1857.5,
            1797.4321,
            1906.1486,
            1793.7565,
            -1862.9388,
            -1835.3907,
            -1830.0468,
            -1752.2075,
            -1827.1566,
            -1793.8264,
            1818.0043,
            1826.383,
            1760.4706,
            1917.9385,
            1811.2101,
            1831.4088,
            -1827.8057,
            1894.3322,
            -1876.4523,
            1844.2185,
            -1871.2894,
            1761.6517,
            -1879.7295,
            -1848.453,
            1862.8223,
            1889.4991,
            -1739.7723,
            1822.169,
            1745.58,
            1724.6978,
            1873.1661,
            -1794.4609,
            1815.8003,
            1821.6508,
            -1765.5266,
            1820.1548,
            1841.0243,
            1725.3827,
            1763.5803,
            1756.7384,
            1762.0444,
            -1754.6888,
            -1736.1929,
            1697.6329,
            -1728.9015,
            1779.6368,
            -1728.5938,
            -1769.4363,
            1703.7745,
            1615.0065,
            1749.0802,
            1746.4912,
            1739.0139,
            -1668.1017,
            1730.727,
            -1710.601,
            1760.3744,
            1680.6567,
            1781.5475,
            -1690.439,
            -1655.212,
            1822.7682,
            -1751.8729,
            -1737.8438,
            -1707.9905,
            -1750.3264,
            1691.3965,
            -1678.2664,
            -1787.3884,
            -1630.0286,
            1712.0481,
            1763.4688,
            1656.4443,
            1669.065,
            -1667.1467,
            1708.0416,
            -1629.9503,
            -1668.4508,
            1670.5121,
            1617.2648,
            -1726.7793,
            -1650.6445,
            -1700.4556,
            -1689.8602,
            1698.3073,
            -1703.7474,
            1656.323,
            1660.322,
            -1691.3893,
            -1730.1201,
            -1716.4463,
            1682.2352,
            -1664.4598,
            1710.6768,
            -1591.3188,
            1657.4857,
            1663.6377,
            1651.2402,
            -1592.0094,
            1580.4333,
            -1664.4246,
            1578.8595,
            1631.3612,
            -1616.4697,
            -1696.31,
            -1594.1962,
            1627.1486,
            1622.9122,
            -1612.2693,
            1672.3738,
            -1683.4764,
            -1610.0054,
            1625.6272,
            -1625.2607,
            1588.2515,
            1572.1575,
            1660.7205,
            -1651.3485,
            1639.0875,
            1561.5599,
            -1655.323,
            -1649.2291,
            1538.7792,
            -1562.8033,
            1656.6597,
            1497.1005,
            1644.681,
            1615.3314,
            1641.217,
            1534.7833,
            1585.4415,
            1640.4703,
            1683.7715,
            1546.2205,
            1522.8816,
            -1509.1516,
            1560.2062,
            1610.3776,
            1541.8741,
            1576.804,
            -1608.4382,
            -1561.9539,
            1568.9382,
            1588.5485,
            -1513.7794,
            1577.4507,
            1504.6926,
            -1572.3297,
            1596.336,
            -1519.7118,
            1557.4878,
            1514.5704,
            -1594.3251,
            -1547.2174,
            -1496.2367,
            -1479.3079,
            1531.1111,
            1530.6241,
            -1511.4178,
            1558.687,
            1485.0059,
            1609.6744,
            1519.7452,
            1548.6978,
            1495.6571,
            1535.4935,
            -1589.7273,
            -1546.1383,
            1500.192,
            -1471.6495,
            1509.3535,
            -1449.0466,
            -1536.0812,
            1522.651,
            1556.9921,
            1498.1724,
            1478.813,
            1469.9126,
            -1543.7834,
            1556.9205,
            1439.8646,
            1536.838,
            -1550.4856,
            1523.6517,
            1473.1705,
            1500.2905,
            -1464.1932,
            1444.8525,
            1479.0289,
            1455.4722,
            -1456.189,
            -1555.9121,
            1473.0304,
            -1469.841,
            -1539.162,
            1431.3096,
            1455.9326,
            -1463.8973,
            1492.6176,
            -1467.183,
            1456.9768,
            -1524.488,
            -1505.4487,
            -1498.333,
            1490.3318,
            -1365.2135,
            -1448.2969,
            1380.7773,
            1483.2645,
            1499.3159,
            -1437.96,
            -1477.7794,
            1416.6342,
            1415.1545,
            1504.4901,
            -1442.7402,
            -1505.5908,
            -1431.5118,
            1485.6733,
            -1449.2294,
            -1469.5148,
            -1489.3604,
            1436.8674,
            1453.6886,
            1435.667,
            1380.298,
            -1417.8859,
            -1406.8949,
            -1393.6196,
            1420.2272,
            1422.5854,
            -1411.3405,
            -1404.7875,
            1378.1626,
            1435.1458,
            1468.8517,
            1363.375,
            -1410.7588,
            1470.4802,
            -1391.4789,
            -1435.4001,
            1391.4194,
            -1394.4366,
            1421.0892,
            1364.1516,
            -1445.405,
            1432.5316,
            1381.2181,
            -1316.3938,
            1444.7582,
            1337.6666,
            1366.8964,
            -1410.7808,
            1403.4641,
            -1423.6947,
            1372.7281,
            -1389.6858,
            1426.1486,
            1363.8207,
            -1452.5917,
            -1414.3578,
            -1363.7573,
            1427.9613,
            1392.1497,
            1313.7887,
            -1387.6971,
            -1397.878,
            1382.5232,
            -1360.2229,
            -1386.6377,
            -1347.6138,
            -1352.9038,
            1345.1124,
            1359.0636,
            1358.4475,
            -1425.7649,
            1312.0239,
            1340.8685,
            -1336.7054,
            1336.2909,
            1327.131,
            1378.4436,
            1371.5453,
            1338.7761,
            -1352.8546,
            1403.0209,
            1284.8369,
            -1306.5784,
            1403.5597,
            -1318.1085,
            -1361.3492,
            -1280.0217,
            1331.5109,
            1402.0499,
            1327.1632,
            -1359.2346,
            1323.2813,
            1350.3905,
            -1335.3475,
            -1335.9572,
            1322.3903,
            1301.5869,
            1314.9618,
            1375.1686,
            -1331.748,
            1282.8546,
            1289.9165,
            1312.3364,
            1283.521,
            1336.8923,
            1321.8082,
            1307.7477,
            1349.2123,
            1303.9056,
            -1335.3492,
            1261.3651,
            -1312.9199,
            1269.7278,
            1321.4963,
            -1268.7156,
            -1294.016,
            1282.9825,
            1311.41,
            -1286.2539,
            1270.1586,
            -1294.9642,
            1311.0021,
            -1260.4895,
            1300.6953,
            1326.2305,
            1279.775,
            -1312.2467,
            -1309.275,
            1267.0886,
            -1303.3905,
            -1287.7974,
            1363.6444,
            1261.6587,
            1325.7776,
            -1244.5668,
            1291.7601,
            1213.6207,
            -1270.9507,
            -1311.3547,
            -1257.8173,
            -1274.2731,
            -1218.6162,
            1222.5334,
            -1268.4061,
            1247.6116,
            -1267.4213,
            -1256.2593,
            1291.809,
            1278.293,
            1201.5177,
            1226.443,
            -1224.234,
            1240.4287,
            1286.5829,
            -1252.789,
            1231.4563,
            1267.9943,
            -1224.7217,
            1205.8022,
            1223.1511,
            1267.244,
            1264.3717,
            -1241.661,
            1293.163,
            -1241.9058,
            1200.4344,
            -1213.6536,
            1260.5121,
            -1200.0321,
            -1199.8796,
            1237.9341,
            1213.7224,
            -1170.493,
            1264.0596,
            1212.0522,
            1217.3555,
            1182.2942,
            -1245.9658,
            1248.7472,
            1207.2621,
            -1232.749,
            1253.177,
            -1231.5399,
            1168.2719,
            1233.9006,
            1215.0771,
            1198.4092,
            1211.7274,
            1203.3046,
            -1207.8157,
            1223.8925,
            1200.599,
            -1208.4126,
            -1219.1199,
            1248.4581,
            -1188.7106,
            1241.1685,
            -1261.0712,
            1173.2284,
            1177.5314,
            -1184.9908,
            -1162.2257,
            -1176.1447,
            -1201.8629,
            1164.8274,
            1192.6191,
            -1180.2343,
            1151.2107,
            -1158.2153,
            1202.9843,
            -1168.6268,
            1195.9128,
            1213.6245,
            -1182.0051,
            1165.4607,
            1170.3297,
            -1170.6603,
            -1184.242,
            -1210.9557,
            1182.0109,
            -1151.3378,
            1189.8884,
            -1173.8962,
            1160.4601,
            1172.1937,
            1162.673,
            1205.9604,
            1176.9143,
            -1153.2698,
            1142.5554,
            -1159.6798,
            1146.2816,
            -1195.5404,
            1129.0797,
            -1129.7887,
            1160.0693,
            -1183.907,
            1166.3104,
            1154.0782,
            1114.4886,
            1135.5266,
            1162.8268,
            -1197.6361,
            1160.1652,
            1153.7654,
            1174.6622,
            1112.1143,
            1129.5171,
            1171.0951,
            1170.3245,
            -1144.171,
            1180.215,
            1107.8467,
            1134.7223,
            1104.9802,
            -1140.3823,
            1149.8721,
            1131.0728,
            -1223.5132,
            1128.9141,
            1164.54,
            1187.3468,
            1159.1761,
            1179.224,
            1139.8037,
            -1161.8778,
            1145.4666,
            -1147.3234,
            1160.6063,
            -1120.0162,
            1143.5673,
            1114.8187,
            -1115.3378,
            1149.6929,
            -1137.5808,
            -1093.2996,
            -1088.3673,
            1163.8451,
            1122.051,
            -1110.493,
            1084.2092,
            1124.6875,
            -1124.5219,
            1103.7158,
            -1126.0331,
            -1139.5757,
            -1133.1599,
            -1103.1823,
            1113.043,
            1106.2838,
            1124.4707,
            -1089.7559,
            -1146.1454,
            1093.8075,
            1105.7732,
            1126.8021,
            1122.819,
            1139.7338,
            1104.6168,
            -1134.5204,
            1131.806,
            1107.8413,
            1116.579,
            -1133.6616,
            -1095.2565,
            1103.6768,
            1102.0427,
            -1119.1167,
            1140.5283,
            -1075.1881,
            1103.9812,
            -1107.7772,
            1099.3848,
            -1104.273,
            1058.4182,
            1093.5425,
            1099.39,
            -1064.9406,
            -1087.431,
            1125.7838,
            -1114.3304,
            1073.8147,
            1113.1871,
            1016.1738,
            1060.6127,
            -1078.555,
            1073.3291,
            1055.0282,
            1050.7151,
            1062.0416,
            1129.2135,
            -1071.0865,
            1041.7448,
            1066.5249,
            1058.5392,
            -1071.212,
            1036.9541,
            1095.9231,
            1033.8276,
            -1059.7197,
            1103.3142,
            1054.6832,
            -1071.4319,
            1043.7651,
            1063.408,
            1011.9666,
            -1062.3025,
            -1065.067,
            -1071.1221,
            1063.0592,
            -1045.325,
            -1086.795,
            -1064.2013,
            -1061.4581,
            -1088.0676,
            -1101.4233,
            -1088.7559,
            1052.847,
            1074.1754,
            1041.1725,
            -1058.8965,
            -1058.4828,
            1053.2446,
            -1021.0283,
            -1070.3934,
            -1023.3195,
            1003.49115,
            1011.5898,
            1009.2378,
            1034.9537,
            -1050.922,
            1076.6113,
            1057.6823,
            -1005.30884,
            1015.2034,
            1058.884,
            -1058.3402,
            1042.4122,
            -1029.6365,
            -1059.5164,
            1031.0997,
            1029.1316,
            -1032.2573,
            -996.77026,
            1014.42126,
            -1036.3954,
            994.0957,
            1013.77,
            -1027.8269,
            -1024.8738,
            1016.3425,
            -996.384,
            1056.4716,
            1019.41425,
            -1015.35046,
            1021.2221,
            -1009.817,
            -1034.8976,
            -1024.011,
            1003.49805,
            1001.4241,
            1021.434,
            -1021.45905,
            997.44885,
            1028.6897,
            999.2341,
            1029.5223,
            1030.1216,
            -1018.7305,
            -1029.5063,
            -1022.9794,
            994.8808,
            -986.5294,
            1034.0322,
            -991.7831,
            978.8939,
            -982.2273,
            -1038.6918,
            1010.2375,
            1032.1693,
            982.6825,
            988.1789,
            1029.6979,
            994.0022,
            -1016.76843,
            988.115,
            967.735,
            999.7673,
            1001.6252,
            -1000.1654,
            -987.53876,
            1020.28827,
            -981.7331,
            997.9227,
            964.7878,
            926.4605,
            972.47375,
            -987.60175,
            1000.7434,
            989.02655,
            974.8689,
            1007.36786,
            985.61646,
            987.3848,
            -985.312,
            -964.71344,
            988.40173,
            1007.4817,
            984.0884,
            -974.55756,
            -993.4585,
            1016.66565,
            975.9804,
            987.3775,
            974.7136,
            -1014.45044,
            947.5716,
            964.4684,
            1001.3405,
            979.436,
            954.1256,
            995.7891,
            1009.2186,
            -1033.1207,
            1029.8702,
            983.143,
            968.5868,
            951.3616,
            986.4625,
            969.32196,
            979.1823,
            -935.768,
            936.63885,
            -979.1223,
            -986.16266,
            -953.2328,
            974.3634,
            -955.5123,
            956.43555,
            -959.29407,
            952.2002,
            981.3308,
            959.96313,
            -970.942,
            939.03845,
            -964.806,
            -946.91736,
            902.2845,
            -915.5202,
            959.55585,
            -908.0847,
            -956.0922,
            964.94183,
            980.71826,
            972.05554,
            -954.78394,
            941.09283,
            -957.6694,
            -954.9365,
            958.1181,
            929.7345,
            929.3982,
            947.631,
            926.4502,
            -918.9294,
            917.36,
            -947.3056,
            905.1351,
            -933.76965,
            -930.2001,
            926.2088,
            929.4951,
            978.62585,
            921.61816,
            -954.9591,
            -958.7937,
            -952.9751,
            938.65436,
            958.6091,
            958.433,
            951.85925,
            950.6212,
            920.13306,
            939.01483,
            896.72943,
            935.96814,
            929.1608,
            -907.4038,
            931.12115,
            915.7568,
            947.3782,
            946.50055,
            914.215,
            930.00854,
            953.78894,
            922.3176,
            924.61017,
            906.073,
            -911.9253,
            -914.0624,
            -910.30756,
            908.1603,
            898.15717,
            909.69324,
            929.81464,
            -907.7089,
            912.5414,
            -912.48486,
            924.64307,
            924.8282,
            -876.87366,
            947.44073,
            901.5356,
            -910.0101,
            928.4643,
            936.1942,
            -922.14716,
            928.68896,
            948.36334,
            906.6905,
            907.609,
            912.3286,
            -905.695,
            919.2551,
            927.9573,
            890.3647,
            -887.3862,
            -919.1535,
            911.66876,
            925.4395,
            900.64197,
            951.39185,
            911.78986,
            896.6964,
            904.62946,
            -901.32007,
            895.7077,
            -878.715,
            911.5193,
            -904.2312,
            877.92664,
            894.2145,
            -865.0423,
            874.6873,
            871.49927,
            898.005,
            -895.23206,
            -888.7015,
            874.3757,
            864.24713,
            914.7581,
            897.4157,
            -849.9275,
            874.47186,
            -869.4902,
            906.44385,
            894.79956,
            -881.2993,
            881.3224,
            891.48016,
            871.37823,
            -865.2369,
            884.4974,
            864.3665,
            904.6083,
            860.24664,
            -896.98126,
            883.4674,
            -888.73553,
            910.001,
            882.97064,
            -896.2621,
            876.6311,
            -889.30023,
            -870.7576,
            878.0517,
            -859.4822,
            870.05365,
            -870.8136,
            876.543,
            897.6649,
            -888.2287,
            866.59845,
            -880.23376,
            853.7795,
            892.2834,
            -874.9878,
            -893.5777,
            872.2791,
            -859.09344,
            -822.73285,
            854.2491,
            858.31793,
            -861.9297,
            875.30743,
            871.5726,
            848.133,
            -880.56995,
            852.624,
            846.7282,
            -889.2334,
            -863.4106,
            -854.0507,
            -828.9909,
            860.42114,
            -868.53845,
            -835.0768,
            841.2282,
            855.8861,
            -834.2339,
            843.8242,
            -827.86096,
            844.6258,
            819.3894,
            -852.71014,
            -821.1786,
            -854.4305,
            837.2882,
            -867.2492,
            823.02625,
            -844.60376,
            -837.7456,
            835.90137,
            844.5527,
            839.75977,
            831.7465,
            -827.36914,
            874.4309,
            862.603,
            -850.70355,
            824.8988,
            858.70056,
            857.44336,
            -829.4387,
            842.4671,
            823.2933,
            -826.8137,
            840.5227,
            -821.0425,
            834.63635,
            829.0877,
            843.05225,
            -866.3062,
            810.5424,
            822.2945,
            855.599,
            833.7888,
            841.0026,
            849.5536,
            -850.3599,
            850.1179,
            831.88116,
            -860.95056,
            821.9851,
            834.9707,
            816.3515,
            796.6011,
            828.87714,
            824.42804,
            822.4254,
            -816.6005,
            831.63556,
            823.99445,
            850.3603,
            -820.9547,
            838.0128,
            817.17834,
            845.77356,
            -812.98596,
            -808.0925,
            805.61584,
            -813.4,
            832.86786,
            -831.0566,
            -812.1587,
            828.90906,
            808.8933,
            -812.90173,
            817.08813,
            -853.18,
            812.6086,
            -833.1607,
            803.9357,
            792.80505,
            795.55,
            -816.072,
            -824.3615,
            822.1304,
            817.67926,
            -792.4524,
            -827.36206,
            791.59814,
            -784.3597,
            -812.4529,
            811.60425,
            -813.99054,
            -775.6505,
            801.83026,
            -811.51654,
            816.1494,
            795.7827,
            -796.5413,
            -810.04114,
            -809.0715,
            -794.9666,
            791.7489,
            -774.4529,
            794.2517,
            777.4296,
            792.137,
            791.9451,
            -798.7749,
            790.9214,
            806.61444,
            798.2286,
            -824.6958,
            -783.90125,
            812.7129,
            808.04987,
            793.7728,
            -819.5705,
            793.0612,
            -807.97797,
            789.097,
            -794.508,
            803.88446,
            -773.22736,
            799.5784,
            770.19006,
            -774.14545,
            799.84064,
            763.8858,
            805.9413,
            -788.2339,
            788.9946,
            777.5544,
            790.9466,
            795.92566,
            -798.908,
            766.284,
            795.622,
            -752.11206,
            759.0477,
            -777.32794,
            791.1953,
            782.1136,
            767.143,
            -750.07684,
            -778.7354,
            780.28735,
            765.9835,
            791.68726,
            -770.8567,
            775.29034,
            773.84064,
            -780.95496,
            790.6589,
            791.9552,
            794.8381,
            -783.62555,
            -766.7159,
            796.10864,
            787.1302,
            784.2013,
            -772.2612,
            -783.32367,
            755.3695,
            781.8367,
            -759.7795,
            -772.02966,
            760.8059,
            -758.7621,
            776.18005,
            759.543,
            765.50287,
            769.752,
            -799.0612,
            -770.4527,
            786.0043,
            -773.66205,
            -774.9783,
            -745.1433,
            759.0883,
            -734.2928,
            748.08624,
            -737.8245,
            763.1586,
            -759.07,
            741.29584,
            -780.35077,
            -757.18317,
            784.23047,
            -763.73816,
            -760.96625,
            737.5987,
            -748.0954,
            737.3591,
            753.3679,
            757.6455,
            741.3308,
            756.4426,
            740.6306,
            -735.8646,
            -745.3518,
            738.5736,
            -739.81573,
            729.3255,
            -745.70654,
            -735.663,
            734.7278,
            728.4408,
            -738.3423,
            727.8389,
            749.3693,
            736.541,
            -732.5302,
            -742.85254,
            -763.21686,
            -746.76996,
            740.8123,
            -773.71295,
            -746.6129,
            -727.063,
            744.0916,
            -750.30347,
            -776.34595,
            -751.569,
            755.106,
            724.6662,
            753.32697,
            -768.065,
            736.592,
            730.17584,
            -753.8692,
            740.25305,
            773.30963,
            -719.7009,
            -724.8938,
            742.196,
            718.0579,
            733.71515,
            -736.31903,
            738.85394,
            757.3292,
            -737.596,
            -746.87854,
            -750.6601,
            -763.2901,
            747.05493,
            -735.5201,
            740.6335,
            -722.9602,
            -723.74994,
            -727.071,
            711.05927,
            729.54767,
            729.1449,
            -726.3928,
            727.65533,
            731.94836,
            736.6952,
            -729.4278,
            725.7129,
            -754.801,
            -724.05585,
            736.93536,
            -746.5711,
            738.24506,
            -737.30444,
            -739.60596,
            734.6775,
            -728.565,
            755.6288,
            -716.65845,
            -739.4514,
            -733.0952,
            -730.32965,
            722.7779,
            730.7372,
            740.0249,
            736.293,
            -739.2616,
            720.8563,
            739.2848,
            -747.69196,
            -722.95374,
            731.2344,
            -722.7598,
            714.5396,
            -716.15656,
            -721.78735,
            720.4919,
            684.9461,
            721.87146,
            -708.72253,
            719.39465,
            -703.43005,
            -719.24475,
            726.7563,
            710.2108,
            -704.3346,
            -704.8415,
            726.6117,
            719.6271,
            724.52814,
            721.60236,
            713.40765,
            719.751,
            703.0673,
            -705.14307,
            709.1342,
            -717.0224,
            -695.1586,
            703.1299,
            691.4213,
            -692.98834,
            -704.59174,
            712.5976,
            -720.1132,
            -731.903,
            709.70215,
            710.0475,
            -705.5099,
            702.5725,
            712.45374,
            -704.06714,
            706.447,
            -710.0871,
            702.5749,
            704.4752,
            -702.8309,
            -714.54987,
            714.0533,
            704.47125,
            693.7522,
            727.9982,
            -696.2999,
            -710.0458,
            717.34265,
            715.8181,
            718.2161,
            -698.12885,
            -706.1894,
            -687.5529,
            -698.292,
            -695.93646,
            693.258,
            682.4606,
            670.70276,
            -667.7726,
            -705.0336,
            704.3343,
            -688.2308,
            -709.4855,
            -691.61896,
            -682.62964,
            681.3668,
            662.80225,
            676.4619,
            -686.37836,
            686.7336,
            692.4645,
            -680.9767,
            -686.4302,
            -702.8287,
            -681.07764,
            689.6664,
            -674.16925,
            680.6819,
            700.0378,
            667.3404,
            -694.4492,
            699.5906,
            675.9194,
            -654.27814,
            -677.43195,
            681.1234,
            698.17694,
            -664.89386,
            680.79443,
            671.6335,
            677.1613,
            678.2434,
            672.7917,
            -673.8952,
            677.1117,
            664.26495,
            -664.3235,
            -670.8351,
            -657.4595,
            671.76355,
            -677.24585,
            675.4446,
            687.25055,
            -685.33044,
            685.9508,
            682.74316,
            663.2503,
            694.97925,
            -683.597,
            674.2843,
            669.4745,
            -662.56104,
            -693.7514,
            669.5715,
            673.91595,
            671.9745,
            660.1266,
            -668.5582,
            -661.85504,
            659.1965,
            668.1986,
            -673.77264,
            658.6337,
            -681.5257,
            684.7526,
            678.51697,
            675.5054,
            -682.01013,
            664.33124,
            668.39374,
            -658.35974,
            -655.0356,
            -652.9233,
            670.0154,
            643.5794,
            661.30334,
            672.4994,
            -671.6595,
            -681.0325,
            663.12714,
            659.6961,
            659.1302,
            647.66425,
            670.14996,
            634.9876,
            682.4534,
            -665.2668,
            -656.7541,
            670.82434,
            680.3447,
            679.6622,
            668.35156,
            674.0805,
            -669.70044,
            651.75226,
            654.74615,
            685.94586,
            -669.4875,
            673.8887,
            -668.2817,
            670.1186,
            -649.78534,
            662.43036,
            668.85156,
            649.0008,
            663.62823,
            660.35706,
            648.8952,
            664.615,
            650.1698,
            655.81055,
            665.49835,
            -656.576,
            645.00806,
            665.59143,
            653.7198,
            -653.3834,
            652.3477,
            667.2516,
            -645.73425,
            654.97894,
            666.85583,
            679.41754,
            670.5785,
            658.6172,
            657.0902,
            -669.8598,
            -679.52527,
            -689.9063,
            677.1002,
            -668.1506,
            643.5501,
            644.65344,
            -646.2723,
            651.3752,
            -631.7877,
            -623.7132,
            642.92487,
            -640.006,
            646.7329,
            630.74945,
            643.13904,
            -636.93097,
            627.9149,
            631.6226,
            638.84845,
            633.30237,
            628.0475,
            640.92706,
            -643.0402,
            660.1328,
            -661.8893,
            672.6191,
            -643.1702,
            -663.6902,
            -640.35223,
            -639.0689,
            -637.18463,
            642.52045,
            -635.20746,
            653.34235,
            653.451,
            649.34827,
            633.0321,
            -655.3416,
            -654.9674,
            -632.92755,
            -635.15686,
            637.8856,
            -659.10516,
            612.5916,
            645.8211,
            621.2943,
            -622.34814,
            -617.7949,
            637.0155,
            -632.36,
            636.5648,
            630.5866,
            620.83716,
            621.25525,
            -621.47644,
            631.47614,
            -644.7851,
            -621.84576,
            622.7476,
            645.9789,
            635.84247,
            631.8987,
            -609.7994,
            -647.111,
            633.3932,
            630.6494,
            617.5917,
            616.57275,
            -640.4858,
            627.04456,
            -627.5338,
            -620.31586,
            626.98267,
            -652.2337,
            628.1045,
            -620.0135,
            617.57477,
            617.5672,
            621.2655,
            632.2673,
            -605.55414,
            604.37213,
            -617.8233,
            611.30115,
            622.3836,
            627.12823,
            611.3365,
            -625.23206,
            -622.5152,
            -623.3692,
            617.60565,
            599.7538,
            634.671,
            613.63684,
            602.15796,
            634.3193,
            603.92004,
            -608.0905,
            604.7012,
            627.63434,
            604.3753,
            622.3324,
            620.43445,
            614.2677,
            626.8562,
            -631.64984,
            616.8202,
            -618.447,
            -607.6326,
            637.558,
            609.76495,
            -627.4152,
            610.17334,
            616.2518,
            -625.4015,
            -610.6476,
            619.3435,
            616.5438,
            610.0732,
            617.3086,
            -615.03,
            -612.5465,
            620.6874,
            614.3082,
            616.20245,
            612.80707,
            608.2581,
            -594.94324,
            -598.5512,
            -613.064,
            588.7167,
            -583.2333,
            594.98346,
            -598.6126,
            600.21936,
            615.1078,
            -605.5146,
            604.44006,
            -619.3138,
            -608.0832,
            615.56274,
            611.8851,
            605.2796,
            618.36304,
            617.8746,
            622.99896,
            -601.7944,
            -613.10474,
            -606.0243,
            603.04614,
            -590.1663,
            593.309,
            -601.8161,
            601.2445,
            604.45667,
            617.0114,
            603.7205,
            -606.1635,
            -593.0761,
            603.38007,
            590.19037,
            -591.82996,
            592.6665,
            603.7015,
            580.368,
            594.2736,
            583.7533,
            -618.0813,
            597.0796,
            602.6566,
            -591.78705,
            -591.7737,
            587.27496,
            -606.8604,
            588.8247,
            -580.8987,
            570.89996,
            580.42804,
            -581.4476,
            -607.359,
            599.74646,
            -604.0373,
            -595.54535,
            590.8665,
            -589.6623,
            -588.81134,
            595.47034,
            589.4277,
            623.0345,
            588.9494,
            599.3737,
            -607.00323,
            588.283,
            -599.47797,
            584.9488,
            607.76465,
            607.2355,
            -597.50055,
            590.1458,
            -591.6192,
            -584.53754,
            582.8141,
            580.1723,
            588.3744,
            -585.8864,
            601.79346,
            -591.9201,
            -585.36926,
            573.66077,
            -573.5816,
            590.649,
            572.72736,
            573.1389,
            -575.0843,
            -576.45087,
            577.1839,
            578.183,
            -581.6608,
            -600.94855,
            596.3209,
            599.3923,
            591.1035,
            -593.29364,
            577.8559,
            590.9475,
            -574.7751,
            588.6516,
            -571.8808,
            593.53754,
            -578.77344,
            574.194,
            589.9936,
            -574.8797,
            590.1989,
            -578.68256,
            -570.8361,
            -576.21857,
            569.97943,
            -580.3461,
            569.9174,
            -552.01855,
            570.79675,
            571.9628,
            563.6671,
            -571.6873,
            580.19434,
            -574.0291,
            -571.022,
            563.55524,
            574.3385,
            593.4214,
            -580.8736,
            573.2439,
            579.03156,
            590.5327,
            578.6101,
            584.0323,
            579.0895,
            576.4621,
            560.5682,
            -572.77606,
            -570.1248,
            -573.7756,
            -579.5179,
            572.7993,
            -588.25037,
            576.50964,
            590.4421,
            577.02814,
            570.94714,
            575.8091,
            -572.31946,
            585.9882,
            -554.1265,
            577.01105,
            573.53973,
            544.1398,
            -556.2262,
            566.5864,
            -561.317,
            557.2127,
            558.5308,
            564.298,
            -566.92755,
            572.53174,
            -570.78076,
            565.8104,
            -564.4901,
            567.43463,
            563.75385,
            563.3584,
            -571.04376,
            559.0625,
            -575.4382,
            570.5159,
            -565.44336,
            -573.4403,
            -565.17285,
            562.57526,
            -566.43567,
            -554.9772,
            547.92053,
            546.23376,
            571.03906,
            -574.2846,
            580.7919,
            582.02167,
            573.6156,
            -579.78534,
            -570.0172,
            572.8284,
            562.9632,
            -567.2613,
            -580.1469,
            -567.5489,
            550.35236,
            552.33154,
            534.41504,
            -523.99744,
            550.87244,
            548.09216,
            539.0656,
            554.3171,
            547.46173,
            552.29944,
            546.55646,
            -552.6117,
            -571.5015,
            -558.69806,
            -550.00256,
            534.3587,
            553.1033,
            546.679,
            -553.3773,
            556.5765,
            549.82245,
            562.2647,
            574.10583,
            572.2904,
            -563.55084,
            553.7361,
            559.5223,
            560.3886,
            -546.9648,
            -562.044,
            555.8813,
            -563.3092,
            -539.4521,
            550.62164,
            -564.2428,
            544.4067,
            -547.47394,
            -561.97656,
            -547.79016,
            559.7267,
            530.1931,
            547.18536,
            533.1203,
            564.2697,
            -541.5784,
            550.9176,
            536.8533,
            -544.834,
            -541.5549,
            -535.3053,
            -525.5542,
            538.8241,
            -517.5649,
            -549.86115,
            -540.04803,
            -548.7781,
            538.58105,
            547.57355,
            550.581,
            549.802,
            550.68506,
            565.2428,
            550.48834,
            -545.8651,
            530.2404,
            523.9981,
            541.4078,
            -511.8434,
            541.6886,
            537.32416,
            536.89124,
            530.4396,
            -544.1161,
            529.0823,
            -537.7617,
            547.9717,
            530.9629,
            544.97864,
            536.4064,
            551.44965,
            542.55005,
            -546.559,
            546.1434,
            544.1798,
            552.3451,
            539.8692,
            -555.1361,
            554.50354,
            539.168,
            537.6874,
            -543.692,
            -561.02765,
            563.2843,
            565.5783,
            557.41327,
            553.46027,
            -550.21643,
            552.2966,
            -561.41223,
            536.5777,
            -554.7057,
            540.2067,
            544.40125,
            524.59296,
            549.54755,
            557.47156,
            -548.2813,
            -535.11316,
            -542.3836,
            516.5466,
            528.51086,
            -521.5181,
            529.922,
            525.4565,
            528.01996,
            -512.876,
            520.7953,
            -522.7507,
            -514.12366,
            -523.11804,
            526.3371,
            538.52795,
            539.1613,
            -533.6701,
            535.0388,
            -534.2268,
            -520.4917,
            -520.2938,
            -525.5434,
            -532.14453,
            520.5757,
            526.4088,
            517.51855,
            520.97595,
            -514.53406,
            515.89386,
            519.502,
            -527.12445,
            529.9221,
            518.3731,
            -501.76974,
            -531.1229,
            -503.83127,
            507.8577,
            -508.5007,
            -509.71518,
            501.9518,
            -504.72113,
            522.7942,
            523.95966,
            -535.3744,
            -534.2942,
            -531.27673,
            -518.82996,
            522.9913,
            -527.62683,
            531.88104,
            523.3235,
            530.2154,
            -547.6814,
            -539.7718,
            -529.60693,
            -520.47687,
            -533.46576,
            513.3005,
            528.40125,
            518.8749,
            -512.90314,
            510.82162,
            -507.8681,
            -501.5726,
            518.9279,
            510.39728,
            -499.36975,
            -515.8085,
            -504.76718,
            -502.36508,
            508.69476,
            -503.13187,
            498.84634,
            520.6397,
            500.19644,
            496.55298,
            510.91782,
            502.7186,
            -514.21606,
            513.30396,
            507.19025,
            520.3861,
            522.64154,
            513.38354,
            -517.40045,
            -520.4223,
            510.2169,
            498.59277,
            -501.84708,
            -480.91272,
            -501.39215,
            -501.11887,
            504.0447,
            509.68546,
            -497.4484,
            496.26627,
            494.24612,
            -492.7467,
            500.048,
            -510.4933,
            -511.9172,
            517.2482,
            -526.10693,
            -508.19278,
            518.16693,
            -515.539,
            527.16095,
            527.14557,
            518.16205,
            -492.76276,
            497.74493,
            -501.71298,
            489.22192,
            495.64798,
            -512.2432,
            -512.71545,
            502.79767,
            510.85062,
            499.51492,
            -499.7447,
            -490.41183,
            499.12793,
            505.67966,
            -494.77988,
            -478.09198,
            488.87296,
            -496.7044,
            508.14774,
            -521.90216,
            522.98254,
            515.3022,
            -517.73456,
            509.45358,
            505.0177,
            502.96097,
            496.95264,
            500.50537,
            -498.08133,
            514.9818,
            506.89932,
            -499.05255,
            -492.39093,
            487.61417,
            499.28482,
            490.9656,
            482.0227,
            -486.94427,
            -481.66168,
            488.30722,
            514.3053,
            -502.82263,
            -496.52838,
            514.17175,
            511.03424,
            -507.43704,
            501.4053,
            490.66254,
            497.79593,
            503.036,
            -496.18008,
            511.82257,
            506.67038,
            490.96033,
            -492.88654,
            491.85672,
            481.87677,
            493.74734,
            -490.10562,
            -503.56516,
            496.84613,
            -489.7852,
            -501.75626,
            499.41296,
            -483.89847,
            -494.1839,
            485.8611,
            -479.57785,
            476.61548,
            -483.43076,
            -496.75122,
            493.08762,
            490.34024,
            496.76184,
            -493.17618,
            -486.6241,
            -498.3037,
            492.98907,
            495.6543,
            498.08423,
            486.45932,
            499.16208,
            -509.80338,
            -502.17584,
            506.31662,
            517.3865,
            501.0279,
            -496.42184,
            499.29178,
            500.4684,
            508.06888,
            -499.62463,
            500.35303,
            495.98233,
            -487.5571,
            496.73022,
            -499.9648,
            -475.11804,
            -486.10208,
            496.0294,
            505.87454,
            495.41977,
            510.68237,
            -505.63083,
            504.22653,
            -500.31317,
            494.53366,
            476.26285,
            490.78122,
            -504.5161,
            497.77396,
            503.01978,
            505.8988,
            -499.41882,
            509.89972,
            520.0903,
            -516.7273,
            497.83926,
            -485.24176,
            473.63196,
            -489.0965,
            490.36038,
            491.35327,
            488.6543,
            481.2537,
            -493.22873,
            -477.31366,
            -482.7174,
            -476.141,
            -476.73282,
            488.61655,
            494.25256,
            470.62943,
            474.74014,
            -485.47073,
            489.52393,
            -475.47824,
            -478.65036,
            472.4127,
            481.27142,
            472.3583,
            -472.27762,
            481.24655,
            476.20602,
            477.64017,
            -480.47162,
            -478.64645,
            473.63498,
            473.866,
            -462.89847,
            462.0376,
            479.30658,
            466.28436,
            -478.8577,
            -464.4895,
            -467.6507,
            473.37564,
            -472.00836,
            470.23895,
            -468.06314,
            -453.43933,
            -472.39108,
            -454.70386,
            452.64984,
            444.18103,
            456.93164,
            449.13037,
            449.34552,
            -447.67303,
            461.19296,
            457.3868,
            461.56073,
            447.79834,
            -453.21753,
            443.21152,
            -448.88373,
            440.6753,
            -439.0856,
            440.0241,
            -446.11758,
            426.01114,
            447.3548,
            436.5469,
            456.23154,
            457.14978,
            457.55405,
            -462.44476,
            463.46146,
            462.89178,
            464.8589,
            472.23312,
            -465.81967,
            -465.22168,
            469.7758,
            458.46335,
            463.24115,
            -458.64236,
            462.4983,
            -466.82452,
            -471.58603,
            467.9233,
            -454.60556,
            452.63705,
            467.51743,
            457.1744,
            -445.74817,
            -461.3619,
            -472.33472,
            -461.7721,
            465.09558,
            -471.61472,
            462.43323,
            -468.29742,
            -463.05396,
            -474.33392,
            477.13336,
            481.75623,
            494.4703,
            -489.17,
            -494.37103,
            498.5981,
            -473.2387,
            -473.30334,
            -466.26965,
            465.065,
            467.00134,
            -461.32593,
            468.44162,
            483.76157,
            489.5038,
            -481.71,
            468.1611,
            -465.6288,
            466.55115,
            -464.48785,
            -463.55096,
            461.2162,
            445.95877,
            -453.04083,
            455.77078,
            456.21667,
            455.6787,
            -467.60608,
            -457.73944,
            450.01553,
            450.2292,
            439.40958,
            -459.95813,
            449.01678,
            -438.36536,
            438.52222,
            436.16708,
            435.5076,
            446.44443,
            -466.4515,
            473.31876,
            465.7865,
            -461.3812,
            463.24323,
            463.4753,
            -463.73767,
            -470.16785,
            -476.3753,
            464.70905,
            450.79346,
            457.8068,
            -456.13077,
            -477.65128,
            -455.8562,
            -468.9453,
            454.9154,
            461.4517,
            449.6903,
            -438.76483,
            -441.51993,
            -432.89075,
            443.7447,
            455.6157,
            -458.28754,
            -458.45557,
            439.41382,
            445.2962,
            445.0248,
            -432.01315,
            -433.2221,
            -454.95016,
            -449.96558,
            457.2777,
            462.4883,
            -465.75723,
            455.64523,
            451.67938,
            454.02536,
            452.41376,
            -453.78192,
            -434.91208,
            453.0816,
            438.13132,
            -452.79913,
            453.871,
            458.84705,
            -454.569,
            448.5178,
            453.88718,
            -452.47363,
            456.07193,
            -447.62744,
            449.50766,
            451.4514,
            -450.4939,
            457.26944,
            -440.44247,
            465.62967,
            -450.08304,
            462.38232,
            -463.2087,
            472.51877,
            -469.99783,
            -468.3512,
            462.26086,
            458.56854,
            448.7809,
            450.5344,
            -457.2194,
            -441.09573,
            446.42505,
            -447.743,
            449.67722,
            -451.5675,
            451.75342,
            457.79572,
            468.81284,
            -470.8448,
            459.32834,
            478.13806,
            -470.37933,
            455.69812,
            457.40115,
            -472.79068,
            460.64914,
            461.20554,
            -473.1002,
            -469.13962,
            465.69,
            473.20032,
            474.78033,
            -473.67892,
            474.21222,
            464.7509,
            -467.00784,
            461.2137,
            -460.9734,
            463.8004,
            461.5255,
            464.56183,
            470.27588,
            473.3563,
            -470.11166,
            468.61053,
            471.30862,
            457.37784,
            462.20654,
            460.18127,
            457.04074,
            468.2272,
            -468.2795,
            465.8439,
            468.00266,
            479.19354,
            -485.08575,
            -480.8793,
            -469.68402,
            471.35306,
            -457.36978,
            453.94913,
            -450.67343,
            434.89835,
            455.07956,
            456.05548,
            454.42664,
            452.27808,
            -453.4835,
            -456.3917,
            -447.06714,
            -438.29398,
            -442.192,
            442.70288,
            -440.7719,
            -441.44482,
            -439.8299,
            -452.54327,
            -451.3418,
            -463.92688,
            -450.34338,
            463.96,
            -459.11447,
            -458.74612,
            -454.56058,
            452.7509,
            452.86746,
            468.86667,
            463.64432,
            463.53943,
            463.01492,
            452.47534,
            -451.8389,
            454.0074,
            456.9525,
            463.81396,
            -466.6045,
            464.0265,
            477.5334,
            -471.8659,
            -451.34763,
            -457.23245,
            -463.06082,
            456.16177,
            -464.86346,
            -449.7553,
            -458.97067,
            -457.9618,
            -462.81442,
            -466.7604,
            464.86258,
            467.238,
            -468.10724,
            -466.60095,
            466.4628,
            472.2242,
            468.82587,
            466.74994,
            460.26718,
            -455.8401,
            -453.67886,
            434.88354,
            -457.138,
            446.29633,
            -454.9957,
            455.1237,
            458.03888,
            464.44504,
            -455.4844,
            458.979,
            460.95715,
            -464.59442,
            459.37924,
            -455.2123,
            -449.98032,
            452.82648,
            459.38736,
            449.7234,
            -446.33127,
            444.44983,
            431.70514,
            451.19727,
            452.43338,
            -450.06845,
            478.25977,
            469.19113,
            -458.36206,
            462.69406,
            -447.7123,
            -446.71814,
            -456.9708,
            451.19427,
            459.01318,
            454.1437,
            457.8742,
            453.75507,
            459.45828,
            464.4122,
            -459.19113,
            465.9317,
            466.9591,
            -460.28467,
            459.92554,
            -441.21576,
            443.9861,
            445.11462,
            441.4239,
            -450.9908,
            -451.62534,
            450.76282,
            447.48184,
            -452.1266,
            -446.678,
            -456.43323,
            -447.01138,
            446.49893,
            -444.2899,
            -452.93402,
            454.7393,
            -454.32635,
            460.4676,
            -457.47824,
            -459.36307,
            450.92957,
            452.44785,
            449.26538,
            447.03427,
            458.71234,
            462.15103,
            460.05402,
            471.24377,
            463.4823,
            455.27017,
            -460.1147,
            -454.4904,
            456.17215,
            451.7903,
            -458.85974,
            448.40167,
            -445.78708,
            442.89063,
            -447.8471,
            448.38437,
            449.85275,
            452.81747,
            -441.8385,
            441.01193,
            -440.66238,
            441.7059,
            -450.47232,
            -449.6956,
            -453.8102,
            444.0211,
            451.41086,
            447.08377,
            448.76627,
            446.18933,
            446.4722,
            -445.15195,
            -447.85733,
            439.4216,
            445.26865,
            447.3228,
            437.3457,
            447.23224,
            456.63226,
            450.15765,
            435.46216,
            -445.49316,
            -447.3363,
            -439.667,
            441.2924,
            442.05643,
            455.11328,
            445.9161,
            445.69,
            436.5914,
            440.6128,
            436.06927,
            -433.84668,
            -442.64893,
            439.29144,
            450.22894,
            -444.34067,
            459.33185,
            -452.58087,
            439.67743,
            -438.9303,
            433.34644,
            422.7012,
            433.08533,
            434.07266,
            448.60193,
            434.53894,
            -444.15115,
            -440.4056,
            433.317,
            433.42627,
            -429.29575,
            -422.36404,
            -416.59512,
            -418.06192,
            416.28238,
            417.4264,
            418.56848,
            413.51843,
            -414.6175,
            -424.1659,
            -418.11795,
            -400.06375,
            397.81235,
            374.47922,
            -378.37576,
            378.07596,
            -381.489,
            -386.97726,
            377.9986,
            384.3808,
            -379.45972,
            382.3476,
            -370.29324,
            378.52972,
            -368.2879,
            -379.98175,
            -374.75464,
            384.32697,
            -382.17365,
            -390.80276,
            381.80954,
            -378.96127,
            385.0802,
            383.77658,
            -389.21008,
            386.73492,
            398.06754,
            399.91815,
            -398.32504,
            390.93378,
            -387.49484,
            -379.56525,
            372.51025,
            377.24612,
            373.46625,
            -369.59766,
            366.7126,
            360.33453,
            -367.17957,
            -385.00693,
            -378.57635,
            -380.84586,
            365.58035,
            -354.11783,
            359.45984,
            -367.5163,
            374.73047,
            383.0175,
            369.4693,
            373.36642,
            -378.52933,
            375.6865,
            384.49707,
            -373.63553,
            -381.3776,
            -372.558,
            -365.24146,
            362.09152,
            -349.67776,
            351.14764,
            -359.65726,
            364.8174,
            384.0212,
            374.74515,
            369.44675,
            -383.457,
            -367.9529,
            377.74493,
            -376.4862,
            368.09818,
            364.62213,
            365.9113,
            365.46146,
            -367.89706,
            -354.9374,
            -360.76944,
            358.86862,
            355.27533,
            -356.30896,
            -366.00897,
            365.2015,
            365.40625,
            372.88156,
            -381.48297,
            -372.71436,
            379.60562,
            373.2545,
            367.72955,
            358.18158,
            -358.64395,
            376.28967,
            393.47647,
            -396.1879,
            -399.05606,
            -399.3083,
            400.39044,
            411.73868,
            419.0634,
            428.0552,
            415.95865,
            -435.12894,
            428.81873,
            -423.2562,
            -413.39026,
            416.86853,
            418.19464,
            420.14163,
            421.267,
            425.9593,
            438.46542,
            -444.30106,
            -441.79034,
            446.37897,
            433.36047,
            449.41638,
            -449.73663,
            -440.75568,
            -432.38693,
            -433.81024,
            -431.4051,
            -423.20636,
            -437.02866,
            -428.42523,
            432.0029,
            -437.44083,
            -439.93597,
            437.5601,
            430.4796,
            -434.2787,
            432.4983,
            420.64227,
            -428.87814,
            433.3222,
            434.99252,
            433.4639,
            437.66476,
            -445.12683,
            -439.79486,
            422.8157,
            -425.1056,
            420.42905,
            -432.68472,
            441.1698,
            429.84656,
            -445.6638,
            448.98303,
            438.26373,
            440.2398,
            -432.01294,
            428.7444,
            422.65118,
            -428.80615,
            -421.77576,
            -428.5981,
            -431.50687,
            432.41223,
            -438.35785,
            433.27386,
            -420.92102,
            427.62326,
            -415.59,
            422.37128,
            -423.88702,
            -433.4621,
            -435.16904,
            437.14667,
            439.20474,
            -436.05307,
            -431.40176,
            -437.41507,
            437.49548,
            434.9472,
            441.0401,
            441.97476,
            -444.38187,
            434.70944,
            444.02466,
            -435.0835,
            431.60846,
            427.6151,
            422.43088,
            -420.79254,
            418.17685,
            -423.73923,
            -419.2672,
            424.72858,
            417.84207,
            426.61493,
            -430.11148,
            432.6784,
            -420.40063,
            428.21423,
            422.0889,
            -416.01895,
            410.77475,
            400.961,
            392.01978,
            401.50928,
            -384.26447,
            389.371,
            -390.96066,
            392.39755,
            381.1437,
            -379.90298,
            361.0271,
            353.5626,
            -354.1602,
            348.96988,
            -342.51483,
            337.94855,
            350.5954,
            337.43277,
            353.8656,
            -344.82355,
            349.0724,
            351.19666,
            -355.87564,
            361.0861,
            -347.99994,
            356.7545,
            -353.27243,
            -352.06717,
            -348.30386,
            338.5362,
            -335.11038,
            335.30884,
            341.66217,
            334.0327,
            334.6776,
            336.8364,
            339.56256,
            -339.39783,
            -337.1009,
            334.698,
            -337.77618,
            -346.8616,
            -352.61417,
            -348.14206,
            335.62595,
            -348.6764,
            -338.09,
            342.41858,
            -329.3593,
            337.65894,
            333.34534,
            -346.24057,
            331.65933,
            331.0176,
            336.09985,
            -340.8932,
            336.768,
            348.6415,
            346.1212,
            -341.49417,
            -346.99866,
            -343.02094,
            -339.981,
            344.1017,
            343.279,
            345.2326,
            -345.90244,
            -348.801,
            -343.38617,
            341.962,
            -340.1914,
            -338.07108,
            -337.67596,
            -337.32468,
            342.95087,
            -337.5723,
            346.68457,
            -342.52823,
            348.98965,
            -341.8645,
            336.96017,
            332.34784,
            -330.22055,
            -336.37616,
            355.70313,
            369.54993,
            -382.02396,
            -404.68225,
            -419.74152,
            419.1486,
            431.09,
            -429.31836,
            415.89294,
            414.78918,
            403.25842,
            411.7036,
            -417.9856,
            413.53528,
            418.18167,
            427.74393,
            -433.4572,
            425.94305,
            419.41687,
            427.75293,
            426.12463,
            -429.5011,
            -420.6224,
            -430.1595,
            -419.21832,
            -421.31857,
            418.21127,
            407.46463,
            -419.0937,
            403.40558,
            413.47543,
            -412.3932,
            416.84677,
            420.12402,
            428.33453,
            -430.67587,
            426.7413,
            416.41837,
            419.77335,
            -427.4713,
            -425.7299,
            -418.69995,
            -407.4205,
            -413.03076,
            399.48557,
            -408.0729,
            -415.2976,
            415.0908,
            421.49652,
            -425.8901,
            426.80566,
            428.15768,
            420.61404,
            -427.43524,
            -417.59845,
            412.29947,
            411.70493,
            407.88443,
            -416.7503,
            422.24133,
            -416.53824,
            426.94238,
            420.432,
            418.4453,
            -420.8348,
            422.94586,
            -429.5352,
            436.20233,
            -432.2623,
            -427.55725,
            -424.40787,
            -415.6864,
            -416.23187,
            -416.09814,
            403.31708,
            398.16727,
            397.55237,
            -401.00308,
            396.81067,
            388.31735,
            386.72205,
            375.36185,
            363.36722,
            351.49228,
            340.5159,
            -336.18924,
            331.30634,
            -326.49872,
            -339.32208,
            -325.61404,
            319.358,
            316.4389,
            322.31168,
            -317.43552,
            -320.10376,
            -325.89435,
            -337.52023,
            -346.30017,
            337.58023,
            339.40906,
            337.48264,
            329.2563,
            -317.18054,
            -313.05838,
            311.06287,
            318.2915,
            318.6456,
            318.75058,
            329.59637,
            -324.1657,
            -324.09958,
            323.98288,
            -326.30713,
            -326.15414,
            337.23218,
            331.0861,
            329.38052,
            -324.20914,
            320.43576,
            312.39502,
            -320.56052,
            -314.0014,
            312.17017,
            314.5609,
            327.09952,
            320.26654,
            -325.70648,
            318.6061,
            -313.14456,
            313.6276,
            -320.63687,
            -314.25543,
            -314.0347,
            -321.54514,
            -314.65338,
            -317.07297,
            -319.59445,
            -327.10455,
            328.43295,
            323.62054,
            -311.60458,
            -308.88846,
            305.33664,
            300.0859,
            293.61746,
            292.1362,
            -290.74564,
            -301.8767,
            309.56744,
            316.82538,
            332.51285,
            -354.88892,
            382.8252,
            400.54572,
            413.0607,
            -429.3919,
            -432.02905,
            -423.9049,
            -432.3712,
            431.2416,
            421.12506,
            404.55026,
            396.0437,
            390.3437,
            -389.2818,
            -398.8236,
            -414.36584,
            -422.2221,
            419.92807,
            -413.9978,
            419.74014,
            -412.28302,
            404.99664,
            -404.47128,
            397.3381,
            -399.42126,
            405.20154,
            -406.13034,
            412.22998,
            -407.55283,
            409.547,
            407.82907,
            -409.6958,
            -418.1934,
            409.22736,
            408.5873,
            408.83414,
            412.39404,
            410.34302,
            -411.72794,
            -404.9215,
            408.4377,
            -415.0654,
            -422.97842,
            430.4564,
            -423.67157,
            -427.56506,
            -426.55484,
            -429.00787,
            428.18466,
            427.79257,
            412.3354,
            399.90744,
            -382.48837,
            374.32327,
            -378.9216,
            388.89496,
            401.1071,
            410.0066,
            -416.73422,
            411.91492,
            419.58215,
            -420.92325,
            -428.48453,
            427.64487,
            -418.039,
            417.5762,
            415.77722,
            -418.7554,
            -415.84293,
            416.33194,
            406.33984,
            413.66812,
            427.9599,
            -418.1702,
            -427.9829,
            425.00662,
            -430.19492,
            -424.38095,
            417.28723,
            428.1206,
            429.56512,
            427.04047,
            434.86026,
            430.5339,
            -439.6879,
            -432.10242,
            427.08087,
            427.05066,
            428.92667,
            424.8353,
            415.9341,
            423.56595,
            -423.95108,
            423.74503,
            -427.66858,
            430.12378,
            -428.07443,
            427.29575,
            424.99615,
            -422.08408,
            420.0397,
            430.11145,
            428.9276,
            428.52286,
            417.1132,
            421.37363,
            426.71478,
            -427.84833,
            -413.3119,
            421.1845,
            -422.42252,
            -431.40225,
            428.83276,
            -424.88205,
            417.36575,
            411.38885,
            -421.22028,
            -420.00113,
            416.8615,
            424.06195,
            421.17313,
            429.3418,
            428.14563,
            427.60928,
            432.71008,
            -431.96967,
            435.87125,
            431.76132,
            -437.4026,
            429.8028,
            -422.9819,
            426.7152,
            424.01404,
            428.17294,
            -425.5997,
            416.71674,
            418.83078,
            -427.7683,
            -429.31625,
            427.95258,
            433.40198,
            -420.5202,
            416.34827,
            423.39584,
            418.34192,
            428.13834,
            419.6245,
            422.08517,
            419.11673,
            -414.4876,
            -427.78656,
            424.6705,
            -421.9469,
            422.148,
            428.27274,
            -424.3021,
            428.3466,
            435.2139,
            429.50073,
            422.33105,
            -423.41586,
            429.87146,
            427.851,
            421.72717,
            -420.86996,
            424.2998,
            -423.3716,
            420.95654,
            -413.06564,
            412.4225,
            -414.79666,
            -415.8197,
            422.21686,
            428.8636,
            -422.3629,
            -421.1622,
            421.2948,
            -421.18466,
            -422.7488,
            428.69968,
            434.9924,
            431.71466,
            430.36777,
            424.75687,
            421.74255,
            419.78613,
            421.70245,
            -425.21536,
            419.3068,
            427.94424,
            426.69876,
            424.43622,
            426.7073,
            -426.26535,
            -422.96713,
            421.35727,
            416.0882,
            432.2658,
            -426.15665,
            -425.0146,
            -424.22592,
            427.52835,
            -430.36526,
            424.03036,
            420.76068,
            -424.79395,
            423.28354,
            417.66406,
            416.78473,
            419.6062,
            419.20776,
            420.98526,
            -421.7496,
            -425.13107,
            428.16522,
            -431.54935,
            423.57892,
            426.15735,
            -421.42365,
            420.84476,
            423.23663,
            -425.58344,
            419.17462,
            -418.1737,
            426.2473,
            423.66,
            -427.9593,
            424.36438,
            425.84006,
            412.11185,
            -419.3354,
            422.8956,
            420.19708,
            423.87183,
            418.2019,
            422.1355,
            417.9483,
            -422.8068,
            422.34116,
            -425.15363,
            -422.9033,
            430.09448,
            424.87036,
            417.76627,
            427.22302,
            -424.67014,
            425.13226,
            423.97397,
            -422.11142,
            422.78806,
            421.1057,
            -426.26425,
            -425.39273,
            -426.14032,
            -430.99063,
            424.89603,
            420.42657,
            414.43466,
            414.20633,
            -418.9255,
            422.20572,
            414.66867,
            413.56924,
            414.9862,
            419.3194,
            416.1853,
            420.32043,
            416.05728,
            423.38013,
            419.4221,
            -416.32858,
            -407.6202,
            420.76172,
            -413.12482,
            420.18222,
            426.71985,
            426.49597,
            421.86557,
            412.74905,
            416.89624,
            416.32373,
            423.16074,
            418.64395,
            418.543,
            418.997,
            422.9039,
            417.56488,
            415.70563,
            422.31903,
            414.0336,
            414.24918,
            415.02087,
            422.8501,
            -415.49722,
            -418.09277,
            -424.96127,
            423.1385,
            -426.66644,
            412.20602,
            -414.62573,
            417.14108,
            424.08725,
            -423.72278,
            -417.16043,
            -416.80048,
            -411.85593,
            -412.20044,
            -411.68115,
            414.18182,
            414.81958,
            -422.13425,
            418.29446,
            418.4035,
            423.04214,
            -414.3861,
            -415.67014,
            -422.6913,
            420.149,
            417.20227,
            423.43256,
            414.22418,
            414.62137,
            -410.32916,
            411.14478,
            412.80493,
            413.0499,
            422.9048,
            415.1404,
            -417.53644,
            -413.44153,
            -409.36322,
            -405.56656,
            -415.49045,
            -416.75806,
            415.41483,
            418.94504,
            419.3756,
            -419.85095,
            419.29605,
            411.89612,
            -412.9225,
            -412.16232,
            410.53244,
            412.80325,
            406.82645,
            -414.84927,
            -417.134,
            414.47186,
            414.26187,
            -416.69925,
            415.99188,
            -419.90875,
            414.04544,
            411.7643,
            409.9584,
            -417.2658,
            -413.0463,
            -417.78098,
            414.3385,
            407.65237,
            -410.67303,
            415.14618,
            -415.2035,
            411.18475,
            -415.49387,
            -412.08017,
            -416.2962,
            -410.44333,
            411.604,
            414.88284,
            410.32813,
            415.10217,
            412.92172,
            409.47992,
            409.5773,
            412.7778,
            407.71594,
            -406.25726,
            406.96442,
            -411.7735,
            -408.39587,
            411.98975,
            -412.8389,
            413.43433,
            -412.15286,
            405.4827,
            405.94678,
            412.7379,
            -414.59634,
            -407.43008,
            407.4848,
            405.50284,
            411.88156,
            -416.50525,
            -413.13107,
            -409.3249,
            -400.70258,
            410.7938,
            -417.2199,
            405.09995,
            -407.83472,
            -405.21735,
            409.51877,
            405.12283,
            411.01315,
            409.22647,
            411.0748,
            412.3997,
            407.40945,
            -413.51456,
            -413.26617,
            -404.60004,
            -407.54153,
            -401.25592,
            403.865,
            -410.9095,
            416.94577,
            -409.03363,
            404.7298,
            407.31107,
            404.69046,
            -404.60608,
            -411.5428,
            413.79428,
            -403.98022,
            408.22366,
            406.87097,
            411.98358,
            411.2444,
            410.14032,
            -409.82617,
            -410.44897,
            -413.44485,
            412.6981,
            415.25803,
            -400.89432,
            402.02808,
            -407.2579,
            406.4745,
            412.05905,
            -407.4225,
            409.48944,
            408.59262,
            407.37384,
            405.17413,
            401.13977,
            405.7775,
            413.03522,
            422.96732,
            -407.77875,
            -410.62695,
            414.17783,
            403.74353,
            408.8695,
            407.961,
            401.54422,
            -403.7875,
            -411.03067,
            -404.2375,
            -411.159,
            410.37158,
            416.12097,
            408.56723,
            -418.73145,
            414.97974,
            -410.86493,
            -411.70853,
            406.5911,
            -406.74243,
            410.03452,
            405.77835,
            -399.44125,
            403.5574,
            404.87756,
            408.15872,
            411.00787,
            -406.41678,
            -412.12708,
            -410.75745,
            -410.80893,
            -401.2495,
            403.6275,
            -401.66916,
            405.6218,
            404.911,
            406.1706,
            -407.73337,
            406.05505,
            410.66724,
            404.38275,
            403.31528,
            403.7274,
            -407.5338,
            -405.68408,
            -407.26782,
            -405.1092,
            405.212,
            408.3835,
            -405.27557,
            -414.86853,
            -404.97467,
            404.98706,
            408.1037,
            408.28058,
            401.908,
            401.8354,
            397.43143,
            -403.83878,
            414.28488,
            407.88428,
            400.9299,
            400.8856,
            -403.76477,
            405.0555,
            404.4124,
            412.98816,
            407.05862,
            412.4699,
            -405.22717,
            402.32425,
            -406.55914,
            397.50732,
            399.0743,
            404.61868,
            -407.87903,
            411.63632,
            408.26202,
            405.9252,
            404.49265,
            -406.75626,
            401.67303,
            -400.83575,
            -398.38516,
            407.79227,
            406.03494,
            407.86234,
            405.65167,
            -407.31802,
            401.79486,
            -398.1402,
            403.99994,
            403.73602,
            400.05884,
            401.78305,
            402.2302,
            405.21664,
            403.56033,
            404.4399,
            406.26205,
            -402.41995,
            -403.04968,
            -398.18723,
            -402.41602,
            -403.78375,
            394.7956,
            405.096,
            405.5551,
            395.56845,
            -409.23816,
            411.49008,
            397.58447,
            394.0712,
            401.48145,
            -399.0738,
            405.9499,
            405.54572,
            -403.11472,
            407.31937,
            405.39777,
            -406.04114,
            404.71713,
            407.9361,
            -405.9866,
            404.84448,
            402.6128,
            -402.92416,
            403.43576,
            401.64566,
            397.77118,
            -402.25363,
            -399.88864,
            399.27957,
            402.93884,
            404.8807,
            409.3034,
            404.15076,
            -399.15738,
            -399.8092,
            -403.30014,
            405.90674,
            407.52292,
            392.13788,
            399.76968,
            -402.94162,
            403.75842,
            395.74213,
            -402.62247,
            -399.24228,
            401.14603,
            399.27258,
            395.815,
            402.5494,
            -394.9988,
            406.1961,
            -411.25412,
            -401.80087,
            -405.34766,
            400.09076,
            -398.98584,
            395.64108,
            396.05704,
            397.7475,
            400.46152,
            400.3886,
            -395.91245,
            -396.70145,
            408.83652,
            -403.1582,
            -404.0482,
            402.3323,
            -397.871,
            398.60223,
            -400.78973,
            407.74255,
            403.13226,
            402.96054,
            -393.44788,
            -402.33868,
            -399.77603,
            392.83514,
            394.53674,
            -391.68222,
            399.35846,
            -398.0027,
            397.8927,
            -404.18243,
            392.79755,
            394.65753,
            391.4358,
            395.81665,
            401.17477,
            404.5696,
            412.22366,
            -403.89258,
            394.9144,
            395.4499,
            398.4484,
            398.39322,
            395.9637,
            395.15356,
            391.75998,
            -405.0895,
            -400.08972,
            -401.12466,
            402.2261,
            401.19888,
            396.30975,
            -397.08807,
            389.20947,
            386.7893,
            391.52072,
            -395.4865,
            -399.40848,
            -396.61395,
            403.9046,
            403.36682,
            399.80048,
            401.53632,
            395.64716,
            392.39728,
            389.3673,
            398.28677,
            401.96912,
            -398.2723,
            -397.95468,
            -395.57596,
            397.9964,
            -387.7714,
            389.27643,
            391.76773,
            -395.1595,
            405.3416,
            403.92838,
            403.82413,
            401.8342,
            -401.81915,
            397.58096,
            391.7494,
            -386.37714,
            386.54645,
            395.10617,
            396.11844,
            -397.05212,
            -396.29346,
            -397.1505,
            -394.9656,
            -397.20886,
            -395.22528,
            -390.10397,
            390.50513,
            399.2318,
            397.80182,
            396.63303,
            398.81293,
            398.17395,
            396.88284,
            396.20706,
            -395.94916,
            -392.74622,
            -388.24863,
            395.14847,
            -398.36038,
            398.87677,
            400.1321,
            -396.6401,
            -395.6342,
            392.26654,
            -390.0497,
            -390.48248,
            393.02017,
            387.9469,
            392.16678,
            -396.99814,
            -395.82608,
            394.6028,
            394.75186,
            389.796,
            393.64655,
            392.40652,
            -399.91364,
            -391.57706,
            -388.05615,
            -390.1692,
            391.9264,
            395.48178,
            -393.83914,
            -398.76602,
            397.10864,
            395.58527,
            389.98163,
            391.1519,
            390.84024,
            396.24194,
            394.83536,
            398.30237,
            399.30066,
            -398.32562,
            388.96973,
            381.32648,
            381.83148,
            380.166,
            384.7484,
            -380.71854,
            391.0964,
            -388.4722,
            388.69235,
            399.40234,
            -398.83142,
            397.6948,
            396.98956,
            -394.0712,
            389.36832,
            391.1704,
            388.25305,
            387.0068,
            -396.2866,
            -392.83252,
            -399.3793,
            -389.3549,
            385.6961,
            383.95883,
            -380.1943,
            -385.39108,
            394.20096,
            398.0972,
            -388.1826,
            388.32016,
            -382.98553,
            384.0952,
            -381.92908,
            -384.18848,
            393.4527,
            -394.24158,
            396.34537,
            401.11868,
            396.0573,
            -393.76813,
            394.1381,
            -389.76474,
            -394.0601,
            -390.95294,
            392.95276,
            -388.0806,
            389.2539,
            -394.47437,
            394.87616,
            -390.37888,
            -389.7303,
            -380.90982,
            390.02194,
            -382.34427,
            -377.94266,
            381.77344,
            392.60757,
            390.85696,
            398.78473,
            398.79068,
            395.66086,
            395.8149,
            391.54236,
            -386.9381,
            389.0397,
            390.03732,
            393.1996,
            397.73114,
            398.6442,
            398.37338,
            -395.58075,
            392.91025,
            -386.35526,
            -392.37622,
            -395.69223,
            394.04697,
            -393.4689,
            395.2006,
            -387.7189,
            390.6512,
            389.4377,
            395.71716,
            -396.65814,
            392.4088,
            398.12036,
            -394.19147,
            -392.94547,
            393.47256,
            387.04666,
            -389.06006,
            -387.52026,
            -389.92563,
            395.94012,
            385.4704,
            387.7743,
            389.1555,
            -395.2375,
            392.2769,
            -397.47717,
            393.9289,
            393.96082,
            401.2813,
            -397.42816,
            393.9779,
            -389.18692,
            -391.17682,
            -392.5273,
            390.11835,
            -389.88354,
            388.62012,
            388.00793,
            -391.3309,
            -390.23743,
            389.07806,
            384.80856,
            -397.522,
            -391.33466,
            -394.7562,
            -398.01968,
            398.72107,
            394.98886,
            397.32037,
            392.14175,
            392.60596,
            392.76306,
            391.70905,
            390.61664,
            -387.93732,
            390.44247,
            383.10654,
            -386.45798,
            388.00357,
            390.9253,
            -390.86426,
            398.16608,
            393.98615,
            398.92303,
            -396.29892,
            -395.66925,
            393.50668,
            396.58078,
            392.79474,
            398.22855,
            389.29956,
            394.08328,
            396.65384,
            -389.40665,
            -397.01965,
            394.00284,
            396.43063,
            390.24368,
            -389.948,
            -383.73285,
            388.2441,
            389.58124,
            -392.97824,
            391.08322,
            -392.634,
            -394.69592,
            392.55725,
            -396.05743,
            393.3466,
            393.2388,
            396.00882,
            -397.75946,
            395.07928,
            -396.62936,
            398.03314,
            398.67337,
            394.71487,
            392.1422,
            386.35886,
            397.4784,
            -396.33328,
            395.68915,
            393.34494,
            -391.47415,
            -393.127,
            399.70786,
            395.23108,
            398.49417,
            392.36475,
            -393.69324,
            -393.22342,
            389.50473,
            -395.62842,
            389.54626,
            394.51605,
            401.36142,
            -395.5423,
            -396.80707,
            -398.83145,
            394.50552,
            392.02554,
            392.8474,
            -396.6016,
            389.4218,
            389.43652,
            393.113,
            391.6111,
            396.20856,
            -390.09018,
            -389.67758,
            -400.50058,
            -390.63058,
            -390.43625,
            391.3907,
            391.42145,
            -390.40274,
            394.8639,
            -391.88724,
            390.71698,
            388.83246,
            -390.3448,
            391.42984,
            -394.52664,
            -394.7262,
            -395.10324,
            393.40744,
            395.80615,
            393.36453,
            395.54993,
            392.14078,
            392.20523,
            -390.63058,
            390.52228,
            389.99802,
            -391.56104,
            389.84537,
            396.12955,
            392.09747,
            396.47662,
            391.27222,
            392.5944,
            394.80576,
            -397.27115,
            392.63913,
            391.98337,
            394.8109,
            388.3831,
            390.68835,
            -388.52603,
            391.6655,
            -393.14377,
            388.6625,
            -387.75198,
            -391.60837,
            -388.7088,
            396.9099,
            396.46527,
            398.1049,
            398.9942,
            -393.02628,
            -393.7763,
            -394.35068,
            -399.15192,
            -391.30066,
            -390.57748,
            -391.1377,
            391.35446,
            -392.5259,
            -391.34616,
            -389.31198,
            393.5981,
            390.63647,
            393.21713,
            395.1655,
            -391.18396,
            -394.14816,
            -388.0758,
            386.91684,
            -385.03372,
            379.9759,
            -384.80527,
            389.165,
            -391.06845,
            388.42844,
            -389.9796,
            394.40652,
            -390.64648,
            387.3412,
            -389.8173,
            -389.76465,
            387.87646,
            390.4836,
            -387.8174,
            387.86685,
            383.73364,
            389.07925,
            -388.06363,
            388.62402,
            382.04517,
            383.22382,
            -380.15347,
            -387.51038,
            -394.73056,
            393.2803,
            393.8343,
            392.58597,
            386.7214,
            -386.42032,
            -383.97217,
            -384.2703,
            387.95554,
            389.45978,
            -396.90106,
            -392.13327,
            391.53152,
            388.7745,
            383.36877,
            -383.3638,
            -389.1496,
            -388.49313,
            387.1387,
            389.55273,
            -392.83963,
            391.24048,
            390.9841,
            -397.56308,
            394.3688,
            391.26706,
            -388.64462,
            -389.16888,
            -392.05682,
            -392.6633,
            396.95032,
            389.8729,
            -389.27335,
            -390.80423,
            394.7831,
            -390.801,
            392.53778,
            391.28033,
            -391.0877,
            391.44858,
            393.83966,
            386.92694,
            389.8614,
            387.6836,
            -388.4552,
            -392.20294,
            392.581,
            383.6151,
            386.2564,
            -393.75146,
            -391.89017,
            392.61548,
            -395.5774,
            392.35645,
            388.69235,
            392.82147,
            -394.1076,
            392.5731,
            385.5967,
            -390.3853,
            390.38098,
            389.14474,
            393.7611,
            -394.08932,
            393.07864,
            -387.0507,
            -388.36176,
            389.32675,
            391.8665,
            -391.63806,
            389.6056,
            -388.39975,
            -386.49997,
            -386.81616,
            391.7475,
            -392.0952,
            -390.39932,
            -387.51306,
            -387.4144,
            -395.41974,
            -383.3646,
            386.29144,
            -389.06717,
            386.77826,
            384.10532,
            -385.81323,
            385.77982,
            383.36026,
            -385.7195,
            -386.65793,
            -387.00696,
            388.43393,
            -390.6636,
            -385.35013,
            385.1817,
            386.41266,
            -387.7082,
            -387.4763,
            389.88965,
            -385.54227,
            386.19803,
            -383.75784,
            -388.77103,
            -389.7146,
            390.3537,
            385.15335,
            389.9075,
            387.28247,
            -388.207,
            -389.33286,
            -388.6685,
            389.72537,
            393.1365,
            -394.0907,
            -392.06952,
            385.70938,
            386.3283,
            388.33734,
            -385.4573,
            391.8993,
            -387.26062,
            390.95718,
            387.3734,
            -386.30084,
            391.0918,
            -388.87915,
            391.25812,
            383.54245,
            388.70468,
            387.5554,
            -386.98907,
            385.6348,
            -383.13168,
            -384.34192,
            388.34836,
            384.36325,
            383.87256,
            387.08536,
            -386.29654,
            -383.8679,
            -385.35367,
            390.59317,
            389.5996,
            -389.446,
            387.01334,
            -385.87753,
            -390.5474,
            -388.12784,
            386.71906,
            387.62515,
            382.9944,
            385.59814,
            382.7072,
            -386.2204,
            -386.1628,
            387.61392,
            380.54007,
            382.96884,
            383.2525,
            -382.40363,
            385.1963,
            -389.2009,
            -387.5893,
            384.76596,
            379.4215,
            -380.7161,
            378.2763,
            383.46484,
            383.3668,
            -387.3693,
            389.0267,
            386.28757,
            -381.1978,
            383.97128,
            384.7349,
            387.98566,
            389.6086,
            -388.30328,
            -387.5342,
            382.5051,
            387.08002,
            387.20456,
            383.05508,
            383.1464,
            -382.02197,
            385.99274,
            -387.9961,
            386.27585,
            381.10638,
            -381.6025,
            -383.1822,
            385.39853,
            384.49075,
            382.3178,
            385.80154,
            -385.90356,
            -383.7985,
            382.1833,
            -380.96268,
            -382.66602,
            -382.5255,
            385.75943,
            385.3583,
            -386.3241,
            -384.91757,
            -386.37744,
            383.73877,
            383.3673,
            -380.4334,
            387.0178,
            383.81027,
            -382.6682,
            -376.08487,
            -384.64612,
            -381.133,
            -381.60022,
            384.20407,
            383.47607,
            378.31427,
            -377.96814,
            371.37073,
            -354.58,
            295.03857,
            -216.77676,
            172.67776,
            163.00786,
            163.3616,
            158.28737,
            -166.61357,
            160.42749,
            159.3129,
            164.69756,
            -157.41113,
            -161.35329,
            -158.7589,
            163.3187,
            166.33546,
            158.58939,
            -161.11302,
            -165.48921,
            -162.86223,
            162.73074,
            160.04504,
            161.06085,
            169.91692,
            -203.98679,
            289.489,
            355.61548,
            376.6304,
            382.796,
            382.25745,
            383.3817,
            382.8036,
            -380.5728,
            -385.14236,
            382.31818,
            383.15234,
            383.05176,
            385.85037,
            -384.90604,
            -381.51578,
            -382.39066,
            383.94904,
            383.51724,
            381.8091,
            383.7834,
            -379.05133,
            380.95953,
            382.04233,
            383.4305,
            -379.0744,
            378.69345,
            377.94534,
            381.74228,
            -379.55353,
            -378.04834,
            381.42178,
            378.94458,
            381.0733,
            380.43613,
            380.52664,
            -378.1059,
            378.65134,
            380.81296,
            379.4922,
            383.88364,
            -377.4457,
            384.09088,
            -379.21313,
            380.24695,
            -379.5364,
            378.84402,
            377.12936,
            -377.22006,
            -376.77545,
            -380.7642,
            382.01886,
            -381.0033,
            -382.07385,
            384.88608,
            384.39825,
            -379.06137,
            379.83908,
            -378.40503,
            -377.9718,
            384.21317,
            384.81204,
            383.67014,
            385.59854,
            381.0807,
            -382.30392,
            -376.5262,
            377.36627,
            378.83096,
            -375.69122,
            379.95517,
            -379.54395,
            383.3658,
            380.38644,
            380.278,
            -378.31418,
            378.24948,
            -378.98605,
            377.60687,
            384.59235,
            -378.401,
            377.7109,
            376.73254,
            376.5872,
            -382.03406,
            -380.69785,
            377.4326,
            378.37152,
            379.12085,
            380.8341,
            376.5656,
            376.4683,
            379.96887,
            -381.52795,
            384.10178,
            376.79138,
            375.5212,
            378.90878,
            -374.84872,
            378.41656,
            -380.8543,
            379.4898,
            -382.62582,
            375.9384,
            385.38525,
            -381.8098,
            380.15964,
            -375.2332,
            -379.98016,
            378.06207,
            374.89294,
            374.52267,
            376.57413,
            -375.05838,
            379.3733,
            382.3058,
            380.4963,
            377.08554,
            375.37393,
            379.98444,
            376.90027,
            -377.77856,
            -379.75906,
            -377.35205,
            377.0278,
            -380.96198,
            -377.50696,
            380.798,
            -376.17242,
            376.69153,
            373.79858,
            -377.1927,
            -376.80643,
            -378.03204,
            -377.8845,
            379.13098,
            -383.4312,
            378.189,
            -380.26285,
            -377.84567,
            375.51306,
            374.51437,
            -380.35117,
            381.92032,
            381.01593,
            377.51385,
            -375.66925,
            376.32742,
            377.28036,
            377.958,
            378.714,
            372.4692,
            -372.24484,
            374.9016,
            374.37418,
            377.42673,
            -380.0339,
            380.6219,
            -377.05664,
            375.45706,
            -377.6172,
            372.44785,
            -376.39325,
            379.48947,
            376.83884,
            -374.96332,
            375.71756,
            -376.28812,
            374.94846,
            -375.74603,
            376.55988,
            -377.29163,
            -381.1273,
            377.97064,
            -377.51425,
            374.03146,
            -377.78247,
            -370.9632,
            -376.5315,
            374.11047,
            -374.8008,
            -375.3607,
            378.93088,
            -377.8367,
            372.8912,
            379.10596,
            -373.0727,
            -372.87048,
            374.90466,
            375.4102,
            -377.39053,
            -376.99652,
            378.95468,
            -376.43024,
            377.80087,
            -377.8594,
            375.6891,
            372.8599,
            370.4453,
            -374.23502,
            374.4502,
            -373.54147,
            376.27014,
            377.35535,
            -374.3815,
            374.0087,
            -377.40863,
            375.19235,
            -371.56226,
            373.70135,
            375.34103,
            372.631,
            374.0884,
            374.8444,
            376.60986,
            378.57602,
            372.59842,
            373.6611,
            373.68945,
            370.77982,
            373.68887,
            374.90402,
            374.99084,
            376.1372,
            -375.51266,
            -377.98608,
            378.74722,
            373.61346,
            374.87268,
            377.51392,
            -374.2321,
            376.07526,
            -371.43832,
            -372.04962,
            -374.70743,
            -379.24747,
            -379.87408,
            378.49142,
            -373.8372,
            373.42987,
            375.9271,
            373.67883,
            374.25998,
            -373.13116,
            377.93674,
            -375.21713,
            374.49377,
            370.8249,
            370.99753,
            379.29242,
            374.7366,
            373.86816,
            -375.23514,
            -373.9346,
            -373.82092,
            365.95877,
            -370.42484,
            -372.78387,
            372.21655,
            -376.0346,
            -375.10413,
            374.19537,
            -371.99786,
            371.7515,
            -370.2992,
            374.43997,
            -373.89816,
            -377.6815,
            -376.3074,
            374.18976,
            375.39545,
            375.94632,
            378.8516,
            -372.81396,
            -374.74496,
            -373.01874,
            373.79773,
            373.80377,
            372.78818,
            376.22354,
            374.73233,
            -372.60547,
            374.6843,
            369.4652,
            -375.01202,
            368.45724,
            -371.34732,
            374.23376,
            374.40152,
            376.2503,
            -377.4556,
            371.9638,
            375.1945,
            371.25488,
            373.28598,
            -375.5964,
            374.93402,
            -375.0105,
            370.441,
            375.04507,
            374.3093,
            370.6179,
            371.79025,
            372.9689,
            -372.1344,
            377.00043,
            -374.80615,
            -374.8151,
            -372.36807,
            -372.8176,
            375.7854,
            377.1134,
            374.84192,
            372.05997,
            372.49863,
            373.61258,
            372.95676,
            374.62894,
            -375.3431,
            372.77103,
            -371.22507,
            373.8732,
            371.76746,
            373.7063,
            371.12015,
            372.59583,
            -368.52902,
            369.1452,
            -370.10287,
            371.31128,
            373.15588,
            -373.1793,
            374.35364,
            373.10364,
            370.60797,
            372.17148,
            -369.56638,
            374.02524,
            370.8317,
            376.54703,
            -371.1776,
            373.8102,
            372.30353,
            374.27365,
            -373.61853,
            -369.97287,
            368.3586,
            372.12753,
            -374.333,
            370.88345,
            -372.34888,
            -372.00952,
            -370.99228,
            -370.62613,
            -370.08182,
            -373.3502,
            -370.9751,
            -370.94986,
            -371.5814,
            371.07095,
            368.5137,
            372.4927,
            370.27347,
            370.08414,
            -369.069,
            -369.22513,
            -369.26523,
            -368.3306,
            -372.0422,
            -367.13754,
            368.7226,
            372.4454,
            -371.63806,
            369.81915,
            369.27936,
            -371.74323,
            -367.79938,
            -370.42285,
            -367.3759,
            368.76852,
            370.14667,
            -371.57016,
            372.78973,
            374.4138,
            -374.21262,
            366.96286,
            -370.4115,
            373.49167,
            371.50787,
            373.35382,
            -373.16898,
            -369.35394,
            370.72064,
            -369.3413,
            371.6045,
            -369.02795,
            374.26184,
            -373.94727,
            370.40924,
            -366.3735,
            -366.8307,
            -371.6919,
            368.77325,
            369.0984,
            369.05478,
            369.0052,
            368.34204,
            369.653,
            371.10394,
            -371.71997,
            -373.35068,
            372.9887,
            -368.62506,
            373.00296,
            -370.19824,
            370.2603,
            369.50412,
            -368.53882,
            -372.2429,
            -368.59442,
            -373.7354,
            -376.3197,
            371.68808,
            371.42575,
            375.58682,
            373.45892,
            370.86993,
            -370.75024,
            -367.50333,
            -368.7321,
            -372.80206,
            369.2965,
            -369.03705,
            368.09573,
            365.8583,
            361.05054,
            -365.97562,
            -370.40927,
            -370.4446,
            368.29742,
            -371.40842,
            368.53265,
            367.2723,
            369.64642,
            -369.52048,
            -369.56473,
            367.29648,
            369.66852,
            -367.73373,
            371.1103,
            367.65024,
            -368.52335,
            373.13895,
            -370.23312,
            370.95132,
            370.05685,
            -367.45935,
            -372.73776,
            -370.88373,
            370.29474,
            371.34308,
            370.3316,
            -367.69513,
            368.4687,
            -370.99683,
            372.27618,
            -365.55777,
            372.28052,
            369.46634,
            -370.033,
            -367.69388,
            368.67502,
            369.58353,
            372.25012,
            -369.65182,
            -368.82764,
            -368.68985,
            -373.27112,
            370.05988,
            367.18372,
            -369.91824,
            367.0828,
            -366.00677,
            -368.46814,
            -365.98404,
            -370.35977,
            -369.35364,
            -368.28305,
            -374.3266,
            368.9141,
            367.6894,
            371.6848,
            -368.9567,
            369.31955,
            368.2952,
            370.2491,
            372.58865,
            -370.44058,
            -368.06342,
            366.94363,
            368.56665,
            369.89105,
            366.01154,
            365.06793,
            -367.3383,
            366.84814,
            370.4129,
            370.7808,
            -367.4286,
            370.86438,
            -362.92908,
            -368.89252,
            -369.0236,
            370.9669,
            -368.89343,
            -368.14642,
            -371.15475,
            -367.88916,
            -367.53613,
            368.729,
            369.32773,
            -365.70605,
            -368.6866,
            362.1463,
            365.2992,
            368.03394,
            -369.66354,
            -365.96378,
            367.01593,
            -368.32434,
            366.00586,
            -366.82706,
            367.1111,
            -368.4954,
            -364.41302,
            364.57986,
            -365.6231,
            365.85257,
            364.65128,
            -362.37906,
            369.30408,
            -365.73874,
            365.1561,
            366.3519,
            365.578,
            368.31738,
            363.4176,
            -363.93253,
            -365.62146,
            -367.3319,
            366.16608,
            -362.91327,
            366.47696,
            363.57056,
            -365.0729,
            -365.3988,
            -365.95477,
            366.11258,
            368.37415,
            -362.7091,
            -368.09836,
            365.5987,
            364.63028,
            371.86884,
            -367.85626,
            362.47107,
            -368.46674,
            -365.3788,
            366.50818,
            -368.05966,
            -364.03857,
            363.56165,
            -362.36386,
            367.30518,
            369.34378,
            367.4562,
            -367.3715,
            367.8612,
            -365.84717,
            -366.82538,
            367.1029,
            -367.0663,
            -363.61945,
            364.53952,
            -366.73923,
            -365.15323,
            365.6244,
            368.3018,
            -366.8564,
            365.75827,
            365.89868,
            366.41296,
            363.81198,
            -363.62714,
            365.3119,
            365.52533,
            365.0162,
            -362.2454,
            365.66193,
            364.54123,
            368.80618,
            366.10162,
            363.90298,
            360.81586,
            363.76154,
            -361.85855,
            362.3179,
            361.37576,
            364.09473,
            364.1907,
            367.63977,
            363.1779,
            -365.87512,
            -365.55908,
            -363.17612,
            361.43726,
            -363.76587,
            362.78192,
            363.56644,
            362.1913,
            361.49072,
            -366.78915,
            364.23697,
            -362.06772,
            361.94888,
            367.1623,
            362.06747,
            362.8327,
            363.17548,
            -366.36584,
            365.08054,
            -365.80493,
            363.3149,
            363.39664,
            365.28082,
            363.6288,
            358.92828,
            363.9698,
            -365.37497,
            364.921,
            363.3273,
            367.7531,
            -364.52417,
            367.17368,
            365.5209,
            361.34097,
            -364.2589,
            364.2558,
            365.8935,
            -365.23306,
            -366.9511,
            -361.36786,
            360.95914,
            364.0647,
            -366.2661,
            -361.04517,
            365.06522,
            362.81454,
            365.3544,
            364.17242,
            -367.6518,
            364.83066,
            -363.69946,
            363.4531,
            367.7298,
            -362.41733,
            364.04666,
            -365.1918,
            -363.12143,
            363.5457,
            -364.5689,
            364.45084,
            -363.616,
            -363.55307,
            359.93237,
            -362.12933,
            362.10098,
            361.32156,
            363.77786,
            360.51205,
            361.57098,
            361.8725,
            360.16327,
            -360.59937,
            363.9991,
            -363.3065,
            -366.85312,
            -366.2978,
            -365.111,
            362.67633,
            -357.97406,
            358.7539,
            -357.6885,
            -362.08316,
            -358.98428,
            -362.25946,
            -358.61505,
            358.99213,
            361.2828,
            363.1716,
            -360.56546,
            -360.50208,
            365.88055,
            -358.88226,
            -363.82724,
            363.3604,
            -359.97272,
            362.2601,
            364.0497,
            -363.8104,
            -365.57977,
            366.7832,
            363.02478,
            366.55838,
            365.31042,
            -361.23657,
            -362.91327,
            361.55212,
            -363.65158,
            360.34818,
            -356.46115,
            360.76883,
            363.24542,
            -362.0574,
            361.0011,
            361.6804,
            -361.06183,
            359.95593,
            355.4274,
            359.95267,
            -359.3058,
            -360.58643,
            -362.4102,
            -360.97644,
            -361.3405,
            359.95972,
            361.26138,
            361.54483,
            361.38107,
            -361.30533,
            -362.74982,
            362.51202,
            360.40244,
            361.81406,
            -360.8645,
            359.51306,
            359.6705,
            360.4306,
            359.72617,
            362.6364,
            363.2867,
            -362.75977,
            361.31027,
            362.61646,
            359.30194,
            362.64633,
            -364.16458,
            -361.65698,
            359.88727,
            -358.18436,
            360.65585,
            361.10526,
            -360.10657,
            -360.27567,
            361.46628,
            359.56024,
            -364.1579,
            360.88217,
            360.19733,
            -359.00293,
            -358.63632,
            -355.82886,
            -357.13965,
            358.85223,
            362.27863,
            362.11566,
            -359.81216,
            -362.13654,
            362.82462,
            -359.8099,
            -359.85156,
            363.3255,
            -360.36908,
            361.04587,
            358.93643,
            357.74362,
            358.90625,
            -356.47775,
            -356.90042,
            -359.69583,
            356.93817,
            358.1153,
            -361.54736,
            361.82602,
            360.1466,
            -358.94458,
            -362.7739,
            -363.02814,
            360.15646,
            359.80713,
            357.92166,
            -359.74,
            -360.24207,
            -360.18466,
            363.35645,
            -362.18875,
            361.3163,
            -360.41855,
            358.91104,
            359.18774,
            364.28302,
            -362.74188,
            356.98566,
            359.8236,
            -357.19107,
            358.50702,
            360.77805,
            357.90234,
            360.18723,
            -360.7631,
            356.7868,
            -356.0483,
            355.85712,
            358.8104,
            -358.41595,
            -356.09296,
            360.72543,
            -361.23267,
            -357.599,
            359.37436,
            360.25894,
            358.17505,
            359.3528,
            -357.58868,
            355.7624,
            -354.72455,
            -359.68225,
            360.2247,
            359.12027,
            -357.72656,
            359.0259,
            361.37048,
            359.73007,
            359.32242,
            -358.04196,
            358.1072,
            358.90567,
            -360.04422,
            357.8306,
            -357.37253,
            -360.22388,
            358.26486,
            356.4401,
            357.9296,
            357.13635,
            359.73633,
            356.29062,
            355.40314,
            357.7816,
            357.22382,
            -359.011,
            359.203,
            357.87323,
            360.9752,
            357.6936,
            358.49893,
            357.8962,
            -357.70148,
            360.6761,
            360.25385,
            -361.08163,
            360.59518,
            357.27652,
            -357.64203,
            -356.37173,
            -357.41977,
            356.3631,
            358.18954,
            -359.0622,
            355.89343,
            360.22568,
            -355.74588,
            356.60934,
            358.481,
            361.00937,
            -357.0944,
            355.9071,
            359.32217,
            -355.54892,
            355.62018,
            358.68826,
            -358.01416,
            -356.03625,
            -358.559,
            356.7077,
            359.4863,
            -361.7854,
            -355.9542,
            -359.41586,
            -361.36148,
            356.8835,
            357.555,
            -354.20532,
            361.41428,
            -356.46265,
            356.97958,
            357.71686,
            -355.4359,
            359.1751,
            358.4274,
            359.28308,
            358.47733,
            -356.41254,
            357.1451,
            356.07086,
            -355.63498,
            359.28952,
            -359.62885,
            -355.87888,
            358.07578,
            356.85202,
            -357.7184,
            -358.16562,
            356.9703,
            358.10703,
            357.86026,
            356.79272,
            356.42657,
            358.9881,
            355.37448,
            353.22302,
            356.18887,
            -357.1546,
            356.64853,
            -356.06046,
            357.91983,
            -356.5869,
            358.4269,
            354.71136,
            -357.679,
            355.11917,
            -358.197,
            -353.80542,
            357.4336,
            355.1279,
            -356.89026,
            353.69128,
            356.30478,
            358.13165,
            -356.85666,
            -354.01395,
            354.8699,
            355.0341,
            355.0896,
            -352.3903,
            -354.97238,
            -355.676,
            356.47137,
            352.4665,
            -355.67618,
            -351.75436,
            -356.60458,
            354.59372,
            354.9808,
            356.59464,
            356.20868,
            355.21634,
            355.5678,
            357.26083,
            355.7088,
            354.56528,
            -353.4529,
            355.91486,
            355.45413,
            356.6106,
            -356.0418,
            -356.62238,
            -354.6358,
            -353.63882,
            351.88785,
            354.56558,
            -354.2928,
            358.33307,
            -356.2082,
            357.49155,
            354.95667,
            353.7863,
            356.39758,
            -360.99753,
            355.24258,
            -353.177,
            353.70673,
            -354.72333,
            355.16235,
            -353.09122,
            354.30875,
            -359.04547,
            353.6456,
            -357.48215,
            -356.36725,
            355.30698,
            356.054,
            354.09753,
            -357.67416,
            -355.51163,
            -353.7853,
            353.18445,
            352.59753,
            -354.0142,
            -357.04694,
            -356.6745,
            356.30383,
            354.11414,
            -353.8728,
            353.27548,
            356.62738,
            -353.983,
            -354.06387,
            -352.60376,
            358.39328,
            -355.12793,
            -355.4909,
            354.70966,
            351.4952,
            -352.9015,
            -353.08194,
            352.78268,
            351.56967,
            355.67758,
            352.4528,
            352.80975,
            -353.87787,
            356.01938,
            -354.474,
            -355.39655,
            -353.34073,
            353.08353,
            354.596,
            355.90308,
            349.11508,
            354.74588,
            -352.3579,
            351.97345,
            -353.97946,
            351.14926,
            -354.726,
            -353.59207,
            349.98196,
            354.34683,
            351.99,
            353.00223,
            351.87692,
            351.76025,
            353.83377,
            352.6558,
            -353.8735,
            355.9252,
            351.5917,
            -351.55954,
            -353.13022,
            352.14777,
            352.3281,
            -353.54773,
            353.41034,
            -355.03787,
            351.71344,
            353.51358,
            -350.6844,
            352.51712,
            354.04675,
            -353.5285,
            -354.10483,
            352.54114,
            357.162,
            355.30267,
            355.90118,
            -356.1284,
            -355.71652,
            -352.84503,
            349.98767,
            354.45535,
            353.64658,
            352.8069,
            352.88724,
            -353.37952,
            353.23007,
            354.19193,
            352.99875,
            -353.18137,
            -352.2553,
            352.58936,
            352.5192,
            355.78754,
            -351.6849,
            349.98996,
            351.86954,
            -352.58713,
            349.21164,
            -352.6498,
            -354.60107,
            351.71497,
            353.03723,
            -353.0108,
            -353.24292,
            352.93307,
            349.85327,
            -353.44843,
            352.53336,
            352.79688,
            -352.45233,
            351.0171,
            -352.91,
            353.9917,
            352.74756,
            354.85056,
            352.6554,
            351.47415,
            348.562,
            351.66312,
            -349.27945,
            354.08936,
            -350.66653,
            351.48917,
            353.56033,
            349.91043,
            -348.96808,
            -349.46332,
            -351.48355,
            355.8276,
            352.22043,
            -350.7747,
            -349.2126,
            351.07532,
            346.34366,
            350.59814,
            -349.5943,
            -349.11423,
            350.93393,
            351.28537,
            351.87265,
            -354.55325,
            -348.99915,
            -352.35312,
            351.93643,
            352.38327,
            346.87585,
            -337.50177,
            255.20522,
            -128.39987,
            -95.81128,
            91.706955,
            96.05519,
            95.3162,
            93.47212,
            95.157295,
            93.9177,
            -91.83697,
            -94.78065,
            91.82196,
            93.798065,
            -98.08121,
            -128.19894,
            -259.04626,
            -343.18457,
            -349.7471,
            -351.0551,
            -347.81168,
            349.14423,
            351.41705,
            349.15747,
            350.0714,
            349.75388,
            349.3261,
            -353.01288,
            350.9969,
            350.11374,
            -347.61414,
            348.3123,
            -350.3884,
            349.20084,
            348.5046,
            348.1063,
            351.6392,
            349.6345,
            -351.09332,
            -353.10858,
            -350.84937,
            -347.63907,
            -348.9927,
            -350.25934,
            -348.4716,
            -348.70413,
            -348.89108,
            -349.33234,
            349.5134,
            350.04636,
            -350.9649,
            -351.2949,
            351.671,
            349.23218,
            349.5816,
            -351.76855,
            350.14713,
            -348.79044,
            349.93137,
            351.46423,
            349.82306,
            -348.74,
            -350.31085,
            346.83102,
            -350.1112,
            -347.94495,
            -351.31506,
            -349.7502,
            348.15613,
            -347.23633,
            -348.12106,
            -347.34756,
            348.32733,
            -347.7546,
            351.41998,
            350.18918,
            -346.5092,
            349.74557,
            351.3013,
            -348.97833,
            -350.76334,
            349.18936,
            348.02463,
            347.64908,
            346.48383,
            -347.14432,
            350.74414,
            -349.0299,
            348.89926,
            -350.1663,
            -347.4924,
            -350.136,
            350.42,
            350.30682,
            350.95993,
            352.00287,
            346.8763,
            -347.75793,
            -349.7989,
            349.69962,
            351.07712,
            -347.8525,
            345.7483,
            -345.13104,
            -350.3361,
            349.3949,
            344.52866,
            -345.04974,
            -346.4907,
            -346.04404,
            347.9267,
            349.41745,
            344.73605,
            347.8434,
            347.4228,
            -347.73685,
            -347.33835,
            -348.916,
            348.67096,
            -348.95468,
            -348.7214,
            348.97647,
            -350.87335,
            346.56445,
            -350.50433,
            346.15305,
            -345.32343,
            -347.7914,
            -349.25372,
            -347.54083,
            -347.3097,
            347.42978,
            349.71155,
            347.70682,
            -348.55643,
            349.92047,
            348.02475,
            347.65582,
            349.43396,
            -346.0006,
            345.35422,
            345.70078,
            -348.291,
            346.67773,
            -350.96588,
            -349.02527,
            -347.00705,
            347.67358,
            346.9815,
            347.5593,
            -348.57587,
            345.47385,
            347.26236,
            -347.15952,
            344.63522,
            344.79456,
            348.1423,
            -347.4558,
            349.08203,
            346.1187,
            346.37418,
            -344.41608,
            -345.1541,
            -346.60553,
            -348.45758,
            -348.24258,
            -348.20273,
            343.34195,
            -343.09766,
            343.57547,
            -344.7336,
            -344.55988,
            343.3374,
            345.49503,
            -346.5227,
            346.00266,
            -343.67062,
            342.56036,
            -347.0789,
            -347.0779,
            -348.03003,
            346.43268,
            -345.52463,
            -347.49686,
            -342.9098,
            344.36066,
            -347.40497,
            344.70938,
            -347.22702,
            -346.63388,
            345.80438,
            345.89417,
            -345.33066,
            -347.605,
            344.94534,
            346.45493,
            346.9371,
            343.4698,
            -346.89255,
            -347.20126,
            -350.11728,
            345.04916,
            -344.60092,
            344.41995,
            348.20993,
            345.25116,
            -345.59952,
            350.333,
            348.9242,
            -345.35706,
            345.42612,
            346.88303,
            347.3273,
            344.99515,
            344.2906,
            345.04703,
            344.20392,
            -345.08395,
            342.2614,
            344.06143,
            -346.11832,
            -346.73248,
            345.1932,
            344.55716,
            344.58704,
            343.60263,
            -344.00253,
            -346.215,
            -343.47452,
            343.88184,
            344.14127,
            344.05533,
            345.03452,
            345.87164,
            -344.80566,
            -344.07007,
            -344.18387,
            -346.38412,
            345.3362,
            -344.38037,
            -343.80634,
            346.7309,
            347.98755,
            -345.89542,
            346.2881,
            -344.58057,
            -344.92545,
            344.81873,
            347.0319,
            344.7667,
            -342.53427,
            345.31158,
            -346.8735,
            345.06885,
            -345.1036,
            -340.92563,
            345.2647,
            344.18527,
            344.27487,
            342.24875,
            346.84207,
            346.21722,
            344.32562,
            -343.9915,
            342.69733,
            345.6167,
            344.08032,
            342.64648,
            -343.32693,
            345.77237,
            344.79205,
            -341.2973,
            347.25616,
            -346.17072,
            343.052,
            -342.26294,
            345.9118,
            -341.76306,
            -343.69763,
            -343.7664,
            -344.35062,
            -345.8614,
            -342.77588,
            345.85916,
            -344.42084,
            342.96704,
            344.6347,
            -342.2408,
            -346.00748,
            343.95404,
            -343.9101,
            343.2312,
            345.52853,
            -342.87823,
            -343.6791,
            344.00858,
            342.16003,
            -342.1209,
            -345.57593,
            342.21576,
            342.72626,
            -344.90372,
            -343.55447,
            343.82184,
            344.20325,
            -344.30875,
            -340.29404,
            343.51215,
            -343.05035,
            343.13632,
            -344.46335,
            344.35928,
            343.40732,
            345.82175,
            343.55783,
            -342.7667,
            -345.5414,
            -344.11063,
            -342.4481,
            342.45596,
            343.27145,
            345.1977,
            342.83334,
            339.0722,
            339.71075,
            -340.52036,
            -343.12885,
            -343.18527,
            -344.90393,
            342.649,
            -341.55386,
            341.41098,
            344.44968,
            -344.4725,
            -343.66342,
            344.13773,
            340.89432,
            342.41486,
            342.6336,
            -341.74786,
            -339.58694,
            341.67773,
            342.67072,
            342.83304,
            343.1996,
            -344.08987,
            344.33215,
            345.45978,
            -342.3266,
            -343.2901,
            -342.18817,
            -343.35873,
            -340.48294,
            342.58047,
            -343.71472,
            340.10822,
            342.06168,
            -342.06387,
            -342.28067,
            341.71393,
            -341.46216,
            -343.54312,
            343.7109,
            -341.82025,
            -340.75244,
            342.05563,
            -341.74823,
            -341.53333,
            342.96085,
            -342.9996,
            -345.38333,
            339.75644,
            341.55484,
            -342.94052,
            341.79492,
            342.27338,
            343.7404,
            339.0348,
            -339.98947,
            339.85596,
            -339.95483,
            343.49567,
            338.72418,
            341.2176,
            -340.44138,
            340.12393,
            -341.40195,
            341.6308,
            -341.19037,
            339.88992,
            -342.36514,
            -340.08328,
            -340.6887,
            340.43228,
            340.9553,
            -339.53152,
            340.26624,
            341.3271,
            -340.91864,
            -342.09958,
            342.2564,
            342.56717,
            339.95886,
            -339.03064,
            338.3236,
            -338.46094,
            -341.65393,
            339.9359,
            341.57672,
            341.55243,
            340.44327,
            -342.7751,
            341.63345,
            -341.3711,
            -344.23868,
            341.16788,
            -341.33618,
            -340.01126,
            342.0016,
            -342.2937,
            341.40427,
            341.3202,
            340.95813,
            338.5334,
            339.70108,
            -341.73883,
            339.71002,
            339.86386,
            339.0698,
            339.6225,
            338.8053,
            -338.3271,
            339.1863,
            342.55887,
            341.52307,
            341.8682,
            -341.5038,
            -342.24136,
            340.4199,
            340.92694,
            341.85757,
            -339.51456,
            341.27383,
            340.42923,
            342.01364,
            -339.79504,
            -340.69186,
            -341.17914,
            339.9535,
            340.39008,
            -341.52008,
            -338.12814,
            -340.1445,
            340.03873,
            -338.06424,
            338.64728,
            340.68732,
            340.2955,
            342.02234,
            341.779,
            338.664,
            342.0748,
            -339.87296,
            337.244,
            338.09225,
            -341.14075,
            340.04565,
            342.08646,
            339.24075,
            -339.44327,
            336.65488,
            -339.2228,
            340.591,
            339.08893,
            338.69852,
            339.61896,
            -339.73135,
            339.99985,
            -340.11884,
            340.71683,
            -340.3059,
            337.48502,
            337.01944,
            340.31265,
            -340.12045,
            336.6928,
            339.92325,
            -338.65927,
            -338.24918,
            -338.96878,
            339.3544,
            -337.5519,
            -337.06448,
            339.1712,
            -338.4996,
            -338.736,
            338.47488,
            -338.3357,
            -338.03656,
            -338.94998,
            -339.72958,
            -341.05948,
            338.6035,
            -338.3012,
            -340.58115,
            336.93127,
            -338.99023,
            -336.99622,
            338.06732,
            -337.18973,
            -338.72345,
            -338.03842,
            -336.29242,
            341.69446,
            337.4448,
            337.87292,
            -338.95728,
            -339.5498,
            339.76,
            336.9487,
            337.21893,
            337.1962,
            338.85617,
            -336.78976,
            337.61542,
            -339.2139,
            -338.02173,
            336.07288,
            -339.21185,
            337.7194,
            338.3549,
            337.29053,
            -338.44962,
            337.97437,
            -339.0391,
            -336.18405,
            335.61725,
            337.08142,
            338.2198,
            338.23355,
            336.75888,
            336.89774,
            338.0019,
            337.4026,
            338.72668,
            -337.68185,
            338.57352,
            338.81033,
            334.46518,
            336.24228,
            340.8334,
            339.36804,
            338.88177,
            -338.0617,
            -338.45175,
            -338.4407,
            -335.75497,
            337.18024,
            -336.4937,
            338.31952,
            337.2317,
            337.9823,
            -337.3708,
            -337.03995,
            -336.68042,
            339.9431,
            -337.00266,
            -334.84055,
            -336.22458,
            -336.60403,
            -337.87564,
            337.8564,
            -336.02002,
            338.76706,
            -338.97385,
            338.18604,
            336.93906,
            336.8162,
            337.07382,
            334.82382,
            334.1725,
            -337.4426,
            335.0496,
            -337.67368,
            -338.21536,
            -335.2981,
            -336.98264,
            -335.45474,
            -334.07516,
            -338.7676,
            337.2673,
            334.6907,
            -338.6392,
            337.80637,
            -334.5454,
            337.1828,
            -338.3685,
            334.69492,
            -337.0072,
            336.23322,
            337.26352,
            -336.5165,
            335.9237,
            -336.9743,
            335.40967,
            -335.98456,
            335.66177,
            337.66565,
            334.9778,
            -336.9397,
            335.3082,
            337.57828,
            -334.94916,
            337.74686,
            -337.83096,
            336.8181,
            -338.20206,
            334.8592,
            336.7408,
            335.16833,
            -335.5224,
            336.37708,
            336.58536,
            -336.04926,
            333.90042,
            336.213,
            336.60037,
            335.94205,
            -336.52817,
            335.21982,
            335.7884,
            -336.21387,
            -336.06485,
            -335.38562,
            -336.011,
            -335.9207,
            -335.64188,
            -336.64102,
            336.12097,
            -333.86536,
            -336.39548,
            334.0553,
            336.6222,
            337.3603,
            336.8358,
            -333.78375,
            334.24738,
            336.16257,
            337.3479,
            -333.22958,
            336.8267,
            335.91953,
            337.86526,
            -334.42075,
            -333.57956,
            -334.5052,
            -334.92993,
            333.0104,
            336.89554,
            335.28348,
            335.4556,
            336.7335,
            335.67654,
            -334.90094,
            334.41135,
            336.3737,
            334.3322,
            -333.16486,
            334.42093,
            -333.87595,
            -335.462,
            334.66568,
            -333.80792,
            -332.67877,
            332.4953,
            -333.8238,
            -332.99918,
            -334.26248,
            -335.1928,
            334.41547,
            334.5691,
            335.0969,
            -337.15594,
            336.60394,
            -333.35028,
            334.89728,
            336.32013,
            -336.5555,
            -334.0797,
            335.52063,
            -334.4752,
            335.41452,
            -333.72986,
            332.7316,
            333.96472,
            333.1104,
            -335.6838,
            334.2015,
            -334.13025,
            -335.06088,
            -334.73904,
            334.44864,
            335.09814,
            -332.94345,
            -333.8405,
            -337.36368,
            335.85278,
            333.80234,
            334.8075,
            331.57364,
            334.08984,
            -327.47165,
            -250.49625,
            99.07374,
            -68.76443,
            69.258644,
            -68.53444,
            -68.16046,
            66.5878,
            65.875244,
            65.94277,
            65.55944,
            -65.42514,
            67.4298,
            -80.843475,
            -194.08539,
            320.594,
            333.928,
            332.96582,
            333.3302,
            332.57504,
            330.50717,
            332.51703,
            335.4411,
            332.33344,
            332.57364,
            -334.9054,
            333.77368,
            333.31357,
            -334.65622,
            -335.08408,
            -331.78372,
            -333.12677,
            -331.12042,
            331.03833,
            -331.22025,
            -334.9187,
            334.10306,
            -335.0991,
            331.99158,
            333.45544,
            -332.04666,
            -333.5858,
            -332.3366,
            334.2067,
            -330.9103,
            333.70883,
            -331.06348,
            332.98343,
            -334.25375,
            333.36853,
            333.87567,
            334.78857,
            330.40103,
            332.41827,
            332.52634,
            333.6886,
            -331.36005,
            331.55585,
            334.14322,
            333.18594,
            -332.6031,
            331.82098,
            332.55533,
            334.27863,
            334.58047,
            331.86148,
            334.38013,
            332.58978,
            332.2416,
            -332.3284,
            -333.40222,
            -331.08026,
            330.78455,
            331.25183,
            332.97906,
            -334.22226,
            -333.4558,
            332.18756,
            -330.41852,
            -330.4512,
            -334.22723,
            -331.56494,
            332.11887,
            332.7653,
            -331.677,
            331.8154,
            331.47797,
            -330.24057,
            332.45688,
            331.90137,
            330.50348,
            332.78387,
            -331.94733,
            330.25067,
            -327.5862,
            331.50623,
            330.8795,
            -332.1837,
            332.16327,
            -332.14984,
            330.10254,
            332.43356,
            -332.04257,
            -330.77194,
            331.60858,
            -331.21005,
            -329.17712,
            332.56738,
            329.63162,
            -332.7419,
            -330.94196,
            331.66025,
            333.4022,
            331.92163,
            -328.84146,
            -331.04922,
            -331.0514,
            -332.22903,
            332.18042,
            333.0164,
            333.0653,
            -332.86337,
            -329.84796,
            -328.90863,
            333.25528,
            331.32062,
            -330.53574,
            332.22516,
            330.7654,
            332.5704,
            332.58658,
            331.33344,
            -331.6384,
            331.2111,
            332.19614,
            329.8935,
            -329.05338,
            330.74997,
            332.5338,
            332.46912,
            -332.49493,
            332.03247,
            332.20883,
            329.58545,
            -330.48972,
            332.3285,
            331.66403,
            330.1018,
            -332.06134,
            331.73196,
            -330.44098,
            332.46005,
            331.84467,
            331.24655,
            -331.70694,
            -328.95374,
            331.11453,
            -330.90753,
            -330.9036,
            -331.02286,
            329.69577,
            -329.83057,
            330.60486,
            -329.86057,
            -327.88544,
            329.02197,
            -331.04037,
            -329.73096,
            331.32855,
            331.75958,
            -332.0981,
            -329.77536,
            -329.67368,
            330.06674,
            328.83746,
            329.69577,
            -332.12143,
            329.27338,
            -329.38052,
            331.09167,
            329.83954,
            329.06424,
            -328.3143,
            330.74103,
            329.73834,
            330.74222,
            -330.2102,
            330.85034,
            330.08356,
            -329.8606,
            330.34277,
            -330.90182,
            -329.02545,
            330.46097,
            -329.83975,
            -328.58688,
            -331.21753,
            -329.6933,
            -328.53445,
            331.53134,
            -330.0552,
            329.511,
            -328.75226,
            330.78214,
            330.03574,
            -329.48505,
            330.83795,
            330.3001,
            328.9625,
            -330.78735,
            329.0697,
            -329.41318,
            331.32227,
            -328.77808,
            329.1766,
            329.57382,
            330.09094,
            -330.29712,
            -329.05765,
            327.7835,
            328.50574,
            -327.8231,
            329.4843,
            328.625,
            328.1739,
            -329.64417,
            329.22577,
            326.6636,
            -328.75977,
            329.9398,
            -328.72668,
            327.58716,
            -329.42984,
            -329.14243,
            330.22958,
            -329.67755,
            327.7582,
            -327.04407,
            331.85364,
            327.35312,
            328.36597,
            329.3429,
            -326.96194,
            327.9431,
            330.53494,
            325.5012,
            327.6657,
            328.49036,
            -328.94205,
            328.47702,
            -330.31372,
            328.65234,
            327.58224,
            -330.15988,
            -327.99225,
            327.51337,
            328.62576,
            -328.32495,
            327.40225,
            -328.4974,
            -327.4916,
            -331.31897,
            326.1246,
            -327.7921,
            -329.8264,
            328.12814,
            328.0378,
            -329.27353,
            -327.48755,
            -330.57813,
            327.12143,
            -326.9938,
            329.599,
            -328.2328,
            -327.6694,
            327.62466,
            -328.6764,
            327.14743,
            327.17123,
            -328.3239,
            -327.5601,
            -327.27402,
            328.07336,
            328.16724,
            326.9444,
            328.27484,
            327.5204,
            325.11118,
            -328.73984,
            328.59943,
            -327.38953,
            326.41733,
            -327.4798,
            326.87137,
            -326.6141,
            -326.895,
            327.51492,
            -327.27884,
            328.93594,
            328.51273,
            -327.0371,
            -327.761,
            -326.95993,
            -327.04398,
            -327.948,
            325.04953,
            326.83737,
            -328.38617,
            326.8453,
            -327.97772,
            -328.63303,
            328.83237,
            -326.98975,
            326.75775,
            325.67554,
            327.50153,
            327.35883,
            327.13913,
            -326.39154,
            -327.00226,
            328.41693,
            -327.979,
            326.71454,
            327.40594,
            327.70486,
            328.57635,
            -327.82077,
            326.03824,
            -326.778,
            -328.7103,
            326.33264,
            -329.28412,
            -328.418,
            -325.77698,
            -329.01453,
            325.60098,
            328.84222,
            325.49506,
            -327.57623,
            -328.44135,
            327.3672,
            328.33914,
            325.7594,
            326.3341,
            -325.86108,
            -326.50198,
            -327.50388,
            -327.56186,
            -324.9798,
            -328.079,
            328.74304,
            -325.39966,
            -324.77707,
            -324.83017,
            -324.8148,
            -329.36307,
            -326.9706,
            -324.96613,
            -327.0758,
            -326.04865,
            327.03592,
            -326.43896,
            325.58047,
            325.82236,
            326.29907,
            324.40735,
            -327.28857,
            -325.39355,
            -326.52887,
            326.02902,
            -326.7284,
            327.42578,
            -326.32584,
            325.46097,
            -325.29474,
            325.82587,
            326.44934,
            -324.29993,
            326.93433,
            325.16245,
            -327.19583,
            -324.55435,
            324.49115,
            323.93658,
            323.1765,
            -326.10373,
            324.52606,
            324.90143,
            -325.84314,
            -324.29785,
            -325.45288,
            -327.83002,
            325.88446,
            -325.66837,
            326.28244,
            -327.08365,
            328.49286,
            325.3215,
            324.7603,
            324.7169,
            -324.7336,
            -327.1978,
            -324.4725,
            326.1083,
            -325.04382,
            -325.08484,
            325.7367,
            325.67838,
            322.19336,
            -326.0693,
            -324.94638,
            -323.53842,
            325.1843,
            -326.61566,
            323.04892,
            326.21777,
            324.0154,
            325.4441,
            -323.6158,
            325.6277,
            324.14056,
            -326.08368,
            -325.98038,
            -325.4744,
            -323.99738,
            -325.3724,
            325.53082,
            326.4129,
            324.10632,
            325.4398,
            -326.91766,
            324.6841,
            326.5461,
            -326.15912,
            323.01086,
            325.5693,
            324.92508,
            324.8828,
            -323.82904,
            -323.78976,
            -324.21072,
            -325.012,
            -322.53525,
            325.61148,
            -322.8154,
            -324.06485,
            -324.09207,
            324.14966,
            -324.66522,
            326.85742,
            324.57166,
            326.7844,
            -324.52975,
            -325.99173,
            322.62228,
            -322.62646,
            322.4441,
            323.46286,
            327.0553,
            -323.80157,
            -322.20847,
            323.7714,
            -324.33774,
            324.5518,
            323.6905,
            -324.23984,
            323.92276,
            324.59326,
            324.1019,
            -325.7709,
            323.07053,
            324.4276,
            -324.2474,
            324.19287,
            -322.95392,
            -323.6299,
            -323.99585,
            323.30295,
            324.05917,
            -323.85538,
            -322.90564,
            324.2564,
            -324.57138,
            324.94348,
            322.80524,
            323.41528,
            -324.1137,
            323.55746,
            322.8684,
            -325.0147,
            -324.1199,
            -322.6547,
            -324.48874,
            325.04535,
            322.86673,
            324.25702,
            324.09738,
            -323.72394,
            323.65134,
            -323.2178,
            322.33078,
            -323.26892,
            324.43564,
            326.65057,
            322.79144,
            -322.5762,
            322.891,
            -324.13126,
            325.08496,
            324.37845,
            -323.04266,
            322.013,
            -322.91388,
            323.02386,
            -322.87906,
            -323.25412,
            -324.1187,
            324.16202,
            323.74826,
            -322.5404,
            -323.4623,
            322.32498,
            322.71054,
            -324.21802,
            323.30273,
            -323.69485,
            322.59235,
            -322.98398,
            -324.5021,
            323.74103,
            -322.71667,
            -323.92505,
            -325.4691,
            -324.0501,
            324.24945,
            -323.34125,
            -323.11725,
            325.32004,
            323.0846,
            -323.73105,
            323.3068,
            322.86542,
            321.3424,
            -324.0736,
            323.59882,
            -322.19214,
            -322.8858,
            -323.58273,
            323.1999,
            -322.5549,
            322.17242,
            -320.60226,
            324.64383,
            323.77664,
            322.01804,
            322.43213,
            -324.93875,
            322.33597,
            -322.8727,
            320.56415,
            321.17047,
            -321.73758,
            322.79297,
            323.48062,
            -321.1708,
            -321.30984,
            -322.58517,
            321.43,
            323.45578,
            -320.94354,
            323.0364,
            -321.32288,
            321.303,
            -321.78165,
            320.972,
            323.45358,
            323.53546,
            320.92252,
            -321.53345,
            -322.82147,
            -322.05576,
            322.76038,
            -323.33728,
            321.20374,
            -319.72464,
            324.2515,
            321.3168,
            -321.94614,
            320.16037,
            -320.85052,
            -323.2971,
            -322.21872,
            323.52963,
            -322.26108,
            -321.16498,
            -321.90656,
            -321.73798,
            -320.53204,
            -321.2072,
            -322.30048,
            -324.14563,
            -320.90256,
            321.32736,
            -322.88644,
            -323.14096,
            -322.37656,
            322.11227,
            321.1233,
            320.94406,
            -321.5468,
            322.95813,
            320.09,
            320.72983,
            -320.60446,
            322.87164,
            -322.08014,
            -321.416,
            -322.44067,
            321.90427,
            -321.4313,
            323.09607,
            321.37082,
            320.32135,
            321.95435,
            -321.22626,
            -320.893,
            320.72067,
            322.43927,
            320.6582,
            322.0574,
            321.32892,
            -322.07547,
            319.90964,
            -322.31232,
            319.73526,
            321.36194,
            322.76083,
            -320.63657,
            319.9881,
            321.8368,
            -320.3449,
            322.82227,
            321.17786,
            -321.8329,
            320.51175,
            322.95413,
            321.49026,
            322.07547,
            -321.6477,
            323.3542,
            320.852,
            321.35413,
            320.05478,
            322.59174,
            320.89618,
            -320.61063,
            320.59195,
            -318.15366,
            -320.71124,
            321.14813,
            -320.6016,
            -319.08786,
            321.8272,
            321.0655,
            -321.92703,
            -322.09204,
            320.1163,
            318.14047,
            -320.2788,
            320.10098,
            318.83362,
            -319.92267,
            321.96353,
            321.63644,
            -320.19925,
            321.41354,
            -320.41064,
            -321.86792,
            -319.66373,
            319.90167,
            321.33603,
            320.34448,
            -318.47794,
            320.99008,
            320.5879,
            320.8086,
            -320.80847,
            321.5076,
            320.36368,
            320.0229,
            -320.51266,
            318.49716,
            320.38657,
            -321.13287,
            321.06757,
            -320.32904,
            -320.1782,
            -319.60272,
            320.24515,
            -319.2161,
            -319.64072,
            320.7557,
            321.17456,
            -319.52734,
            -319.0529,
            -320.08282,
            -322.8562,
            319.54565,
            -320.52603,
            319.72763,
            -319.9375,
            -320.57535,
            -318.6373,
            -319.0069,
            319.5041,
            -319.07895,
            318.07306,
            -320.1642,
            -319.59183,
            -321.07788,
            318.7678,
            318.6707,
            -321.29105,
            318.80408,
            317.87708,
            -320.0367,
            320.25723,
            320.15613,
            320.20596,
            318.81595,
            318.15668,
            319.91235,
            319.83896,
            319.64206,
            -319.43484,
            317.66058,
            319.22067,
            320.32043,
            -318.723,
            319.5402,
            -319.21844,
            -319.56827,
            318.5327,
            318.9296,
            318.2437,
            318.33054,
            -317.8854,
            -318.96973,
            -318.8698,
            -319.8394,
            -319.1516,
            318.6258,
            320.68533,
            318.46948,
            -319.18997,
            318.15063,
            316.6237,
            318.20267,
            318.53256,
            318.32062,
            -318.60358,
            317.80267,
            -318.53543,
            320.15448,
            -318.3831,
            -319.55234,
            -320.16943,
            319.48196,
            319.66162,
            320.8564,
            -317.45352,
            317.8928,
            318.9875,
            319.25793,
            318.76306,
            317.76956,
            318.39282,
            -319.06818,
            -319.0284,
            317.84003,
            -318.81714,
            317.7355,
            318.20914,
            -317.3325,
            317.18585,
            319.15955,
            319.90036,
            317.51978,
            318.62054,
            -318.9678,
            -320.1683,
            -318.4889,
            318.77353,
            319.98074,
            -318.52563,
            318.15607,
            317.07565,
            319.20535,
            318.03452,
            320.1463,
            317.46933,
            -318.98282,
            -317.47522,
            316.3276,
            318.13812,
            -318.74677,
            317.6113,
            -317.74365,
            318.74863,
            -316.7585,
            -317.8091,
            318.87094,
            -318.91043,
            317.0512,
            -317.0691,
            -318.0215,
            318.3123,
            318.7722,
            -317.38385,
            -318.78818,
            318.47098,
            -318.9041,
            -318.84735,
            319.50867,
            320.03442,
            -318.1365,
            318.2172,
            318.05698,
            -319.46667,
            -317.7429,
            317.4056,
            -318.87427,
            319.0087,
            318.85803,
            -318.05887,
            -319.4277,
            317.7459,
            317.27017,
            -317.98926,
            317.64194,
            -318.20386,
            -317.42932,
            317.28128,
            -319.18802,
            316.72794,
            318.65182,
            -317.66443,
            317.6055,
            -316.27908,
            -319.40146,
            -317.62463,
            317.91174,
            317.3144,
            318.72842,
            -317.39224,
            -317.0166,
            -317.10825,
            -317.35034,
            -316.57037,
            317.2088,
            316.9732,
            -318.9888,
            -319.28168,
            -317.82816,
            316.9555,
            317.36093,
            318.24,
            317.95148,
            319.02423,
            -317.91885,
            -317.04654,
            317.26425,
            -317.36346,
            316.40686,
            -316.60965,
            315.8258,
            -315.34137,
            316.59976,
            -316.29474,
            -317.16293,
            318.40134,
            317.05972,
            -316.76642,
            318.51608,
            317.23538,
            316.44116,
            316.71027,
            315.8662,
            -318.14517,
            -315.4306,
            315.88513,
            -315.83588,
            -314.94366,
            -316.48947,
            -314.12238,
            317.78726,
            -318.46603,
            -315.39026,
            316.74747,
            316.8934,
            315.3022,
            317.41336,
            316.43542,
            -316.14996,
            316.38904,
            315.8125,
            -317.1197,
            -316.37317,
            317.1263,
            -316.0972,
            314.68683,
            -317.09195,
            316.73578,
            -315.7379,
            315.8381,
            316.46674,
            -313.57626,
            -316.38538,
            -315.99127,
            317.2411,
            -316.16812,
            316.5346,
            315.29407,
            -317.56082,
            -316.77563,
            315.5798,
            317.3617,
            314.1478,
            315.48602,
            316.03323,
            -316.52957,
            -316.799,
            315.8419,
            -316.1263,
            317.05768,
            316.50403,
            316.91467,
            316.3281,
            316.65265,
            318.51736,
            -314.64557,
            318.07507,
            -317.0261,
            316.7952,
            315.3614,
            -315.0397,
            315.67734,
            316.33868,
            -317.1431,
            -316.75723,
            318.08975,
            315.6554,
            -316.63562,
            315.37097,
            -315.65564,
            316.3064,
            -315.5106,
            315.18503,
            316.60483,
            -316.4164,
            -315.43964,
            315.02158,
            315.66144,
            316.9276,
            -315.18866,
            -315.68564,
            -314.27002,
            315.35895,
            316.19696,
            316.80914,
            316.09586,
            315.74792,
            315.79385,
            317.07162,
            315.79037,
            315.6222,
            -314.83997,
            -314.67007,
            -317.1352,
            -315.78455,
            -313.31546,
            -315.82367,
            -314.84668,
            -314.75937,
            316.27505,
            314.51355,
            -313.51053,
            -314.74512,
            316.18997,
            315.59802,
            -314.42514,
            -314.42703,
            314.96445,
            -314.5428,
            314.98566,
            -316.874,
            315.1346,
            -314.57043,
            -315.56213,
            315.6599,
            314.69638,
            -317.2451,
            -315.706,
            314.8164,
            -314.20142,
            -316.05814,
            315.87402,
            -312.0444,
            315.89673,
            315.1342,
            316.14377,
            314.35477,
            315.5457,
            -314.61896,
            -314.5924,
            314.58072,
            -313.5753,
            313.37787,
            314.596,
            315.86026,
            -316.10413,
            315.73068,
            -315.46066,
            313.62616,
            -314.04245,
            -313.91644,
            315.31506,
            -316.62766,
            314.25656,
            314.4407,
            -315.76245,
            -314.34518,
            314.06155,
            -314.41663,
            -314.6189,
            314.64352,
            312.79227,
            -314.18805,
            -314.57733,
            -314.2413,
            -314.87183,
            313.79102,
            315.24713,
            315.17123,
            314.7242,
            314.5335,
            312.34207,
            315.8554,
            314.00647,
            -314.87466,
            314.44135,
            314.4038,
            -312.80237,
            -313.62848,
            314.6766,
            314.863,
            -315.17166,
            -314.97385,
            314.27975,
            -314.70905,
            313.65286,
            313.81934,
            315.35684,
            316.04684,
            -311.9404,
            313.56064,
            -313.9582,
            -311.999,
            314.19617,
            315.3231,
            -315.1473,
            313.60947,
            -316.0658,
            -313.14694,
            314.94873,
            315.08887,
            313.3025,
            314.355,
            -314.5547,
            -314.0895,
            313.67877,
            -314.15604,
            -313.32642,
            -315.45486,
            -314.62115,
            -313.15732,
            -313.3831,
            -314.51904,
            -313.86636,
            -314.0142,
            314.76016,
            -315.23746,
            312.83093,
            -316.47937,
            -313.68332,
            314.09882,
            -315.81848,
            -313.12534,
            -313.69986,
            -314.19318,
            -314.13974,
            -313.44186,
            -313.17676,
            313.52155,
            -313.92133,
            -314.92487,
            -314.42892,
            -314.18747,
            314.1149,
            313.61612,
            313.2649,
            -312.64175,
            312.88367,
            313.11853,
            -314.69974,
            -312.55222,
            312.58798,
            311.966,
            312.40594,
            312.57404,
            314.24435,
            313.36874,
            -312.0679,
            -313.01312,
            312.7999,
            313.2008,
            313.9382,
            312.12317,
            313.85858,
            -311.9702,
            -313.10934,
            313.3002,
            312.49908,
            -313.3934,
            -311.77762,
            313.29294,
            -314.19974,
            -313.5033,
            314.52234,
            312.09616,
            313.62518,
            312.72116,
            312.96863,
            313.6382,
            -313.5767,
            312.34244,
            313.50897,
            312.3239,
            -313.6829,
            -312.61267,
            -313.8364,
            311.7654,
            -313.05505,
            -310.45212,
            -314.25577,
            -313.57956,
            -313.26636,
            311.529,
            -312.05157,
            -311.71765,
            -312.36966,
            -314.31345,
            -312.38953,
            314.22165,
            -311.57077,
            -311.39966,
            -312.38232,
            313.86597,
            313.76193,
            -312.0037,
            -312.39554,
            -312.62863,
            -312.56445,
            311.48654,
            -312.5986,
            312.39008,
            -312.71396,
            311.62665,
            -312.14447,
            312.5164,
            312.837,
            311.71735,
            -312.99557,
            313.09427,
            -312.77878,
            -312.93018,
            -312.57092,
            311.67844,
            312.10602,
            -312.76218,
            311.7258,
            311.47754,
            -312.82944,
            312.40222,
            311.85913,
            311.4829,
            311.4627,
            312.38342,
            -312.8846,
            -311.61118,
            312.82126,
            -312.0695,
            -313.3853,
            313.78363,
            311.0977,
            -311.0693,
            -312.84256,
            -312.52823,
            -312.06598,
            312.71204,
            -312.79285,
            313.317,
            -312.1976,
            312.38593,
            310.49548,
            312.24683,
            -314.6552,
            312.19666,
            -312.15427,
            312.98953,
            -311.89923,
            312.06833,
            309.35703,
            311.09885,
            -312.0933,
            -311.63974,
            310.80106,
            -312.70538,
            -310.32404,
            -311.86172,
            311.59537,
            -311.04367,
            311.44205,
            312.51572,
            -312.55255,
            313.14175,
            311.9979,
            -312.2837,
            311.52393,
            310.85968,
            -311.5807,
            311.45938,
            -311.84393,
            313.4547,
            310.0102,
            311.8354,
            -310.67474,
            311.66678,
            -312.57697,
            311.80585,
            310.3874,
            -311.68442,
            311.78018,
            309.7177,
            312.55872,
            313.16882,
            312.91776,
            -311.29968,
            311.4292,
            -312.13147,
            310.94452,
            -312.4877,
            310.58615,
            -312.64935,
            -310.3217,
            -312.39697,
            309.91165,
            309.27402,
            310.62338,
            -309.7605,
            310.96707,
            -311.27643,
            -312.12393,
            -310.73883,
            310.10684,
            -311.1507,
            -310.95685,
            -311.85355,
            309.31894,
            311.29163,
            -310.1714,
            312.46808,
            311.2276,
            -311.704,
            312.7542,
            311.58774,
            311.82916,
            -311.10355,
            -312.2336,
            309.71674,
            -311.96964,
            311.58533,
            308.70465,
            309.5827,
            311.7971,
            309.7999,
            310.92242,
            -310.66516,
            310.7001,
            312.49768,
            -311.89597,
            -311.4818,
            -311.19177,
            311.57455,
            -309.59167,
            310.49344,
            310.22495,
            -311.7901,
            -311.41534,
            -311.66016,
            312.72046,
            309.22205,
            -311.26816,
            310.52582,
            -308.52136,
            311.26517,
            311.0447,
            310.743,
            -310.4849,
            -312.36172,
            311.61005,
            -311.4157,
            -310.38306,
            -309.41147,
            -311.58258,
            -309.6777,
            -311.08185,
            310.29346,
            311.5978,
            -311.00568,
            -309.50238,
            311.16574,
            310.04675,
            -311.80365,
            309.40982,
            -309.61807,
            311.04602,
            310.65845,
            -309.6807,
            -308.86963,
            311.26926,
            -309.85498,
            -309.4943,
            310.50906,
            312.217,
            308.96255,
            -310.213,
            -312.06708,
            310.43182,
            -311.6368,
            -310.40213,
            308.90637,
            308.91827,
            309.1892,
            311.553,
            -309.99335,
            -310.3867,
            -309.85995,
            -310.19876,
            -310.25577,
            309.7415,
            310.2111,
            310.3597,
            309.79636,
            310.74295,
            -310.552,
            312.18558,
            307.57294,
            308.6262,
            310.24576,
            -310.77496,
            308.4209,
            309.57843,
            -309.5976,
            308.02557,
            -309.38818,
            -310.85178,
            -308.3454,
            311.02457,
            310.55914,
            309.13522,
            311.69318,
            309.42645,
            -308.75385,
            308.1677,
            310.1233,
            310.66275,
            -309.3496,
            308.06476,
            -309.01883,
            -310.03473,
            -309.3167,
            -308.38766,
            308.86346,
            309.50198,
            308.85504,
            -310.02347,
            309.93033,
            308.27844,
            -310.08032,
            -308.94028,
            309.12155,
            -308.63617,
            309.25372,
            -309.3858,
            307.2476,
            307.33505,
            -308.53903,
            308.4978,
            -309.3644,
            308.55032,
            -307.97827,
            -308.32425,
            309.4576,
            -309.0083,
            -307.7417,
            -309.04153,
            308.25067,
            308.80594,
            308.6248,
            -308.46695,
            -309.89493,
            -309.29828,
            -309.31934,
            309.55826,
            -308.45755,
            309.4602,
            -308.5115,
            309.16708,
            305.56827,
            -309.05252,
            -307.25082,
            308.97055,
            308.51276,
            -308.71024,
            -308.7203,
            -308.2637,
            -308.2953,
            308.57184,
            308.03888,
            -308.40155,
            309.04385,
            -308.64206,
            307.70132,
            310.90955,
            309.49106,
            309.7027,
            308.39584,
            -307.52142,
            -307.07767,
            308.1839,
            -309.0373,
            -308.28955,
            308.8399,
            -306.91943,
            306.80475,
            -308.21954,
            306.66177,
            305.39334,
            308.11612,
            -307.67368,
            -306.64575,
            309.01816,
            309.70587,
            306.83167,
            -307.34277,
            308.61832,
            308.58228,
            307.95224,
            308.3516,
            307.60867,
            309.08444,
            307.6709,
            -307.70187,
            309.13043,
            -307.52957,
            -308.81287,
            306.41177,
            307.50626,
            -307.7245,
            306.21146,
            -307.18335,
            -308.44525,
            -307.99887,
            -306.66226,
            -308.66772,
            306.4527,
            307.05963,
            307.98267,
            -307.9352,
            -307.44376,
            306.12094,
            306.4754,
            308.6427,
            307.66394,
            309.52298,
            -307.52997,
            307.05603,
            -307.12784,
            -307.52673,
            309.2906,
            -307.3641,
            -307.4852,
            -307.48483,
            307.0734,
            -306.74615,
            306.519,
            308.617,
            -306.0051,
            -306.8107,
            308.06378,
            306.88382,
            308.38254,
            305.1839,
            306.06454,
            -307.83545,
            306.80814,
            -306.1767,
            -307.57346,
            307.23666,
            307.3619,
            306.3565,
            307.0738,
            307.67752,
            305.47562,
            -308.4436,
            -309.01138,
            307.1519,
            -308.07535,
            306.32556,
            -307.8972,
            -306.92245,
            -306.56525,
            308.36426,
            -308.094,
            -305.45157,
            -307.3038,
            305.87183,
            306.89914,
            -307.28867,
            306.52728,
            307.53912,
            306.43866,
            -307.70667,
            308.06036,
            306.3274,
            -305.85873,
            306.22156,
            308.73706,
            306.22583,
            -305.67487,
            307.77954,
            307.18823,
            305.68204,
            306.4079,
            306.38303,
            -305.49182,
            -305.5776,
            306.75565,
            -306.45193,
            306.64868,
            -306.22665,
            -306.10938,
            306.29453,
            306.6591,
            -306.41718,
            307.58896,
            306.79044,
            -305.81607,
            306.48578,
            -305.51407,
            304.94278,
            306.1882,
            306.70392,
            -304.819,
            307.64938,
            -306.04742,
            305.13556,
            304.79764,
            -307.04852,
            306.45947,
            -305.6531,
            -307.24924,
            -307.99048,
            -306.52472,
            306.69293,
            -306.7436,
            307.32785,
            -307.4214,
            305.80145,
            305.6163,
            305.2976,
            -305.9597,
            -306.09122,
            -305.35947,
            304.7509,
            306.11182,
            -305.25897,
            -305.7116,
            306.21893,
            -305.92334,
            -304.86542,
            -305.7924,
            306.3133,
            305.12494,
            -304.49985,
            -304.84985,
            -303.8963,
            -306.32397,
            -305.2692,
            -305.42783,
            305.78918,
            306.59552,
            -305.266,
            -305.8638,
            305.5634,
            304.73035,
            304.82257,
            303.209,
            305.5427,
            -306.09637,
            -304.70828,
            304.31082,
            -306.10974,
            -303.44696,
            306.86893,
            -304.10425,
            -304.70236,
            305.54962,
            306.76868,
            -303.13785,
            306.0067,
            306.91028,
            306.93533,
            305.3595,
            305.8713,
            304.94604,
            305.25238,
            305.12683,
            303.65903,
            304.62814,
            305.26233,
            305.63885,
            -304.6767,
            -305.14362,
            -306.59167,
            304.03113,
            -305.12238,
            -306.1,
            306.59024,
            304.51566,
            305.05872,
            305.49948,
            -303.61288,
            305.02762,
            304.86346,
            303.3372,
            305.00018,
            -306.46298,
            304.57172,
            -305.25833,
            -305.89615,
            304.79193,
            304.26694,
            303.3369,
            305.279,
            -304.27438,
            301.39514,
            -304.11703,
            303.70096,
            304.7631,
            -303.93088,
            304.74152,
            -303.93015,
            -304.7626,
            304.20605,
            304.116,
            303.5815,
            303.67358,
            -304.53247,
            305.44064,
            -304.91788,
            304.2001,
            -305.15402,
            304.107,
            -304.8087,
            -303.07813,
            304.61365,
            305.4858,
            303.34158,
            -304.07022,
            303.82254,
            -304.1523,
            304.7421,
            -304.21692,
            -303.48056,
            -303.15674,
            -303.56122,
            302.89246,
            -304.7744,
            -304.6831,
            304.46075,
            304.68726,
            303.3237,
            -304.22607,
            -302.86407,
            -304.12186,
            303.3882,
            -305.5305,
            303.06354,
            -305.1239,
            -303.14645,
            304.86642,
            -303.3391,
            -304.16843,
            -305.03485,
            301.05414,
            -304.44513,
            -303.94287,
            -304.20584,
            303.2933,
            302.70084,
            -302.97723,
            -303.5498,
            303.05768,
            -302.99435,
            -303.33673,
            -302.9571,
            303.11063,
            -302.98584,
            -303.61163,
            -301.59122,
            302.5328,
            302.93597,
            -302.82687,
            -302.64502,
            303.96933,
            -303.2308,
            -303.25668,
            303.64536,
            302.8984,
            -303.015,
            -301.9444,
            303.72705,
            303.79288,
            303.5533,
            303.5211,
            303.36386,
            303.33804,
            -303.34686,
            -303.17496,
            -304.62628,
            303.2466,
            303.1925,
            303.37204,
            301.65665,
            302.72382,
            -303.50748,
            301.436,
            -303.1104,
            -303.03506,
            -304.72504,
            303.55252,
            -303.2816,
            -302.5141,
            -304.08435,
            -303.3564,
            303.93283,
            303.34586,
            -303.19556,
            302.2323,
            -303.88306,
            304.74463,
            -303.3706,
            -303.94403,
            301.63116,
            -302.41782,
            -302.95667,
            303.06442,
            303.24683,
            -302.48297,
            -302.75317,
            -301.66977,
            301.95728,
            -302.02258,
            -302.69293,
            303.20847,
            -302.28317,
            302.77615,
            303.4303,
            301.8124,
            301.17673,
            -302.6948,
            -302.13095,
            302.18942,
            302.29062,
            302.1198,
            302.72794,
            303.72687,
            303.25684,
            -302.47513,
            -301.8101,
            302.5971,
            301.6626,
            -303.34973,
            302.11273,
            -302.658,
            -302.92853,
            -301.82687,
            -303.42923,
            302.4051,
            303.84445,
            -302.42328,
            300.9907,
            300.7793,
            301.65448,
            302.50696,
            302.21848,
            -301.45728,
            303.9806,
            -301.3256,
            302.13425,
            302.68118,
            -302.6243,
            -302.7515,
            300.90607,
            -301.70096,
            302.92493,
            301.8321,
            -301.44275,
            301.18555,
            303.4915,
            301.161,
            -301.45163,
            300.17374,
            -300.8701,
            -301.61148,
            302.024,
            -301.2557,
            302.67892,
            -302.10397,
            -301.1465,
            302.71176,
            301.15338,
            -302.17828,
            300.6226,
            -301.42218,
            301.80432,
            302.8259,
            302.34677,
            301.54916,
            301.6042,
            301.5605,
            -301.90533,
            300.55203,
            302.1209,
            -301.80685,
            -302.3328,
            303.1036,
            303.38477,
            -301.81995,
            301.40207,
            301.6138,
            302.9211,
            300.36993,
            -302.10403,
            -301.72736,
            -301.24814,
            302.2412,
            302.29715,
            302.25888,
            -300.5451,
            -301.67703,
            300.53513,
            300.4627,
            300.6867,
            -300.86533,
            300.78436,
            299.83377,
            302.1718,
            301.72318,
            301.2948,
            301.9133,
            300.55847,
            -300.50195,
            301.03223,
            301.65213,
            301.3785,
            -302.26147,
            -301.31863,
            301.11105,
            -301.57828,
            -301.7742,
            -301.79706,
            -300.9911,
            302.168,
            301.77634,
            -299.069,
            300.5968,
            299.85245,
            -301.62726,
            301.22308,
            -301.3475,
            -299.86484,
            300.51486,
            -300.75433,
            300.88788,
            300.09634,
            300.71332,
            301.20712,
            -301.82864,
            -299.65057,
            -299.28314,
            -300.43304,
            -302.51917,
            -301.30875,
            -300.91144,
            300.11597,
            -300.25177,
            -300.32178,
            301.53302,
            -299.5244,
            -300.10562,
            -301.43945,
            301.02768,
            299.33084,
            300.0301,
            -301.48984,
            -301.6594,
            300.05756,
            299.42596,
            300.63962,
            -300.39597,
            300.277,
            300.24814,
            300.97095,
            300.4749,
            299.4627,
            -300.4111,
            -300.91418,
            299.33264,
            299.03955,
            -300.45084,
            -300.13828,
            300.1785,
            -300.8704,
            299.79773,
            301.21555,
            302.47498,
            -300.67862,
            -299.48737,
            -300.57883,
            298.408,
            -298.3855,
            299.10327,
            298.92636,
            -299.21216,
            -299.54745,
            299.9352,
            -300.73117,
            -297.61374,
            300.35056,
            -298.3225,
            301.21353,
            -299.5491,
            -297.99435,
            -299.6787,
            297.51077,
            -300.26663,
            -300.6573,
            299.14767,
            298.90643,
            -299.33328,
            -300.70676,
            299.7972,
            -299.683,
            299.83252,
            299.39206,
            300.8364,
            298.06052,
            -301.0634,
            299.9837,
            299.93027,
            -300.84662,
            298.84875,
            300.85522,
            -299.922,
            299.57483,
            300.7303,
            299.00354,
            299.58624,
            -298.61948,
            300.093,
            300.8812,
            301.46722,
            -297.77307,
            -298.71985,
            299.62628,
            297.72516,
            -300.9918,
            -298.29343,
            298.3992,
            301.25363,
            300.09595,
            297.62173,
            -299.2788,
            300.57608,
            -300.09238,
            -299.28577,
            -300.6583,
            297.67816,
            299.6354,
            -299.2209,
            299.4506,
            299.29938,
            -299.13766,
            -299.50806,
            -300.72958,
            -298.73526,
            296.70392,
            298.3498,
            299.3506,
            -299.07364,
            299.001,
            -298.7995,
            -299.33688,
            298.2221,
            298.16953,
            299.52258,
            -299.33368,
            -297.24402,
            -301.59244,
            297.30414,
            -296.16724,
            -300.2055,
            -298.42975,
            -299.17706,
            -300.15576,
            299.53912,
            -298.40594,
            298.2866,
            297.7644,
            297.8774,
            -299.2024,
            298.31796,
            -300.44067,
            298.21948,
            298.44882,
            -296.6735,
            -299.46918,
            300.07657,
            298.979,
            298.78815,
            -298.47012,
            297.63245,
            297.25983,
            296.30103,
        ];
        let subdiag = [
            75966.87,
            90517.0,
            -28600.053,
            -44805.504,
            50779.082,
            48102.4,
            26916.666,
            30553.555,
            -40445.168,
            -39799.355,
            -34119.758,
            -24623.617,
            29404.428,
            28766.83,
            23355.906,
            29435.387,
            -25354.402,
            20469.791,
            -23891.117,
            -21821.373,
            -25450.389,
            26994.742,
            18536.871,
            20931.373,
            -20581.998,
            -23872.352,
            -16928.137,
            -14715.533,
            -15743.262,
            17003.936,
            24126.986,
            16676.133,
            -15073.163,
            14225.849,
            -14915.264,
            14920.54,
            12771.989,
            11517.561,
            14307.151,
            -14231.069,
            -12184.051,
            -11663.259,
            -13100.836,
            12602.486,
            -12420.986,
            12235.433,
            -12686.976,
            11934.926,
            -13021.714,
            -12590.817,
            10031.526,
            9575.626,
            9809.1455,
            11332.403,
            -11618.651,
            -11271.73,
            10204.625,
            -9300.663,
            -10251.524,
            10835.755,
            8539.145,
            10642.64,
            9247.583,
            9284.797,
            -10583.971,
            -9560.75,
            -9759.821,
            9235.616,
            10231.511,
            -8999.203,
            8189.451,
            -9394.819,
            -9341.68,
            -8492.3955,
            9081.474,
            -8484.515,
            -8814.421,
            9017.944,
            -9305.994,
            8489.593,
            -8551.214,
            8824.323,
            7878.0757,
            8615.374,
            -7379.3647,
            7500.9946,
            8301.65,
            7777.96,
            7641.793,
            -7766.567,
            -7199.7266,
            7553.522,
            7826.3306,
            7629.1504,
            6765.0903,
            7058.0737,
            7470.6826,
            6884.582,
            6882.6304,
            6416.8223,
            6921.4487,
            -6989.563,
            -7493.316,
            -6094.396,
            -6738.9937,
            -6680.726,
            -7408.283,
            -6731.482,
            -6461.8696,
            -6231.1416,
            6232.864,
            6712.115,
            6970.6646,
            6307.572,
            6240.116,
            5783.8184,
            -6248.868,
            -6372.468,
            6556.387,
            -6282.3696,
            6114.355,
            -6171.378,
            6108.3213,
            -6119.2227,
            5893.202,
            -5937.6367,
            -5686.6704,
            -6050.398,
            5571.7837,
            -5895.436,
            -6239.093,
            -5704.5693,
            -5393.8164,
            5831.407,
            5539.9844,
            -6015.4287,
            5699.5264,
            -5359.445,
            -5101.632,
            -5674.814,
            5343.981,
            5539.192,
            5266.602,
            -5491.2324,
            -5107.81,
            5456.721,
            5365.126,
            -5594.778,
            5289.067,
            -5267.937,
            -5920.7217,
            -5140.093,
            -5169.8247,
            4593.8706,
            -5442.5195,
            5188.0967,
            5090.299,
            -5103.1074,
            -4682.722,
            4798.519,
            -4224.941,
            5298.6567,
            -4701.668,
            -5368.4385,
            4683.02,
            -4670.231,
            5140.556,
            4734.6816,
            -4849.219,
            5012.7324,
            4968.144,
            4282.728,
            4413.4087,
            -4489.7603,
            -5359.7886,
            5031.1724,
            4755.533,
            4660.5156,
            4435.3877,
            -4685.307,
            4759.2676,
            4701.5986,
            -4648.786,
            -4754.1387,
            4474.995,
            -4282.2983,
            4378.597,
            -4665.7505,
            -4682.899,
            -4606.5825,
            4201.1807,
            4756.5054,
            4367.8564,
            -4521.883,
            -4496.5645,
            -4247.077,
            -4413.6245,
            4471.181,
            4326.507,
            -3950.5203,
            4369.902,
            4165.1333,
            -4267.9253,
            4467.994,
            4284.582,
            4320.2915,
            4503.497,
            -4401.7188,
            -4306.4097,
            -4182.373,
            4233.3774,
            -3753.4172,
            3968.8687,
            3987.0168,
            -4256.263,
            4041.56,
            -4386.8066,
            -3975.9802,
            3917.4983,
            -3990.285,
            4159.545,
            -4262.278,
            -4237.0815,
            4063.2275,
            4089.7197,
            -3943.6565,
            -3919.5388,
            -4290.0146,
            -3986.2727,
            3983.634,
            3668.6067,
            3805.6885,
            3896.2878,
            -3606.2083,
            3830.3342,
            -4023.7341,
            -3772.668,
            3863.706,
            -3627.1116,
            -3972.9326,
            3759.0537,
            3738.65,
            3644.0435,
            3792.1826,
            4037.8025,
            -3738.6873,
            3696.268,
            3743.8335,
            -3947.1682,
            -3521.272,
            -3973.97,
            3643.545,
            3686.1704,
            -3427.8552,
            3858.35,
            -3668.4011,
            -3747.2126,
            -3608.7268,
            3476.265,
            -3967.9053,
            3560.937,
            3480.1519,
            3434.5608,
            3682.8264,
            -3560.393,
            3360.9072,
            3596.9565,
            3616.14,
            -3437.7412,
            -3449.3667,
            3496.1423,
            3725.0366,
            -3399.7354,
            3425.3335,
            -3347.7554,
            3285.602,
            3322.7656,
            3365.7864,
            3734.5884,
            -3387.8274,
            3421.9883,
            -3261.1,
            3345.2485,
            -3247.1128,
            -3105.3347,
            -3254.5571,
            -3370.9077,
            3277.5745,
            3234.8784,
            -3387.8809,
            3370.7788,
            3245.0574,
            3270.083,
            3184.9648,
            -3212.865,
            -3060.82,
            -3137.7937,
            3206.9055,
            3162.0874,
            3164.3413,
            3140.2502,
            -3175.9895,
            -3015.2815,
            -2930.3499,
            3203.553,
            3218.852,
            -3213.771,
            3165.8472,
            -3024.4329,
            -3252.4182,
            -2983.0205,
            3406.757,
            -3206.797,
            3012.7554,
            -2983.8792,
            -3230.1633,
            3006.7473,
            -3076.1467,
            2872.7324,
            2987.5532,
            3073.86,
            2985.7852,
            3185.9622,
            -2977.728,
            2912.0188,
            2941.084,
            3042.3865,
            -2898.9448,
            -2829.19,
            2830.12,
            2888.909,
            2876.1138,
            2958.3562,
            -2876.787,
            -2832.1826,
            -2887.1384,
            2946.308,
            -2984.41,
            2802.0405,
            -2953.3784,
            2980.7026,
            2778.6821,
            2948.8425,
            2836.9758,
            2975.0735,
            2836.758,
            -2691.3464,
            2759.2644,
            2993.8416,
            -2578.9888,
            2781.9438,
            2812.2603,
            -2756.9412,
            2791.7283,
            2860.762,
            -2924.9888,
            2854.9846,
            2792.0833,
            -2663.473,
            2830.5405,
            2811.3577,
            2727.0664,
            2779.0493,
            -2691.1357,
            -2686.6296,
            -2842.208,
            2794.3984,
            -2652.2954,
            -2634.4617,
            -2724.5232,
            -2590.9414,
            -2709.7583,
            2640.4165,
            -2591.071,
            2656.1016,
            -2799.0037,
            -2674.401,
            2712.53,
            -2830.7188,
            -2727.73,
            2544.9502,
            -2558.1606,
            -2472.3093,
            2583.72,
            2461.552,
            -2557.0337,
            2502.2876,
            -2597.522,
            -2667.7058,
            2566.9475,
            -2588.2776,
            -2409.991,
            2661.1252,
            -2469.6636,
            -2544.6687,
            2491.0251,
            2733.8381,
            -2463.8525,
            -2477.1812,
            -2485.971,
            -2703.6013,
            2379.3237,
            -2422.679,
            2515.019,
            2398.0955,
            2561.7146,
            2591.5217,
            -2426.0017,
            2428.828,
            2565.5652,
            -2479.7166,
            -2500.434,
            -2329.2012,
            -2363.0916,
            -2463.956,
            2364.883,
            2429.4463,
            2378.9321,
            -2380.6243,
            2531.1538,
            2569.548,
            -2566.1846,
            2512.8467,
            2345.085,
            -2457.4146,
            -2343.811,
            -2445.6794,
            2350.3948,
            -2329.8303,
            -2348.5256,
            -2386.5073,
            2376.581,
            -2329.2573,
            -2387.924,
            2313.8337,
            -2383.2014,
            -2298.3813,
            -2341.7815,
            -2469.6582,
            -2401.7979,
            2413.33,
            2461.3193,
            -2436.3286,
            2426.2407,
            -2340.0789,
            2342.7231,
            2226.729,
            2211.9106,
            -2297.0425,
            -2314.0237,
            2272.8452,
            2433.986,
            -2334.8923,
            -2237.899,
            -2319.6497,
            -2233.1912,
            2308.7634,
            -2343.54,
            2221.9785,
            2250.3733,
            2320.4404,
            2347.3308,
            -2242.453,
            -2199.1646,
            2242.688,
            2319.3816,
            2264.2402,
            -2186.0615,
            2127.2554,
            -2242.5337,
            2322.9873,
            -2142.85,
            2217.5513,
            -2171.133,
            -2159.0422,
            2166.7314,
            -2068.526,
            -2224.5012,
            -2085.3447,
            -2094.4426,
            -2061.936,
            -2221.6946,
            2137.1692,
            -2204.9983,
            2218.9941,
            -2135.0408,
            -2110.4585,
            2169.3777,
            -2201.4358,
            -2278.5457,
            2222.4207,
            -2137.725,
            2139.4358,
            2180.5784,
            -2153.271,
            -2115.0847,
            2138.6287,
            2190.2712,
            -2071.9675,
            -2105.4626,
            -2195.2014,
            2115.462,
            -2070.8293,
            2172.02,
            2103.9578,
            -2103.3174,
            -2111.6902,
            2151.7734,
            -2157.059,
            2159.894,
            -2092.1294,
            2069.9338,
            1987.2493,
            2098.4866,
            2069.6504,
            2060.4958,
            1982.3687,
            -2011.4601,
            -2030.8977,
            2184.5215,
            -2037.7922,
            -1996.3805,
            2101.8835,
            2060.0137,
            1957.7179,
            2030.387,
            -1986.922,
            2065.7617,
            1970.5001,
            -2071.9746,
            -2043.814,
            -1987.6627,
            1962.1224,
            -2033.972,
            1938.1678,
            -2009.4674,
            -2049.601,
            -2109.4326,
            -1951.2262,
            -1961.3093,
            -1986.3759,
            2049.954,
            -2073.4421,
            1961.9696,
            -1924.9506,
            1966.7172,
            -2079.9602,
            -1957.5316,
            1973.8041,
            1997.383,
            2042.5575,
            1988.8346,
            -2021.555,
            -1971.3925,
            -1969.3676,
            1856.1686,
            -2044.8733,
            1977.7422,
            -1916.1929,
            1971.5388,
            -1950.5314,
            -1912.4506,
            -1924.62,
            -1882.2915,
            -1928.3156,
            -1904.268,
            -1961.3215,
            1912.0209,
            -1938.1912,
            1992.2643,
            1923.7118,
            -1959.3447,
            1943.4867,
            -1977.4794,
            -1987.0962,
            -1911.6733,
            -1890.9773,
            -2016.4324,
            1874.4705,
            1916.7831,
            1980.9819,
            1818.3147,
            -1956.4725,
            1933.5768,
            1781.2113,
            1892.4445,
            1819.3871,
            1822.5913,
            -1910.1194,
            -1943.4374,
            -1854.5852,
            -1842.8431,
            -1832.5786,
            1892.0223,
            1874.3986,
            -1858.3289,
            -1813.3544,
            1873.7991,
            -1806.7831,
            1892.1947,
            -1887.9265,
            -1787.2659,
            1740.3956,
            -1892.5443,
            1789.3053,
            1847.1747,
            -1777.1863,
            -1860.3209,
            1844.7102,
            -1822.3488,
            -1847.2294,
            1816.4315,
            -1955.1589,
            1820.9723,
            -1849.8179,
            1833.705,
            1820.3489,
            1879.2974,
            1775.0114,
            1835.1735,
            -1825.0845,
            1739.3917,
            -1743.8708,
            1713.0599,
            1738.798,
            1811.0016,
            -1757.7524,
            -1787.3866,
            1776.096,
            -1702.8818,
            1797.8168,
            1770.7084,
            1721.1184,
            1707.301,
            -1803.3533,
            1709.6318,
            1733.0021,
            1755.4933,
            -1737.1233,
            1707.7036,
            -1791.3313,
            -1850.0905,
            1710.5916,
            -1694.9556,
            -1697.9791,
            1760.9066,
            1772.9174,
            -1722.3754,
            1748.1564,
            -1708.1188,
            1766.4559,
            1698.495,
            1735.8766,
            -1782.1647,
            1612.6394,
            1713.0577,
            1843.9197,
            1669.9109,
            1708.5271,
            1722.397,
            -1694.4882,
            1683.1492,
            -1731.7532,
            -1663.6658,
            -1689.9191,
            1716.7368,
            1709.1829,
            1677.2905,
            -1672.1974,
            1646.2596,
            1594.1847,
            -1690.162,
            1648.5586,
            1618.3663,
            -1680.9994,
            -1615.2839,
            -1668.3872,
            1634.7177,
            1636.6736,
            -1664.419,
            1600.3837,
            -1626.8927,
            1651.9952,
            1621.3624,
            -1717.1796,
            1629.6516,
            -1663.895,
            1698.2699,
            -1600.6716,
            -1613.0347,
            -1645.4004,
            -1641.8057,
            1622.6444,
            -1609.2346,
            -1621.7123,
            -1648.8066,
            -1620.3408,
            -1615.5999,
            1691.7794,
            1649.7057,
            1633.8075,
            1639.3812,
            1561.4551,
            -1632.5607,
            -1699.44,
            1659.4541,
            1598.3564,
            1609.1271,
            -1526.6447,
            1594.4784,
            -1604.031,
            1603.0837,
            1669.2059,
            1568.8635,
            -1558.5409,
            -1672.5272,
            1586.5554,
            -1509.304,
            1640.7487,
            1578.7006,
            -1553.229,
            -1667.6599,
            1594.5271,
            -1590.2114,
            -1569.401,
            1655.6072,
            -1649.3713,
            -1620.4308,
            1560.1105,
            -1483.9817,
            1539.6962,
            -1578.1958,
            -1548.1444,
            -1580.3805,
            -1637.3519,
            1570.6563,
            -1534.7075,
            -1519.3866,
            -1601.7408,
            -1520.3495,
            -1507.8942,
            1562.9414,
            1550.7627,
            -1522.6871,
            -1570.0298,
            -1497.2941,
            1538.0641,
            1564.7454,
            -1518.7836,
            1502.1222,
            1494.9332,
            -1492.4196,
            -1523.96,
            -1575.9282,
            -1470.7698,
            -1544.2068,
            1593.7372,
            -1514.077,
            1541.6749,
            1467.0776,
            -1551.3849,
            1588.2155,
            1502.3629,
            1514.9111,
            1475.1781,
            -1419.4452,
            -1527.9373,
            -1543.379,
            -1480.8685,
            1569.1091,
            -1456.7095,
            -1469.9805,
            -1502.7783,
            1568.8086,
            1465.4727,
            -1488.8335,
            -1558.8433,
            1553.4069,
            1459.187,
            1515.7396,
            1511.9324,
            -1402.1361,
            1444.0685,
            -1477.3513,
            -1459.0907,
            1467.5919,
            -1524.5343,
            -1444.9757,
            -1490.9318,
            1492.9144,
            -1432.7097,
            -1481.0493,
            1520.0177,
            -1410.7373,
            1496.8049,
            1517.3533,
            1519.4554,
            1505.4264,
            1514.8425,
            -1441.0369,
            -1432.8486,
            1464.4773,
            1427.3763,
            1519.7505,
            -1452.0697,
            -1472.9166,
            1451.1055,
            -1424.0175,
            1415.9226,
            1479.0134,
            -1472.8376,
            1419.2305,
            1466.432,
            -1424.1276,
            -1422.1293,
            1483.3932,
            -1439.0183,
            -1445.5137,
            -1412.7505,
            -1403.2493,
            -1419.8416,
            -1392.6016,
            1418.1589,
            -1398.7738,
            -1448.4144,
            -1404.6218,
            1442.2119,
            -1389.3212,
            1394.928,
            -1440.5267,
            -1411.8645,
            1397.9492,
            -1370.8884,
            1440.455,
            -1382.3372,
            1412.921,
            1355.8794,
            1399.095,
            -1400.8479,
            1347.8857,
            1459.9867,
            -1382.4615,
            -1369.4362,
            1381.3855,
            1343.5587,
            -1360.2468,
            1346.7677,
            1416.6532,
            1378.6458,
            -1352.9341,
            -1355.3132,
            -1386.4419,
            1351.2865,
            1321.5056,
            -1406.4244,
            1326.6855,
            1387.8809,
            1391.9213,
            -1311.4484,
            1354.9739,
            1353.0393,
            -1367.9047,
            -1368.4478,
            1381.3164,
            1360.0472,
            1308.6138,
            1369.6653,
            -1328.1051,
            1322.4615,
            1353.7562,
            -1394.3435,
            1294.2621,
            1299.0768,
            -1335.5044,
            1280.701,
            -1317.9142,
            1366.3431,
            1349.096,
            1321.2988,
            1397.3425,
            -1325.9457,
            1264.8564,
            1381.3484,
            1368.1151,
            1323.7443,
            1295.6609,
            1302.6626,
            -1346.5936,
            1341.6611,
            -1349.1821,
            -1289.7178,
            -1345.3038,
            -1299.4794,
            -1301.2695,
            -1321.6726,
            1309.729,
            1258.2902,
            -1314.2899,
            -1343.3467,
            1267.6145,
            1290.3438,
            -1288.8994,
            1288.4977,
            1310.1274,
            -1323.255,
            -1328.8846,
            -1311.8169,
            1348.3531,
            1312.6012,
            -1360.1887,
            -1284.0637,
            1327.3037,
            -1327.151,
            1340.2878,
            1302.1498,
            -1311.5167,
            1348.7522,
            -1297.0265,
            -1309.9618,
            -1354.6658,
            1265.6803,
            1310.1427,
            -1280.7719,
            1300.0614,
            -1322.1797,
            1280.0847,
            1277.1394,
            1306.5543,
            -1292.645,
            1301.1427,
            1309.2468,
            -1363.031,
            -1281.4784,
            1307.9956,
            -1318.0803,
            -1252.8295,
            1277.237,
            -1318.4832,
            -1287.7394,
            -1298.5842,
            -1224.2107,
            -1214.9476,
            -1235.1211,
            -1254.5786,
            -1239.7954,
            1280.556,
            -1250.7856,
            -1278.8466,
            1239.0165,
            1202.9562,
            -1200.6138,
            1253.6932,
            -1258.5322,
            -1289.6166,
            1239.1741,
            -1248.3689,
            1246.8994,
            -1199.8291,
            -1192.9878,
            1241.166,
            -1277.2994,
            1221.3418,
            -1289.8219,
            1241.5018,
            -1211.6877,
            1259.991,
            1224.1454,
            1240.6339,
            1246.1166,
            -1196.7902,
            -1244.6836,
            1201.2599,
            -1212.6063,
            -1239.6698,
            -1221.6451,
            -1206.4203,
            1200.6971,
            1271.67,
            1247.1593,
            1191.2291,
            -1260.1583,
            -1257.7704,
            1171.4323,
            -1242.0264,
            -1220.1563,
            1181.9545,
            -1243.7883,
            -1174.6276,
            -1189.1382,
            1218.17,
            -1197.1184,
            -1203.9617,
            -1194.4878,
            1225.1174,
            1231.5696,
            1169.939,
            1269.3234,
            -1183.6393,
            -1165.1526,
            -1202.0367,
            -1138.505,
            -1144.1512,
            -1212.4812,
            1158.5398,
            1182.114,
            -1191.1802,
            1146.5345,
            -1166.8276,
            -1173.728,
            -1194.282,
            -1186.1737,
            -1209.7018,
            1212.4667,
            -1170.4249,
            -1197.8907,
            1168.2759,
            1195.8561,
            -1213.7474,
            -1177.6385,
            -1172.0764,
            -1209.2926,
            -1151.6892,
            -1200.8351,
            -1164.142,
            -1196.2004,
            1155.4731,
            -1214.1022,
            1158.923,
            1158.5715,
            -1118.301,
            -1170.6516,
            1163.6149,
            -1161.64,
            1129.0493,
            -1124.9103,
            1199.8893,
            1177.0205,
            1179.7894,
            1155.55,
            -1148.094,
            -1163.0635,
            1197.8333,
            -1216.818,
            -1152.5652,
            1210.6123,
            1150.7087,
            -1132.4238,
            1149.1215,
            -1187.7256,
            1169.1359,
            -1158.2926,
            1148.499,
            -1147.5444,
            -1117.958,
            -1140.884,
            -1191.3677,
            -1140.6595,
            -1227.2756,
            1199.8982,
            -1151.2095,
            1186.2502,
            1166.9763,
            -1141.225,
            -1187.4932,
            1129.7269,
            1109.5872,
            -1134.8975,
            1101.6261,
            1110.8146,
            -1071.789,
            1138.0626,
            -1084.1255,
            -1120.6517,
            -1141.1473,
            -1108.6664,
            1083.2842,
            -1105.1094,
            1136.0034,
            -1134.7565,
            -1109.9506,
            1088.6095,
            1104.6171,
            1130.9067,
            1116.9083,
            -1140.5509,
            1118.5634,
            -1132.1016,
            1128.7258,
            -1138.1904,
            -1112.5322,
            1110.1277,
            1125.5737,
            -1101.0524,
            -1104.765,
            -1073.9955,
            -1122.2388,
            1095.0808,
            1097.8473,
            -1077.4949,
            -1108.7399,
            -1093.4358,
            -1091.5922,
            1099.2377,
            1077.8917,
            1058.1016,
            1094.6434,
            -1090.3507,
            1107.7021,
            -1081.9889,
            -1088.3239,
            1086.0898,
            -1087.7128,
            1098.0082,
            -1074.9828,
            -1059.8253,
            1088.3362,
            1074.8004,
            1055.6344,
            -1100.1449,
            -1117.9641,
            1103.632,
            -1093.6831,
            1052.3356,
            -1060.8497,
            1071.9646,
            -1098.6534,
            1077.1373,
            1091.0853,
            -1097.2826,
            -1108.0961,
            1126.1135,
            -1089.3844,
            -1068.5065,
            1072.6292,
            -1065.2595,
            1056.3191,
            -1089.1154,
            1058.5831,
            1067.4773,
            1061.6146,
            1124.3679,
            1074.359,
            -1086.2245,
            -1042.796,
            -1042.2423,
            -1071.3761,
            -1062.4995,
            -1067.2408,
            1064.139,
            1075.7874,
            1019.681,
            -1102.8412,
            1037.0356,
            -1065.6792,
            1063.0665,
            -1079.0427,
            1035.2803,
            1039.9336,
            -1032.6302,
            1029.1039,
            -1006.3442,
            1055.7246,
            -1009.4264,
            1034.0471,
            -1041.6797,
            -985.44354,
            -993.8976,
            -1037.1179,
            -994.85895,
            -1057.6543,
            -1073.4454,
            -1047.2976,
            1041.8835,
            999.69916,
            -1038.5167,
            1050.1166,
            1075.286,
            1030.8494,
            1068.7583,
            -1022.7143,
            -1051.6154,
            -1030.5359,
            -1030.8047,
            -1015.1958,
            -1038.888,
            1017.6914,
            -1023.56793,
            1028.0801,
            -1011.50934,
            1049.6906,
            1018.67163,
            -1022.40546,
            1040.1235,
            1052.4799,
            1036.0823,
            -1022.46545,
            1038.6097,
            1010.9425,
            1043.396,
            -1027.3693,
            -979.1462,
            1037.1039,
            -994.0493,
            -998.8622,
            1029.117,
            -985.8295,
            -1016.6547,
            -1006.1349,
            -1033.272,
            -1018.1792,
            1003.50195,
            965.4858,
            1024.1361,
            1020.08136,
            973.1629,
            989.488,
            -1023.57983,
            1033.4878,
            -997.98206,
            1015.4279,
            979.4022,
            1009.49927,
            -1027.756,
            -998.1713,
            989.78955,
            932.4535,
            997.1778,
            -993.5128,
            -981.12384,
            -1006.7811,
            1012.1561,
            1005.02527,
            998.154,
            988.47345,
            974.5563,
            952.5747,
            -1005.697,
            974.66,
            991.49176,
            -967.69104,
            1000.2308,
            -973.94635,
            -968.0013,
            -972.2719,
            -962.7948,
            -967.3038,
            -994.7712,
            980.18475,
            998.3108,
            983.38,
            -1007.7246,
            1010.929,
            -968.9214,
            965.5831,
            -995.90344,
            -964.80176,
            945.7481,
            -965.6134,
            -977.92566,
            -956.50995,
            965.5732,
            -1004.4384,
            1015.55554,
            -1045.285,
            1009.961,
            -967.79156,
            996.4375,
            956.4088,
            968.8596,
            -989.6403,
            -953.6021,
            923.4807,
            955.4564,
            -979.387,
            -932.50323,
            -968.53723,
            -975.2856,
            917.33105,
            -960.20703,
            -974.76495,
            962.1534,
            -943.7062,
            -981.59357,
            959.5864,
            990.1094,
            973.4367,
            -946.1164,
            -928.7814,
            964.31885,
            965.0164,
            927.51636,
            -980.87555,
            984.8444,
            961.70245,
            -946.07434,
            954.49677,
            -950.6183,
            933.282,
            -941.9696,
            946.31146,
            -929.58075,
            942.0399,
            -915.4528,
            918.5685,
            920.67633,
            935.8459,
            -918.71625,
            936.9712,
            -926.30194,
            931.1625,
            -906.35455,
            946.9437,
            966.7732,
            -911.3971,
            958.5495,
            -929.32623,
            916.31055,
            929.74817,
            -942.19965,
            943.6679,
            -948.6041,
            -916.2319,
            921.03876,
            -928.6614,
            -928.09045,
            932.407,
            923.7414,
            -912.6342,
            -924.6006,
            927.17395,
            935.0406,
            -902.506,
            898.6371,
            925.5028,
            -927.1386,
            -913.93915,
            -903.6713,
            900.3559,
            -933.2694,
            -924.4088,
            -927.8999,
            -933.86847,
            -914.448,
            -934.1565,
            -944.68,
            917.2365,
            928.0723,
            911.37024,
            -931.2944,
            -899.94714,
            895.09863,
            -914.2364,
            -896.0613,
            882.4605,
            -905.88745,
            906.2961,
            904.87286,
            -904.7255,
            -889.5035,
            888.64716,
            -876.00006,
            867.89374,
            -891.6064,
            919.4336,
            -884.2657,
            -881.22485,
            -913.0886,
            907.88586,
            901.8803,
            906.45135,
            -900.3227,
            -915.31226,
            -897.18164,
            883.10223,
            895.2043,
            867.74164,
            903.3214,
            -893.15955,
            -895.6075,
            907.8879,
            -899.08856,
            -888.27075,
            867.648,
            894.8122,
            -879.2559,
            901.22754,
            895.80444,
            -850.0231,
            873.75964,
            892.16504,
            897.37054,
            866.9106,
            -859.11646,
            863.26715,
            -880.82166,
            886.0754,
            -860.41174,
            -877.4143,
            868.9705,
            869.899,
            862.05273,
            -874.2743,
            859.3687,
            -894.89655,
            896.7614,
            -882.2971,
            -884.56726,
            -893.3032,
            -891.1043,
            889.2284,
            -887.2953,
            -888.25543,
            865.39136,
            877.4199,
            866.9547,
            868.63367,
            -872.8977,
            872.4627,
            893.362,
            874.46954,
            -906.5921,
            900.5589,
            -867.89667,
            864.9303,
            -878.78973,
            872.9925,
            899.3579,
            875.7337,
            -882.47577,
            -847.08606,
            -848.9032,
            870.01843,
            -902.2973,
            -874.3307,
            -882.1878,
            -880.7402,
            855.85657,
            864.8686,
            -857.95197,
            -844.8127,
            875.15137,
            874.31915,
            -844.44946,
            851.39795,
            -893.1625,
            -870.54584,
            -843.19684,
            -851.2904,
            -866.77704,
            857.87115,
            839.44574,
            -849.63513,
            -832.4755,
            -848.6766,
            -864.45935,
            840.7987,
            -840.7968,
            -867.4418,
            827.9946,
            841.74524,
            847.2809,
            822.15466,
            844.5499,
            855.3063,
            832.749,
            -855.1358,
            847.963,
            859.3955,
            -881.9448,
            830.713,
            824.8304,
            -873.7929,
            829.2099,
            -849.86597,
            -857.6078,
            840.992,
            -848.8871,
            -843.67065,
            848.9388,
            840.9851,
            854.40643,
            839.61346,
            -842.0358,
            -811.50946,
            838.4164,
            818.81555,
            810.8926,
            830.38934,
            811.50793,
            811.8646,
            -816.4259,
            822.13934,
            818.114,
            -810.2887,
            817.89905,
            -804.3265,
            -841.2483,
            836.9977,
            835.42725,
            843.787,
            817.97076,
            -830.873,
            837.6361,
            819.8931,
            -809.4414,
            -819.3674,
            832.1858,
            -812.4069,
            -813.5168,
            -813.62714,
            -810.848,
            815.4292,
            -820.98785,
            -823.9925,
            -813.60223,
            -821.7091,
            809.5897,
            816.9829,
            -852.39905,
            834.5797,
            842.5067,
            826.93524,
            810.8932,
            -806.28564,
            818.8017,
            -823.2367,
            818.7009,
            -804.619,
            801.1335,
            -769.4572,
            -799.4105,
            -771.6397,
            -777.2547,
            797.07336,
            -799.10626,
            -785.8609,
            792.1161,
            -796.44415,
            -814.295,
            -795.76776,
            791.4558,
            -789.5745,
            -800.3463,
            -797.3075,
            791.7381,
            -792.4418,
            -790.4225,
            801.536,
            812.8696,
            807.8796,
            805.0361,
            -799.2061,
            -786.27856,
            -817.31854,
            -800.56683,
            777.72754,
            790.4428,
            801.00244,
            -783.18524,
            792.80975,
            805.50037,
            -775.68005,
            793.13025,
            -796.8631,
            -779.30676,
            781.6946,
            796.98706,
            -786.846,
            774.151,
            792.57184,
            783.11,
            -798.5381,
            790.4691,
            -802.0934,
            779.3551,
            784.0557,
            788.69525,
            799.1114,
            -778.51794,
            -776.8162,
            -777.21594,
            -744.1401,
            788.3533,
            -794.299,
            -783.621,
            -802.24976,
            -778.64307,
            783.57635,
            -789.3113,
            -783.93134,
            -792.14545,
            785.8738,
            760.5473,
            744.5876,
            -764.93677,
            743.5953,
            768.38654,
            -762.6805,
            -769.113,
            756.0651,
            782.7173,
            -778.2661,
            781.2882,
            -791.1813,
            -775.9502,
            772.7715,
            772.9071,
            -767.2804,
            -790.9778,
            767.0056,
            757.1486,
            786.0883,
            -771.45306,
            -766.90314,
            -757.5292,
            -773.7956,
            759.85077,
            -753.10895,
            -796.10925,
            759.6194,
            -762.51685,
            -778.5711,
            769.3096,
            782.7849,
            763.33026,
            777.05096,
            -788.7407,
            -742.9471,
            -782.08386,
            779.76105,
            -769.36426,
            -789.7026,
            770.94745,
            -760.8829,
            -768.9454,
            751.38715,
            762.80054,
            -771.8086,
            767.5027,
            763.0446,
            -765.84503,
            756.03674,
            -752.237,
            -772.9821,
            753.06683,
            -765.4926,
            730.26495,
            -773.8448,
            734.16174,
            747.89636,
            -749.03937,
            -759.1055,
            758.9139,
            765.25586,
            -766.97675,
            -749.72144,
            760.8874,
            765.37213,
            744.3679,
            753.56134,
            747.39056,
            737.7597,
            714.1368,
            -750.3712,
            748.7617,
            -756.819,
            -739.3151,
            706.95825,
            723.1255,
            751.3307,
            728.89954,
            720.7252,
            -723.50134,
            720.1941,
            760.7319,
            -737.25757,
            718.16895,
            728.27246,
            -753.346,
            -738.70013,
            -778.05035,
            753.2776,
            756.9279,
            -756.14923,
            751.9472,
            726.36145,
            741.215,
            -733.5585,
            -724.80084,
            712.7361,
            741.73865,
            723.72296,
            -735.03723,
            749.3974,
            738.2147,
            -749.91394,
            741.1389,
            -742.1982,
            -747.56824,
            740.4989,
            734.96533,
            741.1893,
            737.88367,
            -738.1322,
            728.2914,
            738.3881,
            -721.63153,
            735.9074,
            707.3141,
            -742.603,
            -698.8532,
            -748.167,
            722.8804,
            724.5726,
            740.18024,
            -725.08484,
            734.6744,
            -709.9171,
            707.0495,
            707.0994,
            708.7966,
            -679.67316,
            680.80286,
            697.00836,
            -688.7914,
            704.8837,
            698.94293,
            697.4539,
            -720.19434,
            -713.4987,
            -736.0016,
            722.1965,
            -714.80994,
            725.58655,
            719.0401,
            721.6393,
            711.9947,
            -723.02905,
            -709.9704,
            -712.5888,
            -706.85034,
            -710.69574,
            734.49133,
            -716.77234,
            -716.32574,
            -723.8265,
            -709.25024,
            -709.12,
            718.8593,
            -715.9313,
            -701.1976,
            710.30963,
            -698.9941,
            698.93665,
            -690.6305,
            688.8652,
            -700.1478,
            700.8586,
            -710.9674,
            694.08044,
            -700.34064,
            705.14746,
            -678.479,
            690.5422,
            715.00385,
            693.4099,
            708.0883,
            702.26227,
            -707.1529,
            716.24585,
            -710.04,
            -719.38873,
            700.2417,
            -680.8137,
            700.65247,
            -681.6115,
            677.8487,
            -693.60974,
            683.25006,
            677.4969,
            693.33954,
            -691.51117,
            677.5874,
            690.0586,
            691.7046,
            -707.6863,
            -685.4846,
            686.21643,
            677.97156,
            695.8221,
            -726.1071,
            -690.76117,
            719.3226,
            726.94403,
            706.2909,
            -711.0948,
            -699.75665,
            -701.0311,
            704.24194,
            708.8482,
            -704.8077,
            -704.8663,
            689.9193,
            -695.92596,
            -698.9569,
            697.2747,
            672.5913,
            669.51965,
            682.4406,
            -670.79755,
            -680.4774,
            688.06067,
            687.764,
            -661.8321,
            -675.3944,
            -673.6605,
            704.30896,
            694.23627,
            -668.62305,
            -673.8608,
            -682.3553,
            -663.21606,
            -705.2538,
            -685.2748,
            -685.57446,
            -712.4423,
            689.08527,
            -684.35895,
            -703.27747,
            -673.72675,
            674.57404,
            -673.1186,
            -697.8335,
            683.61945,
            680.8697,
            -679.44836,
            -659.30865,
            -667.875,
            686.4616,
            674.3613,
            -660.49005,
            -664.7502,
            -677.6768,
            688.91547,
            -670.0622,
            -678.236,
            686.6941,
            661.30975,
            -673.71814,
            -673.76074,
            662.5085,
            685.1826,
            -651.4311,
            661.2469,
            -661.4451,
            -659.6229,
            650.34143,
            650.47577,
            671.12604,
            -668.65045,
            -657.5505,
            -676.57166,
            -659.21674,
            688.5888,
            -669.0778,
            -668.40906,
            668.8142,
            654.02155,
            672.6341,
            -656.51324,
            -634.9576,
            -657.92303,
            646.1431,
            652.94977,
            -651.58496,
            -642.2716,
            -663.0364,
            658.37286,
            -645.7719,
            648.6274,
            -654.54736,
            -662.65686,
            639.15826,
            -650.7091,
            -653.69666,
            625.44073,
            657.7041,
            -653.8836,
            658.83655,
            -644.7239,
            652.86475,
            -639.9274,
            -642.22974,
            649.85034,
            671.6171,
            661.7638,
            -655.79205,
            661.43134,
            -657.1501,
            639.40283,
            632.6191,
            651.7356,
            629.2463,
            -639.5363,
            628.679,
            638.8199,
            639.8489,
            -625.6588,
            641.05396,
            621.2993,
            624.9675,
            -641.23285,
            640.6145,
            -646.0125,
            634.0488,
            -634.2127,
            -634.55457,
            -639.2815,
            -667.4181,
            -659.4523,
            648.008,
            -641.81146,
            636.1255,
            651.79535,
            653.1673,
            -652.3324,
            646.3734,
            -646.76044,
            656.3017,
            -652.76117,
            -668.8992,
            649.8941,
            646.6371,
            658.2375,
            632.5258,
            -650.5393,
            -647.7441,
            627.1283,
            -624.3831,
            637.4382,
            614.4764,
            -629.2012,
            -638.0668,
            633.3897,
            -627.63916,
            633.73004,
            -626.4189,
            620.9196,
            636.1474,
            628.792,
            636.2121,
            -650.8539,
            -634.77124,
            627.1032,
            -626.5604,
            638.8193,
            634.944,
            606.5082,
            -639.20374,
            -641.8606,
            634.9366,
            626.34845,
            647.35156,
            631.1053,
            642.7206,
            625.7494,
            -630.4331,
            641.9803,
            -634.8945,
            -630.0933,
            -626.808,
            624.2618,
            635.19824,
            640.9105,
            -634.30664,
            612.1539,
            -627.8538,
            -628.483,
            -636.87756,
            -609.0295,
            621.4914,
            -633.48517,
            624.8042,
            619.0776,
            -607.4685,
            -618.46454,
            628.85284,
            -614.0832,
            -614.4186,
            613.2371,
            641.83026,
            629.60315,
            -629.3267,
            -622.4759,
            625.3158,
            626.6558,
            628.49115,
            -637.1192,
            -596.802,
            610.9067,
            612.13257,
            619.04956,
            -618.7031,
            -611.6859,
            610.59863,
            -607.63916,
            -614.7776,
            604.0398,
            610.1567,
            610.0263,
            635.41016,
            -618.5877,
            621.4024,
            627.4796,
            608.33716,
            596.6423,
            620.296,
            -613.30676,
            601.4772,
            -627.56195,
            -617.8943,
            616.3053,
            -631.1411,
            -613.58496,
            -592.6203,
            -589.0202,
            589.16785,
            601.45935,
            601.6707,
            598.1563,
            596.1725,
            618.9142,
            616.847,
            613.3226,
            604.6479,
            612.7354,
            -585.21906,
            607.26794,
            -591.8033,
            590.42206,
            -604.2398,
            605.8769,
            598.0737,
            609.37634,
            617.0525,
            603.7294,
            -611.24994,
            -619.71735,
            -597.62006,
            -590.7122,
            -611.81683,
            603.3363,
            598.6256,
            609.3645,
            584.00867,
            597.0369,
            595.62897,
            579.81647,
            590.4571,
            -573.19604,
            587.97424,
            596.56433,
            -605.9302,
            598.2103,
            607.1001,
            -596.57446,
            601.1446,
            -594.1532,
            -588.7258,
            -576.5332,
            599.09656,
            -571.3027,
            577.57263,
            604.5289,
            577.3405,
            -586.37585,
            598.33856,
            -610.3694,
            -605.4943,
            -603.8843,
            606.58264,
            -582.39954,
            574.79016,
            577.6301,
            577.9554,
            586.04877,
            -592.86456,
            586.35986,
            586.56915,
            593.4548,
            -607.661,
            604.24194,
            -602.557,
            605.3791,
            587.2254,
            600.7564,
            -582.24036,
            602.23944,
            -594.50793,
            592.7235,
            -589.1679,
            -584.33417,
            -589.5615,
            -573.00354,
            -582.76276,
            -589.042,
            571.456,
            573.0805,
            575.413,
            584.9157,
            -574.3774,
            -581.2331,
            575.9807,
            -579.46246,
            -577.3129,
            -575.8607,
            -576.46295,
            -585.9434,
            -592.0818,
            -581.40717,
            -576.4776,
            591.16437,
            -575.7273,
            -585.58795,
            -583.8019,
            580.8464,
            569.3207,
            571.97034,
            -585.9675,
            563.2701,
            -581.77673,
            566.9155,
            -599.9631,
            -565.24615,
            573.25507,
            584.973,
            577.6417,
            576.9009,
            -570.2413,
            587.4267,
            -560.70636,
            573.0707,
            593.81946,
            573.5808,
            586.9565,
            587.4178,
            591.98047,
            -567.5631,
            573.28674,
            -581.3752,
            -563.7898,
            -572.19727,
            570.15796,
            -566.4784,
            -568.89655,
            -590.99365,
            571.8937,
            577.13745,
            583.97815,
            592.72754,
            593.3934,
            591.938,
            585.6309,
            -580.9239,
            565.9467,
            -555.6642,
            -568.0,
            -584.0128,
            567.92645,
            584.96423,
            558.57654,
            -564.0053,
            -565.7162,
            550.63965,
            561.7192,
            -557.4541,
            -545.9812,
            551.0469,
            -562.0367,
            568.81525,
            -551.1256,
            -557.1306,
            571.12134,
            545.8154,
            556.1722,
            -550.75323,
            545.6362,
            523.8709,
            547.3914,
            -547.8762,
            557.0339,
            -551.793,
            -579.391,
            557.8462,
            560.7606,
            -574.4689,
            -563.50726,
            -572.4342,
            554.19934,
            -549.2973,
            558.8929,
            -553.49164,
            -556.718,
            560.7881,
            557.63794,
            565.46545,
            559.94586,
            -557.7109,
            561.09607,
            -573.0106,
            -560.6569,
            -575.87805,
            553.26483,
            -560.0789,
            548.47644,
            -559.82904,
            554.9069,
            564.86566,
            -561.36053,
            547.70795,
            558.53735,
            -546.53076,
            550.0405,
            -538.02277,
            -524.36676,
            531.14905,
            533.46716,
            -535.3651,
            -548.8475,
            540.21875,
            -557.37384,
            571.5432,
            -558.8288,
            578.34186,
            567.60913,
            -561.4397,
            -561.6859,
            564.6618,
            -566.6746,
            558.5346,
            -558.9953,
            565.06335,
            564.59534,
            552.3818,
            577.52606,
            575.48145,
            -555.96844,
            540.03,
            544.3095,
            -546.5017,
            539.398,
            544.12274,
            -533.17535,
            -528.3317,
            549.0632,
            544.5562,
            -550.52655,
            -548.4872,
            -546.27875,
            -552.9881,
            549.2304,
            -546.41833,
            549.1945,
            557.3319,
            537.12616,
            -538.23706,
            552.905,
            540.525,
            -546.6889,
            547.5753,
            544.00446,
            568.3825,
            -539.6503,
            550.4041,
            -554.56104,
            560.5238,
            558.96313,
            558.05206,
            -551.20526,
            562.0609,
            562.9642,
            -551.27905,
            557.01874,
            -554.8003,
            543.69867,
            -551.9921,
            -556.3499,
            544.322,
            -536.87494,
            -542.0549,
            -530.40497,
            -534.29,
            -526.9871,
            540.21924,
            -539.4023,
            545.46716,
            -534.34937,
            -538.9225,
            540.51953,
            -554.9402,
            -541.6996,
            562.3874,
            -550.44507,
            -541.7342,
            -526.16205,
            -544.192,
            533.589,
            -532.98755,
            -538.1403,
            528.495,
            -527.5791,
            541.67505,
            527.9352,
            -523.05383,
            -523.8732,
            -515.7874,
            -516.79755,
            -527.9967,
            513.8526,
            536.47363,
            -516.0368,
            511.44107,
            -501.53003,
            510.61508,
            -507.75854,
            -511.55588,
            -506.57074,
            496.0567,
            512.1521,
            505.99533,
            529.3504,
            516.72473,
            511.81857,
            518.4191,
            -514.5001,
            -508.2032,
            -515.6715,
            -530.8997,
            525.974,
            -513.2348,
            -519.246,
            -514.4932,
            519.0922,
            542.32605,
            -539.9395,
            -539.6472,
            534.66266,
            -529.4319,
            523.03754,
            541.2074,
            -530.8432,
            -535.5206,
            -526.06683,
            530.68286,
            -529.28174,
            527.35187,
            531.6481,
            -544.8475,
            -528.5068,
            -520.6238,
            -536.01624,
            526.1134,
            540.33624,
            529.65155,
            -542.00256,
            -539.0523,
            -535.9,
            -543.90216,
            540.63995,
            -540.4342,
            542.2261,
            -527.08746,
            -498.69366,
            -518.8583,
            -520.32794,
            523.7726,
            -535.5287,
            -537.95447,
            544.4594,
            536.00146,
            -519.7601,
            515.4921,
            513.43195,
            -512.6194,
            520.1992,
            -493.13397,
            -513.7,
            -512.42377,
            -519.6959,
            514.73,
            -509.20297,
            518.3377,
            523.2921,
            507.12103,
            -497.10117,
            510.614,
            509.42377,
            -517.2431,
            -521.83386,
            -520.4254,
            521.91315,
            511.71844,
            -531.4488,
            -523.647,
            -543.9963,
            -514.87787,
            -524.92896,
            -517.0374,
            -513.3149,
            -519.35504,
            -522.8448,
            -516.81256,
            526.6357,
            526.1564,
            510.75845,
            522.0227,
            515.1501,
            -518.6063,
            -538.1482,
            506.23364,
            -514.7346,
            -514.1821,
            -505.95575,
            495.51047,
            511.32584,
            509.38983,
            514.67566,
            521.0904,
            514.14746,
            -512.17456,
            -514.4041,
            505.73682,
            506.9376,
            -503.5524,
            510.99197,
            516.65,
            507.38135,
            502.40552,
            -501.15466,
            -506.24707,
            -502.67984,
            -504.31754,
            -511.5958,
            -495.9771,
            -487.87595,
            -492.26764,
            -502.2728,
            506.03943,
            509.65375,
            -515.61786,
            529.618,
            -519.48865,
            -525.8089,
            -521.75446,
            530.46094,
            -511.19623,
            505.35513,
            502.71564,
            -500.20377,
            -504.08954,
            493.23773,
            508.44672,
            509.25906,
            499.32025,
            -485.90338,
            505.28583,
            489.72485,
            -517.98645,
            499.50137,
            508.34366,
            499.35852,
            494.2461,
            507.08627,
            -501.18256,
            503.8492,
            500.36133,
            -508.16306,
            512.3413,
            512.32916,
            -511.7769,
            -496.76355,
            -490.60242,
            493.89435,
            500.4184,
            -494.3082,
            502.23544,
            502.13824,
            484.45923,
            -506.31622,
            -494.67215,
            -467.99796,
            488.206,
            -484.3298,
            501.98636,
            -487.16986,
            498.33694,
            -489.21182,
            -493.9467,
            493.9619,
            -485.18396,
            -507.5589,
            491.64032,
            -499.93042,
            504.7344,
            -495.63907,
            -503.2946,
            -499.3219,
            -490.38815,
            -510.72357,
            -490.8838,
            -506.74457,
            502.54373,
            -503.0407,
            -488.1923,
            -493.51315,
            -486.54825,
            -500.38925,
            -486.16574,
            494.96158,
            -492.41943,
            485.90164,
            -483.0118,
            496.603,
            -490.44598,
            -487.39352,
            -491.87686,
            487.65753,
            482.35184,
            -473.6873,
            471.2701,
            -472.45044,
            -476.24258,
            -469.2579,
            -479.96408,
            -474.5819,
            473.94244,
            -478.2176,
            -476.4759,
            479.99432,
            476.9323,
            -476.425,
            -494.22443,
            -482.59875,
            -493.28705,
            505.34598,
            -503.43494,
            -482.1193,
            487.73502,
            -486.32175,
            481.5431,
            459.6552,
            -465.06174,
            -464.72253,
            473.99524,
            473.154,
            -481.39963,
            -473.38,
            498.12994,
            -480.72125,
            -471.6922,
            479.6334,
            -455.5702,
            458.37564,
            464.65182,
            -475.28418,
            -471.99716,
            456.89392,
            -454.65265,
            458.96112,
            475.6787,
            471.57748,
            469.79007,
            464.77036,
            -474.1691,
            480.76404,
            482.4154,
            489.9263,
            481.75623,
            487.3331,
            494.20657,
            481.42682,
            470.4466,
            476.3392,
            -492.4224,
            -483.19736,
            -480.4583,
            475.54318,
            496.0167,
            -486.2225,
            -491.23026,
            -476.5842,
            -482.90396,
            -470.5881,
            481.72668,
            -480.39728,
            -486.1996,
            -477.37598,
            478.36563,
            -477.42502,
            479.23874,
            -496.04904,
            487.11722,
            498.0869,
            477.77524,
            487.05234,
            492.13052,
            -506.54602,
            489.06998,
            492.211,
            -479.86804,
            488.7023,
            485.20102,
            494.4015,
            499.3367,
            502.89355,
            495.68192,
            -493.60916,
            488.47308,
            -499.13968,
            493.25623,
            486.38107,
            -490.65424,
            -487.9254,
            -498.6137,
            497.2503,
            505.2519,
            -490.435,
            -510.48468,
            -492.74344,
            492.62024,
            501.78085,
            506.00137,
            500.87668,
            -495.29526,
            486.4842,
            481.75568,
            -477.81946,
            -484.90125,
            477.6925,
            -471.79205,
            479.34814,
            468.37738,
            -472.38794,
            470.8929,
            474.6691,
            -469.69284,
            -479.97864,
            -473.9827,
            480.61554,
            -474.18533,
            -475.60028,
            -490.89972,
            -485.801,
            -476.74808,
            473.8857,
            471.1548,
            475.38882,
            471.1861,
            477.4658,
            469.89877,
            -459.8667,
            -455.86786,
            443.36752,
            -453.10406,
            -441.74228,
            -444.62396,
            445.53766,
            441.37524,
            447.6997,
            -459.33,
            -448.78296,
            456.3921,
            450.38455,
            453.76477,
            -442.9566,
            -433.1476,
            438.1461,
            457.92688,
            446.15582,
            -439.4148,
            453.90833,
            459.42465,
            462.95956,
            472.09692,
            -478.30438,
            -470.48288,
            -464.77454,
            472.66354,
            467.46628,
            -458.15747,
            472.849,
            466.54276,
            471.98325,
            462.124,
            -464.20926,
            -455.87967,
            -470.944,
            -469.60904,
            477.12662,
            482.2078,
            -480.5979,
            468.5929,
            458.02475,
            456.12076,
            -446.0087,
            442.11966,
            453.4556,
            462.35458,
            451.86584,
            -448.75287,
            458.523,
            -453.42105,
            443.70535,
            434.66998,
            -439.41663,
            -442.64246,
            -449.5116,
            448.0363,
            -460.51962,
            -475.08545,
            471.82944,
            479.16174,
            -466.0833,
            -466.33606,
            -457.75763,
            453.37402,
            -463.1087,
            -468.16113,
            -453.95322,
            -462.22903,
            481.62247,
            -476.32312,
            -466.97577,
            469.83572,
            483.0136,
            454.925,
            -455.50613,
            462.12424,
            -448.9539,
            -450.41052,
            -450.4017,
            -459.6056,
            460.48364,
            -459.5663,
            -474.61453,
            -450.42264,
            449.10852,
            -456.4261,
            -450.30453,
            446.41486,
            -451.39148,
            448.0351,
            -449.36014,
            456.28574,
            -450.84967,
            -444.04245,
            456.65207,
            -453.1243,
            -455.56256,
            445.4845,
            -444.48163,
            -446.61502,
            -440.42468,
            -438.86172,
            442.42853,
            -429.44962,
            -437.74393,
            -432.67383,
            433.61853,
            440.74738,
            -443.95804,
            461.0655,
            -453.17386,
            440.7872,
            -455.28314,
            453.38034,
            452.56244,
            -451.05353,
            -431.3764,
            -437.66983,
            438.69086,
            428.58377,
            431.86737,
            -437.16882,
            -428.09427,
            -429.78616,
            436.26108,
            444.84842,
            -428.46246,
            -431.3243,
            -429.88986,
            -421.15585,
            -417.08398,
            -416.15204,
            -418.97058,
            -421.11053,
            426.0375,
            425.1173,
            -425.58377,
            -425.03595,
            425.5928,
            -417.75888,
            -421.51974,
            413.98236,
            -420.7321,
            -409.8967,
            415.26477,
            -419.09552,
            418.9855,
            424.22763,
            427.0646,
            436.61462,
            -426.64206,
            -430.1214,
            -416.88565,
            -405.6589,
            -397.32608,
            415.2707,
            -403.4035,
            -394.29758,
            -403.56665,
            -414.49078,
            -429.8176,
            -442.95206,
            -437.98288,
            -430.14444,
            443.754,
            432.735,
            436.20367,
            428.03412,
            451.43704,
            -436.489,
            439.01163,
            -426.19897,
            439.40668,
            442.30536,
            -438.66843,
            -434.32425,
            438.2056,
            -428.28598,
            431.94492,
            -415.95584,
            410.0638,
            -416.26456,
            -424.55505,
            415.09613,
            -421.28732,
            411.0409,
            -415.17023,
            -408.74084,
            -408.21707,
            414.48276,
            -405.34714,
            416.0491,
            411.36136,
            -409.51566,
            -410.00717,
            -404.80917,
            406.9759,
            408.05344,
            417.54666,
            403.67688,
            -402.72147,
            -411.37192,
            -418.76425,
            420.74805,
            -412.18958,
            399.15796,
            412.57672,
            398.96976,
            413.30304,
            401.72665,
            -403.38266,
            407.22568,
            388.64914,
            -386.8015,
            -398.57733,
            393.88605,
            -400.18124,
            -408.2586,
            404.0444,
            -419.45895,
            418.00024,
            -420.68155,
            437.60687,
            414.71844,
            409.59903,
            395.65732,
            -402.39606,
            397.68573,
            -403.74207,
            400.4162,
            -407.71826,
            403.62485,
            -404.75375,
            -413.8179,
            418.45203,
            -424.22928,
            419.20752,
            409.63724,
            -409.33325,
            -397.78296,
            -412.94962,
            -411.3571,
            -411.13565,
            406.65216,
            -401.17438,
            -406.00696,
            -396.02963,
            395.28543,
            402.52673,
            -398.89777,
            -402.84897,
            -401.39056,
            -400.79068,
            397.99414,
            -394.72168,
            -391.97253,
            -396.1612,
            393.38055,
            388.791,
            401.1723,
            395.1628,
            403.5835,
            -399.10623,
            402.80246,
            409.02597,
            406.0894,
            406.00595,
            -401.62433,
            393.2851,
            389.47415,
            396.3233,
            -393.4476,
            -397.63953,
            409.67062,
            -403.72952,
            407.02005,
            392.8845,
            399.2774,
            -389.71774,
            389.9629,
            391.01993,
            399.9077,
            -391.03268,
            406.5924,
            396.29883,
            -395.65323,
            391.9716,
            383.80518,
            376.13373,
            390.8997,
            379.2321,
            -379.144,
            -376.4254,
            -387.0092,
            392.58658,
            389.70395,
            397.00867,
            -401.8375,
            -387.6584,
            -382.82294,
            389.94684,
            -389.49225,
            -390.22662,
            -397.50986,
            398.6746,
            -394.19067,
            -396.98618,
            -399.03064,
            393.34283,
            394.08847,
            -391.21603,
            383.35855,
            -382.16617,
            -395.504,
            382.9412,
            391.16315,
            -374.53284,
            396.75513,
            384.2658,
            -390.61526,
            393.8695,
            -405.73743,
            -396.64658,
            381.18478,
            -391.80124,
            -384.13165,
            -378.32095,
            -383.7535,
            -391.77908,
            393.75928,
            -381.84073,
            379.42252,
            382.26056,
            -390.93463,
            385.44208,
            391.31546,
            -392.3898,
            388.20734,
            -392.94864,
            -390.30695,
            383.9589,
            -385.27826,
            -382.95038,
            -376.7785,
            -381.10632,
            376.54578,
            382.68875,
            -393.56046,
            -399.97763,
            -406.75745,
            392.35767,
            400.33002,
            -388.8769,
            392.9626,
            -389.7421,
            -383.05197,
            -376.37704,
            380.3174,
            -394.1551,
            -399.87872,
            -408.09195,
            406.2508,
            418.1548,
            415.74338,
            -411.90765,
            -404.19147,
            -407.62057,
            -410.58624,
            407.89307,
            427.9656,
            443.94598,
            447.35352,
            455.1233,
            451.68466,
            -445.8928,
            442.1985,
            -432.52362,
            442.69205,
            440.25613,
            -439.11414,
            -438.3867,
            437.5288,
            -441.62115,
            442.53726,
            -434.9989,
            -439.54163,
            -432.6285,
            -439.9983,
            -433.02173,
            -436.0253,
            -444.00317,
            -434.2627,
            427.25943,
            430.09204,
            -431.92447,
            430.27185,
            427.30823,
            422.08453,
            430.4734,
            426.08795,
            441.49594,
            -444.9163,
            -446.5118,
            -440.27213,
            -452.63156,
            434.1377,
            436.07013,
            -443.27228,
            -423.7706,
            -442.76614,
            433.0626,
            -447.2487,
            448.5491,
            -442.9609,
            -447.4191,
            -436.2652,
            -438.76154,
            442.6059,
            443.32288,
            436.65057,
            437.2463,
            -439.8733,
            442.28894,
            -444.74066,
            443.98697,
            -446.02798,
            -440.14795,
            -446.13623,
            -445.79346,
            457.0679,
            442.63104,
            432.65305,
            -430.74,
            -436.20746,
            -424.53882,
            439.72192,
            438.37744,
            441.1582,
            442.84125,
            447.99738,
            448.61087,
            439.10043,
            438.70093,
            440.97934,
            -440.50455,
            440.8449,
            442.01077,
            -439.23285,
            435.7245,
            -428.38412,
            440.52112,
            430.80902,
            -433.47607,
            436.24347,
            434.23108,
            -427.9044,
            427.1936,
            -422.87817,
            -416.1446,
            415.55997,
            -398.53806,
            406.61795,
            392.56854,
            -396.11935,
            395.17523,
            -393.45236,
            383.14563,
            374.2483,
            371.50302,
            373.88373,
            -369.93918,
            -384.54178,
            380.58426,
            -385.5763,
            373.0361,
            362.21048,
            358.48965,
            -359.97162,
            -357.79074,
            -355.42123,
            338.83337,
            -346.28265,
            353.44443,
            359.33783,
            -352.92334,
            358.00308,
            -358.52356,
            359.14056,
            -364.43863,
            365.38535,
            -354.10727,
            363.84222,
            351.84866,
            347.38943,
            -358.61807,
            358.64883,
            -363.0694,
            355.54007,
            -345.6121,
            -353.5801,
            357.5018,
            -351.17807,
            352.3078,
            352.13074,
            360.58047,
            345.78024,
            356.99985,
            357.11478,
            -367.18253,
            359.75726,
            354.4774,
            335.06177,
            345.6844,
            346.728,
            -347.83673,
            356.43353,
            362.95435,
            -366.261,
            368.50974,
            370.54312,
            -363.17593,
            358.74973,
            -351.09473,
            -363.7934,
            -359.30667,
            -362.75018,
            366.24197,
            -360.18268,
            -357.51813,
            355.56973,
            347.79712,
            -350.68195,
            -345.39395,
            344.4821,
            345.89725,
            -352.76575,
            -347.6919,
            -353.63745,
            345.72302,
            -339.54037,
            -339.13745,
            345.22742,
            -340.92133,
            -345.44336,
            349.9482,
            -341.76135,
            352.32394,
            -344.50552,
            359.88058,
            353.04556,
            -362.00226,
            -357.06302,
            360.03064,
            359.80896,
            -349.75375,
            -348.9684,
            356.98495,
            345.04294,
            350.93228,
            355.51028,
            355.48764,
            -358.36362,
            361.17514,
            -359.5954,
            381.82193,
            -377.84164,
            378.9337,
            381.6075,
            -384.3427,
            378.06165,
            388.58167,
            -404.72977,
            410.98502,
            423.1452,
            -430.70377,
            423.58002,
            -418.56592,
            425.61676,
            422.6992,
            -430.89816,
            -434.54047,
            -421.9728,
            416.2294,
            -411.20105,
            -417.88184,
            -423.0287,
            -427.2014,
            419.42734,
            -426.64328,
            -420.12354,
            -415.8865,
            -420.9502,
            417.17148,
            422.5289,
            432.5963,
            -430.8626,
            434.1715,
            434.4788,
            -429.7136,
            -428.8465,
            -437.66678,
            432.4128,
            427.68005,
            431.72485,
            425.51166,
            -416.52365,
            -431.11874,
            428.6689,
            -435.5728,
            430.59125,
            -429.09238,
            430.43265,
            440.73264,
            -436.43665,
            434.0858,
            432.98572,
            433.6293,
            413.7384,
            416.73538,
            416.5427,
            424.20895,
            -426.69073,
            434.34317,
            -433.42413,
            -427.02744,
            419.28137,
            410.66324,
            -413.56525,
            418.2647,
            -420.549,
            414.74408,
            417.58737,
            -412.6022,
            -418.0851,
            429.25864,
            -431.593,
            -422.69565,
            -414.69302,
            403.01938,
            403.4903,
            -410.26083,
            -423.39957,
            424.62152,
            -429.35513,
            423.07404,
            416.76675,
            398.3994,
            -379.18436,
            355.6793,
            -349.6513,
            -335.1569,
            -324.40292,
            327.88806,
            -336.20145,
            340.08078,
            352.3491,
            355.14856,
            350.39594,
            345.17236,
            330.47647,
            -324.9173,
            -335.7977,
            -324.40594,
            322.66074,
            319.45087,
            -327.58087,
            -322.67975,
            326.8497,
            322.3587,
            -328.54672,
            -331.86337,
            -337.6593,
            -334.8997,
            -352.842,
            -346.43436,
            337.52173,
            -335.397,
            341.5111,
            -339.95276,
            335.4871,
            -332.65555,
            -326.94553,
            325.38272,
            -330.244,
            -332.3432,
            -340.95752,
            -338.14145,
            -341.7795,
            353.45502,
            -358.97073,
            -347.44577,
            -344.46735,
            337.44135,
            324.79742,
            -324.66058,
            334.18698,
            -327.4277,
            -316.66986,
            319.45786,
            323.95258,
            -325.5562,
            -332.55865,
            333.37112,
            -331.87427,
            -326.46332,
            324.17465,
            320.4314,
            327.6038,
            -326.04285,
            -321.8108,
            -321.69385,
            318.77997,
            316.90298,
            313.27054,
            -314.47812,
            330.7622,
            -330.5308,
            -332.66888,
            -334.47372,
            339.33615,
            -343.2962,
            344.50525,
            -349.62143,
            -347.26767,
            332.4765,
            -352.91788,
            -362.0221,
            377.4248,
            -387.5238,
            395.50037,
            413.25638,
            -409.28598,
            412.8384,
            -411.0823,
            409.3757,
            -410.8952,
            416.19672,
            -425.41116,
            -424.0809,
            -416.81808,
            -410.6542,
            -400.04306,
            -401.05978,
            389.07083,
            -387.32196,
            -394.80856,
            410.7021,
            -416.1718,
            413.63632,
            -416.38422,
            -419.50934,
            -412.54794,
            -409.36945,
            -413.25958,
            -413.236,
            410.71445,
            413.95886,
            -405.4025,
            -394.19543,
            -393.8285,
            -394.29935,
            399.355,
            -404.95068,
            409.14322,
            410.271,
            424.43686,
            420.0602,
            -415.27594,
            -422.79126,
            419.42218,
            424.01782,
            420.57108,
            423.02188,
            -418.89102,
            -416.54916,
            -419.6106,
            -414.43677,
            -414.36072,
            -418.10718,
            415.4144,
            418.19116,
            -413.7633,
            -406.1822,
            -410.58972,
            408.09964,
            403.71658,
            413.52426,
            -417.96704,
            -432.15146,
            -431.87378,
            -427.568,
            -424.5796,
            418.38248,
            -420.5733,
            405.9362,
            402.80682,
            -377.8568,
            -349.67142,
            -333.8935,
            306.7571,
            -313.21405,
            300.2034,
            291.12296,
            -294.85092,
            -307.55508,
            305.40848,
            311.17587,
            -323.48245,
            -335.3511,
            335.21115,
            -325.79395,
            312.9634,
            -308.8273,
            302.847,
            300.6004,
            -300.14536,
            -311.92728,
            311.0445,
            318.70792,
            326.1976,
            324.97922,
            -323.75928,
            318.76877,
            320.10486,
            310.64102,
            308.27765,
            309.5383,
            -301.89963,
            -306.2876,
            -308.21143,
            -306.11182,
            311.46768,
            -313.35825,
            317.11453,
            316.03766,
            -312.35037,
            316.55902,
            302.4457,
            295.69247,
            -289.47473,
            287.10342,
            285.86792,
            -295.83426,
            -294.20517,
            290.8876,
            293.78815,
            -298.89548,
            -313.55222,
            319.2685,
            -326.06516,
            328.10907,
            -325.7626,
            -317.5211,
            312.2939,
            309.06354,
            297.81302,
            301.1436,
            -283.0397,
            -287.83133,
            286.70486,
            285.63565,
            293.29962,
            -286.79517,
            293.42773,
            291.06128,
            -303.1344,
            302.32025,
            -298.25824,
            -295.90738,
            292.21634,
            -285.10397,
            -288.20172,
            290.8278,
            -282.4277,
            -285.4251,
            -286.09964,
            292.49765,
            -287.49594,
            283.26083,
            272.21658,
            -277.2344,
            283.89758,
            275.95264,
            275.66013,
            276.77237,
            -283.99405,
            282.731,
            281.7947,
            -290.0653,
            280.25647,
            284.62576,
            -283.0026,
            -284.54355,
            -279.35962,
            -279.81506,
            269.3989,
            -277.82648,
            277.69226,
            282.5963,
            -279.45898,
            -279.94595,
            -272.94608,
            280.94525,
            -287.57623,
            -277.17142,
            -271.63324,
            -277.26584,
            -280.91077,
            -289.51443,
            -287.25858,
            -285.4988,
            283.8435,
            -285.3058,
            -285.76035,
            -286.51434,
            -279.6442,
            277.88895,
            275.0678,
            -275.38428,
            -270.75378,
            -264.72617,
            -273.32086,
            -273.48804,
            278.2994,
            -278.60098,
            -287.54095,
            269.46454,
            266.46564,
            -272.96854,
            277.3799,
            -284.94446,
            281.47467,
            269.363,
            274.27823,
            -267.49948,
            267.15005,
            276.1785,
            -279.88815,
            -284.45093,
            -285.83145,
            -283.46713,
            280.19467,
            -273.84244,
            273.0754,
            -271.74127,
            271.24402,
            -273.28375,
            279.0986,
            274.01233,
            269.7654,
            274.26306,
            -268.34003,
            265.1293,
            -271.15372,
            268.3637,
            -272.6178,
            268.6296,
            -275.0506,
            -278.956,
            -272.5282,
            -269.41367,
            -278.45065,
            272.20383,
            -277.72974,
            276.65967,
            -278.0038,
            285.33813,
            279.36746,
            275.12167,
            275.9922,
            -269.3655,
            275.6932,
            -277.4606,
            279.3363,
            268.95834,
            271.87567,
            -271.27536,
            264.1235,
            265.43335,
            -262.27026,
            -261.77615,
            265.6691,
            263.69766,
            -268.83945,
            -267.07214,
            262.91122,
            -268.58337,
            -265.11514,
            267.09018,
            267.06046,
            265.70825,
            266.07736,
            -260.6937,
            -268.81766,
            273.26846,
            -270.5737,
            -270.4043,
            -274.01328,
            -271.0636,
            -263.64343,
            -270.78195,
            272.4096,
            -270.37537,
            264.75464,
            257.43323,
            -264.54245,
            269.46625,
            271.03214,
            269.87927,
            -263.07925,
            -266.83746,
            265.67258,
            266.93924,
            -264.62158,
            -262.28683,
            -258.15564,
            -263.34094,
            -267.65085,
            268.53625,
            -265.0507,
            -264.5309,
            -253.81606,
            254.94537,
            -256.67957,
            260.9995,
            -257.51013,
            258.92834,
            -265.62906,
            -265.86664,
            270.75977,
            263.81662,
            263.1082,
            -260.3636,
            -262.49188,
            -267.30853,
            -266.87164,
            261.3814,
            -261.28967,
            264.33182,
            259.33893,
            257.9902,
            -265.9541,
            -260.6333,
            -265.19138,
            -260.59967,
            266.22586,
            -259.8504,
            -264.05948,
            259.3805,
            257.5456,
            -258.15918,
            260.75922,
            -264.62228,
            -255.4049,
            258.52762,
            258.18225,
            261.04807,
            257.74506,
            254.8714,
            258.61905,
            261.3644,
            -258.8338,
            -259.09625,
            -255.42438,
            264.46085,
            -262.60352,
            -261.67715,
            260.23438,
            264.59952,
            -260.14023,
            -259.80267,
            -256.17935,
            -265.0697,
            254.6174,
            -253.83258,
            -252.10794,
            255.22209,
            259.76733,
            262.85965,
            258.8802,
            -253.21547,
            -265.61667,
            259.4788,
            -257.3032,
            -259.06415,
            262.36777,
            -253.76158,
            -253.31712,
            254.37135,
            261.768,
            -266.5455,
            258.54785,
            264.00427,
            -270.75037,
            -254.37749,
            -256.45966,
            256.92908,
            248.72665,
            -254.15631,
            -245.70293,
            -246.7731,
            251.16711,
            -256.82678,
            -260.20758,
            265.61554,
            258.80292,
            255.48602,
            253.77028,
            -240.2989,
            259.834,
            246.83249,
            247.52748,
            -254.02727,
            252.86227,
            -248.6729,
            -258.01434,
            260.61594,
            -258.63437,
            258.3217,
            -257.2668,
            250.2633,
            -250.65514,
            -254.91011,
            -251.54526,
            256.21896,
            -254.02927,
            -256.1477,
            247.22905,
            252.67429,
            256.2959,
            -248.80588,
            -246.95903,
            245.57901,
            -248.43172,
            -247.47612,
            250.21222,
            -253.60553,
            261.4364,
            257.95908,
            -259.65665,
            252.20224,
            248.44373,
            248.50455,
            245.03537,
            -246.45676,
            -244.90619,
            252.77217,
            -252.78743,
            -251.12898,
            244.61797,
            250.13695,
            247.95924,
            -251.79494,
            -244.92046,
            248.52345,
            249.50787,
            253.0902,
            -256.44943,
            -255.69035,
            249.75897,
            246.86255,
            245.83685,
            -252.3103,
            258.9012,
            258.79837,
            250.78258,
            253.17003,
            245.25224,
            -254.24683,
            -249.70178,
            252.09036,
            248.5258,
            250.85234,
            244.76868,
            246.83478,
            248.47543,
            -247.144,
            249.32942,
            247.8326,
            -250.79892,
            -246.10242,
            -246.44746,
            238.51036,
            -242.1,
            257.66284,
            245.04419,
            -252.22467,
            245.72868,
            -245.51208,
            248.37479,
            247.11682,
            -250.73659,
            -243.75813,
            -249.81429,
            250.41458,
            241.16933,
            -234.87457,
            243.47734,
            244.00734,
            246.25218,
            244.09618,
            -243.68987,
            -245.28108,
            246.07857,
            -243.88443,
            -241.44829,
            249.65564,
            -241.52182,
            246.75162,
            254.97466,
            -251.8217,
            250.5146,
            -247.06018,
            -242.74069,
            -242.39073,
            239.71182,
            -247.295,
            243.38261,
            -246.54948,
            253.74417,
            -251.36386,
            -250.2812,
            246.80098,
            249.92194,
            246.73863,
            -247.48706,
            242.20818,
            -245.17383,
            243.1343,
            -243.77876,
            251.85167,
            -244.70805,
            246.68282,
            -243.44374,
            -241.95815,
            246.0928,
            239.1854,
            -244.8604,
            -241.45045,
            243.76561,
            -243.36365,
            245.45924,
            -243.80429,
            -240.84633,
            251.66461,
            245.98663,
            -238.53305,
            243.29292,
            241.95847,
            237.25037,
            -246.85243,
            238.91258,
            -233.87938,
            -241.29558,
            -239.97311,
            -232.62918,
            244.23488,
            241.48051,
            238.87271,
            -244.57368,
            247.70815,
            -236.8803,
            -236.83913,
            -233.46143,
            237.62389,
            239.91048,
            239.63841,
            -237.20012,
            234.5046,
            242.3922,
            -242.39859,
            235.30289,
            -231.34492,
            -226.78683,
            -228.98503,
            -238.16118,
            243.49438,
            -239.56833,
            240.12265,
            -237.8709,
            239.87158,
            -232.43842,
            234.92653,
            233.1398,
            -244.17769,
            -229.67967,
            -230.6488,
            233.96857,
            238.68425,
            243.30988,
            243.59085,
            -241.05664,
            -235.00766,
            -237.06395,
            -227.38937,
            230.7085,
            235.87134,
            -226.61789,
            -231.89482,
            234.80125,
            -232.00354,
            239.17184,
            -230.85005,
            -228.76837,
            232.2872,
            -235.40718,
            232.93295,
            231.6427,
            -237.06902,
            -231.01633,
            231.68132,
            -227.2056,
            228.8235,
            -231.88121,
            -237.61325,
            -226.1226,
            -244.88423,
            -238.68637,
            -232.27466,
            235.93823,
            -229.86836,
            230.94371,
            -230.2576,
            231.85243,
            -232.9426,
            239.06378,
            242.46233,
            235.64148,
            -231.11098,
            227.74342,
            234.20135,
            -228.76587,
            228.68839,
            233.14157,
            232.90158,
            229.26888,
            -234.05017,
            230.4886,
            226.67892,
            -224.38663,
            227.9943,
            234.21918,
            -235.34581,
            -230.93724,
            -232.8224,
            -225.7703,
            -228.23338,
            227.68642,
            -222.29317,
            -234.4726,
            -225.56166,
            -226.28206,
            233.9316,
            225.71126,
            229.11374,
            -235.6127,
            -225.59743,
            224.17262,
            229.18741,
            -225.11346,
            229.4347,
            -231.52809,
            229.32294,
            -226.46655,
            232.34602,
            222.87128,
            -223.28815,
            -226.59132,
            227.43779,
            220.71806,
            -225.39606,
            -220.29404,
            225.60344,
            -222.27107,
            -224.18666,
            227.97711,
            231.74182,
            225.44583,
            -220.22244,
            223.94373,
            221.682,
            -226.43666,
            -226.71484,
            221.21484,
            223.32985,
            227.78429,
            232.62286,
            -227.6419,
            224.30934,
            -228.98868,
            235.6671,
            222.7609,
            -221.781,
            -224.87044,
            -224.06139,
            -224.04248,
            -227.85028,
            -221.7297,
            227.25398,
            217.52753,
            226.00623,
            -216.40254,
            221.4169,
            229.14784,
            221.35658,
            -224.00754,
            -226.45782,
            -218.38951,
            -229.95685,
            232.58466,
            225.30968,
            -222.2381,
            -223.9859,
            -223.62025,
            223.91306,
            -222.57756,
            226.93628,
            224.39113,
            220.1144,
            228.12648,
            224.21664,
            223.81192,
            230.60661,
            226.71808,
            230.37267,
            -226.24307,
            -225.08205,
            -223.7541,
            -226.19879,
            221.97746,
            -215.51456,
            211.20494,
            -218.21442,
            218.81694,
            224.27226,
            220.77655,
            219.27966,
            221.04274,
            -218.12787,
            215.8869,
            218.1326,
            -223.06792,
            225.16054,
            224.81818,
            220.36208,
            215.92003,
            218.68903,
            212.94699,
            222.20435,
            216.69998,
            222.31808,
            219.6188,
            -217.75212,
            218.38853,
            218.94286,
            220.4267,
            228.40784,
            221.32718,
            219.2606,
            217.0867,
            -214.79346,
            -222.87932,
            222.87286,
            -217.52313,
            -223.33182,
            220.4747,
            -220.07649,
            -218.92519,
            -224.44145,
            215.32721,
            -215.586,
            -219.4621,
            219.90439,
            -214.69852,
            214.36441,
            219.71878,
            212.6203,
            217.94757,
            215.31746,
            214.24756,
            -210.0972,
            212.66798,
            -225.33932,
            -221.76074,
            -221.47217,
            226.50055,
            -223.7125,
            -226.31793,
            218.58994,
            219.26959,
            -209.47177,
            214.57483,
            -215.69844,
            -216.70142,
            214.34631,
            216.80415,
            222.29578,
            224.64537,
            -213.36977,
            -211.75937,
            -211.56888,
            213.84303,
            213.94652,
            221.16852,
            216.55449,
            221.37407,
            222.68948,
            -224.2776,
            215.08237,
            -211.61383,
            215.35724,
            216.04965,
            -218.68039,
            217.8819,
            -218.26768,
            224.15543,
            220.08112,
            -220.27786,
            -213.43114,
            210.67387,
            -206.32858,
            211.74774,
            -218.3853,
            -216.78078,
            -211.46321,
            219.8958,
            221.4893,
            214.65643,
            215.27228,
            212.752,
            -216.46028,
            208.67155,
            215.40709,
            -214.86069,
            212.9092,
            213.93352,
            209.2845,
            206.37611,
            -208.69073,
            -203.91512,
            -214.46497,
            -208.92805,
            212.11287,
            -208.94954,
            212.88959,
            -214.29668,
            220.79898,
            221.13638,
            214.8253,
            -215.9476,
            -214.54765,
            -205.24525,
            -204.39601,
            -209.28592,
            -208.11977,
            207.47746,
            208.05316,
            -207.50534,
            213.24632,
            -215.05997,
            211.50845,
            210.89104,
            208.67566,
            204.88055,
            -212.47614,
            -207.17697,
            222.5235,
            217.91417,
            217.98405,
            214.00407,
            208.81767,
            206.29243,
            -208.2989,
            -216.26015,
            -216.2265,
            -218.71391,
            -214.32838,
            -210.50719,
            205.09692,
            203.48514,
            197.40828,
            -204.78279,
            201.20694,
            204.14578,
            -200.371,
            198.81433,
            -203.08694,
            -207.4331,
            -205.10803,
            205.10748,
            -204.38037,
            -210.94281,
            -212.34558,
            -216.67215,
            -220.5172,
            219.24605,
            -220.40887,
            -216.49962,
            -211.06926,
            -207.98781,
            206.49045,
            202.17993,
            -203.29555,
            -205.9591,
            211.76636,
            -215.27194,
            211.11418,
            -210.04568,
            206.40326,
            202.47124,
            205.51471,
            201.0962,
            -200.5116,
            -204.97514,
            197.61385,
            -197.07346,
            194.75256,
            -205.53983,
            -206.05501,
            -206.79565,
            -211.41391,
            -203.3767,
            206.14166,
            -210.68689,
            206.3185,
            -208.7336,
            202.93759,
            203.2792,
            -196.07245,
            -198.57927,
            196.95332,
            -200.98724,
            206.57729,
            -208.71805,
            -204.6247,
            204.70544,
            -203.3758,
            197.63576,
            -196.93921,
            204.38135,
            -198.99905,
            202.16693,
            -197.20529,
            195.23146,
            196.90262,
            -203.2012,
            203.50514,
            -203.58289,
            -202.43188,
            203.64583,
            203.3038,
            -201.09172,
            202.39238,
            197.43518,
            -200.11864,
            -199.52269,
            197.6306,
            -200.49953,
            197.26578,
            -198.8815,
            -191.0331,
            198.6388,
            -196.66858,
            -194.17444,
            -196.56148,
            196.14207,
            202.77193,
            205.02435,
            203.08235,
            204.98859,
            -196.06653,
            -204.36546,
            203.05551,
            200.24718,
            202.80968,
            -198.01292,
            198.22607,
            -197.70206,
            -193.24329,
            -194.59412,
            -194.07832,
            -192.67432,
            190.2746,
            191.00323,
            191.6737,
            196.183,
            -195.7358,
            -194.78395,
            -195.37965,
            189.86072,
            -187.03044,
            194.99576,
            196.16675,
            -197.12741,
            -196.4912,
            -196.95813,
            -194.61362,
            -201.30678,
            198.17247,
            199.0466,
            194.96129,
            -191.67719,
            -194.0699,
            -188.17116,
            -185.69722,
            188.56483,
            -190.10466,
            190.24554,
            -187.56587,
            189.49277,
            187.11443,
            -190.84734,
            -194.35942,
            -192.09872,
            186.3244,
            189.33363,
            -191.0067,
            189.76093,
            189.53113,
            -187.54276,
            187.67773,
            190.35265,
            192.84052,
            -187.45844,
            -185.66608,
            -187.88966,
            -189.93475,
            190.2627,
            -192.85654,
            186.612,
            187.92348,
            187.67851,
            -188.90816,
            -178.54803,
            184.63545,
            183.68503,
            -186.32962,
            186.17566,
            192.61359,
            190.10028,
            189.02455,
            188.29465,
            -186.46265,
            186.63446,
            187.5275,
            182.85571,
            -190.04147,
            193.3773,
            185.12018,
            186.97768,
            183.80226,
            -181.96289,
            186.62169,
            187.07176,
            185.8317,
            -190.90291,
            188.53723,
            184.71246,
            185.27975,
            -182.88777,
            -183.92946,
            -181.46667,
            -185.19801,
            184.29878,
            181.13481,
            186.12471,
            189.55142,
            -186.45737,
            191.97746,
            185.18742,
            186.17862,
            -187.43433,
            -193.51448,
            -189.3072,
            186.32011,
            181.58871,
            179.1071,
            177.3951,
            182.8384,
            181.89198,
            -179.95163,
            183.85663,
            -182.95436,
            -184.45374,
            187.55952,
            -190.31024,
            -193.22643,
            191.65637,
            187.55789,
            182.01746,
            -178.99081,
            178.68327,
            -183.80121,
            182.5508,
            -186.96086,
            -181.02986,
            -186.89899,
            183.52452,
            -186.87018,
            186.45209,
            180.07045,
            183.60728,
            184.7758,
            -180.4729,
            -186.56786,
            -184.72354,
            -189.20213,
            -183.59479,
            181.78156,
            -180.40466,
            -176.40848,
            -178.24985,
            183.70683,
            186.0323,
            -189.62753,
            185.66446,
            181.89659,
            -183.86569,
            176.7141,
            178.93059,
            177.04706,
            180.94556,
            -181.39928,
            179.5348,
            174.42049,
            175.2553,
            173.90558,
            -177.06886,
            -181.49805,
            -182.2149,
            177.51096,
            -180.92838,
            -177.88815,
            181.15967,
            -179.50183,
            177.41821,
            178.1352,
            -178.6399,
            -181.68411,
            184.20462,
            -181.68849,
            -185.9787,
            -179.3657,
            182.23367,
            180.3672,
            173.68327,
            178.11201,
            175.63017,
            181.08401,
            -182.139,
            -180.40329,
            179.42288,
            -174.31056,
            173.93076,
            -177.22768,
            184.24338,
            178.90924,
            182.03589,
            177.73329,
            175.25581,
            171.69365,
            174.54327,
            -179.37527,
            172.5988,
            -173.65129,
            172.78748,
            -170.57603,
            176.14471,
            174.00566,
            -173.15936,
            -173.92892,
            177.01572,
            -179.58775,
            172.03174,
            -171.38644,
            -174.28612,
            172.75136,
            178.46738,
            -181.17006,
            -172.68724,
            170.5861,
            174.22066,
            -170.07016,
            176.99582,
            181.06512,
            -172.60715,
            -172.11116,
            -179.22386,
            182.63467,
            178.47627,
            -179.25616,
            176.51933,
            -173.61151,
            -173.9081,
            178.64651,
            175.70242,
            -173.87068,
            -166.01454,
            -172.8627,
            164.66086,
            -172.20268,
            -172.0146,
            -174.66643,
            170.78464,
            -171.50177,
            -170.3932,
            179.99275,
            -169.84772,
            172.22824,
            -169.84093,
            168.11047,
            168.07315,
            173.26118,
            173.3567,
            -173.58096,
            173.768,
            -167.99678,
            -172.1079,
            171.46796,
            169.37572,
            172.65042,
            -172.54077,
            170.30374,
            -169.50876,
            173.0708,
            172.17354,
            170.44469,
            -174.16266,
            172.7512,
            -171.23257,
            -173.67142,
            -172.08147,
            -170.51122,
            -169.02554,
            170.21681,
            -170.31505,
            -172.10826,
            -169.5632,
            169.8468,
            168.69629,
            167.84982,
            172.9728,
            170.99057,
            173.82907,
            -170.59485,
            -169.12918,
            166.13297,
            164.82193,
            -164.87753,
            169.43312,
            166.83896,
            -168.52812,
            -167.1005,
            -168.5152,
            -163.92754,
            -167.0244,
            171.99023,
            -167.76303,
            -169.08142,
            -167.53778,
            -165.45901,
            -166.26501,
            -169.86018,
            -166.09207,
            -169.50342,
            167.37912,
            165.80867,
            166.71695,
            169.68408,
            166.65028,
            164.76823,
            -161.86525,
            -168.32822,
            -168.54625,
            -163.4204,
            164.96625,
            167.4744,
            -166.44936,
            166.35565,
            167.812,
            -167.27689,
            165.14247,
            -165.7951,
            -166.51259,
            168.73657,
            166.12798,
            -165.11342,
            164.46579,
            163.36269,
            -160.99275,
            164.7193,
            166.09883,
            168.81255,
            167.97806,
            -166.66536,
            -163.92784,
            165.86012,
            -160.64189,
            161.81348,
            -166.7122,
            -170.45976,
            -170.93434,
            164.66747,
            -165.82011,
            -165.51503,
            161.53741,
            165.43382,
            160.52934,
            -165.33582,
            -164.32243,
            -163.03854,
            -161.01247,
            164.57118,
            163.4492,
            160.94273,
            -166.97842,
            -166.5353,
            -165.92578,
            166.988,
            166.43613,
            -164.76636,
            -168.00008,
            -162.23665,
            163.27821,
            167.11421,
            161.14592,
            -163.76105,
            -160.67075,
            -168.64998,
            -165.07507,
            -164.34868,
            -165.66562,
            162.233,
            164.58264,
            165.80527,
            162.0633,
            161.72806,
            -165.19324,
            -163.2416,
            -165.16194,
            -162.47891,
            164.1091,
            -160.95976,
            -162.18762,
            165.01439,
            -162.29207,
            160.33385,
            166.02773,
            -162.66676,
            -165.21266,
            -164.4178,
            162.12747,
            -166.68333,
            162.04533,
            -162.4284,
            -160.23776,
            -155.29012,
            158.78175,
            165.35808,
            -173.64941,
            211.75507,
            -285.7599,
            358.04688,
            379.5246,
            385.82022,
            -384.66086,
            -386.44427,
            -385.26572,
            380.91776,
            -380.98523,
            -384.76727,
            378.16287,
            -380.68964,
            381.14902,
            384.19153,
            381.07303,
            -380.50302,
            -383.66367,
            -390.85107,
            382.62708,
            -386.0591,
            -381.68204,
            -379.10788,
            364.2377,
            -304.26328,
            213.38318,
            165.9271,
            156.98956,
            -157.46437,
            -157.40637,
            -158.37825,
            157.45079,
            157.07262,
            -157.83113,
            152.51242,
            -156.3175,
            -157.42984,
            160.6131,
            -158.76997,
            -155.95978,
            160.53497,
            -157.27193,
            -157.00851,
            154.8657,
            -158.8176,
            153.32367,
            -159.8769,
            160.17159,
            -156.8206,
            -157.45834,
            159.06357,
            -158.24567,
            160.38875,
            157.6301,
            -156.06548,
            160.13548,
            -156.89441,
            -160.8554,
            160.8893,
            -155.65897,
            154.93192,
            154.71733,
            155.50513,
            -156.70277,
            155.13466,
            -157.9327,
            -159.61612,
            -157.02885,
            159.64111,
            -158.11557,
            -156.00523,
            -158.46323,
            158.02318,
            -157.18263,
            -158.25189,
            -153.31859,
            154.16188,
            -154.3176,
            -157.03438,
            158.4635,
            156.73857,
            -155.75189,
            -152.56458,
            -158.43004,
            160.33698,
            159.9946,
            156.28082,
            151.94417,
            152.2237,
            -150.87718,
            152.4494,
            152.67474,
            -154.55762,
            155.99724,
            -155.39392,
            -152.81181,
            155.89287,
            151.09042,
            -155.75243,
            -154.01216,
            153.1212,
            -153.265,
            -157.28253,
            154.91025,
            155.30992,
            153.95084,
            153.52531,
            -156.35434,
            159.02081,
            160.95078,
            153.96661,
            -156.13962,
            -155.37256,
            157.45805,
            153.08102,
            -154.73656,
            -152.49026,
            -152.74377,
            150.99478,
            150.87735,
            -149.00453,
            -152.62267,
            -152.76015,
            -151.78171,
            -153.9997,
            152.22751,
            150.58792,
            -149.81288,
            156.36119,
            -152.47935,
            149.03763,
            150.10526,
            155.84294,
            149.78522,
            148.28928,
            -152.5677,
            149.10484,
            -148.61098,
            -153.91179,
            154.55788,
            -149.8796,
            -147.98224,
            147.97685,
            -148.7274,
            151.01129,
            -154.02502,
            -156.9357,
            150.04187,
            -153.18854,
            -149.79648,
            -151.9761,
            -148.38991,
            -149.10315,
            148.60588,
            147.34315,
            147.87857,
            -147.16304,
            149.69472,
            149.46313,
            154.07346,
            151.14374,
            -153.60213,
            151.07669,
            154.83914,
            152.8483,
            -153.10272,
            155.5313,
            152.32306,
            148.10522,
            144.19296,
            -145.4064,
            149.32478,
            153.45892,
            -149.18233,
            -147.29803,
            -149.27733,
            -150.34297,
            149.78084,
            146.94498,
            -151.10316,
            149.96907,
            147.95343,
            -148.1724,
            -144.9735,
            146.96353,
            -146.94827,
            -153.76692,
            151.6151,
            152.76717,
            149.70396,
            148.94006,
            152.24644,
            -146.96434,
            -146.52599,
            145.75824,
            -148.12279,
            -148.45009,
            149.39505,
            -144.41719,
            148.5456,
            -145.0233,
            -145.31018,
            -147.25764,
            -147.41028,
            145.51802,
            149.80397,
            152.32535,
            151.75237,
            -150.05539,
            -144.63133,
            142.51062,
            140.02,
            142.77066,
            -145.80159,
            148.42776,
            -145.32042,
            146.21938,
            -142.27888,
            -148.05312,
            146.32283,
            149.31926,
            -147.786,
            149.26038,
            148.2346,
            -147.47,
            -145.2851,
            151.24626,
            -147.9135,
            -146.92479,
            -147.03519,
            -146.52917,
            145.22664,
            -146.03403,
            -145.15033,
            -146.7574,
            -143.1251,
            -146.12541,
            142.17331,
            148.2161,
            -145.31165,
            148.82149,
            -144.4811,
            -144.55502,
            141.46349,
            145.09795,
            142.08641,
            142.17221,
            145.57236,
            -139.2567,
            -148.96918,
            142.71759,
            -147.35622,
            144.09001,
            147.76463,
            -143.86874,
            143.2458,
            -145.36305,
            -143.75107,
            -144.62294,
            -142.17012,
            145.42274,
            142.98384,
            -142.71143,
            143.36244,
            -145.81064,
            -144.13705,
            144.89912,
            144.34549,
            142.08151,
            -140.78844,
            141.74866,
            138.93178,
            142.10695,
            144.60048,
            -144.00742,
            -141.33917,
            141.694,
            146.10414,
            144.33345,
            143.88023,
            146.37157,
            -143.40724,
            -141.90857,
            144.52724,
            143.95764,
            143.33441,
            141.5378,
            141.44157,
            -144.00613,
            143.16962,
            -146.56248,
            -145.34407,
            144.86359,
            -144.40686,
            -144.51453,
            143.09329,
            -141.8243,
            138.66106,
            -138.9095,
            136.32066,
            137.46085,
            -139.66638,
            140.90517,
            143.12471,
            -143.22014,
            142.84152,
            -139.70808,
            139.49704,
            142.17029,
            -134.01132,
            141.09651,
            -138.99477,
            -141.40819,
            137.28824,
            -142.36859,
            142.01666,
            135.08856,
            -140.57443,
            141.96614,
            -139.40381,
            138.11662,
            137.27328,
            139.56534,
            142.97293,
            142.10979,
            139.65724,
            -138.0631,
            138.33688,
            -134.4597,
            143.23949,
            137.01929,
            -137.66936,
            139.48724,
            -139.57234,
            141.56937,
            142.51515,
            139.61252,
            -135.15332,
            136.57498,
            140.94089,
            137.26248,
            137.34796,
            141.21614,
            -136.80428,
            135.14511,
            133.56839,
            136.51408,
            -136.99745,
            136.44662,
            139.39299,
            136.97726,
            138.61177,
            -136.58446,
            138.84766,
            134.324,
            -135.25938,
            137.11813,
            -135.47289,
            134.44286,
            -138.09775,
            -137.78606,
            -138.01581,
            -139.09401,
            -141.06242,
            -135.4274,
            137.20691,
            -135.68611,
            138.18832,
            -140.83774,
            -134.1234,
            -140.29623,
            -139.66629,
            140.2519,
            -136.09868,
            140.59428,
            -134.32016,
            136.02089,
            -137.76482,
            -142.64449,
            140.01498,
            -142.54724,
            137.38124,
            137.30789,
            -136.62514,
            -137.11285,
            136.34593,
            -137.7142,
            -136.58313,
            -136.23093,
            134.94269,
            -134.95613,
            -134.38051,
            -132.29237,
            -132.17993,
            -134.82138,
            -137.17316,
            -134.41737,
            133.88045,
            135.34999,
            -136.17538,
            -138.11449,
            -133.47345,
            -134.41309,
            135.16628,
            134.87254,
            135.1633,
            133.50267,
            138.274,
            133.75612,
            -133.13263,
            -134.84476,
            -131.04884,
            135.00368,
            133.86996,
            131.50188,
            -133.66292,
            -131.28531,
            131.96327,
            -131.10948,
            128.8557,
            133.32928,
            -134.7189,
            -128.25107,
            129.78857,
            128.32417,
            134.92883,
            135.32532,
            132.83585,
            -136.31589,
            -131.4852,
            127.84897,
            130.85461,
            -131.70648,
            -130.66295,
            -137.54729,
            131.1999,
            135.82895,
            134.53297,
            -128.78163,
            -128.11905,
            -129.4095,
            -130.75258,
            132.96146,
            130.41937,
            132.96823,
            -133.57323,
            -131.97165,
            -134.06071,
            135.69743,
            131.00093,
            134.61809,
            -127.58087,
            129.14442,
            130.15273,
            -132.07834,
            131.7968,
            -131.80399,
            -131.31752,
            -128.80919,
            128.18745,
            -129.94177,
            130.66536,
            127.462715,
            128.80963,
            129.75887,
            -126.17332,
            130.12468,
            128.78522,
            128.09056,
            -130.15004,
            -129.68976,
            129.41655,
            129.30983,
            -129.37863,
            -128.53978,
            -130.76213,
            -128.49326,
            -129.85182,
            127.67108,
            -129.04547,
            128.08644,
            -130.2219,
            127.99679,
            131.31496,
            129.53638,
            -131.3213,
            128.00572,
            128.77132,
            -130.90614,
            128.3694,
            128.98138,
            -131.06873,
            133.5194,
            -128.14491,
            127.13293,
            -127.57453,
            -128.40366,
            128.92268,
            126.36239,
            126.84662,
            -131.77103,
            128.63263,
            -130.16273,
            126.931114,
            -126.815414,
            126.47061,
            -128.35826,
            -125.56895,
            -130.13275,
            132.4114,
            -126.991554,
            -128.20317,
            126.315315,
            125.88332,
            126.15446,
            -127.8113,
            -125.09127,
            -127.04279,
            -124.851326,
            129.95052,
            128.28178,
            -126.11006,
            -125.163315,
            -125.46725,
            124.874214,
            123.070404,
            -124.51335,
            -126.37133,
            129.16403,
            125.658905,
            132.61487,
            128.19577,
            127.00247,
            128.928,
            123.26501,
            -127.80801,
            -127.698204,
            -127.49338,
            -124.67574,
            126.93529,
            -122.16911,
            121.46954,
            -123.945755,
            -124.25397,
            122.57128,
            -124.24737,
            124.892815,
            120.59638,
            -125.79225,
            126.312546,
            -127.13586,
            122.721115,
            129.49599,
            -126.35783,
            124.79983,
            126.47694,
            -121.91915,
            121.765816,
            125.83068,
            124.1087,
            127.093994,
            -123.29887,
            -122.92482,
            -124.92027,
            129.62798,
            -127.04796,
            126.33585,
            -124.528275,
            120.740875,
            -126.19499,
            121.728806,
            -123.077385,
            -128.2238,
            -126.683754,
            126.49137,
            -123.35667,
            127.48237,
            -124.65593,
            -125.165085,
            -125.36363,
            127.62683,
            121.686165,
            -122.814896,
            -123.64556,
            123.47728,
            122.16787,
            -125.88668,
            123.59328,
            117.98443,
            -123.67262,
            120.75265,
            -121.566986,
            -124.954575,
            -121.46782,
            124.11117,
            -120.50828,
            120.74166,
            122.97992,
            -123.599625,
            -119.33378,
            -119.20808,
            120.07201,
            119.10218,
            -124.6028,
            123.229324,
            123.26673,
            -120.46597,
            -121.34475,
            -119.16217,
            119.7616,
            122.443306,
            -122.586395,
            -118.32613,
            124.4913,
            -120.73421,
            123.46946,
            121.267975,
            120.178444,
            119.95886,
            -120.257866,
            121.41734,
            118.72967,
            121.811424,
            120.13746,
            120.8711,
            -120.23297,
            118.29063,
            -121.18248,
            122.98087,
            120.65002,
            -117.69398,
            118.820305,
            -117.935356,
            -120.43247,
            121.93559,
            -117.98075,
            123.280945,
            -120.64274,
            -120.78925,
            119.00283,
            -120.967094,
            119.99865,
            -118.845795,
            -119.72188,
            -119.75367,
            121.564125,
            118.605576,
            -116.34857,
            117.666306,
            117.37903,
            115.323586,
            -117.833916,
            -119.34903,
            -121.65791,
            121.46991,
            120.562065,
            -115.25611,
            118.61089,
            119.11826,
            -116.77403,
            -121.10264,
            -121.71884,
            -119.66675,
            -119.60386,
            -119.10082,
            -119.43242,
            117.88954,
            119.86655,
            114.03601,
            -114.84991,
            117.58956,
            120.23181,
            117.85117,
            119.286385,
            116.357765,
            121.53454,
            118.19008,
            -118.38689,
            121.7402,
            116.99861,
            -119.33675,
            118.74747,
            -118.85119,
            -121.07945,
            -119.27082,
            -116.09521,
            -114.956696,
            -118.6763,
            -123.063896,
            116.98634,
            119.37735,
            118.04289,
            116.27047,
            116.03668,
            119.36052,
            117.34972,
            -119.64768,
            119.08196,
            -115.987564,
            -118.51721,
            117.08424,
            117.535484,
            119.13024,
            -117.22343,
            118.23763,
            114.010414,
            -120.144356,
            114.07075,
            118.29277,
            -118.9237,
            -116.59178,
            -118.93763,
            115.39877,
            -114.26845,
            114.733055,
            116.12185,
            -116.85362,
            116.47418,
            -118.41259,
            116.97738,
            113.135635,
            115.14518,
            -115.80184,
            116.85162,
            -114.2757,
            116.948135,
            113.93525,
            -112.88292,
            116.88664,
            -114.90927,
            -113.6513,
            115.83742,
            115.81072,
            -115.987656,
            112.9226,
            -116.947235,
            111.33178,
            -116.74218,
            113.99002,
            -115.05573,
            115.10701,
            114.535194,
            -117.15753,
            -115.00101,
            -114.8573,
            -118.033646,
            -116.48353,
            111.939156,
            111.6864,
            114.4101,
            -112.36827,
            -113.51528,
            114.06645,
            -115.53988,
            -117.16242,
            -115.13042,
            114.34127,
            113.17746,
            111.085335,
            112.87489,
            -113.71082,
            -113.840576,
            -110.9598,
            -111.676186,
            -110.59781,
            113.0391,
            -110.52045,
            112.96727,
            110.3703,
            112.77954,
            110.97359,
            -111.17386,
            112.34912,
            -116.88913,
            115.35111,
            -113.67482,
            -111.82556,
            -109.628685,
            112.33731,
            -116.146416,
            112.63369,
            113.38885,
            -114.04488,
            -112.80399,
            -112.28743,
            -114.54472,
            109.25865,
            112.37511,
            -109.78721,
            112.52113,
            -111.78066,
            -112.33146,
            110.86829,
            -114.31308,
            108.64261,
            110.87101,
            -109.85589,
            -114.31863,
            -113.75142,
            114.28352,
            -111.73794,
            -112.14915,
            -109.315834,
            109.3814,
            -116.74302,
            115.25766,
            -112.49878,
            113.80198,
            -110.48467,
            -109.69234,
            108.94321,
            -108.03104,
            -111.756874,
            -109.98285,
            110.69545,
            -111.9099,
            -110.01003,
            110.65376,
            108.462944,
            -109.80283,
            -114.4538,
            112.04381,
            112.166084,
            -110.539085,
            -108.46714,
            112.75971,
            108.46807,
            111.71693,
            -109.15468,
            -107.469765,
            105.67645,
            107.9631,
            -105.09608,
            110.52046,
            113.24259,
            107.036934,
            110.14115,
            -109.27771,
            107.22255,
            110.863594,
            105.6628,
            -108.24264,
            -110.94666,
            108.94012,
            107.49802,
            -107.86777,
            107.37636,
            -109.203125,
            104.15043,
            -106.45268,
            112.80793,
            -105.35912,
            108.487526,
            -106.869995,
            -107.44013,
            108.58205,
            107.99932,
            107.70336,
            105.47852,
            110.14374,
            110.920395,
            -108.34515,
            -107.94602,
            -106.55609,
            -106.3628,
            109.05158,
            -108.223724,
            -106.00521,
            -108.38299,
            -107.53249,
            -105.96496,
            108.15683,
            107.348236,
            -111.60116,
            108.31238,
            -106.89647,
            107.93215,
            -106.2246,
            -108.913704,
            106.891,
            -102.12663,
            105.19869,
            -107.36248,
            -105.38825,
            109.134224,
            105.0773,
            107.48139,
            108.70011,
            -105.82342,
            -106.29433,
            -106.1842,
            -107.047714,
            -108.92946,
            108.97367,
            107.903725,
            105.681854,
            105.69796,
            -105.272804,
            104.409485,
            106.36993,
            -109.26666,
            -105.04115,
            -102.70962,
            -105.52154,
            106.60236,
            101.96415,
            105.2109,
            -106.76907,
            109.643974,
            -105.353676,
            106.31063,
            -106.128914,
            -103.34652,
            107.3799,
            -108.91009,
            107.77875,
            -105.53402,
            -105.321205,
            -104.678856,
            -108.66936,
            -103.86952,
            -102.67325,
            -109.12146,
            -105.1544,
            -104.27208,
            104.91245,
            106.62003,
            -106.28188,
            -103.42334,
            105.41271,
            104.339,
            -104.12508,
            -105.4941,
            107.758354,
            105.41596,
            105.05452,
            -103.85287,
            -103.640915,
            -103.151764,
            -102.95554,
            -105.04507,
            -105.80864,
            -102.166725,
            104.49895,
            -103.95953,
            103.95963,
            103.24905,
            -102.996086,
            -107.18496,
            104.648926,
            -104.90489,
            105.72478,
            105.210464,
            104.98848,
            -102.694176,
            104.4933,
            -106.8483,
            103.26807,
            104.59778,
            101.48828,
            101.9384,
            105.037445,
            103.34981,
            -103.01299,
            -106.80041,
            105.93158,
            103.70652,
            -103.09115,
            104.52143,
            -100.85082,
            -101.66821,
            102.30512,
            104.89556,
            104.277115,
            -101.40657,
            101.98332,
            -101.98073,
            -98.199936,
            99.861305,
            -101.057205,
            -101.84975,
            -102.84708,
            102.44077,
            102.22276,
            101.16945,
            -103.02072,
            97.32799,
            100.94339,
            -102.72653,
            101.41147,
            99.27803,
            101.80556,
            103.94754,
            -100.8464,
            -102.01047,
            -101.301216,
            105.57613,
            -101.90015,
            100.645775,
            -104.05422,
            -101.15637,
            -99.890465,
            99.04808,
            -99.20652,
            -96.37288,
            -102.07944,
            100.02381,
            -100.944954,
            -99.7504,
            -97.35875,
            100.58121,
            -104.04822,
            -101.866615,
            -97.815216,
            97.50091,
            101.46999,
            -99.68648,
            -96.470985,
            -101.80356,
            -101.62921,
            101.29038,
            101.46938,
            102.7934,
            -100.87329,
            -101.721695,
            -99.27412,
            -102.50104,
            -100.414085,
            -98.47753,
            96.86875,
            -98.16764,
            100.06029,
            -100.52508,
            -100.97012,
            102.2048,
            101.43538,
            99.31789,
            -98.749405,
            -101.53772,
            -98.679756,
            99.77626,
            -99.036644,
            -102.07656,
            100.633316,
            -100.02495,
            -99.586754,
            97.36436,
            -94.79099,
            -101.18694,
            -99.742615,
            94.86944,
            98.25539,
            98.515686,
            -100.80821,
            -98.78222,
            -98.78241,
            -102.1262,
            -98.54468,
            98.38833,
            97.52239,
            -98.04995,
            98.40163,
            96.522385,
            97.92608,
            98.807236,
            -97.253746,
            97.433655,
            94.972496,
            95.74109,
            98.5483,
            95.78574,
            95.75899,
            -95.64574,
            -98.38195,
            -96.78473,
            95.81292,
            97.408875,
            -98.40609,
            -98.971985,
            -99.056725,
            99.40599,
            -96.67848,
            -98.88583,
            96.19627,
            95.48844,
            98.51451,
            -96.22626,
            -98.40151,
            -95.731735,
            97.18636,
            96.29459,
            97.63344,
            -98.56134,
            -96.891716,
            -99.04935,
            -96.92572,
            -100.346405,
            100.13822,
            -99.80563,
            99.5454,
            -98.694336,
            98.85404,
            -96.80993,
            -96.05681,
            99.189766,
            -95.66408,
            99.40204,
            96.34406,
            -96.3171,
            98.86548,
            -94.510544,
            99.075455,
            -94.97187,
            95.83365,
            -95.34122,
            -100.73694,
            -97.56664,
            -94.03507,
            -93.31319,
            -93.807884,
            94.73025,
            -94.59172,
            -95.24796,
            -94.81514,
            -97.050674,
            -94.99643,
            97.258125,
            94.95773,
            -97.60928,
            -95.20339,
            96.6527,
            98.15663,
            95.97436,
            94.07466,
            -96.339905,
            96.163994,
            -95.29939,
            97.477745,
            95.45707,
            -95.11727,
            -96.070915,
            -92.20477,
            94.473045,
            -94.232765,
            94.67896,
            92.72593,
            92.56671,
            95.81422,
            -90.648155,
            90.30834,
            92.05798,
            97.13261,
            95.23712,
            96.025345,
            -94.4945,
            -95.31811,
            -92.9349,
            90.91211,
            -95.09839,
            -91.074,
            92.3753,
            -94.31313,
            93.35193,
            95.26035,
            95.1175,
            -93.08659,
            -96.015305,
            -94.99451,
            95.32347,
            -94.30218,
            -127.73743,
            260.25467,
            -341.74344,
            349.4803,
            -349.57162,
            355.66507,
            -350.47467,
            -348.82397,
            -354.3843,
            350.74683,
            351.52988,
            350.07416,
            351.49542,
            -351.53363,
            341.69772,
            253.98016,
            -123.92908,
            96.35082,
            91.06472,
            -90.83071,
            89.76168,
            -93.020065,
            92.122505,
            91.788315,
            94.38587,
            93.2701,
            96.80618,
            -94.33329,
            91.74874,
            -94.97966,
            87.78467,
            -93.466286,
            -90.549126,
            90.2092,
            90.39333,
            90.930084,
            -93.7383,
            -90.96982,
            94.94126,
            94.76527,
            90.39415,
            -87.88898,
            -92.357895,
            -93.17701,
            91.43986,
            -93.88028,
            -92.42176,
            92.392624,
            93.52929,
            -92.29189,
            91.806244,
            94.16847,
            92.72645,
            -90.631905,
            91.111626,
            94.20956,
            -92.746666,
            -89.8205,
            93.25323,
            91.76561,
            90.80371,
            -91.73216,
            90.36026,
            -90.24914,
            -89.0429,
            -90.13848,
            -94.63617,
            92.552635,
            -87.8357,
            88.97698,
            90.01937,
            -91.474174,
            90.30491,
            91.25903,
            -93.857956,
            -90.12232,
            89.17976,
            91.00141,
            -90.17809,
            90.44932,
            -91.48446,
            -90.670494,
            87.89067,
            -89.41945,
            89.39461,
            90.80413,
            91.835106,
            88.74135,
            -90.64168,
            89.46758,
            -88.2384,
            -91.41103,
            -91.4638,
            89.76086,
            -92.090225,
            89.94992,
            88.91854,
            -86.75279,
            90.903206,
            88.51376,
            -91.19053,
            -88.91907,
            -87.29996,
            -89.30956,
            -90.085464,
            -88.7723,
            85.93712,
            88.46164,
            -83.98686,
            92.1561,
            90.61922,
            90.60643,
            88.29586,
            -90.373726,
            89.47969,
            89.84404,
            -90.12294,
            -88.99007,
            -90.773796,
            89.895035,
            89.396355,
            90.91523,
            89.007034,
            88.42618,
            88.93815,
            -84.84777,
            86.013374,
            90.48436,
            89.58643,
            87.310684,
            89.27171,
            -91.34719,
            -89.86893,
            88.32109,
            89.54652,
            90.15279,
            -87.985,
            -90.942924,
            -88.68795,
            88.035515,
            -87.79725,
            -87.88885,
            -89.45489,
            90.15026,
            90.73036,
            -90.23115,
            -88.16208,
            -86.85605,
            87.18841,
            -88.03007,
            -88.335625,
            -87.95267,
            87.09243,
            86.33641,
            -84.23684,
            84.31417,
            -87.97882,
            -89.21101,
            -86.57653,
            87.97382,
            -86.883156,
            86.987076,
            88.23792,
            88.28264,
            88.055435,
            88.7569,
            -87.08417,
            -86.50282,
            -86.385475,
            -86.87072,
            86.16303,
            82.86042,
            -82.65116,
            -84.59029,
            85.517456,
            -84.718506,
            -82.145935,
            84.246376,
            85.20564,
            -84.13139,
            -85.9367,
            -85.08595,
            -84.7581,
            87.0059,
            -85.032875,
            -86.77656,
            86.90054,
            86.026566,
            87.221115,
            -86.26172,
            86.4634,
            86.41637,
            -85.39422,
            -85.31764,
            -85.189606,
            -84.90777,
            -86.256355,
            -83.18882,
            -87.727005,
            -85.79906,
            87.294624,
            84.57546,
            82.941574,
            -83.255394,
            85.95687,
            -82.73426,
            -90.193474,
            89.60817,
            -86.0904,
            85.95933,
            -84.07504,
            87.80966,
            85.69658,
            84.62348,
            -82.54377,
            -85.21901,
            -82.16161,
            -82.71387,
            83.04886,
            86.36565,
            82.9928,
            85.00881,
            -82.7519,
            -83.604454,
            -82.370804,
            83.74907,
            84.239624,
            -84.368195,
            84.148,
            -81.51137,
            81.932755,
            -85.02969,
            83.56593,
            -85.48574,
            -83.848274,
            84.33688,
            82.15279,
            -83.81886,
            84.41443,
            84.06285,
            -84.37137,
            -85.60574,
            85.33517,
            82.704414,
            -84.43552,
            83.24981,
            84.94968,
            84.61126,
            84.84621,
            -82.86623,
            82.846436,
            85.56009,
            -84.661354,
            -83.77089,
            82.254265,
            82.31268,
            -85.87284,
            -80.98326,
            84.81416,
            81.994095,
            82.889465,
            -83.39743,
            -83.63264,
            -79.44416,
            83.19508,
            -84.858864,
            -80.28624,
            80.9126,
            84.133484,
            -82.64657,
            -80.207954,
            79.66913,
            -82.94163,
            -81.11187,
            -77.97519,
            84.524155,
            82.257095,
            -79.528534,
            -80.2672,
            81.18492,
            81.99004,
            -83.27858,
            82.14135,
            -83.39237,
            -83.06615,
            -82.6775,
            83.69522,
            82.61325,
            83.03886,
            -82.964806,
            80.28323,
            -82.70617,
            81.55505,
            82.62689,
            -81.54999,
            -83.288445,
            79.40487,
            83.0974,
            82.23155,
            -80.889946,
            -80.68272,
            83.868225,
            79.985176,
            81.85757,
            -82.41381,
            81.395546,
            80.20574,
            -82.060814,
            -81.84468,
            81.18411,
            -80.886894,
            -80.649765,
            -81.30469,
            82.321205,
            80.191536,
            84.61131,
            81.21629,
            -80.34335,
            -78.150856,
            80.39544,
            -82.69081,
            -82.74622,
            -78.952034,
            80.569725,
            78.868324,
            80.97109,
            82.39051,
            -81.287674,
            81.39008,
            -80.637115,
            79.60062,
            -80.71599,
            -81.09341,
            82.34627,
            79.46911,
            -81.17929,
            80.08569,
            80.5542,
            81.15088,
            -79.19147,
            80.55516,
            78.09883,
            79.44612,
            -80.55906,
            -79.54021,
            82.0611,
            -82.54996,
            -80.51378,
            80.581215,
            -79.69647,
            -81.58751,
            -79.28042,
            -80.3295,
            -78.97799,
            80.45219,
            -78.25956,
            80.62249,
            81.35787,
            -78.44103,
            -80.145935,
            -81.24776,
            81.921906,
            79.5934,
            -79.02228,
            -78.73589,
            -80.0795,
            -80.172646,
            -80.184044,
            79.16296,
            83.076645,
            79.233604,
            -78.102844,
            -79.173004,
            -79.72988,
            -80.565895,
            -79.90308,
            -77.79583,
            -76.38421,
            -76.78437,
            79.24359,
            79.88505,
            81.30208,
            -77.83368,
            79.55301,
            -78.68695,
            77.54906,
            77.8912,
            -79.59458,
            78.91945,
            -79.635345,
            -78.160324,
            -79.82514,
            78.59701,
            77.92807,
            -79.09881,
            76.00108,
            -78.826035,
            -77.45325,
            -77.16714,
            79.28417,
            -79.78572,
            -76.800095,
            -77.71749,
            -77.2929,
            -77.97058,
            79.8381,
            -78.60236,
            78.12665,
            -79.699554,
            -78.99051,
            -79.329796,
            -79.25856,
            -76.49233,
            80.88603,
            78.49744,
            77.81647,
            78.812416,
            -76.72318,
            -79.075836,
            77.68909,
            79.331055,
            78.658745,
            78.02314,
            77.15896,
            78.60537,
            -78.69562,
            -78.80923,
            79.25679,
            75.91706,
            75.11976,
            -75.87312,
            73.95707,
            -80.145226,
            -79.04994,
            76.65267,
            79.70713,
            -77.242165,
            77.55094,
            78.32411,
            -77.07651,
            -77.91076,
            76.83914,
            -77.3901,
            -77.644325,
            -76.49085,
            76.36798,
            77.91761,
            79.0298,
            76.29232,
            74.87124,
            -76.22,
            -76.02609,
            -75.38951,
            76.50349,
            -77.187096,
            76.19159,
            77.39131,
            -78.70244,
            77.38244,
            78.41321,
            -77.07883,
            76.43525,
            -75.767456,
            74.44681,
            -75.84576,
            -75.97663,
            75.55867,
            76.259605,
            73.46658,
            -73.80196,
            -74.27307,
            -75.86652,
            77.07545,
            74.43983,
            74.7049,
            74.95663,
            -75.206215,
            -75.01795,
            77.50314,
            -77.41953,
            -74.899956,
            -72.96663,
            -76.19365,
            75.75745,
            74.02557,
            -75.407875,
            -77.357925,
            -75.31136,
            -75.2119,
            75.92347,
            74.23217,
            73.85335,
            75.070854,
            74.47896,
            -73.95039,
            75.333405,
            -72.6283,
            77.355644,
            -76.183975,
            75.772285,
            -75.16089,
            76.77479,
            -74.54663,
            74.87822,
            -76.11697,
            -74.50048,
            72.28355,
            75.55376,
            74.82278,
            74.04297,
            -74.9815,
            74.052505,
            -75.04484,
            75.12025,
            -74.90497,
            75.85481,
            74.29219,
            -74.06376,
            72.54926,
            -73.43396,
            -73.38735,
            -74.48375,
            -73.55257,
            -73.36559,
            72.93185,
            -75.974396,
            -72.80575,
            70.755,
            -76.149254,
            72.728745,
            73.07653,
            72.74797,
            -74.95305,
            -73.94901,
            74.202225,
            -71.21828,
            72.56897,
            -74.77349,
            71.26564,
            72.68357,
            -71.498146,
            70.48202,
            -72.88817,
            73.35673,
            74.395164,
            72.47055,
            -74.59806,
            -74.70931,
            -71.71865,
            73.61764,
            76.19678,
            75.380844,
            -72.21378,
            72.07689,
            -72.30549,
            -73.49598,
            71.19539,
            -72.05799,
            72.33064,
            -73.9156,
            74.17439,
            -75.414894,
            -72.378784,
            71.93644,
            72.44817,
            -71.60836,
            -71.81433,
            -70.76069,
            70.777054,
            73.56991,
            71.94234,
            72.53193,
            72.93343,
            72.86537,
            72.74121,
            72.11129,
            -73.22758,
            -72.06369,
            -70.38438,
            -71.34106,
            -71.44477,
            -70.70951,
            72.26927,
            72.872635,
            73.3186,
            -73.02237,
            -71.26323,
            -69.22005,
            72.953,
            -70.95219,
            -70.069046,
            -72.47905,
            -72.458244,
            -68.86792,
            72.133354,
            -72.68041,
            71.093475,
            -70.9711,
            -72.86007,
            -72.268,
            -70.930565,
            -71.88302,
            70.31842,
            70.86613,
            71.655975,
            70.42835,
            -69.75377,
            -71.13375,
            70.29335,
            -69.693726,
            -70.81545,
            -67.74293,
            -69.46521,
            -71.93547,
            -69.84587,
            -71.80558,
            -70.05457,
            70.56154,
            71.12821,
            69.91909,
            71.61483,
            -69.65105,
            -70.39462,
            -69.469734,
            70.449486,
            69.937096,
            -71.144226,
            -70.54533,
            70.11328,
            71.29071,
            -69.854454,
            -70.91939,
            -70.98229,
            -69.840935,
            69.26822,
            -69.87094,
            -69.29793,
            69.3881,
            70.88049,
            70.26642,
            -69.761375,
            -69.581154,
            70.48262,
            -71.828384,
            -69.964584,
            -69.38986,
            -68.68623,
            -70.11886,
            70.157036,
            -68.86573,
            70.52035,
            70.83671,
            69.7847,
            66.936035,
            69.15989,
            -67.36494,
            70.41017,
            68.649185,
            -71.90204,
            69.32142,
            69.000946,
            -70.38872,
            68.33695,
            -68.13874,
            70.2949,
            69.76437,
            -67.50457,
            -68.03895,
            68.13617,
            -67.10649,
            69.68628,
            69.36103,
            68.38164,
            -68.16916,
            69.29988,
            -68.4245,
            -68.94712,
            -68.26835,
            -69.47455,
            -66.99262,
            -70.00193,
            -69.40488,
            -70.14462,
            -67.75439,
            68.44613,
            -68.71845,
            68.35278,
            -70.64959,
            68.88618,
            68.16405,
            68.027725,
            68.76396,
            -67.35636,
            -67.25182,
            -68.16319,
            68.45427,
            -69.866844,
            67.62333,
            69.081985,
            68.032234,
            69.20445,
            -68.78568,
            -68.52335,
            -67.7596,
            69.59933,
            -70.736404,
            -69.15359,
            67.94652,
            -66.11997,
            66.69308,
            -71.158806,
            -92.041016,
            231.08226,
            324.70526,
            -333.03546,
            -334.43903,
            -335.46698,
            333.45938,
            334.62564,
            -332.41196,
            332.15567,
            -332.00067,
            -332.80338,
            -328.52148,
            281.2267,
            -119.65801,
            71.8037,
            68.06933,
            -66.93665,
            -67.499756,
            65.951164,
            -65.37945,
            67.26902,
            -66.30977,
            64.28277,
            65.61381,
            -66.264656,
            -68.14484,
            -67.29079,
            69.7587,
            67.42381,
            66.89194,
            -64.71225,
            64.562775,
            65.67886,
            65.76945,
            -66.33217,
            67.80407,
            -67.63457,
            65.52989,
            -65.09694,
            65.59624,
            67.20258,
            -65.033035,
            67.684296,
            -65.27838,
            -65.26468,
            -64.487236,
            66.78817,
            -66.592545,
            65.89892,
            -67.31905,
            66.613075,
            64.3811,
            67.235245,
            66.846924,
            65.48897,
            65.92112,
            66.692635,
            67.49501,
            67.29881,
            -64.18599,
            -65.79231,
            -68.663246,
            65.64258,
            -65.70898,
            -67.285675,
            65.43699,
            65.23334,
            -65.52679,
            -64.29991,
            -64.97779,
            64.11614,
            64.73528,
            64.00484,
            -67.13088,
            -65.9453,
            -65.58337,
            -64.35583,
            -64.27854,
            66.12662,
            -66.02766,
            -66.095955,
            -64.565994,
            64.73806,
            66.29461,
            -64.70423,
            63.56164,
            67.38833,
            -66.18791,
            65.844986,
            63.174294,
            65.166405,
            -64.694885,
            -61.488792,
            -63.06972,
            -64.2428,
            -65.39542,
            63.77387,
            64.46927,
            -63.22631,
            64.7399,
            -63.915657,
            65.05627,
            -64.77071,
            64.71118,
            -64.63742,
            -63.74226,
            63.48005,
            63.82106,
            66.09993,
            64.65256,
            65.18518,
            65.39836,
            61.853493,
            -63.508408,
            -64.71605,
            61.894123,
            -64.583496,
            64.042534,
            -66.28644,
            -65.066895,
            64.98415,
            62.65375,
            -62.255634,
            -65.86777,
            63.193882,
            -66.12467,
            63.569206,
            -63.50182,
            62.83416,
            -64.61915,
            -64.389626,
            64.7604,
            64.86286,
            -65.59103,
            -62.882114,
            62.65169,
            63.56714,
            66.48186,
            64.8083,
            -66.77501,
            -64.26336,
            -65.55548,
            -61.96553,
            -63.296265,
            -65.08464,
            64.25163,
            -60.967392,
            63.347042,
            64.17836,
            64.26562,
            -65.6274,
            -64.36799,
            62.1847,
            -62.761307,
            60.92705,
            -63.17035,
            -62.06719,
            -63.718285,
            62.05478,
            62.352608,
            -62.331245,
            -66.29286,
            -63.160656,
            -60.978054,
            63.147846,
            64.28054,
            63.689342,
            63.31716,
            64.87646,
            -63.20856,
            62.414078,
            61.89374,
            -62.218636,
            63.771355,
            62.260418,
            65.06731,
            64.784584,
            -62.951447,
            -63.28442,
            61.777092,
            -63.37267,
            -62.55635,
            -61.255222,
            -63.660915,
            -62.08081,
            62.949707,
            62.842434,
            -62.55368,
            62.68292,
            60.497936,
            -63.529316,
            -63.132904,
            62.05171,
            60.478596,
            62.59098,
            -63.24812,
            -60.008293,
            63.062668,
            -62.201138,
            60.96691,
            63.848236,
            62.775627,
            61.851776,
            61.271557,
            62.27516,
            63.071896,
            63.277603,
            -62.48521,
            -60.98878,
            61.59893,
            61.150654,
            -62.190945,
            61.32636,
            -60.16258,
            61.897472,
            -61.69285,
            60.496426,
            -60.89022,
            -60.025024,
            -61.13412,
            -60.67038,
            -61.10369,
            61.004402,
            -59.533176,
            -62.343338,
            58.819622,
            58.195183,
            64.38082,
            -60.921814,
            -60.592365,
            -61.509975,
            61.605618,
            -59.830334,
            -61.252686,
            60.229366,
            59.959335,
            -62.035686,
            61.558056,
            59.588005,
            -60.015526,
            59.640064,
            -58.842533,
            59.68948,
            61.236702,
            58.57436,
            -57.400898,
            59.601303,
            60.717796,
            58.912563,
            -61.85384,
            -58.782658,
            -59.990715,
            60.05714,
            -60.642475,
            -61.4685,
            60.91623,
            -60.17083,
            60.92505,
            -59.20606,
            -60.161797,
            62.27073,
            -59.536686,
            -59.911636,
            60.410213,
            -59.74624,
            -58.26161,
            60.290924,
            -59.455032,
            -60.217163,
            -58.946995,
            -59.013885,
            60.54459,
            57.320435,
            59.919056,
            -59.45035,
            -59.220726,
            59.88595,
            -60.270184,
            60.76744,
            -59.44252,
            59.990715,
            59.423798,
            -60.256428,
            -58.48267,
            59.84404,
            58.900166,
            57.41948,
            -60.988968,
            -59.456554,
            58.471775,
            57.903732,
            -58.981503,
            59.777027,
            57.3705,
            59.929394,
            -57.918007,
            -59.599293,
            -59.114246,
            -58.35086,
            58.729256,
            59.918324,
            57.231167,
            60.561768,
            -59.28017,
            58.230564,
            -56.713352,
            -57.391148,
            -57.624237,
            56.94549,
            59.962383,
            60.07252,
            57.885117,
            59.356865,
            -57.028473,
            -60.453293,
            -59.73046,
            58.735783,
            -57.72108,
            59.42319,
            -58.29658,
            55.861176,
            58.314487,
            -59.117928,
            59.19085,
            57.419262,
            58.619247,
            57.21083,
            -59.275192,
            -59.675728,
            -58.576157,
            -58.789677,
            -57.030655,
            -58.444366,
            -57.748665,
            56.611393,
            58.01725,
            56.501766,
            59.810402,
            57.977673,
            57.90439,
            -59.617855,
            56.06747,
            -56.139687,
            56.009506,
            58.567158,
            -60.147278,
            -56.65019,
            -58.200478,
            58.765873,
            -57.735703,
            57.677,
            56.1061,
            -55.762844,
            58.607937,
            55.622917,
            -56.453278,
            -58.034866,
            56.46552,
            -59.921177,
            59.38206,
            -56.67311,
            56.797844,
            -57.824425,
            -57.772755,
            56.303474,
            55.74289,
            -57.3449,
            -56.843407,
            57.16767,
            57.081963,
            57.38467,
            54.607773,
            -58.777126,
            -56.250454,
            55.79301,
            -57.751236,
            -55.56857,
            -56.99917,
            -57.04792,
            -56.70896,
            -57.956936,
            -55.43354,
            -55.966434,
            56.314552,
            -56.528652,
            54.47373,
            -54.519558,
            -54.47164,
            -55.440014,
            54.329662,
            -56.50605,
            -56.83198,
            56.74138,
            57.924957,
            58.54454,
            55.308674,
            -55.890938,
            -56.504765,
            -55.299488,
            55.162365,
            -56.924744,
            -56.491333,
            57.53533,
            56.305397,
            -55.942123,
            55.74184,
            55.196445,
            53.559444,
            -55.551273,
            -54.113426,
            -55.70367,
            -54.80118,
            -57.32062,
            54.29332,
            -56.80971,
            55.145985,
            -56.080906,
            -55.94465,
            54.04921,
            57.980137,
            55.741753,
            -57.28793,
            -55.16048,
            -54.72234,
            -55.368958,
            -56.0487,
            55.46438,
            56.10757,
            56.435604,
            -55.34951,
            -54.970623,
            -58.27684,
            -53.39056,
            55.44464,
            -54.622833,
            -56.896893,
            -54.95775,
            -54.48379,
            -55.166634,
            56.22221,
            53.39795,
            -53.70989,
            -56.187912,
            55.372643,
            54.788532,
            55.075893,
            53.29854,
            55.93729,
            -56.697468,
            -55.464188,
            -56.109665,
            -55.36043,
            56.177116,
            53.147476,
            -53.859314,
            -51.781357,
            -56.21095,
            56.804558,
            -54.099457,
            53.33531,
            54.91902,
            -54.522366,
            -51.999752,
            -55.08044,
            54.609074,
            56.88193,
            -53.301125,
            54.46277,
            55.278152,
            52.46688,
            -53.961376,
            54.200363,
            -54.673664,
            54.278038,
            53.95392,
            -52.99581,
            54.659702,
            -53.649307,
            53.064312,
            55.51794,
            54.503696,
            -55.84548,
            -53.081627,
            -54.228004,
            -53.91296,
            53.285847,
            -53.653267,
            54.47919,
            -56.033924,
            52.689674,
            54.576515,
            -54.54106,
            51.95917,
            -52.399063,
            -54.94123,
            -53.51873,
            52.64861,
            -53.302204,
            53.121704,
            -52.877956,
            53.931137,
            54.221893,
            -55.18216,
            -52.936268,
            52.120983,
            -52.41044,
            54.038506,
            -55.591064,
            52.379227,
            52.75694,
            52.718616,
            -53.02861,
            -53.423355,
            -54.264988,
            53.121815,
            53.40805,
            -54.79009,
            52.880295,
            -54.0601,
            -52.20287,
            -51.891785,
            -53.043816,
            54.69293,
            -53.78516,
            52.779446,
            51.39454,
            53.934227,
            53.60644,
            -52.942192,
            52.699352,
            52.898746,
            -53.493187,
            -51.633965,
            54.951473,
            52.350994,
            53.661934,
            -52.80656,
            51.94119,
            -52.625347,
            -53.25174,
            51.334805,
            -52.810165,
            51.65204,
            53.260883,
            -51.838215,
            53.247086,
            -51.569607,
            -51.841537,
            -52.288494,
            -50.845333,
            52.323967,
            53.37318,
            -52.157864,
            -52.350574,
            53.652233,
            51.32308,
            -51.50196,
            51.191807,
            -50.560276,
            -50.84457,
            -52.477154,
            -52.115448,
            -50.400757,
            50.19494,
            -51.210762,
            -51.163643,
            51.20712,
            51.81419,
            -51.55599,
            -52.304447,
            -50.98035,
            51.252872,
            -51.538773,
            -51.564213,
            52.50828,
            -50.91249,
            51.56053,
            -51.61225,
            -50.844406,
            -51.3291,
            -52.449833,
            50.234283,
            51.123226,
            50.04786,
            -51.769417,
            50.220562,
            -50.07235,
            -50.79774,
            -50.28532,
            52.806038,
            -52.222393,
            52.216297,
            -50.63641,
            -52.298927,
            -49.264915,
            -50.322258,
            51.06881,
            51.616043,
            51.862923,
            -52.12038,
            -50.76811,
            -51.273438,
            -51.05564,
            -51.571705,
            51.59184,
            52.53952,
            49.993767,
            48.532074,
            52.052498,
            -51.78115,
            50.086456,
            50.513695,
            50.379738,
            -52.942215,
            50.293858,
            -51.91599,
            50.731655,
            51.82843,
            -51.41331,
            51.395645,
            50.080826,
            51.20889,
            50.483112,
            -50.811302,
            49.551735,
            49.839706,
            -51.13547,
            -51.577324,
            -48.659943,
            -50.77657,
            -49.341568,
            -48.872734,
            -48.19588,
            -50.145634,
            -50.713776,
            51.734226,
            49.62988,
            49.686188,
            48.88209,
            48.79107,
            52.392372,
            -50.126793,
            -49.881477,
            50.347332,
            -50.53587,
            -49.45681,
            -51.790497,
            49.02249,
            -50.919823,
            -50.67473,
            50.013847,
            48.741627,
            50.285767,
            -49.784267,
            -50.41475,
            -49.534275,
            -49.2526,
            -49.701122,
            48.642296,
            48.86104,
            48.631393,
            50.155037,
            48.51894,
            50.24726,
            -49.565666,
            48.122772,
            48.463554,
            -50.096443,
            47.145283,
            -49.521038,
            50.153687,
            50.481842,
            -50.671722,
            -48.91783,
            -49.504536,
            48.17283,
            -50.312664,
            -48.35523,
            48.975147,
            -48.038795,
            -48.27251,
            47.849182,
            48.535797,
            49.151943,
            -48.926727,
            -49.419857,
            46.56042,
            -49.222366,
            48.666767,
            48.0982,
            47.80731,
            49.590984,
            48.436836,
            -49.957615,
            -48.503162,
            -46.913486,
            -50.800495,
            -47.20456,
            -49.582336,
            -49.635807,
            47.991688,
            -47.734688,
            48.355007,
            -48.314728,
            48.849377,
            -48.14541,
            48.054863,
            48.388065,
            48.772995,
            48.911476,
            -46.216057,
            -46.74434,
            48.873646,
            47.74667,
            46.30486,
            48.68959,
            -47.11332,
            -49.268135,
            46.377434,
            47.808056,
            -47.873726,
            48.556038,
            -46.585762,
            -46.124817,
            -48.33751,
            -48.580387,
            -46.15132,
            -46.259792,
            -48.858047,
            47.213367,
            -46.79796,
            49.112213,
            -48.707535,
            47.665325,
            47.664368,
            48.075615,
            -45.283165,
            49.18755,
            47.081966,
            47.488163,
            -48.652462,
            -46.785885,
            -46.447292,
            46.951733,
            -45.405552,
            47.958538,
            47.21284,
            -46.22256,
            -49.84512,
            46.037987,
            46.160503,
            47.441956,
            48.60405,
            -46.97724,
            -47.656406,
            45.77682,
            46.44605,
            -46.971306,
            -49.524895,
            -46.69847,
            44.96312,
            -46.564106,
            -47.918236,
            46.343662,
            47.492134,
            48.610195,
            -46.15675,
            -47.6147,
            46.92869,
            -46.391346,
            46.997017,
            46.40978,
            46.47373,
            -46.535713,
            -45.619762,
            -46.97801,
            46.03568,
            -46.62114,
            -46.944237,
            46.895844,
            46.103916,
            46.068924,
            -44.942722,
            -45.93867,
            -46.884995,
            -46.354385,
            -45.226646,
            -46.993694,
            47.024464,
            -46.734417,
            47.50222,
            -46.92279,
            45.133717,
            -46.420418,
            46.27433,
            46.80673,
            47.39968,
            46.069256,
            47.280743,
            46.331333,
            -46.252827,
            -45.15528,
            -44.866295,
            -47.498653,
            -45.525974,
            45.891415,
            45.859116,
            45.9038,
            44.309074,
            46.17377,
            -44.170994,
            45.22179,
            -45.964657,
            46.441715,
            45.40729,
            -45.965187,
            -44.94565,
            46.17016,
            -46.628696,
            -47.137512,
            46.080906,
            -46.694168,
            -46.744625,
            45.98134,
            44.8557,
            -46.097492,
            -46.90532,
            46.38653,
            46.7977,
            43.329227,
            -47.004097,
            -47.162212,
            45.78914,
            -46.115356,
            -46.667324,
            -44.1862,
            -45.46821,
            43.295395,
            45.48365,
            45.32552,
            -44.12527,
            45.49453,
            43.971424,
            -44.904667,
            -45.552795,
            44.53859,
            -43.14468,
            -44.782677,
            44.636192,
            45.537296,
            45.09318,
            -45.125347,
            45.816772,
            -45.298107,
            43.91961,
            43.90676,
            -45.012405,
            -45.16706,
            45.52459,
            45.68922,
            45.748573,
            -45.277336,
            45.375214,
            44.864235,
            -45.392254,
            45.778614,
            -44.37366,
            -45.001255,
            44.473053,
            -44.748898,
            -43.083515,
            44.265186,
            45.611076,
            44.023937,
            42.665672,
            -44.21854,
            45.227818,
            -43.890648,
            -45.000404,
            -44.207653,
            42.926674,
            -44.10266,
            44.47683,
            44.3519,
            43.093636,
            -43.207485,
            -44.532074,
            -44.452915,
            42.84351,
            -44.994236,
            42.94961,
            42.656685,
            42.547276,
            43.018272,
            45.27783,
            -44.25572,
            -42.957874,
            44.699093,
            42.1646,
            43.893578,
            -43.67679,
            -45.12426,
            -43.599686,
            -44.512375,
            43.47061,
            -42.97484,
            -44.43453,
            -44.60641,
            -41.483677,
            43.593204,
            -44.39043,
            45.444805,
            42.81428,
            43.78051,
            -44.224766,
            43.73068,
            -42.720108,
            42.223583,
            42.882626,
            -44.368416,
            42.356182,
            -43.595875,
            43.713875,
            41.905396,
            42.50446,
            43.57044,
            43.908558,
            43.197697,
            44.531277,
            43.131363,
            -43.722466,
            42.680603,
            44.01798,
            44.303925,
            43.374634,
            42.567173,
            -43.76496,
            43.008736,
            41.671677,
            44.768856,
            43.362778,
            -41.38935,
            -42.384697,
            -42.1272,
            43.265266,
            43.206673,
            -44.369934,
            -44.66826,
            -44.167137,
            43.638924,
            43.122425,
            43.15785,
            -41.934433,
            41.958073,
            -44.423683,
            -42.40803,
            42.384563,
            43.331303,
            41.768806,
            -42.08247,
            41.498405,
            43.257122,
            42.6949,
            -41.806065,
            -42.10601,
            -42.41431,
            43.190044,
            -43.5285,
            42.983284,
            -41.428288,
            42.20565,
            43.230118,
            -42.409477,
            43.213783,
            -41.80626,
            40.407486,
            -43.205296,
            41.599735,
            42.5235,
            40.037548,
            40.952007,
            41.934334,
            40.80946,
            -41.05753,
            41.028122,
            -40.987606,
            -43.146618,
            41.880882,
            40.989353,
            39.274345,
            41.381546,
            40.809994,
            40.999866,
            -41.934223,
            -41.39226,
            40.647133,
            -42.408703,
            42.722622,
            -41.12638,
            40.97031,
            43.248165,
            -41.30438,
            40.317867,
            -41.960407,
            -42.300484,
            -39.836613,
            -40.74965,
            40.48735,
            41.86928,
            -40.583374,
            42.137894,
            -42.94585,
            -40.956066,
            40.83663,
            -40.543797,
            40.131092,
            -38.485676,
            -40.88662,
            -43.288277,
            -42.18036,
            42.840057,
            -40.132343,
            -39.763935,
            -40.16013,
            -42.103294,
            -43.10429,
            -41.99978,
            -41.35397,
            39.89788,
            41.04974,
            41.14855,
            41.213684,
            -39.92466,
            43.000282,
            -40.143917,
            40.292908,
            -40.653236,
            39.23038,
            -40.015,
            -40.681087,
            -39.99084,
            40.61369,
            40.037807,
            -40.43901,
            -40.000496,
            40.227688,
            40.459705,
            -40.979454,
            39.67306,
            39.975048,
            39.69051,
            -39.942425,
            -39.905396,
            -40.15388,
            40.24249,
            40.382263,
            -41.505074,
            -40.773243,
            40.14854,
            -39.97207,
            -40.239628,
            -41.527298,
            -40.086693,
            -38.978745,
            -40.846893,
            -41.579075,
            -38.987553,
            -40.872147,
            40.443584,
            39.7088,
            39.81461,
            -39.215237,
            39.98649,
            40.769062,
            -39.717648,
            39.84296,
            -39.805782,
            40.023552,
            -41.349556,
            -38.901505,
            39.29431,
            -40.53066,
            -41.138897,
            39.74836,
            40.411915,
            39.421944,
            -40.13014,
            40.34992,
            -39.427517,
            41.224205,
            40.34954,
            39.53093,
            40.389862,
            37.554096,
            -40.660862,
            -40.211014,
            -36.18389,
            40.157333,
            39.800686,
            -38.975918,
            -38.601982,
            -38.847324,
            40.282776,
            -37.971348,
            40.489876,
            -38.28919,
            39.137768,
            -39.19003,
            -37.722572,
            40.023266,
            -39.041393,
            -38.91149,
            -39.912422,
            38.88615,
            -40.140984,
            36.468143,
            -37.61809,
            37.789383,
            -39.87151,
            39.49618,
            -39.007153,
            -38.636738,
            -37.209476,
            -41.525196,
            39.13404,
            -39.710278,
            38.552723,
            39.01164,
            -38.14346,
            38.293053,
            -37.63972,
            38.512585,
            -38.060516,
            -38.212784,
            -38.765606,
            39.480335,
            -39.07722,
            39.340107,
            -38.484684,
            -39.10186,
            38.907192,
            -37.280613,
            40.50522,
            -37.626713,
            -39.307915,
            38.75883,
            38.948807,
            36.456097,
            38.294613,
            -37.394184,
            38.152542,
            37.239586,
            -37.756805,
            38.39467,
            37.913074,
            -38.41641,
            -36.717754,
            35.884804,
            -38.143566,
            39.416515,
            -38.202915,
            39.2804,
            37.61172,
            37.369102,
            -36.813297,
            38.95118,
            -37.888412,
            38.63375,
            37.89961,
            -37.47989,
            36.92326,
            -38.651157,
            37.409725,
            36.918633,
            38.53891,
            36.92197,
            -36.82119,
            -37.57641,
            38.54012,
            -38.382076,
            37.741096,
            37.6194,
            -38.963276,
            36.07887,
            -38.022892,
            36.63073,
            37.23948,
            37.910606,
            -37.411583,
            37.514503,
            -36.689613,
            36.094135,
            -37.376873,
            -36.58403,
            -36.37373,
            36.56085,
            -36.73921,
            -37.246174,
            -37.260258,
            37.814568,
            36.73932,
            -36.944565,
            -36.410744,
            36.826523,
            36.64007,
            -37.98357,
            -37.739166,
            -38.14403,
            37.703133,
            -36.15552,
            38.50709,
            36.222446,
            -38.439686,
            -36.94812,
            -38.588474,
            -36.943676,
            -37.82799,
            -36.83316,
            -36.791485,
            -35.45296,
            -36.858692,
            -36.007874,
            -36.341705,
            -36.78996,
            37.251114,
            37.693913,
            -37.104618,
            35.215176,
            -36.261192,
            -36.447556,
            35.539776,
            -36.147728,
            36.677418,
            -37.103664,
            35.882076,
            36.838882,
            36.055058,
            -34.57409,
            35.342438,
            -34.093166,
            -36.442905,
            36.28809,
            -36.446686,
            35.7988,
            -35.971195,
            35.918148,
            36.398647,
            -37.50459,
            -36.23687,
            36.65114,
            -35.660316,
            34.900352,
            -36.55769,
            35.80538,
            37.083103,
            -35.52382,
            -36.462227,
            -37.273983,
            36.63352,
            37.591236,
            35.678185,
            -35.960373,
            -35.670284,
            37.482777,
            -35.21657,
            34.92688,
            34.650536,
            34.324142,
            -34.637066,
            -33.77396,
            35.643497,
            -35.509254,
            -32.419,
            34.76715,
            36.830833,
            -34.813015,
            -34.946033,
            35.23648,
            35.491272,
            35.27057,
            -36.395813,
            36.303715,
            -36.46968,
            36.72392,
            -36.21142,
            -34.16627,
            36.337955,
            -35.546803,
            -35.746254,
            34.627556,
            -35.476585,
            33.09323,
            -35.39167,
            36.066273,
            36.13751,
            -34.178463,
            -34.641335,
            -36.61888,
            -35.82327,
            -34.291527,
            -34.4799,
            36.023148,
            34.85249,
            -34.045338,
            -34.47761,
            34.733665,
            -37.211006,
            36.923546,
            -36.040203,
            -34.941986,
            -34.639977,
            -34.790306,
            33.604492,
            -31.81209,
            -36.049522,
            35.708557,
            34.877213,
            36.199554,
            34.301247,
            35.00406,
            -34.625072,
            34.780617,
            33.77536,
            34.714325,
            -34.670765,
            34.838425,
            -35.42052,
            -34.48308,
            34.711613,
            35.00458,
            -34.958057,
            -34.838387,
            33.432777,
            -33.744793,
            -31.287567,
            -36.057514,
            -34.28488,
            32.995777,
            -34.059177,
            34.087364,
            33.899628,
            -33.553814,
            34.55223,
            -34.457607,
            -34.341152,
            -34.5571,
            -33.775936,
            -35.49817,
            34.70468,
            -33.027554,
            31.849003,
            32.977127,
            34.18585,
            35.858604,
            -34.367245,
            -34.61926,
            33.585373,
            -31.781242,
            33.272175,
            34.527905,
            -31.987043,
            33.025623,
            33.17216,
            -33.370777,
            -33.4558,
            -33.133835,
            -33.341755,
            -32.12612,
            -32.24556,
            -35.157795,
            32.994213,
            31.910812,
            34.552742,
            -33.895462,
            -33.000515,
            -33.45753,
            32.968742,
            -32.031464,
            33.423107,
            -34.650845,
            33.17395,
            31.726944,
            30.953575,
            -31.970306,
            -33.97211,
            33.117954,
            32.28745,
            -31.895449,
            -32.5756,
            -32.095135,
            -32.457233,
            31.050642,
            -31.147562,
            -33.366867,
            32.116486,
            33.32806,
            32.254726,
            -32.260197,
            -33.140984,
            -30.889256,
            32.0393,
            31.882895,
            32.75333,
            31.143332,
            -33.09392,
            -30.009386,
            30.787113,
            -33.339108,
            32.614822,
            31.012415,
            30.795176,
            29.792004,
            31.305706,
            31.855455,
            30.973139,
            -32.123188,
            31.392853,
            32.133446,
            -29.888304,
            -32.709206,
            31.71242,
            31.879986,
            -31.882822,
            30.17586,
            31.036304,
            31.480633,
            -30.655924,
            -28.976881,
            30.338938,
            31.201317,
            31.402294,
            32.356693,
            28.845987,
            -30.182487,
            31.427748,
            30.42216,
            31.526087,
            -29.045801,
            31.853163,
            30.317568,
            30.84234,
            -30.81503,
            -30.495241,
            32.48303,
            32.21937,
            -30.520937,
            -31.368418,
            30.125137,
            30.725592,
            30.451647,
            -30.747774,
            30.910732,
            -30.517927,
            -29.998098,
            30.812983,
            -30.13568,
            28.61693,
            29.196804,
            30.661661,
            30.64854,
            -30.013744,
            32.27757,
            -29.8201,
            -30.339035,
            30.264498,
            -30.754797,
            30.954144,
            -30.447659,
            -28.656723,
            -30.412117,
            29.911278,
            29.921438,
            -29.672329,
            29.555115,
            -28.478745,
            30.193811,
            28.7603,
            30.101809,
            -29.59805,
            28.69221,
            29.239067,
            -30.256538,
            29.870003,
            -31.300125,
            -29.186705,
            30.052607,
            -30.477633,
            -28.509178,
            -29.437614,
            27.79892,
            27.680151,
            -29.28622,
            30.052015,
            -30.837675,
            29.833332,
            -28.853884,
            29.257282,
            -29.423237,
            -29.079248,
            -28.119976,
            29.62141,
            -29.11661,
            29.401525,
            -29.221376,
            -29.879747,
            28.773125,
            -29.059008,
            27.623686,
            30.505547,
            -29.410467,
            -30.208376,
            -29.774042,
            -28.060894,
            28.397211,
            -29.913773,
            -29.747223,
            -29.192825,
            -29.69196,
            29.313372,
            28.754686,
            29.159193,
            -27.580318,
            -28.149645,
            -28.329975,
            -29.891872,
            -28.363176,
            30.252888,
            -28.76835,
            -28.863476,
            -29.614668,
            -28.435562,
            27.717299,
            -30.30045,
            -29.14059,
            26.21242,
            -27.381935,
            -28.690332,
            -27.168844,
            28.460077,
            28.294722,
            -27.78939,
            27.950241,
            -28.885513,
            27.133524,
            -27.581285,
            -27.946981,
            -29.671556,
            28.265358,
            26.514894,
            -27.505241,
            -28.021866,
            27.2838,
            26.527103,
            -28.453314,
            27.257673,
            -26.606903,
            -27.669737,
            26.170334,
            -28.304752,
            -26.767576,
            26.993483,
            26.381462,
            -26.056843,
            -28.16423,
            -27.7475,
            -28.270092,
            -27.532589,
            -28.353012,
            28.010593,
            -27.51598,
            26.180649,
            27.59772,
            26.608152,
            -26.663883,
            -26.476503,
            -26.010275,
            27.426645,
            -26.957573,
            -27.370659,
            26.35108,
            -26.336765,
            27.382397,
            -27.120937,
            27.491453,
            26.574505,
            28.534557,
            25.396715,
            -24.657919,
            -27.281057,
            -25.968575,
            26.986238,
            26.851357,
            25.351294,
            -24.811699,
            -26.455048,
            25.31453,
            27.12486,
            -26.290617,
            -24.707758,
            -26.492044,
            -25.37665,
            27.218956,
            25.857452,
            25.010582,
            25.74132,
            -24.603825,
            -26.998611,
            27.182388,
            25.956007,
            -25.392885,
            27.171558,
            -25.911974,
            25.221624,
            25.772224,
            -25.004072,
            24.821686,
            -25.689215,
            -24.965363,
            -24.817196,
            25.996204,
            -24.044888,
            -26.329958,
            -24.678827,
            25.031462,
            -25.763708,
            23.955523,
            25.955847,
            -27.01721,
            25.110664,
            24.664402,
            -24.857994,
            25.457146,
            24.858606,
            -26.10812,
            -24.700935,
            -25.279324,
            -23.31582,
            24.776356,
            -24.833946,
            -25.530878,
            -26.17203,
            24.208616,
            25.781763,
            24.762354,
            -26.748314,
            -24.880295,
            25.581,
            -23.899817,
            -24.761583,
            -23.182499,
            24.315647,
            -23.424265,
            -23.10058,
            -23.093334,
            23.372002,
            -23.58001,
            -25.36628,
            22.698244,
            24.250032,
            -24.350895,
            24.200851,
            -24.490881,
            24.298817,
            24.728647,
            26.099566,
            23.354229,
            -22.419796,
            24.578444,
            22.985409,
            -24.635992,
            -24.465786,
            -24.380924,
            -22.060099,
            24.800343,
            24.652311,
            23.70349,
            -22.86156,
            -23.211422,
            24.083405,
            -23.034983,
            -24.870186,
            -25.41847,
            -22.505625,
            -24.150051,
            -22.620749,
            24.202171,
            -24.070671,
            24.123972,
            -22.306017,
            -23.109879,
            -22.612953,
            23.826782,
            -22.061918,
            -22.732925,
            -25.809828,
            23.13845,
            -21.17716,
            24.38903,
            22.76799,
            -22.85122,
            23.142405,
            -23.258512,
            -21.8386,
            -23.038994,
            -22.029102,
            -22.33958,
            -22.854563,
            -21.350061,
            21.295145,
            22.0148,
            -22.58052,
            -23.752443,
            22.515787,
            23.647753,
            22.777422,
            21.895208,
            20.410484,
            22.057735,
            -23.609251,
            21.34194,
            21.537447,
            21.758053,
            23.733662,
            23.139519,
            21.59562,
            -21.816534,
            20.957243,
            22.39922,
            -21.992886,
            -22.596125,
            21.484953,
            22.476046,
            22.28179,
            20.56944,
            21.369059,
            23.95593,
            -21.186287,
            20.700514,
            -21.01825,
            -22.232525,
            -22.192186,
            -22.166142,
            22.187279,
            22.107141,
            20.261753,
            -21.311663,
            20.322632,
            20.68756,
            -19.986012,
            21.745272,
            -22.193138,
            -19.655708,
            23.04614,
            21.646076,
            -21.285385,
            -21.501797,
            -21.177645,
            -20.751154,
            -21.176815,
            -21.94505,
            20.871584,
            20.921474,
            -21.154175,
            20.748665,
            -20.23704,
            22.69392,
            -21.404348,
            20.957844,
            20.893536,
            19.643711,
            -19.942427,
            -22.312801,
            22.280987,
            19.692207,
            -18.74975,
            -18.558893,
            -21.483658,
            -20.577776,
            -20.70883,
            -19.300907,
            20.689089,
            -20.84186,
            -19.722076,
            -19.318686,
            -20.032907,
            19.948275,
            19.680094,
            -20.330805,
            18.910946,
            -20.18199,
            18.97037,
            -18.40464,
            19.664425,
            -20.64357,
            17.702402,
            -18.717436,
            -18.72082,
            20.624723,
            -20.12477,
            -19.53634,
            -18.959038,
            19.404184,
            20.703804,
            18.302767,
            18.062002,
            19.434486,
            -19.440222,
            -19.101992,
            18.342573,
            -17.984497,
            -19.415133,
            18.389938,
            -17.6461,
            18.499989,
            -17.713743,
            -17.766268,
            18.824297,
            -19.24355,
            -20.037247,
            -17.926432,
            -17.313885,
            19.241762,
            -18.060534,
            17.535849,
            -20.494013,
            18.634495,
            20.360277,
            18.122232,
            19.708231,
            -18.45016,
            17.79138,
            -19.80097,
            16.103918,
            -17.561771,
            17.180012,
            18.539572,
            -17.594992,
            18.344173,
            -17.480448,
            18.564596,
            -18.678995,
            -17.674484,
            16.527126,
            -16.145184,
            -18.258238,
            -17.158642,
            15.792081,
            -17.780651,
            18.85165,
            16.67864,
            15.916081,
            -16.487375,
            -17.1369,
            -16.052519,
            17.06169,
            18.204042,
            -16.939472,
            -17.86801,
            18.704916,
            16.57862,
            -18.12842,
            16.799587,
            17.183907,
            -15.72932,
            17.165134,
            -16.812313,
            16.603537,
            -15.699383,
            17.705671,
            16.73992,
            -17.08696,
            -16.631987,
            -16.574892,
            -17.52014,
            -14.653339,
            -17.821966,
            -15.960413,
            18.164534,
            -15.830348,
            16.516827,
            -16.435858,
            15.024074,
            -16.516254,
            15.027513,
            16.495335,
            16.048187,
            -16.427458,
            16.223967,
            14.30592,
            17.266712,
            15.133361,
            16.313887,
            14.676913,
            16.204798,
            15.16019,
            -15.521066,
            14.664936,
            -14.445678,
            16.320278,
            -16.264875,
            13.977014,
            -14.2027235,
            -15.8703985,
            -13.793635,
            14.68962,
            14.985089,
            -14.122217,
            -15.467843,
            -14.381159,
            -15.198908,
            15.0566225,
            -14.044708,
            -14.453376,
            -14.531279,
            15.387726,
            -16.49144,
            13.799187,
            15.628975,
            14.502868,
            -14.477395,
            15.219666,
            14.817227,
            13.417116,
            -14.533744,
            -14.184732,
            -14.343338,
            -14.624256,
            14.418963,
            -12.746217,
            -13.092987,
            14.221765,
            -14.472001,
            13.830579,
            -14.081556,
            12.3345995,
            15.442802,
            13.083994,
            -12.615521,
            12.557031,
            14.267156,
            13.834671,
            14.159422,
            13.716201,
            13.871092,
            -14.321626,
            15.624995,
            -12.66488,
            -13.46883,
            -13.313092,
            -14.604291,
            12.367549,
            14.003825,
            -11.776432,
            13.016178,
            13.210076,
            12.458779,
            -12.176744,
            11.816758,
            -13.192253,
            -10.817635,
            11.124778,
            -11.46464,
            -14.623574,
            12.711826,
            12.330635,
            11.836184,
            -11.171112,
            -11.349251,
            -12.006696,
            12.625569,
            -12.104921,
            -11.150993,
            11.581493,
            12.931788,
            11.837559,
            -11.625236,
            11.742782,
            12.65295,
            -11.298203,
            11.872695,
            -10.497248,
            11.491383,
            11.729223,
            10.813235,
            -12.435286,
            11.224647,
            10.229562,
            -10.676235,
            10.146114,
            10.200087,
            -11.227423,
            11.559691,
            12.017453,
            10.545389,
            10.754475,
            11.25537,
            -9.243484,
            9.34558,
            -8.924784,
            -8.925159,
            -10.515249,
            10.518326,
            -8.65893,
            -8.284538,
            -9.82869,
            9.683651,
            7.625086,
            9.539605,
            -8.833434,
            9.518975,
            -8.631654,
            -8.174976,
            9.136648,
            7.503391,
            8.263506,
            8.046428,
            -9.575889,
            -7.386245,
            -6.946397,
            6.437481,
            -6.9055467,
            6.309657,
            -7.283715,
            8.89724,
            -7.821819,
            -7.697759,
            -7.1857715,
            -8.031479,
            6.18595,
            7.2908583,
            -6.514728,
            4.7587204,
            -6.6548014,
            7.916879,
            5.719551,
            -6.765744,
            4.9986734,
            6.940376,
            5.3977227,
            4.7982287,
            4.86791,
            -5.954917,
            4.455157,
            4.2256975,
            4.768814,
            6.1262717,
            -4.6334615,
            -4.0402575,
            -3.6827855,
            -3.6857526,
            -4.1707764,
            2.3570395,
            3.080712,
            -0.28484115,
            -1.1142883,
            0.0,
        ];

        type T = f32;

        let n = diag.len();
        let max = T::max(
            crate::col::from_slice(&diag).norm_max(),
            crate::col::from_slice(&subdiag).norm_max(),
        );
        let mut u = Mat::from_fn(n + 1, n + 1, |_, _| T::NAN);
        let mut v = Mat::from_fn(n, n, |_, _| T::NAN);
        let s = {
            let mut diag = diag.clone();
            let mut subdiag = subdiag.clone();
            compute_bidiag_real_svd(
                &mut diag,
                &mut subdiag,
                Some(u.as_mut()),
                Some(v.as_mut()),
                4,
                128,
                T::EPSILON,
                T::MIN_POSITIVE,
                Parallelism::None,
                make_stack!(bidiag_real_svd_req::<T>(
                    n,
                    4,
                    true,
                    true,
                    Parallelism::None
                )),
            );
            Mat::from_fn(n + 1, n, |i, j| if i == j { diag[i] } else { 0.0 })
        };

        let reconstructed = &u * &s * v.transpose();
        let mut err = 0.0;

        for j in 0..n {
            for i in 0..n + 1 {
                let target = if i == j {
                    diag[j]
                } else if i == j + 1 {
                    subdiag[j]
                } else {
                    0.0
                };

                err = T::max(err, (reconstructed.read(i, j) - target).abs());
                assert_approx_eq!(reconstructed.read(i, j), target, 1e-2 * max);
            }
        }
    }

    #[test]
    fn test_deflation43() {
        let approx_eq = crate::ApproxEq::<f32>::eps();

        let eps = f32::EPSILON;
        let zero_threshold = f32::MIN_POSITIVE;

        let n = 7;
        let k = 3;
        let diag_orig = &*vec![1.0, 5.0, 3.0, 1e-7, 4.0, 2.0, 2e-7_f32];
        let col_orig = &*vec![1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0_f32];

        let n_jacobi = 2;
        let jacobi_coeffs = &mut *vec![JacobiRotation { c: 0.0, s: 0.0 }; n_jacobi];
        let jacobi_indices = &mut *vec![0usize; n_jacobi];
        let perm = &mut *vec![0usize; n];
        let diag = &mut *(diag_orig.to_vec());
        let col = &mut *(col_orig.to_vec());

        let (jacobi_0i, jacobi_ij) = deflate(
            diag,
            col,
            jacobi_coeffs,
            jacobi_indices,
            &mut *vec![0usize; n],
            perm,
            k,
            eps,
            zero_threshold,
            PodStack::new(bytemuck::cast_slice_mut(&mut *vec![0usize; 2 * n])),
        );
        assert!(all(jacobi_0i == 2, jacobi_ij == 0));

        let perm_inv = &mut *vec![0usize; n];
        for (i, p) in perm.iter().enumerate() {
            perm_inv[*p] = i;
        }

        let P = crate::perm::PermRef::new_checked(perm, perm_inv);

        let mut M_orig = Mat::<f32>::zeros(n, n);
        for i in 0..n {
            M_orig[(i, i)] = diag_orig[i];
            M_orig[(i, 0)] = col_orig[i];
        }

        let mut M = Mat::<f32>::zeros(n, n);
        for i in 0..n {
            M[(i, i)] = diag[i];
            M[(i, 0)] = col[i];
        }

        M = P.inverse() * M * P;

        for (&rot, &i) in core::iter::zip(&*jacobi_coeffs, &*jacobi_indices).rev() {
            let (x, y) = M.two_rows_mut(0, i);
            rot.apply_on_the_left_in_place(x, y);
        }

        for j in 0..n {
            for i in 0..n {
                assert!(M[(i, j)] ~ M_orig[(i, j)]);
            }
        }
    }

    #[test]
    fn test_deflation44() {
        let approx_eq = crate::ApproxEq::<f32>::eps();

        let eps = f32::EPSILON;
        let zero_threshold = f32::MIN_POSITIVE;

        let n = 7;
        let k = 3;
        let diag_orig = &*vec![1.0, 5.0, 3.0, 1.0, 4.0, 2.0, 1.0];
        let col_orig = &*vec![1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0_f32];

        let n_jacobi = 1;
        let jacobi_coeffs = &mut *vec![JacobiRotation { c: 0.0, s: 0.0 }; n_jacobi];
        let jacobi_indices = &mut *vec![0usize; n_jacobi];
        let perm = &mut *vec![0usize; n];
        let diag = &mut *(diag_orig.to_vec());
        let col = &mut *(col_orig.to_vec());

        let (jacobi_0i, jacobi_ij) = deflate(
            diag,
            col,
            jacobi_coeffs,
            jacobi_indices,
            &mut *vec![0usize; n],
            perm,
            k,
            eps,
            zero_threshold,
            PodStack::new(bytemuck::cast_slice_mut(&mut *vec![0usize; 2 * n])),
        );
        assert!(all(jacobi_0i == 0, jacobi_ij == 1));

        let perm_inv = &mut *vec![0usize; n];
        for (i, p) in perm.iter().enumerate() {
            perm_inv[*p] = i;
        }

        let P = crate::perm::PermRef::new_checked(perm, perm_inv);

        let mut M_orig = Mat::<f32>::zeros(n, n);
        for i in 0..n {
            M_orig[(i, i)] = diag_orig[i];
            M_orig[(i, 0)] = col_orig[i];
        }

        let mut M = Mat::<f32>::zeros(n, n);
        for i in 0..n {
            M[(i, i)] = diag[i];
            M[(i, 0)] = col[i];
        }

        M = P.inverse() * M * P;

        for (&rot, &i) in core::iter::zip(&*jacobi_coeffs, &*jacobi_indices).rev() {
            let (i, j) = (i - 1, i);
            let (pi, pj) = (perm[i], perm[j]);

            let (x, y) = M.two_rows_mut(pi, pj);
            rot.apply_on_the_left_in_place(x, y);

            let (x, y) = M.two_cols_mut(pi, pj);
            rot.transpose().apply_on_the_right_in_place(x, y);
        }

        for j in 0..n {
            for i in 0..n {
                assert!(M[(i, j)] ~ M_orig[(i, j)]);
            }
        }
    }

    #[test]
    fn test_both_deflation() {
        let approx_eq = crate::ApproxEq::<f32>::eps();

        let eps = f32::EPSILON;
        let zero_threshold = f32::MIN_POSITIVE;

        let n = 7;
        let k = 3;
        let diag_orig = &*vec![1.0, 5.0, 3.0, 2.0, 4.0, 2.0, 0.0];
        let col_orig = &*vec![1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0_f32];

        let n_jacobi = 2;
        let jacobi_coeffs = &mut *vec![JacobiRotation { c: 0.0, s: 0.0 }; n_jacobi];
        let jacobi_indices = &mut *vec![0usize; n_jacobi];
        let perm = &mut *vec![0usize; n];
        let diag = &mut *(diag_orig.to_vec());
        let col = &mut *(col_orig.to_vec());

        let (jacobi_0i, jacobi_ij) = deflate(
            diag,
            col,
            jacobi_coeffs,
            jacobi_indices,
            &mut *vec![0usize; n],
            perm,
            k,
            eps,
            zero_threshold,
            PodStack::new(bytemuck::cast_slice_mut(&mut *vec![0usize; 2 * n])),
        );
        assert!(all(jacobi_0i == 1, jacobi_ij == 1));

        let perm_inv = &mut *vec![0usize; n];
        for (i, p) in perm.iter().enumerate() {
            perm_inv[*p] = i;
        }

        let P = crate::perm::PermRef::new_checked(perm, perm_inv);

        let mut M_orig = Mat::<f32>::zeros(n, n);
        for i in 0..n {
            M_orig[(i, i)] = diag_orig[i];
            M_orig[(i, 0)] = col_orig[i];
        }

        let mut M = Mat::<f32>::zeros(n, n);
        for i in 0..n {
            M[(i, i)] = diag[i];
            M[(i, 0)] = col[i];
        }

        M = P.inverse() * M * P;

        for (&rot, &i) in core::iter::zip(&jacobi_coeffs[1..], &jacobi_indices[1..]).rev() {
            let (i, j) = (i - 1, i);
            let (pi, pj) = (perm[i], perm[j]);

            let (x, y) = M.two_rows_mut(pi, pj);
            rot.apply_on_the_left_in_place(x, y);

            let (x, y) = M.two_cols_mut(pi, pj);
            rot.transpose().apply_on_the_right_in_place(x, y);
        }
        for (&rot, &i) in core::iter::zip(&jacobi_coeffs[..1], &jacobi_indices[..1]).rev() {
            let (x, y) = M.two_rows_mut(0, i);
            rot.apply_on_the_left_in_place(x, y);
        }

        for j in 0..n {
            for i in 0..n {
                assert!(M[(i, j)] ~ M_orig[(i, j)]);
            }
        }
    }
}
