% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/residuals.flexsurvreg.R
\name{coxsnell_flexsurvreg}
\alias{coxsnell_flexsurvreg}
\title{Cox-Snell residuals from a parametric survival model}
\usage{
coxsnell_flexsurvreg(x)
}
\arguments{
\item{x}{Object returned by \code{\link{flexsurvreg}} or \code{\link{flexsurvspline}} representing a fitted survival model}
}
\value{
A data frame with a column called \code{est} giving the Cox-Snell residual, defined as the fitted cumulative hazard at each data point.
 fitted cumulative hazard at the given observed data point, and other columns indicating the observation time
 and covariate values defining the data at this point.   

An extra column \code{"(qexp)"} gives the equally-spaced quantiles of a standard 
exponential distribution in the same order as \code{est}.   To check the fit of the model, 
\code{"(qexp)"} is plotted against \code{est}, and the points should form a straight line 
through the origin with slope 1.
}
\description{
Cox-Snell residuals from a parametric survival model
}
\examples{

  fitg <- flexsurvreg(formula = Surv(futime, fustat) ~ age, data = ovarian, dist = "gengamma")
  cs <- coxsnell_flexsurvreg(fitg)
  
  ## Model doesn't appear to fit well since the cumulative hazards are underestimated.
  ## In this example, this is probably because the dataset is small, 
  ## hence the point estimate is noisy.
  plot(cs$"(qexp)", cs$est, pch=19, xlab="Theoretical quantiles", ylab="Cumulative hazard")
  abline(a=0,b=1,col="red",lwd=2)
  
  ## Alternative way to produce the same plot using "qqplot"
  qy <- qexp(ppoints(nrow(cs),0))
  qqplot(qy, cs$est)
  abline(a=0,b=1, col="red", lwd=2)
  
  ## A log transform may or may not bring out the pattern more clearly
  plot(log(cs$"(qexp)"), log(cs$est), pch=19)
  abline(a=0,b=1, col="red", lwd=2)
  
  ## In the model `fitg`, the fitted cumulative hazard is lower than the true cumulative hazard
  ## Another way to show this is to compare parametric vs nonparametric estimates of 
  ## the cumulative hazard 
  plot(fitg, type="cumhaz", ci=FALSE)
  
  ## Alternative example: where the true model is fitted to simulated data
  ## The model fits well
  y <- rweibull(10000, 2, 2)
  fite <- flexsurvreg(Surv(y) ~ 1, dist="weibull")
  cs <- coxsnell_flexsurvreg(fite)
  plot(cs$"(qexp)", cs$est, pch=19, xlab="Theoretical quantiles", ylab="Cumulative hazard")
  abline(a=0,b=1,col="red",lwd=2)
  
}
