#' Collect occurrence records from a pre-existing DOI or URL
#'
#' @description `r lifecycle::badge("experimental")` 
#' 
#' Download occurrence records using an existing DOI or URL. Pre-existing DOIs 
#' and URLs come from previously generated downloads using `atlas_occurrences` 
#' or online.
#'
#' @param url `string`: Retrieve occurrence
#' records previously downloaded from the ALA, using the URL provided via email.
#' @param doi `string`: Retrieve occurrence
#' records previously downloaded from the ALA, using the DOI generated by the
#' data.
#'
#' @return An object of class `tbl_df` and `data.frame` (aka a tibble) of 
#' occurrences 
#' 
#' @export collect_occurrences

collect_occurrences <- function(url, doi){

  if(missing(doi) & missing(url)){
    abort("one of either `doi` or `url` must be specified")
  }

  if(!missing(doi)){
    if(is.null(doi)){
      abort("Please specify a valid `doi`")
    }else if(grepl("^http", doi)){
      bullets <- c(
        "It looks like you have supplied a URL to the `doi` argument",
        i = "The `doi` argument does not accept DOIs formatted as URLs",
        i = "If you are supplying an ALA download url, pass it to `url` instead")
      abort(bullets)
    }else{
      result <- doi_download(doi)
    }
  }
  
  if(!missing(url)){
    if(is.null(url)){
      abort("Please specify a valid `url`")
    }else{
      result <- url_download(url)
    }
  } 
    
  if(is.null(result)){
    system_down_message("collect_occurrences")
    return(tibble())
  }else{
    return(tibble(result))
  }
  
}


doi_download <- function(doi, error_call = caller_env()) {
  # strip useful part of DOI
  doi_str <- str_split(doi, "ala.")[[1]][2]
  if (is.na(doi_str)) {
    bullets <- c(
      "DOI has not been generated by the ALA.",
      i = "DOIs created by the ALA have a prefix of 10.26197/ala."
    )
    abort(bullets, call = error_call)
  }
  
  verbose <- getOption("galah_config")$verbose
  if(verbose) {
    cat("Downloading\n")
  }

  path <- atlas_url("doi_download", doi_string = doi_str) |>
          atlas_download(ext = ".zip", cache_file = tempfile(pattern = "data"))

  if(is.null(path)){
    inform("Download failed")
    return(tibble())
  }else{
    record_file <- grep("^records", unzip(path, list=TRUE)$Name, 
                        ignore.case=TRUE, value=TRUE)
    result <- read.csv(unz(path, record_file), stringsAsFactors = FALSE)
    
    # return tibble with correct info
    result <- as_tibble(result)
    attr(result, "doi") <- doi
    attr(result, "call") <- "atlas_occurrences"
    
    return(result)
  }
}

# TODO: fix multiple file import
url_download <- function(url){
  
  verbose <- getOption("galah_config")$verbose
  if(verbose) {
    cat("Downloading\n")
  }
  
  local_file <- atlas_download(url, 
    cache_file = tempfile(pattern = "data"), 
    ext = ".zip")
  
  if(is.null(local_file)){
    inform("Download failed")
    return(tibble())
  }
  
  # unzip files and read in any named "data.csv" or similar
  local_file_uz <- unzip(local_file, list = TRUE)$Name
  data_files <- local_file_uz[
    grepl("data", local_file_uz) & grepl(".csv$", local_file_uz)]
  
  if(length(data_files) < 1){
    inform("There was a problem reading the occurrence data and it looks like no data were returned.")
  }else{
    result <- do.call(rbind, 
                      lapply(data_files, 
                             function(a){read.csv(unz(local_file, a))})) |> 
              as_tibble()
  }
  
  # rename cols so they match requested cols
  names(result) <- rename_columns(names(result), type = "occurrence")
  
  # replace 'true' and 'false' with boolean
  valid_assertions <- show_all_assertions()$id
  assertions_check <- names(result) %in% valid_assertions
  if(any(assertions_check)){
    result <- fix_assertion_cols(result, names(result)[assertions_check])
  }
  
  # get DOI, if one exists
  doi_lookup <- grepl("doi.txt", local_file_uz)
  if(any(doi_lookup)){
    doi_text <- as.character(
      read.table(unz(local_file, "doi.txt"))$V1)
    attr(result, "doi") <- doi_text
  }
  
  # return tibble with correct info
  attr(result, "data_type") <- "occurrences"
  attr(result, "call") <- "atlas_occurrences"
  
  return(result)
}