#' Generate a citation for occurrence data
#'
#' If a `data.frame` was generated using [atlas_occurrences()],
#' and the `mint_doi` argument was set to `TRUE`, the DOI associated
#' with that dataset is appended to the resulting `data.frame` as an
#' attribute. This function simply formats that DOI as a citation that can be 
#' included in a scientific publication. Please also consider citing this 
#' package, using the information in `citation("galah")`.
#' @param data data.frame: occurrence data generated by 
#' [atlas_occurrences()]
#' @return A string containing the citation for that dataset.
#' @examples \dontrun{
#' atlas_citation(doi)
#' }
#' @export

atlas_citation <- function(data) {
  if (is.na(attributes(data)$doi)) {
    if (is.null(attributes(data)$search_url) || 
        is.na(attributes(data)$search_url)) {
      bullets <- c(
        "This data does not have a DOI or associated search url.",
        i = "Did you set `atlas_occurrences(mint_doi = TRUE)`?",
        i = "`atlas_citation` extracts this citation info when present."
      )
      abort(bullets, call = caller_env())
    }
    search_url <- attributes(data)$search_url
    return(glue("
                ALA occurrence download accessed from R with galah {galah_version_string()} \\
                (https://github.com/AtlasOfLivingAustralia/galah/) on \\
                {Sys.Date()}. 
                Search url: {search_url})
                "
    ))
  }
  search_url <- attributes(data)$search_url
  glue("
       ALA occurrence download {attributes(data)$doi}.
       Accessed from R with {galah_version_string()} \\
       (https://github.com/AtlasOfLivingAustralia/galah/) on {Sys.Date()}.
       ")
}
