
#' Filter data cube pixels by a user-defined predicate on band values
#' 
#' Create a proxy data cube, which evaluates a predicate over all pixels of a data cube. For all pixels that fulfill the predicate, the original
#' band values are returned. Other pixels are simply filled with NANs. The predicate may access band values by name.
#'
#' @param cube source data cube
#' @param pred predicate to be evaluated over all pixels
#' @return a proxy data cube object
#' @details gdalcubes uses and extends the \href{https://github.com/codeplea/tinyexpr}{tinyexpr library} to evaluate expressions in C / C++, you can look at the \href{https://github.com/codeplea/tinyexpr#functions-supported}{library documentation}
#' to see what kind of expressions you can execute. Pixel band values can be accessed by name.
#' @examples 
#' # create image collection from example Landsat data only 
#' # if not already done in other examples
#' if (!file.exists(file.path(tempdir(), "L8.db"))) {
#'   L8_files <- list.files(system.file("L8NY18", package = "gdalcubes"),
#'                          ".TIF", recursive = TRUE, full.names = TRUE)
#'   create_image_collection(L8_files, "L8_L1TP", file.path(tempdir(), "L8.db"), quiet = TRUE) 
#' }
#' 
#' L8.col = image_collection(file.path(tempdir(), "L8.db"))
#' v = cube_view(extent=list(left=388941.2, right=766552.4, 
#'               bottom=4345299, top=4744931, t0="2018-01", t1="2018-06"),
#'               srs="EPSG:32618", nx = 497, ny=526, dt="P1M")
#' L8.cube = raster_cube(L8.col, v) 
#' L8.cube = select_bands(L8.cube, c("B04", "B05")) 
#' L8.ndvi = apply_pixel(L8.cube, "(B05-B04)/(B05+B04)", "NDVI") 
#' L8.ndvi.filtered = filter_pixel(L8.ndvi, "NDVI > 0.5") 
#' L8.ndvi.filtered
#' \donttest{
#' plot(L8.ndvi.filtered)
#' }
#' @note This function returns a proxy object, i.e., it will not start any computations besides deriving the shape of the result.
#' @export
filter_pixel <- function(cube, pred) {
  stopifnot(is.cube(cube))

  x = gc_create_filter_predicate_cube(cube, pred)
  class(x) <- c("filter_pixel_cube", "cube", "xptr")
  return(x)
}



is.filter_pixel_cube  <- function(obj) {
  if(!("filter_pixel_cube" %in% class(obj))) {
    return(FALSE)
  }
  if (gc_is_null(obj)) {
    warning("GDAL data cube proxy object is invalid")
    return(FALSE)
  }
  return(TRUE)
}




