% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gdalraster_proc.R
\name{read_ds}
\alias{read_ds}
\title{Convenience wrapper for \code{GDALRaster$read()}}
\usage{
read_ds(
  ds,
  bands = NULL,
  xoff = 0,
  yoff = 0,
  xsize = ds$getRasterXSize(),
  ysize = ds$getRasterYSize(),
  out_xsize = xsize,
  out_ysize = ysize,
  as_list = FALSE,
  as_raw = FALSE
)
}
\arguments{
\item{ds}{An object of class \code{GDALRaster} in open state.}

\item{bands}{Integer vector of band numbers to read. By default all bands
will be read.}

\item{xoff}{Integer. The pixel (column) offset to the top left corner of the
raster region to be read (zero to start from the left side).}

\item{yoff}{Integer. The line (row) offset to the top left corner of the
raster region to be read (zero to start from the top).}

\item{xsize}{Integer. The width in pixels of the region to be read.}

\item{ysize}{Integer. The height in pixels of the region to be read.}

\item{out_xsize}{Integer. The width in pixels of the output buffer into
which the desired region will be read (e.g., to read a reduced resolution
overview).}

\item{out_ysize}{Integer. The height in pixels of the output buffer into
which the desired region will be read (e.g., to read a reduced resolution
overview).}

\item{as_list}{Logical. If \code{TRUE}, return output as a list of band vectors.
If \code{FALSE} (the default), output is a vector of pixel data interleaved by
band.}

\item{as_raw}{Logical. If \code{TRUE} and the underlying data type is Byte,
return output as R's raw vector type. This maps to the setting
\verb{$readByteAsRaw} on the \code{GDALRaster} object, which will be temporarily
updated in this function. To control this behavior in a persistent way on
a dataset see \code{$readByteAsRaw} in \code{\link[=GDALRaster]{GDALRaster-class}}.}
}
\value{
If \code{as_list = FALSE} (the default), a vector of \code{raw}, \code{integer},
\code{double} or \code{complex} containing the values that were read. It is organized
in left to right, top to bottom pixel order, interleaved by band.
If \code{as_list = TRUE}, a list with number of elements equal to the number of
bands read. Each element contains a vector of \code{raw}, \code{integer}, \code{double} or
\code{complex} containing the pixel values that were read for the band.
}
\description{
\code{read_ds()} will read from a raster dataset that is already open in a
\code{GDALRaster} object. By default, it attempts to read the full raster
extent from all bands at full resolution. \code{read_ds()} is sometimes more
convenient than \code{GDALRaster$read()}, e.g., to read specific multiple bands
for display with \code{\link[=plot_raster]{plot_raster()}}, or simply for the default arguments that
read an entire raster into memory (see Note).
}
\details{
\code{NA} will be returned in place of the nodata value if the raster dataset has
a nodata value defined for the band. Data are read as R \code{integer} type when
possible for the raster data type (Byte, Int8, Int16, UInt16, Int32),
otherwise as type \code{double} (UInt32, Float32, Float64).

The output object has attribute \code{gis}, a list containing:
\preformatted{
  $type = "raster"
  $bbox = c(xmin, ymin, xmax, ymax)
  $dim = c(xsize, ysize, nbands)
  $srs = <projection as WKT2 string>
  $datatype = <character vector of data type name by band>
}
The WKT version used for the projection string can be overridden by setting
the \code{OSR_WKT_FORMAT} configuration option. See \code{\link[=srs_to_wkt]{srs_to_wkt()}} for a list of
supported values.
}
\note{
There is small overhead in calling \code{read_ds()} compared with
calling \code{GDALRaster$read()} directly. This would only matter if calling
the function repeatedly to read a raster in chunks. For the case of reading
a large raster in many chunks, it will be optimal performance-wise to call
\code{GDALRaster$read()} directly.

By default, this function will attempt to read the full raster into memory.
It generally should not be called on large raster datasets using the default
argument values. The memory size in bytes of the returned vector will be,
e.g., (xsize * ysize * number of bands * 4) for data read as
\code{integer}, or (xsize * ysize * number of bands * 8) for data read as
\code{double} (plus small object overhead for the vector).
}
\examples{
# read three bands from a multi-band dataset
lcp_file <- system.file("extdata/storm_lake.lcp", package="gdalraster")
ds <- new(GDALRaster, lcp_file)

# as a vector of pixel data interleaved by band
r <- read_ds(ds, bands=c(6,5,4))
typeof(r)
length(r)
object.size(r)

# as a list of band vectors
r <- read_ds(ds, bands=c(6,5,4), as_list=TRUE)
typeof(r)
length(r)
object.size(r)

# gis attributes
attr(r, "gis")

ds$close()
}
\seealso{
\code{\link[=GDALRaster]{GDALRaster$read()}}
}
