% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculate_expression_quantitative_trait_loci.R
\name{calculate_expression_quantitative_trait_loci}
\alias{calculate_expression_quantitative_trait_loci}
\title{Calculate Expression Quantitative Trait Loci}
\usage{
calculate_expression_quantitative_trait_loci(
  tissueSiteDetailId,
  gencodeId,
  variantId,
  datasetId = "gtex_v8",
  .return_raw = FALSE
)
}
\arguments{
\item{tissueSiteDetailId}{String. The ID of the tissue of interest. Can be a
GTEx specific ID (e.g. "Whole_Blood"; use \code{\link[=get_tissue_site_detail]{get_tissue_site_detail()}}
to see valid values) or an Ontology ID.}

\item{gencodeId}{String. A Versioned GENCODE ID of a gene, e.g.
"ENSG00000065613.9".}

\item{variantId}{String. A gtex variant ID.}

\item{datasetId}{String. Unique identifier of a dataset. Usually includes a
data source and data release. Options: "gtex_v8", "gtex_snrnaseq_pilot".}

\item{.return_raw}{Logical. If \code{TRUE}, return the raw API JSON response.
Default = \code{FALSE}}
}
\value{
A tibble. Or a list if \code{.return_raw = TRUE}.
}
\description{
Calculate your own eQTLs
\itemize{
\item This service calculates the gene-variant association for any given pair of gene and variant, which may or may not be significant.
\item This requires as input a GENCODE ID, GTEx variant ID, and tissue site detail ID.
}

By default, the calculation is based on the latest GTEx release.

\href{https://gtexportal.org/api/v2/redoc#tag/Dynamic-Association-Endpoints/operation/calculate_expression_quantitative_trait_loci_api_v2_association_dyneqtl_get}{GTEx Portal API documentation}.
}
\details{
Notes on output:
\itemize{
\item Beta and standard error are recorded in columns \code{nes} and \code{error} respectively (see \href{https://gtexportal.org/home/faq#nes_beta}{GTEx FAQs})
\item \code{variantId} contains (in order) chromosome, position, reference allele, alternative allele and human genome build separated by underscores. The reference and alternative alleles for "chr1_13550_G_A_b38" for example are "G" and "A" respectively.
\item See examples for how to calculate minor and alternative allele frequencies.
}

Notes on input:
\itemize{
\item Argument \code{variantId} also accepts RSIDs.
}
}
\examples{
\dontrun{
# perform request - returns a tibble with a single row
calculate_expression_quantitative_trait_loci(
  tissueSiteDetailId = "Whole_Blood",
  gencodeId = "ENSG00000203782.5",
  variantId = "rs79641866"
)

# unnest list columns with tidyr::unnest()
calculate_expression_quantitative_trait_loci(
  tissueSiteDetailId = "Whole_Blood",
  gencodeId = "ENSG00000203782.5",
  variantId = "rs79641866"
) |>
  tidyr::unnest(c("data", "genotypes"))

# to calculate minor and alternative allele frequencies
calculate_expression_quantitative_trait_loci(
  tissueSiteDetailId = "Liver",
  gencodeId = "ENSG00000237973.1",
  variantId = "rs12119111"
) |>
  dplyr::bind_rows(.id = "rsid") |>
  tidyr::separate(
    col = "variantId",
    into = c(
      "chromosome",
      "position",
      "reference_allele",
      "alternative_allele",
      "genome_build"
    ),
    sep = "_"
  ) |>
  # ...then ascertain alternative_allele frequency
  dplyr::mutate(
    alt_allele_count = (2 * homoAltCount) + hetCount,
    total_allele_count = 2 * (homoAltCount + hetCount + homoRefCount),
    alternative_allele_frequency = alt_allele_count / total_allele_count
  ) |>
  dplyr::select(
    rsid,
    beta = nes,
    se = error,
    pValue,
    minor_allele_frequency = maf,
    alternative_allele_frequency,
    chromosome:genome_build,
    tissueSiteDetailId
  )
}
}
\seealso{
Other Dynamic Association Endpoints: 
\code{\link{calculate_ieqtls}()},
\code{\link{calculate_isqtls}()},
\code{\link{calculate_splicing_quantitative_trait_loci}()}
}
\concept{Dynamic Association Endpoints}
