% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/router.R
\name{gtfs_route}
\alias{gtfs_route}
\title{gtfs_route}
\usage{
gtfs_route(
  gtfs,
  from,
  to,
  start_time = NULL,
  day = NULL,
  route_pattern = NULL,
  earliest_arrival = TRUE,
  include_ids = FALSE,
  grep_fixed = TRUE,
  max_transfers = NA,
  from_to_are_ids = FALSE,
  quiet = FALSE
)
}
\arguments{
\item{gtfs}{A set of GTFS data returned from \link{extract_gtfs} or, for more
efficient queries, pre-processed with \link{gtfs_timetable}.}

\item{from}{Names, IDs, or approximate (lon, lat) coordinates of start
stations (as \code{stop_name} or \code{stop_id} entry in the \code{stops} table, or a vector
of two numeric values). See Note.}

\item{to}{Corresponding Names, IDs, or coordinates of end station.}

\item{start_time}{Desired departure time at \code{from} station, either in seconds
after midnight, a vector of two or three integers (hours, minutes) or (hours,
minutes, seconds), an object of class \link{difftime}, \pkg{hms}, or
\pkg{lubridate}. If not provided, current time is used.}

\item{day}{Day of the week on which to calculate route, either as an
unambiguous string (so "tu" and "th" for Tuesday and Thursday), or a number
between 1 = Sunday and 7 = Saturday. If not given, the current day will be
used. (Not used if \code{gtfs} has already been prepared with
\link{gtfs_timetable}.)}

\item{route_pattern}{Using only those routes matching given pattern, for
example, "^U" for routes starting with "U" (as commonly used for underground
or subway routes. To negate the \code{route_pattern} -- that is, to include all
routes except those matching the pattern -- prepend the value with "!"; for
example "!^U" will include all services except those starting with "U". (This
parameter is not used at all if \code{gtfs} has already been prepared with
\link{gtfs_timetable}.)}

\item{earliest_arrival}{If \code{FALSE}, routing will be with the first-departing
service, which may not provide the earliest arrival at the \code{to} station. This
may nevertheless be useful for bulk queries, as earliest arrival searches
require two routing queries, while earliest departure searches require just
one, and so will be generally twice as fast.}

\item{include_ids}{If \code{TRUE}, result will include columns containing
GTFS-specific identifiers for routes, trips, and stops.}

\item{grep_fixed}{If \code{FALSE}, match station names (when passed as character
string) with \code{grep(..., fixed = FALSE)}, to allow use of \code{grep} expressions.
This is useful to refine matches in cases where desired stations may match
multiple entries.}

\item{max_transfers}{If not \code{NA}, specify a desired maximum number of
transfers for the route (including but not exceeding this number). This
parameter may be used to generate alternative routes with fewer transfers,
although actual numbers of transfers may still exceed this number if the
value specified is less than the minimal feasible number of transfers.}

\item{from_to_are_ids}{Set to \code{TRUE} to enable \code{from} and \code{to} parameter to
specify entries in \code{stop_id} rather than \code{stop_name} column of the \code{stops}
table.}

\item{quiet}{Set to \code{TRUE} to suppress screen messages (currently just
regarding timetable construction).}
}
\value{
For single (from, to) values, a \code{data.frame} describing the route,
with each row representing one stop. For multiple (from, to) values, a list
of \code{data.frames}, each of which describes one route between the i'th start
and end stations (\code{from} and \code{to} values). Origin and destination stations
for which no route is possible return \code{NULL}.
}
\description{
Calculate single route between a start and end station departing at or after
a specified time.
}
\note{
This function will by default calculate the route that arrives earliest
at the specified destination, although this may depart later than the
earliest departing service. Routes which depart at the earliest possible time
can be calculated by setting \code{earliest_arrival = FALSE}.
}
\examples{
# Examples must be run on single thread only:
data.table::setDTthreads (1)

berlin_gtfs_to_zip () # Write sample feed from Berlin, Germany to tempdir
f <- file.path (tempdir (), "vbb.zip") # name of feed
gtfs <- extract_gtfs (f)
from <- "Innsbrucker Platz" # U-bahn station, not "S"
to <- "Alexanderplatz"
start_time <- 12 * 3600 + 120 # 12:02

route <- gtfs_route (gtfs, from = from, to = to, start_time = start_time)

# Specify day of week
route <- gtfs_route (
    gtfs,
    from = from,
    to = to,
    start_time = start_time,
    day = "Sunday"
)

# specify travel by "U" = underground only
route <- gtfs_route (
    gtfs,
    from = from,
    to = to,
    start_time = start_time,
    day = "Sunday",
    route_pattern = "^U"
)
# specify travel by "S" = street-level only (not underground)
route <- gtfs_route (
    gtfs,
    from = from,
    to = to,
    start_time = start_time,
    day = "Sunday",
    route_pattern = "^S"
)

# Route queries are generally faster if the GTFS data are pre-processed with
# `gtfs_timetable()`:
gt <- gtfs_timetable (gtfs, day = "Sunday", route_pattern = "^S")
route <- gtfs_route (gt, from = from, to = to, start_time = start_time)
}
\seealso{
Other main: 
\code{\link{gtfs_route_headway}()},
\code{\link{gtfs_traveltimes}()}
}
\concept{main}
