
#' Calculate and Summarize Distribution of Susceptibilities by Gene
#'
#' @description This function will calculate the distribution of
#' susceptibilities by gene.
#' @param x a `data.frame` containing the data.
#' @param cutoff value for percent susceptible cutoff. Numeric.
#' @param control value used to denote the susceptible control in the `gene`
#'  field. Character.
#' @param sample field providing the unique identification for each sample being
#'  tested. Character.
#' @param gene field providing the gene(s) being tested. Character.
#' @param perc_susc field providing the percent susceptible reactions.
#'  Character.
#'
#' @examples
#'
#' # locate system file for import
#' Ps <- system.file("extdata", "practice_data_set.csv", package = "hagis")
#'
#' # import 'practice_data_set.csv'
#' Ps <- read.csv(Ps)
#' head(Ps)
#'
#' # calculate susceptibilities with a 60 % cutoff value
#'susc <- summarize_gene(x = Ps,
#'                      cutoff = 60,
#'                      control = "susceptible",
#'                      sample = "Isolate",
#'                      gene = "Rps",
#'                      perc_susc = "perc.susc")
#' susc
#'
#' # plot susceptibilities
#' plot(susc, type = "percentage")
#'
#' @return  returns an object of [class()] `hagis.gene.summary`
#' An object of class `hagis.summaries` is a\cr [data.table::data.table()]
#'  containing the following components fields
#'   \describe{
#'     \item{gene}{the gene}
#'     \item{N_susc}{the total number susceptible for a given gene in the
#'     gene field}
#'     \item{percent_pathogenic}{the frequency with which a gene is pathogenic}
#'   }
#' @importFrom data.table ":="
#' @export summarize_gene

summarize_gene <- function(x,
                          cutoff,
                          control,
                          sample,
                          gene,
                          perc_susc) {
  # check inptuts and rename fields to work with this package
  x <- .check_inputs(
    .x = x,
    .cutoff = cutoff,
    .control = control,
    .sample = sample,
    .gene = gene,
    .perc_susc = perc_susc
  )
  
  # CRAN NOTE avoidance
  susceptible.1 <- percent_pathogenic <- N_susc <- NULL
  
  # summarise the reactions, create susceptible.1 field, see
  # internal_functions.R
  x <- .binary_cutoff(.x = x, .cutoff = cutoff)
  
  # create new data.table with percentages
  y <- x[, list(N_susc = sum(susceptible.1)), by = list(gene)]
  y[, percent_pathogenic := round((N_susc) / max(N_susc) * 100, 2)]
  
  # Set new class
  class(y) <- union("hagis.gene.summary", class(y))
  return(y)
}

#' @importFrom ggplot2 autoplot
#' @export
ggplot2::autoplot

#' Plot \pkg{hagis} Gene Summaries
#' @return A \link[ggplot2]{ggplot2} plot
#' @method autoplot hagis.gene.summary
#' @noRd

autoplot.hagis.gene.summary <-
  function(object, type, color = NULL, ...) {
    # CRAN NOTE avoidance
    gene <- percent_pathogenic <- NULL
    
    plot_percentage <- function(.data, .color) {
      perc_plot <- ggplot2::ggplot(data = .data,
                                   ggplot2::aes(x = as.factor(gene),
                                                y = percent_pathogenic)) +
        ggplot2::labs(y = "Percent of samples",
                      x = "Gene") +
        ggplot2::ggtitle(expression("Percentage of samples pathogenic"))
      
      if (!is.null(.color)) {
        perc_plot +
          ggplot2::geom_col(fill = .color,
                            colour = .color)
      } else {
        perc_plot +
          ggplot2::geom_col()
      }
    }
    
    plot_count <- function(.data, .color) {
      N_susc <- NULL
      num_plot <- ggplot2::ggplot(data = .data,
                                  ggplot2::aes(x = as.factor(gene),
                                               y = N_susc)) +
        ggplot2::labs(y = "Number of samples",
                      x = "Gene") +
        ggplot2::ggtitle(expression("Number of samples pathogenic"))
      
      if (!is.null(.color)) {
        num_plot +
          ggplot2::geom_col(fill = .color,
                            colour = .color)
      } else {
        num_plot +
          ggplot2::geom_col()
      }
    }
    
    if (type == "percentage") {
      plot_percentage(.data = object, .color = color)
    } else if (type == "count") {
      plot_count(.data = object, .color = color)
    } else
      stop(.call = FALSE,
           "You have entered an invalid `type`.")
  }

#' Creates Simple Plot Function for hagis Complexities Object
#'
#' @description Creates a \pkg{ggplot2} object of the gene summaries calculated
#'  by [summarize_gene()]
#' @param object a `hagis.gene.summary` object generated by [summarize_gene()].
#'  Character.
#' @param type a vector of values for which the bar plot is desired. Specify
#'  whether to return a graph of the percent pathogenic isolates, `percentage`,
#'  or as the count, `count`. Character.
#' @param color a named or hexadecimal color value to use for the bar color
#' @param ... passed to the chosen `geom(s)`
#'
#' @examples
#  # locate system file for import
#' Ps <- system.file("extdata", "practice_data_set.csv", package = "hagis")
#'
#' # import 'practice_data_set.csv'
#' Ps <- read.csv(Ps)
#'
#' # calculate susceptibilities with a 60 % cutoff value
#' susc <- summarize_gene(x = Ps,
#'                        cutoff = 60,
#'                        control = "susceptible",
#'                        sample = "Isolate",
#'                        gene = "Rps",
#'                        perc_susc = "perc.susc")
#'
#' # Visualize the summary of genes
#' plot(susc, type = "percentage")
#'
#' @param x A \pkg{hagis} `autoplot.hagis.gene.summary` object generated by
#'  [autoplot.hagis.gene.summary()]. Character.
#' @method plot hagis.gene.summary
#' @noRd
#' @export
#' @importFrom graphics plot
plot.hagis.gene.summary <- function(x, ...) {
  print(autoplot(x, ...))
}
