/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the COPYING file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
#ifndef H5_SUBFILING_COMMON_H
#define H5_SUBFILING_COMMON_H
#include <stdatomic.h>
#include "H5private.h"
#include "H5FDprivate.h"
#include "H5Iprivate.h"
#include "H5Pprivate.h"
#include "H5FDsubfiling.h"
#include "H5FDioc.h"
#ifndef PATH_MAX
#define PATH_MAX 4096
#endif
#ifndef H5_SUBFILING_PREFER_ALLGATHER_TOPOLOGY
#if !H5_CHECK_MPI_VERSION(3, 0)
#error "MPI 3 required for MPI_Comm_split_type"
#endif
#endif
#define H5FD_SUBFILING_CONFIG_PROP "H5FD_SUBFILING_CONFIG_PROP"
#define H5FD_SUBFILING_STUB_FILE_ID "H5FD_SUBFILING_STUB_FILE_ID"
#define COLL_FUNC (0x1 << 3)
#define ACK_PART  (0x01 << 8)
#define DATA_PART (0x02 << 8)
#define READY     (0x04 << 8)
#define COMPLETED (0x08 << 8)
#define INT32_MASK 0x07FFFFFFFFFFFFFFF
#define READ_INDEP  (READ_OP)
#define READ_COLL   (COLL_FUNC | READ_OP)
#define WRITE_INDEP (WRITE_OP)
#define WRITE_COLL  (COLL_FUNC | WRITE_OP)
#define GET_EOF_COMPLETED (COMPLETED | GET_EOF_OP)
#define TRUNC_COMPLETED   (COMPLETED | TRUNC_OP)
#define SET_LOGGING (LOGGING_OP)
#define WRITE_INDEP_ACK 0
#define READ_INDEP_ACK  1
#define READ_INDEP_DATA 2
#define WRITE_DATA_DONE 3
#define IO_TAG_BASE     4
typedef enum {
    SF_BADID    = (-1),
    SF_TOPOLOGY = 1,
    SF_CONTEXT  = 2,
    SF_NTYPES
} sf_obj_type_t;
typedef enum io_ops {
    READ_OP    = 1,
    WRITE_OP   = 2,
    OPEN_OP    = 3,
    CLOSE_OP   = 4,
    TRUNC_OP   = 5,
    GET_EOF_OP = 6,
    FINI_OP    = 8,
    LOGGING_OP = 16
} io_op_t;
typedef struct {
    int rank;
    int node_local_rank;
    int node_local_size;
    int node_lead_rank;
} layout_t;
typedef struct app_layout_t {
    layout_t *layout;
    int      *node_ranks;
    int       node_count;
    int       world_rank;
    int       world_size;
    int       node_local_rank;
    int       node_local_size;
} app_layout_t;
typedef struct topology {
    app_layout_t               *app_layout;
    MPI_Comm                    app_comm;
    bool                        rank_is_ioc;
    int                         ioc_idx;
    int                         n_io_concentrators;
    int                        *io_concentrators;
    H5FD_subfiling_ioc_select_t selection_type;
} sf_topology_t;
typedef struct {
    int64_t        sf_context_id;
    uint64_t       h5_file_id;
    bool           threads_inited;
    int            file_ref;
    int           *sf_fids;
    int            sf_num_fids;
    int            sf_num_subfiles;
    size_t         sf_write_count;
    size_t         sf_read_count;
    haddr_t        sf_eof;
    int64_t        sf_stripe_size;
    int64_t        sf_blocksize_per_stripe;
    int64_t        sf_base_addr;
    MPI_Comm       sf_msg_comm;
    MPI_Comm       sf_data_comm;
    MPI_Comm       sf_eof_comm;
    MPI_Comm       sf_node_comm;
    MPI_Comm       sf_group_comm;
    int            sf_group_size;
    int            sf_group_rank;
    char          *subfile_prefix;
    char          *config_file_prefix;
    char          *h5_filename;
    void          *ioc_data;
    sf_topology_t *topology;
#ifdef H5_SUBFILING_DEBUG
    char  sf_logfile_name[PATH_MAX];
    FILE *sf_logfile;
#endif
} subfiling_context_t;
typedef struct {
    int64_t header[3];
    int     tag;
    int     source;
    int     ioc_idx;
    int64_t context_id;
    double  start_time;
} sf_work_request_t;
extern MPI_Datatype H5_subfiling_rpc_msg_type;
static inline void *
H5FD__subfiling_cast_to_void(const void *data)
{
    union {
        const void *const_ptr_to_data;
        void       *ptr_to_data;
    } eliminate_const_warning;
    eliminate_const_warning.const_ptr_to_data = data;
    return eliminate_const_warning.ptr_to_data;
}
#ifdef __cplusplus
extern "C" {
#endif
H5_DLL herr_t H5FD__subfiling_open_stub_file(const char *name, unsigned flags, MPI_Comm file_comm,
                                             H5FD_t **file_ptr, uint64_t *file_id);
H5_DLL herr_t H5FD__subfiling_open_subfiles(const char *base_filename, uint64_t file_id,
                                            H5FD_subfiling_params_t *subfiling_config, int file_acc_flags,
                                            MPI_Comm file_comm, int64_t *context_id_out);
H5_DLL herr_t H5FD__subfiling_close_subfiles(int64_t subfiling_context_id, MPI_Comm file_comm);
H5_DLL void  *H5FD__subfiling_get_object(int64_t object_id);
H5_DLL herr_t H5FD__subfiling_free_object(int64_t object_id);
H5_DLL herr_t H5FD__subfiling_get_config_from_file(FILE *config_file, int64_t *stripe_size,
                                                   int64_t *num_subfiles);
H5_DLL herr_t H5FD__subfiling_resolve_pathname(const char *filepath, MPI_Comm comm, char **resolved_filepath);
H5_DLL herr_t H5FD__subfiling_set_config_prop(H5P_genplist_t                *plist_ptr,
                                              const H5FD_subfiling_params_t *vfd_config);
H5_DLL herr_t H5FD__subfiling_get_config_prop(H5P_genplist_t *plist_ptr, H5FD_subfiling_params_t *vfd_config);
H5_DLL herr_t H5FD__subfiling_set_file_id_prop(H5P_genplist_t *plist_ptr, uint64_t file_id);
H5_DLL herr_t H5FD__subfiling_get_file_id_prop(H5P_genplist_t *plist_ptr, uint64_t *file_id);
H5_DLL herr_t H5FD__subfile_fid_to_context(uint64_t file_id, int64_t *context_id_out);
H5_DLL herr_t H5FD__subfiling_validate_config_params(const H5FD_subfiling_params_t *subf_config);
H5_DLL herr_t H5FD__subfiling_get_default_ioc_config(H5FD_ioc_config_t *config);
H5_DLL herr_t H5FD__subfiling_terminate(void);
#ifdef H5_SUBFILING_DEBUG
H5_DLL void H5FD__subfiling_log(int64_t sf_context_id, const char *fmt, ...);
H5_DLL void H5FD__subfiling_log_nonewline(int64_t sf_context_id, const char *fmt, ...);
#endif
#ifdef __cplusplus
}
#endif
#endif
