% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/04-hdnom-calibrate.R
\name{hdnom.calibrate}
\alias{hdnom.calibrate}
\title{Calibrate High-Dimensional Cox Models}
\usage{
hdnom.calibrate(x, time, event, model.type = c("lasso", "alasso", "flasso",
  "enet", "aenet", "mcp", "mnet", "scad", "snet"), alpha, lambda,
  pen.factor = NULL, gamma, lambda1, lambda2, method = c("fitting",
  "bootstrap", "cv", "repeated.cv"), boot.times = NULL, nfolds = NULL,
  rep.times = NULL, pred.at, ngroup = 5, seed = 1001, trace = TRUE)
}
\arguments{
\item{x}{Matrix of training data used for fitting the model;
on which to run the calibration.}

\item{time}{Survival time.
Must be of the same length with the number of rows as \code{x}.}

\item{event}{Status indicator, normally 0 = alive, 1 = dead.
Must be of the same length with the number of rows as \code{x}.}

\item{model.type}{Model type to calibrate. Could be one of \code{"lasso"},
\code{"alasso"}, \code{"flasso"}, \code{"enet"}, \code{"aenet"},
\code{"mcp"}, \code{"mnet"}, \code{"scad"}, or \code{"snet"}.}

\item{alpha}{Value of the elastic-net mixing parameter alpha for
\code{enet}, \code{aenet}, \code{mnet}, and \code{snet} models.
For \code{lasso}, \code{alasso}, \code{mcp}, and \code{scad} models,
please set \code{alpha = 1}.
\code{alpha=1}: lasso (l1) penalty; \code{alpha=0}: ridge (l2) penalty.
Note that for \code{mnet} and \code{snet} models,
\code{alpha} can be set to very close to 0 but not 0 exactly.}

\item{lambda}{Value of the penalty parameter lambda to use in the
model fits on the resampled data. From the Cox model you have built.}

\item{pen.factor}{Penalty factors to apply to each coefficient.
From the built \emph{adaptive lasso} or \emph{adaptive elastic-net} model.}

\item{gamma}{Value of the model parameter gamma for
MCP/SCAD/Mnet/Snet models.}

\item{lambda1}{Value of the penalty parameter lambda1 for fused lasso model.}

\item{lambda2}{Value of the penalty parameter lambda2 for fused lasso model.}

\item{method}{Calibration method.
Options including \code{"fitting"}, \code{"bootstrap"}, \code{"cv"},
and \code{"repeated.cv"}.}

\item{boot.times}{Number of repetitions for bootstrap.}

\item{nfolds}{Number of folds for cross-validation and
repeated cross-validation.}

\item{rep.times}{Number of repeated times for repeated cross-validation.}

\item{pred.at}{Time point at which calibration should take place.}

\item{ngroup}{Number of groups to be formed for calibration.}

\item{seed}{A random seed for resampling.}

\item{trace}{Logical. Output the calibration progress or not.
Default is \code{TRUE}.}
}
\description{
Calibrate High-Dimensional Cox Models
}
\examples{
library("survival")

# Load imputed SMART data
data("smart")
x = as.matrix(smart[, -c(1, 2)])
time = smart$TEVENT
event = smart$EVENT
y = Surv(time, event)

# Fit Cox model with lasso penalty
fit = hdcox.lasso(x, y, nfolds = 5, rule = "lambda.1se", seed = 11)

# Model calibration by fitting the original data directly
cal.fitting = hdnom.calibrate(x, time, event, model.type = "lasso",
                              alpha = 1, lambda = fit$lasso_best_lambda,
                              method = "fitting",
                              pred.at = 365 * 9, ngroup = 5,
                              seed = 1010)

# Model calibration by bootstrap
# Normally boot.times should be set to 200 or more,
# we set it to 3 here only to save example running time.
cal.boot = hdnom.calibrate(x, time, event, model.type = "lasso",
                           alpha = 1, lambda = fit$lasso_best_lambda,
                           method = "bootstrap", boot.times = 3,
                           pred.at = 365 * 9, ngroup = 5,
                           seed = 1010)

# Model calibration by 5-fold cross-validation
cal.cv = hdnom.calibrate(x, time, event, model.type = "lasso",
                         alpha = 1, lambda = fit$lasso_best_lambda,
                         method = "cv", nfolds = 5,
                         pred.at = 365 * 9, ngroup = 5,
                         seed = 1010)

# Model calibration by repeated cross-validation
cal.repcv = hdnom.calibrate(x, time, event, model.type = "lasso",
                            alpha = 1, lambda = fit$lasso_best_lambda,
                            method = "repeated.cv", nfolds = 3, rep.times = 3,
                            pred.at = 365 * 9, ngroup = 5,
                            seed = 1010)

print(cal.fitting)
summary(cal.fitting)
plot(cal.fitting)

print(cal.boot)
summary(cal.boot)
plot(cal.boot)

print(cal.cv)
summary(cal.cv)
plot(cal.cv)

print(cal.repcv)
summary(cal.repcv)
plot(cal.repcv)
}

