###########################################################################
# Copyright 2009 Nobody                                                   #
#                                                                         #
# This file is part of hergm.                                             #
#                                                                         # 
#    hergm is free software: you can redistribute it and/or modify        #
#    it under the terms of the GNU General Public License as published by #
#    the Free Software Foundation, either version 3 of the License, or    #
#    (at your option) any later version.                                  #
#                                                                         # 
#    hergm is distributed in the hope that it will be useful,             #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of       #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        #
#    GNU General Public License for more details.                         #
#                                                                         #
#    You should have received a copy of the GNU General Public License    #
#    along with hergm.  If not, see <http://www.gnu.org/licenses/>.       #
#                                                                         # 
###########################################################################

hergm.postprocess <- function(sample = NULL,
                              seed = NULL,
                              burnin = 0, 
                              thinning = 1,
                              relabel = 0,
                              number_runs = 1,
                              extract = NULL,
                              ...)
# input: MCMC sample generated by function hergm
# output: postprocessed MCMC sample
{
  if (is.null(seed) == FALSE) set.seed(seed)

  if (is.null(sample$extract)) extract <- TRUE
  else extract <- sample$extract

  if (extract == TRUE) # Extract MCMC sample from argument sample
  { # Begin: extract

  # Extract 
  n <- sample$n
  max_number <- sample$max_number
  number_fixed <- sample$number_fixed
  d1 <- sample$d1
  d2 <- sample$d2
  parallel <- sample$parallel
  simulate <- sample$simulate
  sample_size <- sample$sample_size
  mcmc <- sample$sample 
  predictions <- sample$predictions

  # Check
  if (burnin > sample_size) burnin <- 0 
  if ((d2 == 0) || (max_number == 1) || (number_fixed == n)) relabel <- 0 
  
  # Preprocess MCMC sample: delete burn-in iterations and transform vector into matrix, where rows correspond to MCMC draws
  d <- d1 + d2
  terms <- length_mcmc(d1, d2, max_number, n, predictions)  
  mcmc_sample <- NULL
  count <- 0
  for (i in 1:parallel) 
    {
    first <- count + (burnin * terms) + 1
    last <- count + (sample_size * terms)
    mcmc_sample <- append(mcmc_sample, mcmc[first:last])
    count <- count + (sample_size * terms)
    }
  mcmc_sample_size <- parallel * (sample_size - burnin)
  mcmc_sample <- matrix(mcmc_sample, nrow = mcmc_sample_size, ncol = terms, byrow = TRUE)
  if (thinning > 1)
    {
    for (i in sample_size:1) 
      {
      if (trunc(i / thinning) != (i / thinning)) mcmc_sample <- mcmc_sample[-i,] 
      }
    mcmc_sample_size <- nrow(mcmc_sample)
    }

  # Initialize arrays
  output <- list()
  if (d1 > 0) 
    {
    output$ergm_theta <- matrix(data = 0, nrow = mcmc_sample_size, ncol = d1) 
    }
  if (d2 > 0)
    {
    output$eta_mean <- matrix(data = 0, nrow = mcmc_sample_size, ncol = d2)
    output$eta_precision <- matrix(data = 0, nrow = mcmc_sample_size, ncol = d2)
    output$hergm_theta <- matrix(data = 0, nrow = mcmc_sample_size, ncol = d2 * (max_number + 1))
    output$indicator <- matrix(data = 0, nrow = mcmc_sample_size, ncol = n)
    output$size <- matrix(data = 0, nrow = mcmc_sample_size, ncol = max_number)
    output$p_k <- matrix(data = 0, nrow = mcmc_sample_size, ncol = max_number)
    output$alpha <- matrix(data = 0, nrow = mcmc_sample_size, ncol = 1)
    }
  if (predictions == TRUE) output$prediction <- matrix(data = 0, nrow = mcmc_sample_size, ncol = d)

  # Process MCMC sample
  for (row in 1:mcmc_sample_size)
    {
    column <- 0
    if (d1 > 0)
      {
      for (i in 1:d1) 
        {
        column <- column + 1
        output$ergm_theta[row,i] <- mcmc_sample[row,column]
        }
      }
    if (d2 > 0)
      {
      for (i in 1:d2) 
        {
        column <- column + 1
        output$eta_mean[row,i] <- mcmc_sample[row,column]
        }
      for (i in 1:d2) 
        {
        column <- column + 1
        output$eta_precision[row,i] <- mcmc_sample[row,column]
        }
      for (i in 1:(d2 * (max_number + 1))) 
        {
        column <- column + 1
        output$hergm_theta[row,i] <- mcmc_sample[row,column]
        }
      for (i in 1:n) 
        {
        column <- column + 1
        output$indicator[row,i] <- mcmc_sample[row,column]
        }
      for (i in 1:max_number) 
        {
        column <- column + 1
        output$size[row,i] <- mcmc_sample[row,column]
        }
      for (i in 1:max_number) 
        {
        column <- column + 1
        output$p_k[row,i] <- mcmc_sample[row,column]
        }
      column <- column + 1
      output$alpha[row,1] <- mcmc_sample[row,column]
      } 
    if (predictions == TRUE)
      {
      for (i in 1:d) 
        {
        column <- column + 1 
        output$prediction[row,i] <- mcmc_sample[row,column]
        }
      }
    }

  } # End: extract

  # Relabel sample
  if (relabel > 0) # Minimizing posterior expected loss
    {
    if (relabel == 1) minimizer <- hergm.relabel_1(max_number, output$indicator, number_runs) # Minimize posterior expected loss of Schweinberger and Handcock (2015)
    else minimizer <- hergm.relabel_2(max_number, output$indicator) # Minimize posterior expected loss of Peng and Carvalho (2015) 
    output$loss <- minimizer$loss
    output$indicator_min <- minimizer$indicator
    output$p_i_k <- minimizer$p
    output$hergm_theta_min <- matrix(0, nrow=nrow(output$hergm_theta), ncol=ncol(output$hergm_theta))
    if (simulate == FALSE)
      {
      output$hergm_theta_min <- matrix(0, nrow=nrow(output$hergm_theta), ncol=ncol(output$hergm_theta))
      index1 <- 1
      index2 <- max_number
      theta <- output$hergm_theta[,index1:index2]
      output$hergm_theta_min[,index1:index2] <- hergm.permute_mcmc(theta, max_number, minimizer$min_permutations) # Within-block parameters 
      index2 <- index2 + 1 # Between-block parameters
      output$hergm_theta_min[,index2] <- output$hergm_theta[,index2] # Between-block parameters
      if (d2 > 1)
        {
        for (h_term in 2:d2) # Relabel block-dependent parameters of block-dependent model terms one by one
          {
          index1 <- index2 + 1 # Increment starting index
          index2 <- index2 + max_number # Increment stopping index
          theta <- output$hergm_theta[,index1:index2] # Within-block parameters
          output$hergm_theta_min[,index1:index2] <- hergm.permute_mcmc(theta, max_number, minimizer$min_permutations) # Within-block parameters 
          index2 <- index2 + 1 # Between-block parameters
          output$hergm_theta_min[,index2] <- output$hergm_theta[,index2] # Copy between-block parameters
          }
        }
      cat("\n")
      }
    }

  # Store
  output$extract <- FALSE # If hergm.postprocess() is called hereafter, we will not extract the MCMC sample again, because otherwise we will get runtime errors
  output$n <- sample$n
  output$network <- sample$network
  output$model <- sample$model 
  output$max_number <- sample$max_number
  output$number_fixed <- sample$number_fixed  
  output$d1 <- sample$d1
  output$d2 <- sample$d2
  output$hyper_prior <- sample$hyper_prior
  output$ergm_theta <- sample$ergm_theta
  output$parallel <- sample$parallel
  output$simulate <- sample$simulate
  output$sample_size <- mcmc_sample_size
  if (simulate == TRUE)
    {
    output$sample <- sample$sample
    output$heads <- sample$heads
    output$tails <- sample$tails
    }
  output$predictions <- sample$predictions
  output$sample
 
  output
}

