

#' Create an inlabru model object from model components
#'
#' The [inlabru] syntax for model formulae is different from what
#' `INLA::inla` considers a valid.
#' In inla most of the effects are defined by adding an `f(...)` expression to the formula.
#' In [inlabru] the `f` is replaced by an arbitrary (exception: `offset`) string that will
#' determine the label of the effect. See Details for further information.
#'
#' @details
#' For instance
#'
#' `y ~ f(myspde, ...)`
#'
#' in INLA is equivalent to
#'
#' `y ~ myspde(...)`
#'
#' in inlabru.
#'
#' A disadvantage of the inla way is that there is no clear separation between the name of the covariate
#' and the label of the effect. Furthermore, for some models like SPDE it is much more natural to
#' use spatial coordinates as covariates rather than an index into the SPDE vertices. For this purpose
#' [inlabru] provides the new `main` agument. For convenience, the `main` argument can be used
#' like the first argument of the f function, e.g., and is the first argument of the component definition.
#'
#' `y ~ f(temperature, model = 'linear')`
#'
#' is equivalent to
#'
#' `y ~ temperature(temperature, model = 'linear')`
#' and
#' `y ~ temperature(main = temperature, model = 'linear')`
#' as well as
#' `y ~ temperature(model = 'linear')`
#' which sets `main = temperature`.
#'
#' On the other hand, map can also be a function mapping, e.g the [coordinates] function of the
#' [sp] package :
#'
#' `y ~ mySPDE(coordinates, ...)`
#'
#' This exctract the coordinates from the data object, and maps it to the latent
#' field via the information given in the `mapper`, which by default is
#' extracted from the `model` object, in the case of `spde` model
#' objects.
#'
#' Morevover, `main` can be any expression that evaluates within your data as an environment.
#' For instance, if your data has columns 'a' and 'b', you can create a fixed effect of 'sin(a+b)' by
#' setting `map` in the following way:
#'
#' `y ~ myEffect(sin(a+b))`
#'
#'
#' @export
#' @param components A [component_list] object
#' @param lhoods A list of one or more `lhood` objects
#' @return A [bru_model] object
#' @keywords internal

bru_model <- function(components, lhoods) {
  stopifnot(inherits(components, "component_list"))

  # Back up environment
  env <- environment(components)

  # Complete the component definitions based on data
  components <- component_list(components, lhoods)

  # Create joint formula that will be used by inla
  formula <- BRU_response ~ -1
  linear <- TRUE
  included <- character(0)
  for (lh in lhoods) {
    linear <- linear && lh[["linear"]]
    included <- union(
      included,
      parse_inclusion(
        names(components),
        include = lh[["include_components"]],
        exclude = lh[["exclude_components"]]
      )
    )
  }

  for (cmp in included) {
    if (linear ||
      !identical(components[[cmp]][["main"]][["type"]], "offset")) {
      formula <- update.formula(formula, components[[cmp]]$inla.formula)
    }
  }

  # Restore environment
  environment(formula) <- env

  # Make model
  mdl <- list(effects = components, formula = formula)
  class(mdl) <- c("bru_model", "list")
  return(mdl)
}





#' Evaluate or sample from a posterior result given a model and locations
#'
#' @export
#' @param model A [bru] model
#' @param state list of lists, as generated by [evaluate_state()]
#' @param data A `list`, `data.frame`, or `Spatial*DataFrame`, with coordinates
#' and covariates needed to evaluate the predictor.
#' @param A Precomputed A-matrices
#' @param predictor A formula or an expression to be evaluated given the
#' posterior or for each sample thereof. The default (`NULL`) returns a
#' `data.frame` containing the sampled effects. In case of a formula the right
#' hand side is used for evaluation.
#' @param format character; determines the storage format of predictor output.
#' Available options:
#' * `"auto"` If the first evaluated result is a vector or single-column matrix,
#'   the "matrix" format is used, otherwise "list".
#' * `"matrix"` A matrix where each column contains the evaluated predictor
#' expression for a state.
#' * `"list"` A list where each element contains the evaluated predictor
#' expression for a state.
#' @param n Number of samples to draw.
#' @param seed If seed != 0L, the random seed
#' @param num.threads Specification of desired number of threads for parallel
#' computations. Default NULL, leaves it up to INLA.
#' When seed != 0, overridden to "1:1"
#' @param include Character vector of component labels that are needed by the
#'   predictor expression; Default: NULL (include all components that are not
#'   explicitly excluded)
#' @param exclude Character vector of component labels that are not used by the
#'   predictor expression. The exclusion list is applied to the list
#'   as determined by the `include` parameter; Default: NULL (do not remove
#'   any components from the inclusion list)
#' @param \dots Additional arguments passed on to `inla.posterior.sample`
#' @details * `evaluate_model` is a wrapper to evaluate model state, A-matrices,
#' effects, and predictor, all in one call.
#'
#' @keywords internal
#' @rdname evaluate_model
evaluate_model <- function(model,
                           state,
                           data = NULL,
                           A = NULL,
                           predictor = NULL,
                           format = NULL,
                           include = NULL,
                           exclude = NULL,
                           ...) {
  included <- parse_inclusion(names(model$effects), include, exclude)

  if (is.null(state)) {
    stop("Not enough information to evaluate model states.")
  }
  if (is.null(A) && !is.null(data)) {
    A <- amatrix_eval(model$effects[included], data = data)
  }
  if (is.null(A)) {
    effects <- NULL
  } else {
    effects <- evaluate_effect_multi(
      model$effects[included],
      state = state,
      data = data,
      A = A
    )
  }

  if (is.null(predictor)) {
    return(effects)
  }

  values <- evaluate_predictor(
    model,
    state = state,
    data = data,
    effects = effects,
    predictor = predictor,
    format = format
  )

  values
}


#' @details * `evaluate_state` evaluates model state properties or samples
#' @param result A `bru` object from [bru()] or [lgcp()]
#' @param property Property of the model components to obtain value from.
#' Default: "mode". Other options are "mean", "0.025quant", "0.975quant",
#' "sd" and "sample". In case of "sample" you will obtain samples from the
#' posterior (see `n` parameter). If `result` is `NULL`, all-zero vectors are
#' returned for each component.
#' @param internal_hyperpar logical; If `TRUE`, return hyperparameter properties
#' on the internal scale. Currently ignored when `property="sample"`.
#' Default is `FALSE`.
#' @export
#' @rdname evaluate_model
evaluate_state <- function(model,
                           result,
                           property = "mode",
                           n = 1,
                           seed = 0L,
                           num.threads = NULL,
                           internal_hyperpar = FALSE,
                           ...) {
  # Evaluate random states, or a single property
  if (property == "sample") {
    state <- inla.posterior.sample.structured(result,
      n = n, seed = seed,
      num.threads = num.threads,
      ...
    )
  } else if (is.null(result)) {
    state <- list(lapply(
      model[["effects"]],
      function(x) {
        rep(0.0, ibm_n(x[["mapper"]]))
      }
    ))
  } else {
    state <- list(extract_property(
      result = result,
      property = property,
      internal_hyperpar = internal_hyperpar
    ))
  }

  state
}




#' @export
#' @rdname evaluate_effect
evaluate_effect_single <- function(...) {
  UseMethod("evaluate_effect_single")
}
#' @export
#' @rdname evaluate_effect
evaluate_effect_multi <- function(...) {
  UseMethod("evaluate_effect_multi")
}

#' Evaluate a component effect
#'
#' Calculate latent component effects given some data and the state of the
#' component's internal random variables.
#'
#' @export
#' @keywords internal
#' @param component A `bru_component`.
#' @param data A `data.frame` or Spatial* object of covariates and/or point locations.
#' @param state Specification of one (for `evaluate_effect_single`) or several
#' (for `evaluate_effect_multi`) latent variable states:
#' * `evaluate_effect_single.component`: A vector of the latent component state.
#' * `evaluate_effect_single.component_list`: list of named vectors.
#' * `evaluate_effect_multi`: list of lists of the `evaluate_effect_single.component_list` type.
#' For `evaluate_effect_multi.component`,
#' only the label for the given component needs to be included
#' @param A A matrix overriding the default projection matrix or matrices
#' (named list of matrices for `evaluate_effect.component_list`)
#' @param ... Unused.
#' @return * `evaluate_effect_single.component`: A numeric vector of the component effect values
#' state.
#' @author Fabian E. Bachl \email{bachlfab@@gmail.com} and
#' Finn Lindgren \email{finn.lindgren@@gmail.com}
#' @rdname evaluate_effect

evaluate_effect_single.component <- function(component, state, data, A = NULL, ...) {
  # Make A-matrix (if not provided)
  if (is.null(A)) {
    A <- amatrix_eval(component, data)
  }

  # Determine component depending on the type of latent model
  if (component$main$type %in% c("offset")) {
    values <- A
  } else {
    values <- A %*% state
  }

  as.matrix(values)
}


#' @return * `evaluate_effect_single.component_list`: A data.frame of evaluated component
#' effect values
#' @export
#' @rdname evaluate_effect
#' @keywords internal
evaluate_effect_single.component_list <- function(components, state, data,
                                                  A = NULL, ...) {
  if (is.null(A)) {
    A <- amatrix_eval(components, data = data)
  }
  result <- list()
  for (label in names(components)) {
    result[[label]] <- evaluate_effect_single(
      components[[label]],
      data = data,
      state = state[[label]],
      A = A[[label]]
    )
  }
  as.data.frame(result)
}

#' @return * `evaluate_effect_multi.component`: A list of numeric vectors
#' of evaluated component effects.
#' @export
#' @rdname evaluate_effect
#' @keywords internal
evaluate_effect_multi.component <- function(component, state, data,
                                            A = NULL, ...) {
  if (is.null(A)) {
    A <- amatrix_eval(component, data = data)
  }
  lapply(
    state,
    function(x) {
      evaluate_effect_single(
        component,
        state = x,
        data = data,
        A = A,
        ...
      )
    }
  )
}


#' @return * `evaluate_effect_multi.component_list`: A list of data.frames of
#' evaluated component effects, one data.frame for each state
#' @export
#' @rdname evaluate_effect
#' @keywords internal
evaluate_effect_multi.component_list <- function(components, state, data,
                                                 A = NULL, ...) {
  if (is.null(A)) {
    A <- amatrix_eval(components, data = data)
  }
  lapply(
    state,
    function(x) {
      evaluate_effect_single(
        components,
        state = x,
        data = data,
        A = A,
        ...
      )
    }
  )
}




#' Evaluate component effects or expressions
#'
#' Evaluate component effects or expressions, based on a bru model and one or
#' several states of the latent variables and hyperparameters.
#'
#' @param data A `list`, `data.frame`, or `Spatial*DataFrame`, with coordinates
#' and covariates needed to evaluate the model.
#' @param state A list where each element is a list of named latent state
#' information, as produced by [evaluate_state()]
#' @param effects A list where each element is list of named evaluated effects,
#' as computed by [evaluate_effect_multi.component_list()]
#' @param predictor Either a formula or expression
#' @param format character; determines the storage format of the output.
#' Available options:
#' * `"auto"` If the first evaluated result is a vector or single-column matrix,
#'   the "matrix" format is used, otherwise "list".
#' * `"matrix"` A matrix where each column contains the evaluated predictor
#' expression for a state.
#' * `"list"` A list where each column contains the evaluated predictor
#' expression for a state.
#'
#' Default: "auto"
#' @return A list or matrix is returned, as specified by `format`
#' @keywords internal
#' @rdname evaluate_predictor
evaluate_predictor <- function(model,
                               state,
                               data,
                               effects,
                               predictor,
                               format = "auto") {
  stopifnot(inherits(model, "bru_model"))
  format <- match.arg(format, c("auto", "matrix", "list"))
  pred.envir <- environment(predictor)
  if (inherits(predictor, "formula")) {
    predictor <- parse(text = as.character(predictor)[length(as.character(predictor))])
  }
  formula.envir <- environment(model$formula)
  enclos <-
    if (!is.null(pred.envir)) {
      pred.envir
    } else if (!is.null(formula.envir)) {
      formula.envir
    } else {
      parent.frame()
    }

  envir <- new.env(parent = enclos)
  # Find .data. first,
  # then data variables,
  # then pred.envir variables (via enclos),
  # then formula.envir (via enclos if pred.envir is NULL):
  #  for (nm in names(pred.envir)) {
  #    assign(nm, pred.envir[[nm]], envir = envir)
  #  }
  if (is.list(data)) {
    for (nm in names(data)) {
      assign(nm, data[[nm]], envir = envir)
    }
  } else {
    data_df <- as.data.frame(data)
    for (nm in names(data_df)) {
      assign(nm, data_df[[nm]], envir = envir)
    }
  }
  assign(".data.", data, envir = envir)

  # Rename component states from label to label_latent
  state_names <- as.list(expand_labels(
    names(state[[1]]),
    names(model$effects),
    suffix = "_latent"
  ))
  names(state_names) <- names(state[[1]])

  # Construct _eval function names
  eval_names <- as.list(expand_labels(
    intersect(names(state[[1]]), names(model$effects)),
    intersect(names(state[[1]]), names(model$effects)),
    suffix = "_eval"
  ))
  names(eval_names) <- intersect(names(state[[1]]), names(model$effects))

  eval_fun_factory <-
    function(.comp, .envir, .enclos) {
      .is_offset <- .comp$main$type %in% c("offset")
      .is_iid <- .comp$main$type %in% c("iid")
      .mapper <- .comp$mapper
      .label <- paste0(.comp$label, "_latent")
      .iid_precision <- paste0("Precision_for_", .comp$label)
      .iid_cache <- list()
      .iid_cache_index <- NULL
      eval_fun <- function(main, group = NULL, replicate = NULL, .state = NULL) {
        if (is.null(group)) {
          group <- rep(1, NROW(main))
        }
        if (is.null(replicate)) {
          replicate <- rep(1, NROW(main))
        }
        .A <- ibm_amatrix(
          .mapper,
          input = list(
            main = main,
            group = group,
            replicate = replicate
          )
        )
        if (.is_offset) {
          .values <- .A
        } else {
          if (is.null(.state)) {
            .state <- eval(parse(text = .label),
                           envir = .envir,
                           enclos = .enclos)
          }
          .values <- .A %*% .state
          if (.is_iid) {
            ok <- ibm_valid_input(
              .mapper,
              input = list(
                main = main,
                group = group,
                replicate = replicate
              )
            )
            if (any(!ok)) {
              .cache_state_index <- eval(parse(text = ".cache_state_index"),
                                         envir = .envir,
                                         enclos = .enclos)
              if (!identical(.cache_state_index, .iid_cache_index)) {
                .iid_cache_index <<- .cache_state_index
                .iid_cache <<- list()
              }
              key <- as.character(main[!ok])
              not_cached <- !(key %in% names(.iid_cache))
              if (any(not_cached)) {
                .prec <- eval(parse(text = .iid_precision),
                              envir = .envir,
                              enclos = .enclos)
                for (k in unique(key[not_cached])) {
                  .iid_cache[k] <<- rnorm(1, mean = 0, sd = .prec^-0.5)
                }
              }
              .values[!ok] <- vapply(key,
                                     function(k) .iid_cache[[k]],
                                     0.0)
            }
          }
        }
        
        as.matrix(.values)
      }
      eval_fun
    }
  for (nm in names(eval_names)) {
    assign(eval_names[[nm]],
           eval_fun_factory(model$effects[[nm]], .envir = envir, .enclos = enclos),
           envir = envir
    )
  }

  # Remove problematic objects:
  problems <- c(".Random.seed")
  remove(list = intersect(names(envir), problems), envir = envir)

  n <- length(state)
  for (k in seq_len(n)) {
    # Keep track of the iteration index so the iid cache can be invalidated
    assign(".cache_state_index", k, envir = envir)

    for (nm in names(state[[k]])) {
      assign(state_names[[nm]], state[[k]][[nm]], envir = envir)
    }
    for (nm in names(effects[[k]])) {
      assign(nm, effects[[k]][[nm]], envir = envir)
    }

    result_ <- eval(predictor, envir = envir, enclos = enclos)
    if (k == 1) {
      if (identical(format, "auto")) {
        if ((is.vector(result_) && !is.list(result_)) ||
          (is.matrix(result_) && (NCOL(result_) == 1))) {
          format <- "matrix"
        } else {
          format <- "list"
        }
      }
      if (identical(format, "matrix")) {
        result <- matrix(0.0, NROW(result_), n)
        rownames(result) <- row.names(as.matrix(result_))
      } else if (identical(format, "list")) {
        result <- vector("list", n)
      }
    }
    if (identical(format, "list")) {
      result[[k]] <- result_
    } else {
      result[, k] <- result_
    }
  }

  result
}






#' Compute all A-matrices
#'
#' Computes the A matrices for included components for each model likelihood
#'
#' @param model A `bru_model` object
#' @param lhoods A `bru__like_list` object
#' @rdname evaluate_A
evaluate_A <- function(model, lhoods) {
  stopifnot(inherits(model, "bru_model"))
  lapply(
    lhoods,
    function(lh) {
      included <- parse_inclusion(
        names(model[["effects"]]),
        lh[["include_components"]],
        lh[["exclude_components"]]
      )

      amatrix_eval(model$effects[included], data = lh[["data"]])
    }
  )
}

#' Compute all index values
#'
#' Computes the index values matrices for included components
#'
#' @param model A `bru_model` object
#' @param lhoods A `bru__like_list` object
#' @rdname evaluate_index
evaluate_index <- function(model, lhoods) {
  stopifnot(inherits(model, "bru_model"))
  included <-
    unique(do.call(
      c,
      lapply(
        lhoods,
        function(lh) {
          parse_inclusion(
            names(model[["effects"]]),
            lh[["include_components"]],
            lh[["exclude_components"]]
          )
        }
      )
    ))

  index_eval(model[["effects"]][included])
}
