## -----------------------------------------------------------------------------
library(isoorbi) #load isoorbi R package
library(forcats) #better ordering of factor variables in plots

# Load a path to a system test file
file_path <-system.file("extdata", "testfile_dual_inlet_new.isox", package = "isoorbi")


## ---- message=FALSE, results='hide'-------------------------------------------

# Read .isox test data
df <- orbi_read_isox(file = file_path)

# Keep only most important columns; equivalent to simplify check box in IsoX
df.simple <- orbi_simplify_isox(dataset = df)

# Filter the data
df.filtered <- df.simple |> orbi_filter_isox(
                              time_min = 0,
                              time_max = 75,
                              compounds = "NO3",
                              isotopocules = c("M0", "15N", "18O", "17O")
                            )

# Clean the data by removing noise and outliers
df.clean <- df.filtered |> orbi_filter_satellite_peaks() |>
                            orbi_filter_weak_isotopocules(min_percent = 10) |>
                            orbi_filter_scan_intensity(outlier_percent = 2)


## -----------------------------------------------------------------------------
# change a global setting
orbi_set_settings(di_ref_name = "my ref")

# define blocks
df_w_blocks <-
  # general definition
  orbi_define_blocks_for_dual_inlet(
    df.simple,
    ref_block_time.min = 10,
    sample_block_time.min = 10,
    startup_time.min = 5,
    change_over_time.min = 2,
    sample_block_name = "sample"
  ) |> 
  # fine adjustments
  orbi_adjust_block(block = 1, shift_start_time.min = 2) |>
  orbi_adjust_block(block = 4, set_start_time.min = 38, set_end_time.min = 46)

# get blocks info
blocks_info <- df_w_blocks |> orbi_get_blocks_info()
blocks_info |> knitr::kable()

## ---- fig.width=8, fig.height=4-----------------------------------------------
# visualization
library(ggplot2)
library(dplyr)

df_w_blocks |> 
  filter(isotopocule == "15N") |>
  # data
  ggplot() +
  aes(x = time.min, y = ions.incremental, group = data_group, color = data_type) +
  geom_line() +
  # scales
  scale_fill_brewer(palette = "Set1") +
  scale_x_continuous(breaks = scales::breaks_pretty(8), expand = c(0, 0)) +
  theme_bw() +
  labs(x = "time [min]", y = "intensity")

## ---- fig.width=8, fig.height=4-----------------------------------------------
ggplot() +
  # blocks background
  geom_rect(
    data = blocks_info,
    map = aes(
      xmin = start_time.min,
      xmax = end_time.min,
      ymin = -Inf,
      ymax = Inf,
      # use sample name for data blocks, otherwise data type
      fill = ifelse(data_type == "data", sample_name, data_type)
    ), alpha = 0.5
  ) +
  # data
  geom_line(
    data = df_w_blocks |> filter(isotopocule == "15N"),
    map = aes(x = time.min, y = ions.incremental, group = data_group)
  ) +
  # scales
  scale_fill_brewer(palette = "Set1") +
  scale_x_continuous(breaks = scales::breaks_pretty(8), expand = c(0, 0)) +
  theme_bw() +
  labs(x = "time [min]", y = "intensity", fill = "block")

## -----------------------------------------------------------------------------
# calculate results
df_results <- 
  df_w_blocks |>
  # define base peak
  orbi_define_basepeak(basepeak_def = "M0") |> 
  # segment (optional)
  orbi_segment_blocks(into_segments = 3) |>
  # calculate results
  orbi_summarize_results(ratio_method = "sum")

## ---- fig.width=8, fig.height=7-----------------------------------------------
library(forcats)

df_results |>
  filter(data_type == "data") |>
  mutate(block_seg = paste0(block, ".", segment) |> fct_inorder()) |>
  # data
  ggplot() +
  aes(x = block_seg, y = ratio, color = sample_name) +
  geom_point(size = 2) +
  facet_grid(isotopocule ~ ., scales = "free_y") +
  # scales
  scale_color_brewer(palette = "Set1") +
  theme_bw() +
  labs(x = "block.segment", y = "ratio")

## ---- fig.width=8, fig.height=6-----------------------------------------------
# demonstrate use of a base plot
base_plot <-
  ggplot() +
  # blocks background
  geom_rect(
    data = blocks_info,
    map = aes(
      xmin = start_time.min, xmax = end_time.min, ymin = -Inf, ymax = Inf,
      fill = data_type
    ), alpha = 0.5
  ) +
  # data
  geom_point(
    data = function(df) df |> filter(panel == "ratio"),
    map = aes(x = mean_time.min, y = ratio, shape = sample_name),
    size = 3
  ) +
  geom_line(
    data = function(df) df |> filter(panel == "intensity"),
    map = aes(x = time.min, y = ions.incremental)
  ) +
  facet_grid(panel ~ ., scales = "free_y") +
  # scales
  scale_fill_brewer(palette = "Set1") +
  scale_x_continuous(breaks = scales::breaks_pretty(8), expand = c(0, 0)) +
  theme_bw() +
  labs(x = "time [min]", y = NULL, fill = "block")

# with 15N data
df_plot <-
  df_w_blocks |>
  mutate(panel = "intensity") |>
  bind_rows(df_results |> mutate(panel = "ratio"))

base_plot %+% filter(df_plot, isotopocule == "15N") + labs(title = "15N/M0")

## ---- fig.width=8, fig.height=6-----------------------------------------------
# same but with 18O
base_plot %+% filter(df_plot, isotopocule == "18O") + labs(title = "18O/M0")

## -----------------------------------------------------------------------------
# calculate ratios segmented with 5 scans at a time
df_ratios <- 
  df_w_blocks |>
  # define base peak
  orbi_define_basepeak(basepeak_def = "M0") |> 
  # segment (optional)
  orbi_segment_blocks(by_scans = 5) |>
  # calculate results
  orbi_summarize_results(ratio_method = "mean")

## ---- fig.width=8, fig.height=4-----------------------------------------------
ggplot() +
  # blocks background
  geom_rect(
    data = blocks_info,
    map = aes(
      xmin = start_time.min, xmax = end_time.min, ymin = -Inf, ymax = Inf,
      fill = data_type
    ), alpha = 0.5
  ) +
  # data
  geom_line(
    data = df_ratios |> filter(data_type == "data", isotopocule == "17O"),
    map = aes(x = mean_time.min, y = ratio, group = block)
  ) +
  # scales
  scale_fill_brewer(palette = "Set1") +
  scale_x_continuous(breaks = scales::breaks_pretty(8), expand = c(0, 0)) +
  theme_bw() +
  labs(x = "time [min]", y = "ratio", fill = "block")

