% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/evaluate_po.R
\name{evaluate_po}
\alias{evaluate_po}
\title{Evaluate the model based on presence-only data.}
\usage{
evaluate_po(
  model,
  occ_pred,
  bg_pred = NULL,
  var_pred,
  threshold = NULL,
  visualize = FALSE
)
}
\arguments{
\item{model}{(\code{isolation_forest}) The extended isolation forest SDM.
It could be the item \code{model} of \code{POIsotree} made by
function \code{\link{isotree_po}}.}

\item{occ_pred}{(\code{vector} of \code{numeric}) A \code{vector} contains predicted values
at occurrence locations.}

\item{bg_pred}{(\code{vector} of \code{numeric}) the vector contains predicted values
with same number of background points.}

\item{var_pred}{(\code{vector} of \code{numeric}) the vector contains predicted values
of the whole area.}

\item{threshold}{(\code{numeric} or \code{NULL}) The threshold to calculate
threshold-based evaluation metrics. If \code{NULL}, a recommended threshold
will be calculated based on optimal TSS value. The default is \code{NULL}.}

\item{visualize}{(\code{logical}) If \code{TRUE}, plot the evaluation figures.
The default is \code{FALSE}.}
}
\value{
(\code{POEvaluation}) A list of
\itemize{
\item{
\bold{po_evaluation} is presence-only evaluation metrics. It is a list of
\itemize{
\item{cvi (\code{list}) A list of CVI with 0.25, 0.5, and 0.75 as threshold}
\item{boyce (\code{list}) A list of items related to continuous Boyce index (CBI)}
\item{roc_ratio (\code{list}) A list of ROC ratio and AUC ratio}
}}
\item{
\bold{pb_evaluation} is presence-background evaluation metrics.
It is a list of
\itemize{
\item{confusion matrix (\code{table}) A table of confusion matrix. The columns are
true values, and the rows are predicted values.}
\item{sensitivity (\code{numeric}) The sensitivity or TPR}
\item{specificity (\code{numeric}) The specificity or TNR}
\item{TSS (\code{list}) A list of info related to true skill statistic (TSS)
\itemize{
\item{cutoff (\code{vector} of \code{numeric}) A vector of cutoff threshold values}
\item{tss (\code{vector} of \code{numeric}) A vector of TSS for each cutoff threshold}
\item{Recommended threshold (\code{numeric}) A recommended threshold
according to TSS}
\item{Optimal TSS (\code{numeric}) The best TSS value}}}
\item{roc (\code{list}) A list of ROC values and AUC value}
\item{Jaccard's similarity index (\code{numeric}) The Jaccard's similarity index}
\item{Sørensen's similarity index (\code{numeric}) The Sørensen's similarity index
or F-measure}
\item{Overprediction rate (\code{numeric}) The Overprediction rate}
\item{Underprediction rate (\code{numeric}) The Underprediction rate}
}}}
}
\description{
This function will calculate two major types of evaluation
metrics in terms of presence-only data. The first type is presence-only
customized metrics, such as Contrast Validation Index (CVI),
continuous Boyce index (CBI), and ROC_ratio.
The second type is presence-background evaluation metrics by
extracting background points as pseudo absence observations.
}
\details{
\itemize{
\item{
\bold{CVI} is the proportion of presence points falling in cells having
a threshold (\code{0.5} for example) habitat suitability index minus
the proportion of cells within this range of threshold of the model.
Here we used varied thresholds: \code{0.25}, \code{0.5}, and \code{0.75}.}
\item{\bold{continuous Boyce index (CBI)} is made with a 100 resolution of
moving windows and Kendall method.}
\item{
\bold{ROC_ratio} curve plots the proportion of presences falling above a
range of thresholds against the proportion of cells falling
above the range of thresholds. The area under the modified
ROC curve was then called \bold{AUC_ratio}.}
\item{\bold{Sensitivity (TPR)} = TP/(TP + FN)}
\item{\bold{Specificity (TNR)} = TN/(TN + FP)}
\item{\bold{True skill statistic (TSS)} = Sensitivity + specificity - 1}
\item{\bold{Jaccard's similarity index} = TP/(FN + TP + FP)}
\item{\bold{Sørensen's similarity index (F-measure)} = 2TP/(FN + 2TP + FP)}
\item{\bold{Overprediction rate} = FP/(TP + FP)}
\item{\bold{Underprediction rate} = FN/(TP + FN)}
}
}
\examples{
# Using a pseudo presence-only occurrence dataset of
# virtual species provided in this package
library(dplyr)
library(sf)
library(stars)
library(itsdm)

data("occ_virtual_species")
occ_virtual_species <- occ_virtual_species \%>\%
  mutate(id = row_number())

set.seed(11)
occ <- occ_virtual_species \%>\% sample_frac(0.7)
occ_test <- occ_virtual_species \%>\% filter(! id \%in\% occ$id)
occ <- occ \%>\% select(-id)
occ_test <- occ_test \%>\% select(-id)

env_vars <- system.file(
  'extdata/bioclim_tanzania_10min.tif',
  package = 'itsdm') \%>\% read_stars() \%>\%
  slice('band', c(1, 5, 12, 16))

mod <- isotree_po(
  occ = occ, occ_test = occ_test,
  variables = env_vars, ntrees = 50,
  sample_size = 0.8, ndim = 2L,
  seed = 123L, response = FALSE,
  spatial_response = FALSE,
  check_variable = FALSE)

eval_train <- evaluate_po(mod$model,
  occ_pred = mod$pred_train$prediction,
  var_pred = na.omit(as.vector(mod$prediction[[1]])))
print(eval_train)
plot(eval_train)

}
\references{
\itemize{
\item{Peterson,
A. Townsend, Monica Papeş, and Jorge Soberón. "Rethinking receiver operating
characteristic analysis applications in ecological niche modeling."
\emph{Ecological modelling} 213.1 (2008): 63-72.
\doi{10.1016/j.ecolmodel.2007.11.008}}
\item{Hirzel,
Alexandre H., et al. "Evaluating the ability of habitat suitability models
to predict species presences." \emph{Ecological modelling}
199.2 (2006): 142-152.
\doi{10.1016/j.ecolmodel.2006.05.017}}
\item{Hirzel, Alexandre
H., and Raphaël Arlettaz. "Modeling habitat suitability for complex species
distributions by environmental-distance geometric mean."
\emph{Environmental management} 32.5 (2003): 614-623.
\doi{10.1007/s00267-003-0040-3}}
\item{Leroy, Boris, et al.
"Without quality presence-absence data, discrimination metrics such as
TSS can be misleading measures of model performance."
\emph{Journal of Biogeography} 45.9 (2018): 1994-2002.
\doi{10.1111/jbi.13402}}
}
}
\seealso{
\code{\link{print.POEvaluation}}, \code{\link{plot.POEvaluation}}
}
