% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/learning_rate_schedules.R
\name{learning_rate_schedule_cosine_decay}
\alias{learning_rate_schedule_cosine_decay}
\title{A LearningRateSchedule that uses a cosine decay schedule}
\usage{
learning_rate_schedule_cosine_decay(
  initial_learning_rate,
  decay_steps,
  alpha = 0,
  ...,
  name = NULL
)
}
\arguments{
\item{initial_learning_rate}{A scalar \code{float32} or \code{float64} Tensor or a
R number. The initial learning rate.}

\item{decay_steps}{A scalar \code{int32} or \code{int64} \code{Tensor} or an R number.
Number of steps to decay over.}

\item{alpha}{A scalar \code{float32} or \code{float64} Tensor or an R number.
Minimum learning rate value as a fraction of initial_learning_rate.}

\item{...}{For backwards and forwards compatibility}

\item{name}{String. Optional name of the operation.  Defaults to
'CosineDecay'.}
}
\description{
A LearningRateSchedule that uses a cosine decay schedule
}
\details{
See \href{https://arxiv.org/abs/1608.03983}{Loshchilov & Hutter, ICLR2016},
SGDR: Stochastic Gradient Descent with Warm Restarts.

When training a model, it is often useful to lower the learning rate as
the training progresses. This schedule applies a cosine decay function
to an optimizer step, given a provided initial learning rate.
It requires a \code{step} value to compute the decayed learning rate. You can
just pass a TensorFlow variable that you increment at each training step.

The schedule is a 1-arg callable that produces a decayed learning
rate when passed the current optimizer step. This can be useful for changing
the learning rate value across different invocations of optimizer functions.
It is computed as:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{decayed_learning_rate <- function(step) \{
  step <- min(step, decay_steps)
  cosine_decay = <- 0.5 * (1 + cos(pi * step / decay_steps))
  decayed <- (1 - alpha) * cosine_decay + alpha
  initial_learning_rate * decayed
\}
}\if{html}{\out{</div>}}

Example usage:

\if{html}{\out{<div class="sourceCode R">}}\preformatted{decay_steps <- 1000
lr_decayed_fn <-
  learning_rate_schedule_cosine_decay(initial_learning_rate, decay_steps)
}\if{html}{\out{</div>}}

You can pass this schedule directly into a keras Optimizer
as the \code{learning_rate}.
}
\seealso{
\itemize{
\item \url{https://www.tensorflow.org/api_docs/python/tf/keras/optimizers/schedules/CosineDecay}
}
}
