% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tree_detection.R
\name{tree_segmentation}
\alias{tree_segmentation}
\title{Preprocessing and segmentation of raster image for tree identification}
\usage{
tree_segmentation(
  dem,
  nl_filter = "Closing",
  nl_size = 5,
  sigma = 0.3,
  dmin = 0,
  dprop = 0.05,
  hmin = 5,
  crown_prop = 0.3,
  crown_hmin = 2,
  dtm = NULL
)
}
\arguments{
\item{dem}{raster object or string indicating location of raster file 
(typically a canopy height model or a digital surface model; in the latter 
case the dtm parameter should be provided)}

\item{nl_filter}{string. specifies the non-linear filter for image pre-processing, 
should be an option of function \code{\link{dem_filtering}}}

\item{nl_size}{numeric. width of kernel of non-linear filter in pixels}

\item{sigma}{numeric or matrix. if a single number is provided, sigmap is the 
standard deviation of Gaussian filter in meters, 0 corresponds to no smoothing. 
In case of matrix, the first column corresponds to the standard deviation of 
the filter, and the second to thresholds for image values (e.g. a filter of 
standard deviation specified in line \code{i} is applied to pixels in image 
which values are between thresholds indicated in lines \code{i} and 
\code{i+1}). Threshold values should be ordered in increasing order.}

\item{dmin}{numeric. treetop minimum distance to next higher pixel in meters}

\item{dprop}{numeric. number defining the treetop minimum distance as 
proportion of height to next higher pixel}

\item{hmin}{numeric. minimum treetop height}

\item{crown_prop}{numeric. minimum height of tree crown as proportion of 
treetop height}

\item{crown_hmin}{numeric. minimum crown height}

\item{dtm}{raster object or string indicating location of raster file with 
the terrain model. If provided, the maxima extraction and watershed segmentation 
are performed on the dem (this avoids the deformation of crown because of the 
normalisation with terrain), but maxima selection and segment adjustment are 
performed on 'dem-dtm' because the selection criteria is the height to terrain.}
}
\value{
A SpatRaster with 4 layers: selected local maxima (values = 
distance to higher pixel), segments, non-linear preprocessed dem, smoothed 
preprocessed dem
}
\description{
global function for preprocessing (filtering), maxima detection and selection, 
segmentation and segmentation adjustment of a raster image.
}
\examples{
data(chm_chablais3)
chm_chablais3 <- terra::rast(chm_chablais3)

# tree segmentation
segments <- tree_segmentation(chm_chablais3)
segments2 <- tree_segmentation(chm_chablais3,
  nl_filter = "Median", nl_size = 3,
  sigma = cbind(c(0.2, 0.8), c(0, 15)), dmin = 0, dprop = 0, hmin = 10, 
  crown_prop = 0.5, crown_hmin = 5
)

# plot initial image segments
terra::plot(chm_chablais3, main = "Initial image")
terra::plot(segments$smoothed_dem, main = "Filtered image")
terra::plot(segments$local_maxima, main = "Local maxima")
#
# replace segment with id 0 (not a tree) with NA
segments$segments_id[segments$segments_id == 0] <- NA
terra::plot(segments$segments_id \%\% 8, main = "Segments", col = rainbow(8))
#
# plot segmentation with other parameters
segments2$segments_id[segments2$segments_id == 0] <- NA
terra::plot(segments2$segments_id \%\% 8, main = "Segments2", col = rainbow(8))
}
\references{
Monnet, J.-M. 2011. Using airborne laser scanning for mountain 
forests mapping: Support vector regression for stand parameters estimation 
and unsupervised training for treetop detection. Ph.D. thesis. University of 
Grenoble, France. Section 6.2 
\url{https://tel.archives-ouvertes.fr/tel-00652698/document}

Monnet, J.-M., Mermin, E., Chanussot, J., Berger, F. 2010. Tree top detection 
using local maxima filtering: a parameter sensitivity analysis. Silvilaser 2010, 
the 10th International Conference on LiDAR Applications for Assessing Forest 
Ecosystems, September 14-17, Freiburg, Germany, 9 p. 
\url{https://hal.archives-ouvertes.fr/hal-00523245/document}
}
\seealso{
\code{\link{dem_filtering}}, \code{\link{maxima_detection}}, 
\code{\link{maxima_selection}}, \code{\link{segmentation}}, 
\code{\link{seg_adjust}}, \code{\link{tree_extraction}},
\code{\link{tree_detection}}
}
