# ============================================================================
# Main logitr functions
# ============================================================================

#' The main function for estimating logit models
#'
#' Use this function to estimate multinomial (MNL) and mixed logit (MXL)
#' models with "Preference" space or "Willingness-to-pay" (WTP) space utility
#' parameterizations. The function includes an option to run a multistart
#' optimization loop with random starting points in each iteration, which is
#' useful for non-convex problems like MXL models or models with WTP space
#' utility parameterizations. The main optimization loop uses the `nloptr()`
#' function to minimize the negative log-likelihood function.
#' @keywords logitr mnl mxl wtp willingness-to-pay mixed logit
#'
#' @param data The choice data, formatted as a `data.frame` object.
#' @param choiceName The name of the column that identifies the choice variable.
#' @param obsIDName The name of the column that identifies the `obsID` variable.
#' @param parNames The names of the parameters to be estimated in the model.
#' Must be the same as the column names in the `data` argument. For WTP space
#' models, do not include price in `parNames`.
#' @param priceName The name of the column that identifies the price variable.
#' Only required for WTP space models. Defaults to `NULL`.
#' @param randPars A named vector whose names are the random parameters and
#' values the distribution: `'n'` for normal or `'ln'` for log-normal.
#' Defaults to `NULL`.
#' @param randPrice The random distribution for the price parameter: `'n'` for
#' normal or `'ln'` for log-normal. Only used for WTP space MXL models.
#' Defaults to `NULL`.
#' @param modelSpace Set to `'wtp'` for WTP space models. Defaults to `"pref"`.
#' @param weightsName The name of the column that identifies the weights to be
#' used in model estimation. Optional. Defaults to `NULL`.
#' @param options A list of options.
#'
#' @details
#' The following options control the detailed behavior of the optimization
#' algorithm. They must be provided as a named list to the `options` argument,
#' e.g. `options = list(...)`.
#'
#' |    Argument    |    Description    |    Default    |
#' |:---------------|:------------------|:--------------|
#' |`numMultiStarts`|Number of times to run the optimization loop, each time starting from a different random starting point for each parameter between `startParBounds`. Recommended for non-convex models, such as WTP space models and MXL models.|`1`|
#' |`keepAllRuns`|Set to `TRUE` to keep all the model information for each multistart run. If `TRUE`, the `logitr()` function will return a list with two values: `models` (a list of each model), and `bestModel` (the model with the largest log-likelihood value).|`FALSE`|
#' |`startParBounds`|Set the `lower` and `upper` bounds for the starting parameters for each optimization run, which are generated by `runif(n, lower, upper)`.|`c(-1, 1)`|
#' |`startVals`|A vector of values to be used as starting values for the optimization. Only used for the first run if `numMultiStarts > 1`.|`NULL`|
#' |`useAnalyticGrad`|Set to `FALSE` to use numerically approximated gradients instead of analytic gradients during estimation (which is slower).|`TRUE`|
#' |`scaleInputs`|By default each variable in `data` is scaled to be between 0 and 1 before running the optimization routine because it usually helps with stability, especially if some of the variables have very large or very small values (e.g. `> 10^3` or `< 10^-3`). Set to `FALSE` to turn this feature off.|`TRUE`|
#' |`standardDraws`|By default, a new set of standard normal draws are generated during each call to `logitr` (the same draws are used during each multistart iteration). The user can override those draws by providing a matrix of standard normal draws if desired.|`NULL`|
#' |`numDraws`|The number of draws to use for MXL models for the maximum simulated likelihood.|`50`|
#' |`printLevel`|The print level of the `nloptr` optimization loop. Use `nloptr::nloptr.print.options()` for more details.|`0`|
#' |`xtol_rel`|The relative `x` tolerance for the `nloptr` optimization loop. Use `nloptr::nloptr.print.options()` for more details.|`1.0e-6`|
#' |`xtol_abs`|The absolute `x` tolerance for the `nloptr` optimization loop. Use `nloptr::nloptr.print.options()` for more details.|`1.0e-6`|
#' |`ftol_rel`|The relative `f` tolerance for the `nloptr` optimization loop. Use `nloptr::nloptr.print.options()` for more details.|`1.0e-6`|
#' |`ftol_abs`|The absolute `f` tolerance for the `nloptr` optimization loop. Use `nloptr::nloptr.print.options()` for more details.|`1.0e-6`|
#' |`maxeval`|The maximum number of function evaluations for the `nloptr` optimization loop. Use `nloptr::nloptr.print.options()` for more details. |`1000`|
#' |`algorithm`|The optimization algorithm that `nloptr` uses.|`"NLOPT_LD_LBFGS"`|
#'
#' @return
#' The function returns a list object containing the following objects.
#'
#' |    Value    |    Description    |
#' |:------------|:------------------|
#' |`coef`|The model coefficients at convergence.|
#' |`standErrs`|The standard errors of the model coefficients at convergence.|
#' |`logLik`|The log-likelihood value at convergence.|
#' |`nullLogLik`|The null log-likelihood value (if all coefficients are 0).|
#' |`gradient`|The gradient of the log-likelihood at convergence.|
#' |`hessian`|The hessian of the log-likelihood at convergence.|
#' |`numObs`|The number of observations.|
#' |`numParams`|The number of model parameters.|
#' |`startPars`|The starting values used.|
#' |`multistartNumber`|The multistart run number for this model.|
#' |`time`|The user, system, and elapsed time to run the optimization.|
#' |`iterations`|The number of iterations until convergence.|
#' |`message`|A more informative message with the status of the optimization result.|
#' |`status`|An integer value with the status of the optimization (positive values are successes). Use [statusCodes()] for a detailed description.|
#' |`modelSpace`|The model space (`'pref'` or `'wtp'`).|
#' |`standardDraws`|The draws used during maximum simulated likelihood (for MXL models).|
#' |`randParSummary`|A summary of any random parameters (for MXL models).|
#' |`parSetup`|A summary of the distributional assumptions on each model parameter (`"f"`="fixed", `"n"`="normal distribution", `"ln"`="log-normal distribution").|
#' |`options`|A list of all the model options.|
#' |`multistartSummary`|A summary of the log-likelihood values for each multistart run.|
#'
#' @export
#' @examples
#' \dontrun{
#'
#' # For more detailed examples, visit
#' # https://jhelvy.github.io/logitr/articles/
#'
#' library(logitr)
#'
#' # Run a MNL model in the Preference Space:
#' mnl_pref <- logitr(
#'   data = yogurt,
#'   choiceName = "choice",
#'   obsIDName = "obsID",
#'   parNames = c("price", "feat", "dannon", "hiland", "yoplait")
#' )
#'
#' # Run a MNL model in the WTP Space:
#' mnl_wtp <- logitr(
#'   data = yogurt,
#'   choiceName = "choice",
#'   obsIDName = "obsID",
#'   parNames = c("feat", "dannon", "hiland", "yoplait"),
#'   priceName = "price",
#'   modelSpace = "wtp"
#' )
#' }
logitr <- function(data, choiceName, obsIDName, parNames, priceName = NULL,
                   randPars = NULL, randPrice = NULL, modelSpace = "pref",
                   weightsName = NULL, options = list()) {
  modelInputs <- getModelInputs(
    data, choiceName, obsIDName, parNames,
    randPars, priceName, randPrice, modelSpace, weightsName,
    options
  )
  allModels <- runMultistart(modelInputs)
  if (modelInputs$options$keepAllRuns) {
    models <- appendAllModelsInfo(allModels, modelInputs)
    message("Done!")
    return(models)
  } else {
    bestModel <- getBestModel(allModels, modelInputs)
    if (modelInputs$options$numMultiStarts > 1) {
      bestModel$multistartSummary <- getMultistartSummary(allModels)
      class(bestModel) <- c("logitr", "logitr.multistart")
    }
    message("Done!")
    return(bestModel)
  }
}

appendAllModelsInfo <- function(allModels, modelInputs) {
  models <- list()
  for (i in seq_len(length(allModels))) {
    models[[i]] <- appendModelInfo(allModels[[i]], modelInputs)
  }
  bestModel <- getBestModel(allModels, modelInputs)
  multistartSummary <- getMultistartSummary(models)
  result <- structure(list(
    models = models, bestModel = bestModel,
    multistartSummary = multistartSummary
  ),
  class = c("logitr", "logitr.multistart", "logitr.allRuns")
  )
  return(result)
}

getMultistartSummary <- function(models) {
  summary <- as.data.frame(matrix(0, ncol = 4, nrow = length(models)))
  colnames(summary) <- c("run", "logLik", "iterations", "status")
  for (i in seq_len(length(models))) {
    summary[i, 1] <- i
    summary[i, 2] <- round(models[[i]]$logLik, 5)
    summary[i, 3] <- models[[i]]$iterations
    summary[i, 4] <- models[[i]]$status
  }
  return(summary)
}

getBestModel <- function(models, modelInputs) {
  logLikVals <- getLogLikVals(models)
  bestModel <- models[[which(logLikVals == max(logLikVals))[1]]]
  bestModel <- appendModelInfo(bestModel, modelInputs)
  return(bestModel)
}

getLogLikVals <- function(models) {
  logLikVals <- matrix(0)
  for (i in seq_len(length(models))) {
    logLikVals[i] <- models[[i]]$logLik
  }
  return(logLikVals)
}
