% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pipe.R
\name{\%>\%}
\alias{\%>\%}
\title{Pipe}
\usage{
lhs \%>\% rhs
}
\arguments{
\item{lhs}{A value or the magrittr placeholder.}

\item{rhs}{A function call using the magrittr semantics.}
}
\description{
Pipe an object forward into a function or call expression.
}
\details{
\subsection{Using \verb{\%>\%} with unary function calls}{

When functions require only one argument, \code{x \%>\% f} is equivalent
to \code{f(x)} (not exactly equivalent; see technical note below.)

}
\subsection{Placing \code{lhs} as the first argument in \code{rhs} call}{

The default behavior of \verb{\%>\%} when multiple arguments are required
in the \code{rhs} call, is to place \code{lhs} as the first argument, i.e.
\code{x \%>\% f(y)} is equivalent to \code{f(x, y)}.
}
\subsection{Placing \code{lhs} elsewhere in \code{rhs} call}{

Often you will want \code{lhs} to the \code{rhs} call at another position than the first.
For this purpose you can use the dot (\code{.}) as placeholder. For example,
\code{y \%>\% f(x, .)} is equivalent to \code{f(x, y)} and
\code{z \%>\% f(x, y, arg = .)} is equivalent to \code{f(x, y, arg = z)}.
}

\subsection{Using the dot for secondary purposes}{

Often, some attribute or property of \code{lhs} is desired in the \code{rhs} call in
addition to the value of \code{lhs} itself, e.g. the number of rows or columns.
It is perfectly valid to use the dot placeholder several times in the \code{rhs}
call, but by design the behavior is slightly different when using it inside
nested function calls. In particular, if the placeholder is only used
in a nested function call, \code{lhs} will also be placed as the first argument!
The reason for this is that in most use-cases this produces the most readable
code. For example, \code{iris \%>\% subset(1:nrow(.) \%\% 2 == 0)} is
equivalent to \code{iris \%>\% subset(., 1:nrow(.) \%\% 2 == 0)} but
slightly more compact. It is possible to overrule this behavior by enclosing
the \code{rhs} in braces. For example, \code{1:10 \%>\% {c(min(.), max(.))}} is
equivalent to \code{c(min(1:10), max(1:10))}.
}

\subsection{Using \verb{\%>\%} with call- or function-producing \code{rhs}}{

It is possible to force evaluation of \code{rhs} before the piping of \code{lhs} takes
place. This is useful when \code{rhs} produces the relevant call or function.
To evaluate \code{rhs} first, enclose it in parentheses, i.e.
\code{a \%>\% (function(x) x^2)}, and \code{1:10 \%>\% (call("sum"))}.
Another example where this is relevant is for reference class methods
which are accessed using the \code{$} operator, where one would do
\code{x \%>\% (rc$f)}, and not \code{x \%>\% rc$f}.
}

\subsection{Using lambda expressions with \verb{\%>\%}}{

Each \code{rhs} is essentially a one-expression body of a unary function.
Therefore defining lambdas in magrittr is very natural, and as
the definitions of regular functions: if more than a single expression
is needed one encloses the body in a pair of braces, \code{\{ rhs \}}.
However, note that within braces there are no "first-argument rule":
it will be exactly like writing a unary function where the argument name is
"\code{.}" (the dot).

}
\subsection{Using the dot-place holder as \code{lhs}}{

When the dot is used as \code{lhs}, the result will be a functional sequence,
i.e. a function which applies the entire chain of right-hand sides in turn
to its input. See the examples.
}
}
\section{Technical notes}{

The magrittr pipe operators use non-standard evaluation. They capture
their inputs and examines them to figure out how to proceed. First a function
is produced from all of the individual right-hand side expressions, and
then the result is obtained by applying this function to the left-hand side.
For most purposes, one can disregard the subtle aspects of magrittr's
evaluation, but some functions may capture their calling environment,
and thus using the operators will not be exactly equivalent to the
"standard call" without pipe-operators.

Another note is that special attention is advised when using non-magrittr
operators in a pipe-chain (\verb{+, -, $,} etc.), as operator precedence will impact how the
chain is evaluated. In general it is advised to use the aliases provided
by magrittr.
}

\examples{
# Basic use:
iris \%>\% head

# Use with lhs as first argument
iris \%>\% head(10)

# Using the dot place-holder
"Ceci n'est pas une pipe" \%>\% gsub("une", "un", .)
  
# When dot is nested, lhs is still placed first:
sample(1:10) \%>\% paste0(LETTERS[.])

# This can be avoided:
rnorm(100) \%>\% {c(min(.), mean(.), max(.))} \%>\% floor

# Lambda expressions: 
iris \%>\%
{
  size <- sample(1:10, size = 1)
  rbind(head(., size), tail(., size))
}

# renaming in lambdas:
iris \%>\%
{
  my_data <- .
  size <- sample(1:10, size = 1)
  rbind(head(my_data, size), tail(my_data, size))
}

# Building unary functions with \%>\%
trig_fest <- . \%>\% tan \%>\% cos \%>\% sin

1:10 \%>\% trig_fest
trig_fest(1:10)

}
\seealso{
\code{\link{\%<>\%}}, \code{\link{\%T>\%}}, \code{\link{\%$\%}}
}
