% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/comparisons.R
\name{comparisons}
\alias{comparisons}
\title{Contrasts Between Adjusted Predictions}
\usage{
comparisons(
  model,
  newdata = NULL,
  variables = NULL,
  type = NULL,
  vcov = TRUE,
  conf_level = 0.95,
  transform_pre = "difference",
  transform_post = NULL,
  cross = FALSE,
  by = NULL,
  wts = NULL,
  hypothesis = NULL,
  eps = NULL,
  ...
)
}
\arguments{
\item{model}{Model object}

\item{newdata}{\code{NULL}, data frame, string, or \code{datagrid()} call. Determines the predictor values for which to compute contrasts.
\itemize{
\item \code{NULL} (default): Unit-level contrasts for each observed value in the original dataset.
\item data frame: Unit-level contrasts for each row of the \code{newdata} data frame.
\item string:
\itemize{
\item "mean": Contrasts at the Mean. Contrasts when each predictor is held at its mean or mode.
\item "median": Contrasts at the Median. Contrasts when each predictor is held at its median or mode.
\item "marginalmeans": Contrasts at Marginal Means.
\item "tukey": Contrasts at Tukey's 5 numbers.
\item "grid": Contrasts on a grid of representative numbers (Tukey's 5 numbers and unique values of categorical predictors).
}
\item \code{\link[=datagrid]{datagrid()}} call to specify a custom grid of regressors. For example:
\itemize{
\item \code{newdata = datagrid(cyl = c(4, 6))}: \code{cyl} variable equal to 4 and 6 and other regressors fixed at their means or modes.
\item \code{newdata = datagrid(mpg = fivenum)}: \code{mpg} variable held at Tukey's five numbers (using the \code{fivenum} function), and other regressors fixed at their means or modes.
\item See the Examples section and the \link{datagrid} documentation.
}
}}

\item{variables}{\code{NULL}, character vector, or named list. The subset of variables for which to compute contrasts.
\itemize{
\item \code{NULL}: compute contrasts for all the variables in the model object (can be slow).
\item Character vector: subset of variables (usually faster).
\item Named list: names identify the subset of variables of interest, and values define the type of contrast to compute. Acceptable values depend on the variable type:
\itemize{
\item Factor or character variables:
\itemize{
\item "reference": Each factor level is compared to the factor reference (base) level
\item "all": All combinations of observed levels
\item "sequential": Each factor level is compared to the previous factor level
\item "pairwise": Each factor level is compared to all other levels
\item Vector of length 2 with the two values to compare.
}
\item Logical variables:
\itemize{
\item NULL: contrast between TRUE and FALSE
}
\item Numeric variables:
\itemize{
\item Numeric of length 1: Contrast for a gap of \code{x}, computed at the observed value plus and minus \code{x / 2}. For example, estimating a \code{+1} contrast compares adjusted predictions when the regressor is equal to its observed value minus 0.5 and its observed value plus 0.5.
\item Numeric vector of length 2: Contrast between the 2nd element and the 1st element of the \code{x} vector.
\item Function which accepts a numeric vector and returns a data frame with two columns of "low" and "high" values to compare. See examples below.
\item "iqr": Contrast across the interquartile range of the regressor.
\item "sd": Contrast across one standard deviation around the regressor mean.
\item "2sd": Contrast across two standard deviations around the regressor mean.
\item "minmax": Contrast between the maximum and the minimum values of the regressor.
}
\item Examples:
\itemize{
\item \code{variables = list(gear = "pairwise", hp = 10)}
\item \code{variables = list(gear = "sequential", hp = c(100, 120))}
\item See the Examples section below for more.
}
}
}}

\item{type}{string indicates the type (scale) of the predictions used to
compute marginal effects or contrasts. This can differ based on the model
type, but will typically be a string such as: "response", "link", "probs",
or "zero". When an unsupported string is entered, the model-specific list of
acceptable values is returned in an error message. When \code{type} is \code{NULL}, the
default value is used. This default is the first model-related row in
the \code{marginaleffects:::type_dictionary} dataframe.}

\item{vcov}{Type of uncertainty estimates to report (e.g., for robust standard errors). Acceptable values:
\itemize{
\item FALSE: Do not compute standard errors. This can speed up computation considerably.
\item TRUE: Unit-level standard errors using the default \code{vcov(model)} variance-covariance matrix.
\item String which indicates the kind of uncertainty estimates to return.
\itemize{
\item Heteroskedasticity-consistent: \code{"HC"}, \code{"HC0"}, \code{"HC1"}, \code{"HC2"}, \code{"HC3"}, \code{"HC4"}, \code{"HC4m"}, \code{"HC5"}. See \code{?sandwich::vcovHC}
\item Heteroskedasticity and autocorrelation consistent: \code{"HAC"}
\item Mixed-Models degrees of freedom: "satterthwaite", "kenward-roger"
\item Other: \code{"NeweyWest"}, \code{"KernHAC"}, \code{"OPG"}. See the \code{sandwich} package documentation.
}
\item One-sided formula which indicates the name of cluster variables (e.g., \code{~unit_id}). This formula is passed to the \code{cluster} argument of the \code{sandwich::vcovCL} function.
\item Square covariance matrix
\item Function which returns a covariance matrix (e.g., \code{stats::vcov(model)})
}}

\item{conf_level}{numeric value between 0 and 1. Confidence level to use to build a confidence interval.}

\item{transform_pre}{string or function. How should pairs of adjusted predictions be contrasted?
\itemize{
\item string: shortcuts to common contrast functions.
\itemize{
\item Supported shortcuts strings: difference, differenceavg, differenceavgwts, dydx, eyex, eydx, dyex, dydxavg, eyexavg, eydxavg, dyexavg, dydxavgwts, eyexavgwts, eydxavgwts, dyexavgwts, ratio, ratioavg, ratioavgwts, lnratio, lnratioavg, lnratioavgwts, lnor, lnoravg, lnoravgwts, expdydx, expdydxavg, expdydxavgwts
\item See the Transformations section below for definitions of each transformation.
}
\item function: accept two equal-length numeric vectors of adjusted predictions (\code{hi} and \code{lo}) and returns a vector of contrasts of the same length, or a unique numeric value.
\itemize{
\item See the Transformations section below for examples of valid functions.
}
}}

\item{transform_post}{(experimental) A function applied to unit-level estimates and confidence intervals just before the function returns results.}

\item{cross}{TRUE or FALSE
\itemize{
\item \code{FALSE}: Contrasts represent the change in adjusted predictions when one predictor changes and all other variables are held constant.
\item \code{TRUE}: Contrasts represent the changes in adjusted predictions when all the predictors specified in the \code{variables} argument are manipulated simultaneously (a "cross-contrast").
}}

\item{by}{Compute group-wise average estimates. Valid inputs:
\itemize{
\item Character vector of column names in \code{newdata} or in the data frame produced by calling the function without the \code{by} argument.
\item Data frame with a \code{by} column of group labels, and merging columns shared by \code{newdata} or the data frame produced by calling the same function without the \code{by} argument.
\item See examples below.
}}

\item{wts}{string or numeric: weights to use when computing average
contrasts or marginaleffects. These weights only affect the averaging in
\code{tidy()} or \code{summary()}, and not the unit-level estimates themselves.
\itemize{
\item string: column name of the weights variable in \code{newdata}. When supplying a column name to \code{wts}, it is recommended to supply the original data (including the weights variable) explicitly to \code{newdata}.
\item numeric: vector of length equal to the number of rows in the original data or in \code{newdata} (if supplied).
}}

\item{hypothesis}{specify a hypothesis test or custom contrast using a vector, matrix, string, or string formula.
\itemize{
\item String:
\itemize{
\item "pairwise": pairwise differences between estimates in each row.
\item "reference": differences between the estimates in each row and the estimate in the first row.
\item "sequential": difference between an estimate and the estimate in the next row.
\item "revpairwise", "revreference", "revsequential": inverse of the corresponding hypotheses, as described above.
}
\item String formula to specify linear or non-linear hypothesis tests. If the \code{term} column uniquely identifies rows, terms can be used in the formula. Otherwise, use \code{b1}, \code{b2}, etc. to identify the position of each parameter. Examples:
\itemize{
\item \code{hp = drat}
\item \code{hp + drat = 12}
\item \code{b1 + b2 + b3 = 0}
}
\item Numeric vector: Weights to compute a linear combination of (custom contrast between) estimates. Length equal to the number of rows generated by the same function call, but without the \code{hypothesis} argument.
\item Numeric matrix: Each column is a vector of weights, as describe above, used to compute a distinct linear combination of (contrast between) estimates. The column names of the matrix are used as labels in the output.
\item See the Examples section below and the vignette: https://vincentarelbundock.github.io/marginaleffects/articles/hypothesis.html
}}

\item{eps}{NULL or numeric value which determines the step size to use when
calculating numerical derivatives: (f(x+eps)-f(x))/eps. When \code{eps} is
\code{NULL}, the step size is 0.0001 multiplied by the difference between
the maximum and minimum values of the variable with respect to which we
are taking the derivative. Changing \code{eps} may be necessary to avoid
numerical problems in certain models.}

\item{...}{Additional arguments are passed to the \code{predict()} method
supplied by the modeling package.These arguments are particularly useful
for mixed-effects or bayesian models (see the online vignettes on the
\code{marginaleffects} website). Available arguments can vary from model to
model, depending on the range of supported arguments by each modeling
package. See the "Model-Specific Arguments" section of the
\code{?marginaleffects} documentation for a non-exhaustive list of available
arguments.}
}
\description{
Difference, ratio, or function of adjusted predictions, calculated for
meaningfully different predictor values. The \code{tidy()} and \code{summary()}
functions can be used to aggregate and summarize the output of
\code{comparisons()}. To learn more, read the contrasts vignette, visit the
package website, or scroll down this page for a full list of vignettes:
\itemize{
\item \url{https://vincentarelbundock.github.io/marginaleffects/articles/contrasts.html}
\item \url{https://vincentarelbundock.github.io/marginaleffects/}
}
}
\details{
A "contrast" is a difference, ratio of function of adjusted predictions,
calculated for meaningfully different predictor values (e.g., College
graduates vs. Others). Uncertainty estimates are computed using the delta
method.

The \code{newdata} argument can be used to control the kind of contrasts to report:
\itemize{
\item Average Contrasts
\item Adjusted Risk Ratios
\item Adjusted Risk Differences
\item Group-Average Contrasts
\item Contrasts at the Mean
\item Contrasts at User-Specified values (aka Contrasts at Representative values, MER).
\item Custom contrasts using arbitrary functions
}
}
\section{Vignettes and documentation}{


Vignettes:
\itemize{
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/predictions.html}{Adjusted Predictions}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/contrasts.html}{Contrasts}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/marginaleffects.html}{Marginal Effects}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/marginalmeans.html}{Marginal Means}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/hypothesis.html}{Hypothesis Tests and Custom Contrasts using the Delta Method}
}

Case studies:
\itemize{
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/brms.html}{Bayesian Analyses with \code{brms}}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/gformula.html}{Causal Inference with the g-Formula}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/elasticity.html}{Elasticity}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/experiments.html}{Experiments}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/gam.html}{Generalized Additive Models}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/lme4.html}{Mixed effects models}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/mlogit.html}{Multinomial Logit and Discrete Choice Models}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/multiple_imputation.html}{Multiple Imputation}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/plot.html}{Plots: interactions, predictions, contrasts, and slopes}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/python.html}{Python NumPyro models in \code{marginaleffects}}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/logistic_contrasts.html}{Unit-level contrasts in logistic regressions}
}

Tips and technical notes:
\itemize{
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/supported_models.html}{71 Supported Classes of Models}
\item \href{https://vincentarelbundock.github.io/marginaleffects/reference/index.html}{Index of Functions and Documentation}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/extensions.html}{Extending \code{marginaleffects}: add new models or modify existing ones}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/sandwich.html}{Standard Errors and Confidence Intervals}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/modelsummary.html}{Tables and Plots}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/performance.html}{Performance}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/alternative_software.html}{Alternative Software}
\item \href{https://vincentarelbundock.github.io/marginaleffects/articles/faq.html}{Frequently Asked Questions}
}
}

\section{Model-Specific Arguments}{


Some model types allow model-specific arguments to modify the nature of
marginal effects, predictions, marginal means, and contrasts.\tabular{llll}{
   Package \tab Class \tab Argument \tab Documentation \cr
   \code{brms} \tab \code{brmsfit} \tab \code{ndraws} \tab \link[brms:posterior_predict.brmsfit]{brms::posterior_predict} \cr
    \tab  \tab \code{re_formula} \tab  \cr
   \code{lme4} \tab \code{merMod} \tab \code{include_random} \tab \link[insight:get_predicted]{insight::get_predicted} \cr
    \tab  \tab \code{re.form} \tab \link[lme4:predict.merMod]{lme4::predict.merMod} \cr
    \tab  \tab \code{allow.new.levels} \tab \link[lme4:predict.merMod]{lme4::predict.merMod} \cr
   \code{glmmTMB} \tab \code{glmmTMB} \tab \code{re.form} \tab \link[glmmTMB:predict.glmmTMB]{glmmTMB::predict.glmmTMB} \cr
    \tab  \tab \code{allow.new.levels} \tab \link[glmmTMB:predict.glmmTMB]{glmmTMB::predict.glmmTMB} \cr
    \tab  \tab \code{zitype} \tab \link[glmmTMB:predict.glmmTMB]{glmmTMB::predict.glmmTMB} \cr
   \code{mgcv} \tab \code{bam} \tab \code{exclude} \tab \link[mgcv:predict.bam]{mgcv::predict.bam} \cr
   \code{robustlmm} \tab \code{rlmerMod} \tab \code{re.form} \tab \link[robustlmm:rlmerMod-class]{robustlmm::predict.rlmerMod} \cr
    \tab  \tab \code{allow.new.levels} \tab \link[robustlmm:rlmerMod-class]{robustlmm::predict.rlmerMod} \cr
}
}

\section{Transformations}{


The following transformations can be applied by supplying one of the shortcut strings to the
\code{transform_pre} argument.
\code{hi} is a vector of adjusted predictions for the "high" side of the
contrast. \code{lo} is a vector of adjusted predictions for the "low" side of the
contrast. \code{y} is a vector of adjusted predictions for the original data. \code{x}
is the predictor in the original data. \code{eps} is the step size to use to
compute derivatives and elasticities.\tabular{ll}{
   Shortcut \tab Function \cr
   difference \tab \(hi, lo) hi - lo \cr
   differenceavg \tab \(hi, lo) mean(hi) - mean(lo) \cr
   differenceavgwts \tab \(hi, lo, w) wmean(hi, w) - wmean(lo, w) \cr
   dydx \tab \(hi, lo, eps) (hi - lo)/eps \cr
   eyex \tab \(hi, lo, eps, y, x) (hi - lo)/eps * (x/y) \cr
   eydx \tab \(hi, lo, eps, y, x) ((hi - lo)/eps)/y \cr
   dyex \tab \(hi, lo, eps, x) ((hi - lo)/eps) * x \cr
   dydxavg \tab \(hi, lo, eps) mean((hi - lo)/eps) \cr
   eyexavg \tab \(hi, lo, eps, y, x) mean((hi - lo)/eps * (x/y)) \cr
   eydxavg \tab \(hi, lo, eps, y, x) mean(((hi - lo)/eps)/y) \cr
   dyexavg \tab \(hi, lo, eps, x) mean(((hi - lo)/eps) * x) \cr
   dydxavgwts \tab \(hi, lo, eps, w) wmean((hi - lo)/eps, w) \cr
   eyexavgwts \tab \(hi, lo, eps, y, x, w) wmean((hi - lo)/eps * (x/y), w) \cr
   eydxavgwts \tab \(hi, lo, eps, y, x, w) wmean(((hi - lo)/eps)/y, w) \cr
   dyexavgwts \tab \(hi, lo, eps, x, w) wmean(((hi - lo)/eps) * x, w) \cr
   ratio \tab \(hi, lo) hi/lo \cr
   ratioavg \tab \(hi, lo) mean(hi)/mean(lo) \cr
   ratioavgwts \tab \(hi, lo) wmean(hi)/wmean(lo) \cr
   lnratio \tab \(hi, lo) log(hi/lo) \cr
   lnratioavg \tab \(hi, lo) log(mean(hi)/mean(lo)) \cr
   lnratioavgwts \tab \(hi, lo) log(wmean(hi)/wmean(lo)) \cr
   lnor \tab \(hi, lo) log((hi/(1 - hi))/(lo/(1 - lo))) \cr
   lnoravg \tab \(hi, lo) log((mean(hi)/(1 - mean(hi)))/(mean(lo)/(1 - mean(lo)))) \cr
   lnoravgwts \tab \(hi, lo, w) log((wmean(hi, w)/(1 - wmean(hi, w)))/(wmean(lo, w)/(1 - wmean(lo, w)))) \cr
   expdydx \tab \(hi, lo, eps) ((exp(hi) - exp(lo))/exp(eps))/eps \cr
   expdydxavg \tab \(hi, lo, eps) mean(((exp(hi) - exp(lo))/exp(eps))/eps) \cr
   expdydxavgwts \tab \(hi, lo, eps, w) wmean(((exp(hi) - exp(lo))/exp(eps))/eps, w) \cr
}
}

\examples{

library(marginaleffects)
library(magrittr)

# Linear model
tmp <- mtcars
tmp$am <- as.logical(tmp$am)
mod <- lm(mpg ~ am + factor(cyl), tmp)
comparisons(mod, variables = list(cyl = "reference")) \%>\% tidy()
comparisons(mod, variables = list(cyl = "sequential")) \%>\% tidy()
comparisons(mod, variables = list(cyl = "pairwise")) \%>\% tidy()

# GLM with different scale types
mod <- glm(am ~ factor(gear), data = mtcars)
comparisons(mod, type = "response") \%>\% tidy()
comparisons(mod, type = "link") \%>\% tidy()

# Contrasts at the mean
comparisons(mod, newdata = "mean")

# Contrasts between marginal means
comparisons(mod, newdata = "marginalmeans")

# Contrasts at user-specified values
comparisons(mod, newdata = datagrid(am = 0, gear = tmp$gear))
comparisons(mod, newdata = datagrid(am = unique, gear = max))

# Numeric contrasts
mod <- lm(mpg ~ hp, data = mtcars)
comparisons(mod, variables = list(hp = 1)) \%>\% tidy()
comparisons(mod, variables = list(hp = 5)) \%>\% tidy()
comparisons(mod, variables = list(hp = c(90, 100))) \%>\% tidy()
comparisons(mod, variables = list(hp = "iqr")) \%>\% tidy()
comparisons(mod, variables = list(hp = "sd")) \%>\% tidy()
comparisons(mod, variables = list(hp = "minmax")) \%>\% tidy()

# using a function to specify a custom difference in one regressor
dat <- mtcars
dat$new_hp <- 49 * (dat$hp - min(dat$hp)) / (max(dat$hp) - min(dat$hp)) + 1
modlog <- lm(mpg ~ log(new_hp) + factor(cyl), data = dat)
fdiff <- \(x) data.frame(x, x + 10)
comparisons(modlog, variables = list(new_hp = fdiff)) \%>\% summary()

# Adjusted Risk Ratio: see the contrasts vignette
mod <- glm(vs ~ mpg, data = mtcars, family = binomial)
cmp <- comparisons(mod, transform_pre = "lnratioavg")
summary(cmp, transform_avg = exp)

# Adjusted Risk Ratio: Manual specification of the `transform_pre`
cmp <- comparisons(mod, transform_pre = function(hi, lo) log(mean(hi) / mean(lo)))
summary(cmp, transform_avg = exp)
# cross contrasts
mod <- lm(mpg ~ factor(cyl) * factor(gear) + hp, data = mtcars)
cmp <- comparisons(mod, variables = c("cyl", "gear"), cross = TRUE)
summary(cmp)

# variable-specific contrasts
cmp <- comparisons(mod, variables = list(gear = "sequential", hp = 10))
summary(cmp)

# hypothesis test: is the `hp` marginal effect at the mean equal to the `drat` marginal effect
mod <- lm(mpg ~ wt + drat, data = mtcars)

comparisons(
    mod,
    newdata = "mean",
    hypothesis = "wt = drat")

# same hypothesis test using row indices
comparisons(
    mod,
    newdata = "mean",
    hypothesis = "b1 - b2 = 0")

# same hypothesis test using numeric vector of weights
comparisons(
    mod,
    newdata = "mean",
    hypothesis = c(1, -1))

# two custom contrasts using a matrix of weights
lc <- matrix(c(
    1, -1,
    2, 3),
    ncol = 2)
comparisons(
    mod,
    newdata = "mean",
    hypothesis = lc)


# `by` argument
mod <- lm(mpg ~ hp * am * vs, data = mtcars)
cmp <- comparisons(mod, variables = "hp", by = c("vs", "am"))
summary(cmp)

library(nnet)
mod <- multinom(factor(gear) ~ mpg + am * vs, data = mtcars, trace = FALSE)
by <- data.frame(
    group = c("3", "4", "5"),
    by = c("3,4", "3,4", "5"))
comparisons(mod, type = "probs", by = by)

}
