% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/est.mean.R
\name{est.mean}
\alias{est.mean}
\title{Estimating Sample Mean using Quantiles}
\usage{
est.mean(
   min = NULL, 
   q1 = NULL, 
   med = NULL, 
   q3 = NULL, 
   max = NULL, 
   n = NULL, 
   method = "gld/sld", 
   opt = TRUE
   )
}
\arguments{
\item{min}{numeric value representing the sample minimum.}

\item{q1}{numeric value representing the first quartile of the sample.}

\item{med}{numeric value representing the median of the sample.}

\item{q3}{numeric value representing the third quartile of the sample.}

\item{max}{numeric value representing the sample maximum.}

\item{n}{numeric value specifying the sample size.}

\item{method}{character string specifying the approach used to estimate the sample means. The options are the following:
\describe{
\item{\code{'gld/sld'}}{The default option. The method proposed by De Livera et al. (2024). Estimation using the Generalized Lambda Distribution (GLD) for 5-number summaries (\eqn{S_3}), and the Skew Logistic Distribution (SLD)  for 3-number summaries (\eqn{S_1} and \eqn{S_2}).}
\item{\code{'luo'}}{Method of Luo et al. (2018).}
\item{\code{'hozo/wan/bland'}}{The method proposed by Wan et al. (2014). i.e., the method of Hozo et al. (2005) for \eqn{S_1}, method of Wan et al. (2014) for \eqn{S_2}, and method of Bland (2015) for \eqn{S_3}.}
\item{\code{'bc'}}{Box-Cox method proposed by McGrath et al. (2020).}
\item{\code{'qe'}}{Quantile Matching Estimation method proposed by McGrath et al. (2020).}
}}

\item{opt}{logical value indicating whether to apply the optimization step of \code{'gld/sld'} method, in estimating their parameters using theoretical quantiles.
The default value is \code{TRUE}.}
}
\value{
\code{mean}: numeric value representing the estimated mean of the sample.
}
\description{
This function estimates the sample mean from a study presenting quantile summary measures with the sample size (\eqn{n}). The quantile summaries can fall into one of the following categories:
\itemize{
\item \eqn{S_1}: \{ minimum, median, maximum \}
\item \eqn{S_2}: \{ first quartile, median, third quartile \}
\item \eqn{S_3}: \{ minimum, first quartile, median, third quartile, maximum \}
}

The \code{est.mean} function implements newly proposed flexible quantile-based distribution methods for estimating sample mean (De Livera et al., 2024).
It also incorporates existing methods for estimating sample means as described by Luo et al. (2018) and McGrath et al. (2020).
}
\details{
The \code{'gld/sld'} method (i.e., the method of De Livera et al., (2024)) of \code{est.mean} uses the following quantile based distributions:
\itemize{
\item Generalized Lambda Distribution (GLD) for estimating the sample mean using 5-number summaries (\eqn{S_3}).
\item Skew Logistic Distribution (SLD) for estimating the sample mean using 3-number summaries (\eqn{S_1} and \eqn{S_2}).
}
The generalised lambda distribution (GLD) is a four parameter family of distributions defined by its quantile function under the FKML parameterisation (Freimer et al., 1988).
De Livera et al. propose that the GLD quantlie function can be used to approximate a sample's distribution using 5-point summaries.
The four parameters of GLD quantile function include: a location parameter (\eqn{\lambda_1}), an inverse scale parameter (\eqn{\lambda_2}>0), and two shape parameters (\eqn{\lambda_3} and \eqn{\lambda_4}).

The quantile-based skew logistic distribution (SLD), introduced by Gilchrist (2000) and further modified by van Staden and King (2015)
is used to approximate the sample's distribution using 3-point summaries.
The SLD quantile function is defined using three parameters: a location parameter (\eqn{\lambda}), a scale parameter (\eqn{\eta}), and a skewing parameter (\eqn{\delta}).

For \code{'gld/sld'} method, the parameters of the generalized lambda distribution (GLD) and skew logistic distribution (SLD) are estimated
by formulating and solving a set of simultaneous equations. These equations relate the estimated sample quantiles to their theoretical counterparts
of the respective distribution (GLD or SLD). Finally, the mean for each scenario is calculated by integrating functions of the estimated quantile function.
}
\examples{
#Generate 5-point summary data
set.seed(123)
n <- 1000
x <- stats::rlnorm(n, 4, 0.3)
quants <- c(min(x), stats::quantile(x, probs = c(0.25, 0.5, 0.75)), max(x))
obs_mean <- mean(x)

#Estimate sample mean using s3 (5 number summary)
est_mean_s3 <- est.mean(min = quants[1], q1 = quants[2], med = quants[3], q3 = quants[4], 
                        max = quants[5], n=n, method = "gld/sld")
est_mean_s3

#Estimate sample mean using s1 (min, median, max)
est_mean_s1 <- est.mean(min = quants[1], med = quants[3], max = quants[5],
                        n=n, method = "gld/sld")
est_mean_s1

#Estimate sample mean using s2 (q1, median, q3)
est_mean_s2 <- est.mean(q1 = quants[2], med = quants[3], q3 = quants[4],
                        n=n, method = "gld/sld")
est_mean_s2

}
\references{
Alysha De Livera, Luke Prendergast, and Udara Kumaranathunga. A novel density-based approach for estimating unknown means, distribution visualisations, and meta-analyses of quantiles. \emph{Submitted for Review}, 2024, pre-print available here: \url{https://arxiv.org/abs/2411.10971}

Dehui Luo, Xiang Wan, Jiming Liu, and Tiejun Tong. Optimally estimating the sample mean from the sample size, median, mid-range, and/or mid-quartile range. \emph{Statistical methods in medical research}, 27(6):1785–1805,2018.

Xiang Wan, Wenqian Wang, Jiming Liu, and Tiejun Tong. Estimating the sample mean and standard deviation from the sample size, median, range and/or interquartile range. \emph{BMC medical research methodology}, 14:1–13, 2014.

Sean McGrath, XiaoFei Zhao, Russell Steele, Brett D Thombs, Andrea Benedetti, and DEPRESsion Screening Data (DEPRESSD) Collaboration. Estimating the sample mean and standard deviation from commonly reported quantiles in meta-analysis. \emph{Statistical methods in medical research}, 29(9):2520–2537, 2020b.

Marshall Freimer, Georgia Kollia, Govind S Mudholkar, and C Thomas Lin. A study of the generalized tukey lambda family. \emph{Communications in Statistics-Theory and Methods}, 17(10):3547–3567, 1988.

Warren Gilchrist. \emph{Statistical modelling with quantile functions}. Chapman and Hall/CRC, 2000.

P. J. van Staden and R. A. R. King. The quantile-based skew logistic distribution.  \emph{Statistics & Probability Letters}, 96:109–116, 2015.
}
